/**
 * @file vipw.c
 * Edit the user- or group-database with proper locking
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <wait.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "vipw"		/**< Name shown by --help etc */
#define PRG_NAME2 "vigr"	/**< If called as this, default to -g */
#define PRG_NAME3 "visp"	/**< If called as this, default to -s */
#define PRG_NAME4 "visg"	/**< If called as this, default to -g -s */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Program synopsis */
static char doc[] =
	N_("Safely edit " SYSCONFDIR "/{passwd,shadow,group,gshadow}.\n"
	   "\n"
	   "The program can be called as any of vipw, vigr, "
	   "visp, or visg, and will behave accordingly.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "group", 'g', 0, 0,
	  N_("Edit the group database"), 0 },
	{ "no-shadow", 'n', 0, 0,
	  N_("Edit the normal database"), 0 },
	{ "passwd", 'p', 0, 0,
	  N_("Edit the user database"), 0 },
	{ "shadow", 's', 0, 0,
	  N_("Edit the shadow database"), 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	/**
	 * Password file to edit;
	 * F_PASSWD - For /etc/passwd
	 * F_GROUP - For /etc/group
	 */
	int passwd;
	/**
	 * Password file to edit;
	 * F_NORMAL - For the normal file
	 * F_SHADOW - For the shadow file
	 */
	int shadow;
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'g':
		if (args->passwd == F_PASSWD)
			argp_error(state,
				   _("`-p' cannot be combined with `-g'"));

		args->passwd = F_GROUP;
		break;

	case 'n':
		if (args->shadow == F_SHADOW)
			argp_error(state,
				   _("`-n' cannot be combined with `-s'"));

		args->shadow = F_NORMAL;
		break;

	case 'p':
		if (args->passwd == F_GROUP)
			argp_error(state,
				   _("`-p' cannot be combined with `-g'"));

		args->passwd = F_PASSWD;
		break;

	case 's':
		if (args->shadow == F_NORMAL)
			argp_error(state,
				   _("`-n' cannot be combined with `-s'"));

		args->shadow = F_SHADOW;
		break;

	case ARGP_KEY_INIT:
		args->passwd = F_DEFAULT;
		args->shadow = F_DEFAULT;
		break;

	case ARGP_KEY_ARG:
		(void)arg;
		argp_usage(state);

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	error_t status = 0;
	int waitstat = 0;

	pid_t pid;

	const char *fname = NULL;
	char *editor = NULL;
	char *bname = NULL;
	char *ename = NULL;
	char *cmd = NULL;
	char *c;

	int passwd = F_DEFAULT;
	int shadow = F_DEFAULT;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Setup mode of operation based on program-name */
	if (!strcmp(argv[0], PRG_NAME)) {
		passwd = F_PASSWD;
		shadow = F_NORMAL;
	} else if (!strcmp(argv[0], PRG_NAME2)) {
		passwd = F_GROUP;
		shadow = F_NORMAL;
	} else if (!strcmp(argv[0], PRG_NAME3)) {
		passwd = F_PASSWD;
		shadow = F_SHADOW;
	} else if (!strcmp(argv[0], PRG_NAME4)) {
		passwd = F_GROUP;
		shadow = F_SHADOW;
	}

	/* Find a sensible editor to use */
	if ((c = getenv("VISUAL")) || (c = getenv("EDITOR"))) {
		if (!(editor = strdup(c))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	} else {
		if (!access(EDITOR, X_OK)) {
			if (!(editor = strdup(EDITOR))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "strdup()", strerror(errno));
				status = errno;
				goto EXIT;
			}
		} else if (!access(DEFAULT_EDITOR, X_OK)) {
			if (!(editor = strdup(DEFAULT_EDITOR))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "strdup()", strerror(errno));
				status = errno;
				goto EXIT;
			}
		} else {
			fprintf(stderr,
				_("%s: could not find any editor\n"
				  "Please define the environment "
				  "variable $EDITOR.\n"),
				progname);
			status = EINVAL;
			goto EXIT;
		}
	}

	if (args.passwd != -1)
		passwd = args.passwd;

	if (passwd == F_DEFAULT)
		passwd = F_PASSWD;

	if (args.shadow != -1)
		shadow = args.shadow;

	if (shadow == F_DEFAULT)
		shadow = F_NORMAL;

	/* Create the proper file-name to copy to */
	if (passwd == F_PASSWD) {
		if (shadow)
			fname = SHADOW_FILE;
		else
			fname = PASSWD_FILE;
	} else {
		if (shadow)
			fname = GSHADOW_FILE;
		else
			fname = GROUP_FILE;
	}

	/* Make sure the caller has root privileges */
	if ((status = is_root())) {
		if (status == EPERM) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be root to edit `%s'.\n"),
				progname, fname);
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_root", strerror(errno));
		}

		goto EXIT;
	}

	/* Create filename /etc/<file>- */
	if (!(bname = create_filename(fname, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/<file>.edit */
	if (!(ename = create_filename(fname, EDIT_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create the command-line for file editing */
	if (!(cmd = strconcat(editor, " ", ename, NULL))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strconcat", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT;

	/* Backup file to /etc/<file>- */
	if ((status = backup_file(fname, bname)))
		goto EXIT2;

	/* Copy permissions from /etc/<file> to /etc/<file>- */
	if ((status = copy_file_modes(fname, bname)))
		goto EXIT2;


	/* Copy /etc/<source> to /etc/<file>.copy */
	if ((status = copy_file(fname, ename)))
		goto EXIT2;

	/* Copy permissions from /etc/<file> to /etc/<file>.edit */
	if ((status = copy_file_modes(fname, ename)))
		goto EXIT2;

	/* Fork off a process for editing */
	if ((pid = fork()) == -1) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fork()", strerror(errno));
		status = errno;
		goto EXIT2;
	} else if (!pid) {	/* child */
		return system(cmd);
	}

	/* Wait for the child to return control to the parent */
	do {
		if ((pid = waitpid(pid, &waitstat, WUNTRACED)) == -1) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "waitpid()", strerror(errno));
			exit(errno);
		}

		if (WIFSTOPPED(waitstat)) {
			kill(getpid(), SIGSTOP);
			kill(getpid(), SIGCONT);
		} else {
			break;
		}
	} while (1);

	/* Make sure nothing failed */
	if (pid == -1 || !WIFEXITED(waitstat) || WEXITSTATUS(waitstat)) {
		fprintf(stderr,
			_("%s: failed to edit `%s'; %s\n"),
			progname, fname, strerror(errno));
		status = errno;
		goto EXIT2;
	}

	/* Finally, move the new file in place */
	status = replace_file(ename, fname);

EXIT2:
	/* This file might not exist, but that's ok */
	status = unlink_file(ename, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT:
	/* Free all allocated memory */
	free(editor);
	free(bname);
	free(ename);
	free(cmd);

	return status;
}
