/**
 * @file rmgroup.c
 * Remove the specified groups
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <grp.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gshadow.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "rmgroup"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("GROUPS");

/** Program synopsis */
static char doc[] =
	N_("Remove the specified groups.\n"
	   "\n"
	   "GROUPS should be a comma-separated list of groups.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "verbose", 'v', 0, 0,
	  N_("Warn if the specified groups do not exist"), -2 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	const char *groups;	/**< Comma separated list of groups to remove */
	int verbose;		/**< Warn about non-existing groups */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'v':
		args->verbose = 1;
		break;

	case ARGP_KEY_INIT:
		args->groups = NULL;
		args->verbose = 0;
		break;

	case ARGP_KEY_ARG:
		if (args->groups)
			argp_usage(state);

		args->groups = arg;
		break;

	case ARGP_KEY_NO_ARGS:
		argp_usage(state);

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	FILE *grrfp = NULL;
	FILE *grwfp = NULL;
	struct group *gr;

	FILE *sgrfp = NULL;
	FILE *sgwfp = NULL;
	struct sgrp *sg;

	int empty = 1;

	error_t status = 0;

	char **grparray = NULL;

	char *grwname = NULL;
	char *grbname = NULL;
	char *sgwname = NULL;
	char *sgbname = NULL;

	gid_t i; /* We're scanning <= max(nrofgroups), hence gid_t */

	mode_t oldmask;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Make sure the caller is a user-admin */
	if ((status = is_useradmin())) {
		if (status == EPERM) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be a user-administrator to %s.\n"),
				progname, _("remove groups"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_useradmin", strerror(errno));
		}

		goto EXIT;
	}

	/* Get the list of groups to remove */
	if (!(grparray = strsplit(args.groups, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Verify that none of the listed groups is the primary
	 * group for any user
	 */
	for (i = 0; grparray[i]; i++) {
		char *tmp = NULL;

		/* Get the group entry for the group */
		if (!(gr = getgrnam(grparray[i])) && errno) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getgrnam()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* Ignore non-existing groups */
		if (!gr) {
			if (args.verbose)
				fprintf(stderr,
					_("%s: warning: %s `%s' "
					  "does not exist ... skipping\n"),
					progname, _("group"), grparray[i]);

			continue;
		}

		if ((tmp = get_group_primary_members(gr)) && strlen(tmp) > 0) {
			fprintf(stderr,
				_("%s: cannot remove `%s'; "
				  "it is the primary group\n"
				  "for the following users:\n"
				  "%s\n"),
				progname, gr->gr_name, tmp);
			status = EINVAL;
		} else if (errno) {
			status = errno;
		}

		free(tmp);

		if (status)
			goto EXIT;
	}

	/* Create filename /etc/group.write */
	if (!(grwname = create_filename(GROUP_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/group- */
	if (!(grbname = create_filename(GROUP_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/gshadow.write */
	if (!(sgwname = create_filename(GSHADOW_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/gshadow- */
	if (!(sgbname = create_filename(GSHADOW_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT;

	/* Change umask */
	oldmask = umask(0077);

	/* Open /etc/group */
	if (!(grrfp = open_file(GROUP_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/group to /etc/group- */
	if ((status = backup_file(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Copy permissions from /etc/group to /etc/group- */
	if ((status = copy_file_modes(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Open /etc/group.write */
	if (!(grwfp = open_file(grwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Open /etc/gshadow */
	if (!(sgrfp = open_file(GSHADOW_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/gshadow to /etc/gshadow- */
	if ((status = backup_file(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Copy permissions from /etc/gshadow to /etc/gshadow- */
	if ((status = copy_file_modes(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Open /etc/gshadow.write */
	if (!(sgwfp = open_file(sgwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Perform changes on /etc/group */
	while ((gr = fgetgrent(grrfp))) {
		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* If the entry is not part of the array of groups to remove,
		 * write it to the new file
		 */
		if (!is_in_array(grparray, gr->gr_name)) {
			/* Write the entry */
			if ((status = fputgrent(gr, grwfp))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "fputgrent", strerror(errno));
				goto EXIT2;
			}
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetgrent()", strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/group.write */
	if ((status = close_file(&grwfp)))
		goto EXIT2;

	/* Close /etc/group */
	if ((status = close_file(&grrfp)))
		goto EXIT2;

	/* Perform changes on /etc/gshadow */
	while ((sg = fgetsgent(sgrfp))) {
		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* If the entry is not part of the array of groups to remove,
		 * write it to the new file
		 */
		if (!is_in_array(grparray, sg->sg_name)) {
			/* Write the entry */
			if ((status = fputsgent(sg, sgwfp))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "fputsgent", strerror(errno));
				goto EXIT2;
			}
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetsgent", strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/gshadow.write */
	if ((status = close_file(&sgwfp)))
		goto EXIT2;

	/* Close /etc/gshadow */
	if ((status = close_file(&sgrfp)))
		goto EXIT2;

	/* Everything is in order, move the new files in place */
	if ((status = replace_file(grwname, GROUP_FILE)))
		goto EXIT2;

	if ((status = replace_file(sgwname, GSHADOW_FILE)))
		goto EXIT2;

	/* Set file permissions properly */
	if ((status = copy_file_modes(grbname, GROUP_FILE)))
		goto EXIT2;

	if ((status = copy_file_modes(sgbname, GSHADOW_FILE)))
		goto EXIT2;

EXIT2:
	/* Restore umask */
	umask(oldmask);

	/* These files might not exist, but that's ok */
	status = unlink_file(grwname, status);
	status = unlink_file(sgwname, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT:
	/* Free all allocated memory */
	strfreev(grparray);
	free(grwname);
	free(grbname);
	free(sgwname);
	free(sgbname);

	return status;
}
