/**
 * @file mkuser.c
 * Create a new user
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <grp.h>
#include <pwd.h>
#include <errno.h>
#include <stdio.h>
#include <shadow.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gshadow.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "mkuser"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("ATTRIBUTE=VALUE... USER");

/** Program synopsis */
static char doc[] =
	N_("Create a new user, with the specified attributes.\n"
	   "\n"
	   "Valid attributes:\n"
	   "\n"
	   "account_locked | locked        Lock/unlock account\n"
	   "admgroups                      List of the groups that the "
	   "user administers\n"
	   "fname | fullname               The user's full name "
	   "for the user\n"
	   "gecos                          The user's gecos-field\n"
	   "groups                         The user's group membership "
	   "list\n"
	   "home                           The user's home-directory\n"
	   "hphone                         The user's home phone\n"
	   "id | uid                       The user's id\n"
	   "other                          Other information\n"
	   "pgrp                           The user's primary group\n"
	   "room                           The user's room number\n"
	   "shell                          The user's login shell\n"
	   "wphone                         The user's work phone\n"
	   "\n"
	   "If you do not specify a user id, the user will be assigned "
	   "the lowest available id.\n"
	   "\n"
	   "Note, that the attribute `gecos' cannot be combined with any of "
	   "the attributes `fname', `room', `wphone', `hphone', or `other'.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "firstuid", 'F', N_("ID"), 0,
	  N_("The first id in the range from which the id is taken"), 0 },
	{ "lastuid", 'L', N_("ID"), 0,
	  N_("The last id in the range from which the id is taken"), 0 },
	{ "system", 's', 0, 0,
	  N_("The user should, if possible, "
	     "get an id in the system-range"), 0 },
	{ "force", 'f', 0, 0,
	  N_("Allow non-existing groups to be specified "
	     "as groups and/or admgroups"), -2 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	char *const *args;	/**< Arguments */
	int nargs;		/**< Number of arguments */
	int firstid;		/**< First UID in the range to choose from */
	int lastid;		/**< Last UID in the range to choose from */
	int usesysid;		/**< Use a UID from the system range */
	int force;		/**< Allow non-existing groups/admgroups */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'F':
		if ((status = is_uid_t(arg))) {
			fprintf(stderr,
				_("%s: invalid value supplied to `%s'\n"),
				progname, "-F | --firstuid");
			goto EXIT;
		}

		if ((status = string_to_uid_t(arg, (void **)&args->firstid))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_uid_t",
				strerror(errno));
			goto EXIT;
		}

		break;

	case 'L':
		if ((status = is_uid_t(arg))) {
			fprintf(stderr,
				_("%s: invalid value supplied to `%s'\n"),
				progname, "-L | --lastuid");
			goto EXIT;
		}

		if ((status = string_to_uid_t(arg, (void **)&args->lastid))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_uid_t",
				strerror(errno));
			goto EXIT;
		}

		break;

	case 's':
		args->usesysid = 1;
		break;

	case 'f':
		args->force = 1;
		break;

	case ARGP_KEY_INIT:
		args->args = NULL;
		args->nargs = 0;
		args->firstid = 65535;
		args->lastid = 65535;
		args->usesysid = 0;
		args->force = 0;
		break;

	case ARGP_KEY_ARG:
		status = ARGP_ERR_UNKNOWN;
		break;

	case ARGP_KEY_ARGS:
		args->args = state->argv + state->next;
		args->nargs = state->argc - state->next;
		break;

	case ARGP_KEY_NO_ARGS:
		argp_usage(state);
		break;

	default:
		(void)arg;
		status = ARGP_ERR_UNKNOWN;
		break;
	}

EXIT:

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	FILE *pwrfp = NULL;
	FILE *pwwfp = NULL;
	struct passwd *pw;
	struct passwd pw2;

	FILE *sprfp = NULL;
	FILE *spwfp = NULL;
	struct spwd *sp;
	struct spwd sp2;

	FILE *grrfp = NULL;
	FILE *grwfp = NULL;
	struct group *gr;

	FILE *sgrfp = NULL;
	FILE *sgwfp = NULL;
	struct sgrp *sg;

	int empty = 1;
	int changed = 0;

	int j;

	error_t status = 0;

	int dirflags = DF_EXISTSOK;
	int usergroups = DEFAULT_USERGROUPS;
	int letterhomes = DEFAULT_LETTERHOMES;
	int grouphomes = DEFAULT_GROUPHOMES;

	const char *default_sysgroup = DEFAULT_SYSTEM_GROUP;
	const char *default_group = DEFAULT_GROUP;
	const char *homepath = DEFAULT_HOME;

	mode_t homeumask = DEFAULT_UMASK;

	char *pwwname = NULL;
	char *pwbname = NULL;
	char *spwname = NULL;
	char *spbname = NULL;
	char *grwname = NULL;
	char *grbname = NULL;
	char *sgwname = NULL;
	char *sgbname = NULL;

	uid_t firstsysuid = DEFAULT_FIRSTSUID;
	uid_t lastsysuid = DEFAULT_LASTSUID;
	uid_t firstnormaluid = DEFAULT_FIRSTUID;
	uid_t lastnormaluid = DEFAULT_LASTUID;

	int newlocked = 1;		/* Default is to lock the group */
	uid_t newid = 65535;		/* Default is the first free id */
	char *newpgrp = NULL;		/* Default is policy based */
	gid_t newpgrpgid = 65535;
	char *newname = NULL;		/* Must always be specified */
	char *newshell = NULL;		/* Default is /bin/bash for users,
					 * /bin/false for system-users
					 */
	char *newhomedir = NULL;	/* Must be specified for systme-users,
					 * default is policy-based for users
					 */
	char *newgroups = NULL;		/* Default is no groups */
	char *newadmgroups = NULL;	/* Default is no admgroups */
	char **newgroupsv = NULL;
	char **newadmgroupsv = NULL;

	/* GECOS information */
	char *newgecos = NULL;
	char *newfname = NULL;
	char *newroom = NULL;
	char *newwphone = NULL;
	char *newhphone = NULL;
	char *newother = NULL;

	struct attr attributes[] = {
		{
			.attribute	= AS_A_LOCKED,
			.value		= (void **)&newlocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= AS_ADMGROUPS,
			.value		= (void **)&newadmgroups,
			.validator	= is_valid_namelist,
			.converter	= string_to_string
		}, {
			.attribute	= GS_FNAME,
			.value		= (void **)&newfname,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= GS_FULLNAME,
			.value		= (void **)&newfname,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= AS_GECOS,
			.value		= (void **)&newgecos,
			.validator	= is_valid_gecos,
			.converter	= string_to_string
		}, {
			.attribute	= AS_GROUPS,
			.value		= (void **)&newgroups,
			.validator	= is_valid_namelist,
			.converter	= string_to_string
		}, {
			.attribute	= AS_HOME,
			.value		= (void **)&newhomedir,
			.validator	= is_valid_path,
			.converter	= string_to_string
		}, {
			.attribute	= GS_HPHONE,
			.value		= (void **)&newhphone,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= AS_ID,
			.value		= (void **)&newid,
			.validator	= is_uid_t,
			.converter	= string_to_uid_t
		}, {
			.attribute	= AS_LOCKED,
			.value		= (void **)&newlocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= GS_OTHER,
			.value		= (void **)&newother,
			.validator	= is_valid_gecos_other,
			.converter	= string_to_string
		}, {
			.attribute	= AS_PGRP,
			.value		= (void **)&newpgrp,
			.validator	= is_valid_name,
			.converter	= string_to_string
		}, {
			.attribute	= GS_ROOM,
			.value		= (void **)&newroom,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= AS_SHELL,
			.value		= (void **)&newshell,
			.validator	= is_valid_shell,
			.converter	= string_to_string
		}, {
			.attribute	= AS_UID,
			.value		= (void **)&newid,
			.validator	= is_uid_t,
			.converter	= string_to_uid_t
		}, {
			.attribute	= AS_USERNAME,
			.value		= (void **)&newname,
			.validator	= is_valid_name,
			.converter	= string_to_string
		}, {
			.attribute	= GS_WPHONE,
			.value		= (void **)&newwphone,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= NULL,
			.value		= NULL,
			.validator	= NULL,
			.converter	= NULL
		}
	};

	mode_t oldmask;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Make sure the caller is a user-admin */
	if ((status = is_useradmin())) {
		if (status == EPERM) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "you must be a user-administrator to %s\n"),
				progname, _("create users"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_useradmin", strerror(errno));
		}

		goto EXIT;
	}

	/* Make sure we get the correct number of arguments */
	if (args.nargs < 1) {
		fprintf(stderr,
			_("%s: too few arguments\n"
			  "Try `%s --help' for more information.\n"),
			progname, progname);
		status = EINVAL;
		goto EXIT;
	}

	/* Parse /etc/mkuser.default */
	/* TODO */

	/* Parse the attribute=value pairs */
	for (j = 0; j < (args.nargs - 1); j++) {
		if ((status = parse_key_pairs(args.args[j], attributes))) {
			if (status == ENOENT) {
				fprintf(stderr,
					_("%s: invalid attributes\n"
					  "Valid attributes are:\n"
					  "- `%s'\n"
					  "- `%s' | `%s'\n"
					  "- `%s' | `%s'\n"
					  "- `%s'\n"
					  "Try `%s --help' for more "
					  "information.\n"),
					progname,
					AS_ADMS,
					AS_G_LOCKED, AS_LOCKED,
					AS_ID, AS_GID,
					AS_USERS,
					progname);
				status = EINVAL;
			}

			goto EXIT;
		}
	}

	/* Merge the homedir flags into the dirflags variable */
	if (letterhomes)
		dirflags |= DF_LETTERHOMES;

	if (grouphomes)
		dirflags |= DF_GROUPHOMES;

	if (usergroups)
		dirflags |= DF_USERGROUPS;

	/* `gecos' cannot be combined with any of `fname', `room',
	 * `wphone', `hphone', `other'.
	 */
	if (newgecos &&
	    (newfname || newhphone || newwphone || newother || newroom)) {
		fprintf(stderr,
			_("%s: `%s' cannot be combined with any of\n"
			  "`%s', `%s', `%s', `%s', or `%s'\n"),
			progname,
			"gecos", "fname", "room", "wphone", "hphone", "other");
		status = EINVAL;
		goto EXIT;
	}

	/* If any commands that require `--force' are specified
	 * without the use of `--force', abort.
	 */
	if (!args.force && (newid != 65535 || newname)) {
		fprintf(stderr,
			_("%s: these changes require `--force' to be "
			  "specified\n"),
			progname);
		status = EINVAL;
		goto EXIT;
	}

	/* The last argument is the name of the new user */

	/* Verify that the username is valid */
	if ((status = is_valid_name(args.args[args.nargs - 1]))) {
		fprintf(stderr,
			_("%s: the specified %s is invalid\n"),
			progname, _("username"));
		goto EXIT;
	}

	/* The username is valid; point to it */
	newname = args.args[args.nargs - 1];

	/* Verify that the requested username is unused */
	if ((status = is_free_username(newname))) {
		if (status == EEXIST) {
			fprintf(stderr,
				_("%s: the specified %s already exists\n"),
				progname, _("username"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_free_username", strerror(errno));
		}

		goto EXIT;
	}

	/* Verify that the requested uid is free */
	if (newid != 65535 && (status = is_free_uid(newid))) {
		if (status == EEXIST) {
			fprintf(stderr,
				_("%s: the specified %s already exists\n"),
				progname, _("group id"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_free_uid", strerror(errno));
		}

		goto EXIT;
	}

	/* Verify that !((lastid || firstid) && newid) */
	if ((args.lastid != 65535 || args.firstid != 65535) && newid != 65535) {
		fprintf(stderr,
			_("%s: `%s' cannot be combined with `%s'\n"
			  "or `%s'\n"),
			progname,
			"id | uid", "-f | --firstuid", "-l | --lastuid");
		status = EINVAL;
		goto EXIT;
	}

	/* Verify that firstid <= lastid */
	if (args.firstid != 65535 && args.lastid != 65535
	    && args.firstid > args.lastid) {
		fprintf(stderr,
			_("%s: `%s' cannot be larger than `%s'\n"),
			progname, "firstuid", "lastuid");
		status = EINVAL;
		goto EXIT;
	}

	/* Set the ranges */
	if (args.firstid == 65535)
		args.firstid = (args.usesysid) ? firstsysuid : firstnormaluid;

	if (args.lastid == 65535)
		args.lastid = (args.usesysid) ? lastsysuid : lastnormaluid;

	if (newid != 65535)
		args.firstid = args.lastid = newid;

	/* Make sure that newid isn't unreasonably large */
	if (newid != 65535 && args.lastid > LASTUID) {
		fprintf(stderr,
			_("%s: the %s cannot be larger than %d\n"),
			progname, _("user id"), LASTUID);
		status = EINVAL;
		goto EXIT;
	}

	/* Try to obtain a uid in the specified range */
	if ((newid = get_free_uid(args.firstid, args.lastid)) == 65535) {
		if (errno) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "get_free_uid", strerror(errno));
			status = errno;
		} else {
			fprintf(stderr,
				_("%s: could not find a free %s\n"),
				progname, _("user id"));
			errno = EINVAL;
		}

		goto EXIT;
	}

	/* If newgroups or newadmgroups are empty, free and NULL them */
	if (newgroups && !strlen(newgroups)) {
		free(newgroups);
		newgroups = NULL;
	}

	if (newadmgroups && !strlen(newadmgroups)) {
		free(newadmgroups);
		newadmgroups = NULL;
	}

	/* Either get a new gid or the gid of an existing group */
	if (newpgrp) {
		/* Verify that the specified primary group exists */
		if (!(gr = getgrnam(newpgrp)) && errno) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getgrnam()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* Abort on non-existing groups */
		if (!gr) {
			fprintf(stderr,
				_("%s: the specified %s does not exist\n"),
				progname, _("primary group"));
			status = ENOENT;
			goto EXIT;
		} else {
			newpgrpgid = gr->gr_gid;
		}
//	} else if (dirflags & DF_USERGROUPS) {
	} else {
		/* Use the default group */
		if (!(gr = getgrnam(args.usesysid ? default_sysgroup :
						    default_group)) && errno) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getgrnam()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* Abort on non-existing groups */
		if (!gr) {
			fprintf(stderr,
				_("%s: the default %s does not exist\n"),
				progname, _("primary group"));
			status = ENOENT;
			goto EXIT;
		} else {
			newpgrpgid = gr->gr_gid;
		}
	}

	/* Make sure that the list of groups is unique */
	if (!is_uniq_list(newgroups)) {
		fprintf(stderr,
			_("%s: the list of %s contains duplicate entries\n"),
			progname, "groups");
		status = EINVAL;
		goto EXIT;
	}

	/* Make sure that the list of admgroups is unique */
	if (!is_uniq_list(newadmgroups)) {
		fprintf(stderr,
			_("%s: the list of %s contains duplicate entries\n"),
			progname, "admgroups");
		status = EINVAL;
		goto EXIT;
	}

	/* Split the list of groups */
	if (newgroups && !(newgroupsv = strsplit(newgroups, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Split the list of admgroups */
	if (newadmgroups && !(newadmgroupsv = strsplit(newadmgroups, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Verify that all groups exist */
	if (!args.force && newgroupsv) {
		uid_t i;

		for (i = 0; newgroupsv[i]; i++) {
			if (!(status = is_free_groupname(newgroupsv[i]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("groups"));
				status = ENOENT;
				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_groupname",
					strerror(errno));
				status = errno;
				goto EXIT;
			}
		}
	}

	/* Verify that all admgroups exist */
	if (!args.force && newadmgroupsv) {
		uid_t i;

		for (i = 0; newadmgroupsv[i]; i++) {
			if (!(status = is_free_groupname(newadmgroupsv[i]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("admgroups"));
				status = ENOENT;
				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_groupname",
					strerror(errno));
				status = errno;
				goto EXIT;
			}
		}
	}

	/* If no shell has been specified, use defaults */
	if (!newshell) {
		if (!(newshell = strdup(args.usesysid ? DEFAULT_SYS_SHELL :
							DEFAULT_USER_SHELL))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* If the GECOS information is specified as separate arguments,
	 * join it before using it
	 */
	if (!newgecos) {
		if (!(newgecos = join_gecos(newfname, newroom, newwphone,
					    newhphone, newother))) {
			status = errno;
			goto EXIT;
		}
	}

	/* Create the new passwd entry */
	pw2.pw_name = newname;
	pw2.pw_passwd = (char *)"x";
	pw2.pw_uid = newid;
	pw2.pw_gid = newpgrpgid;
	pw2.pw_gecos = newgecos;
	pw2.pw_shell = newshell;

	/* Create the home-directory and copy /etc/skel there */
	if (newhomedir) {
		/* Nothing needs to be done apart from making sure the
		 * specified directory exists
		 */
		/* XXX */;
	} else {
		if (!(newhomedir = create_home_directory(&pw2,
							 dirflags, homeumask,
							 homepath))) {
			fprintf(stderr,
				_("%s: failed to create a home-directory; "
				  "%s\n"),
				progname, strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* Fill out the rest of the passwd entry */
	pw2.pw_dir = newhomedir;

	/* Create the new shadow group entry */
	sp2.sp_namp = newname;
	sp2.sp_pwdp = (char *)((newlocked) ? "!" : "");
	if ((sp2.sp_lstchg = get_current_date()) == -1) {
		status = errno;
		goto EXIT;
	}
	sp2.sp_min = 0;				/* Allow change immediately */
	sp2.sp_max = (long)-1;			/* No outer limit */
	sp2.sp_warn = 7;			/* Warn a week before expiry */
	sp2.sp_inact = (long)-1;		/* Never inactivate */
	sp2.sp_expire = (long)-1;		/* Never expire */
	sp2.sp_flag = ~((unsigned long)0);	/* reserved for future use */

	/* Create filename /etc/passwd.write */
	if (!(pwwname = create_filename(PASSWD_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/passwd- */
	if (!(pwbname = create_filename(PASSWD_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/group.write */
	if (!(grwname = create_filename(GROUP_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/group- */
	if (!(grbname = create_filename(GROUP_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/shadow.write */
	if (!(spwname = create_filename(SHADOW_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/shadow- */
	if (!(spbname = create_filename(SHADOW_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/gshadow.write */
	if (!(sgwname = create_filename(GSHADOW_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/gshadow- */
	if (!(sgbname = create_filename(GSHADOW_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT;

	/* Change umask */
	oldmask = umask(0077);

	/* Open /etc/passwd */
	if (!(pwrfp = open_file(PASSWD_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/passwd to /etc/passwd- */
	if ((status = backup_file(PASSWD_FILE, pwbname)))
		goto EXIT2;

	/* Copy permissions from /etc/passwd to /etc/passwd- */
	if ((status = copy_file_modes(PASSWD_FILE, pwbname)))
		goto EXIT2;

	/* Open /etc/passwd.write */
	if (!(pwwfp = open_file(pwwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Open /etc/group */
	if (!(grrfp = open_file(GROUP_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/group to /etc/group- */
	if ((status = backup_file(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Copy permissions from /etc/group to /etc/group- */
	if ((status = copy_file_modes(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Open /etc/group.write */
	if (!(grwfp = open_file(grwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Open /etc/shadow */
	if (!(sprfp = open_file(SHADOW_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/shadow to /etc/shadow- */
	if ((status = backup_file(SHADOW_FILE, spbname)))
		goto EXIT2;

	/* Copy permissions from /etc/shadow to /etc/shadow- */
	if ((status = copy_file_modes(SHADOW_FILE, spbname)))
		goto EXIT2;

	/* Open /etc/shadow.write */
	if (!(spwfp = open_file(spwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Open /etc/gshadow */
	if (!(sgrfp = open_file(GSHADOW_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/gshadow to /etc/gshadow- */
	if ((status = backup_file(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Copy permissions from /etc/gshadow to /etc/gshadow- */
	if ((status = copy_file_modes(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Open /etc/gshadow.write */
	if (!(sgwfp = open_file(sgwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Perform changes on /etc/passwd */
	while ((pw = fgetpwent(pwrfp))) {
		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Write the entry */
		if ((status = fputpwent(pw, pwwfp))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputpwent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Write the new entry */
	if ((!errno || errno == ENOENT) && (status = fputpwent(&pw2, pwwfp))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fputpwent", strerror(errno));
		goto EXIT2;
	} else {
		/* Set as an indication that the file has at least 1 entry */
		empty = 0;
		changed = 1;
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetpwent()",
			strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/passwd.write */
	if ((status = close_file(&pwwfp)))
		goto EXIT2;

	/* Close /etc/passwd */
	if ((status = close_file(&pwrfp)))
		goto EXIT2;

	/* Perform changes on /etc/group */
	while ((gr = fgetgrent(grrfp))) {
		static struct group gr2;
		char *memlist = NULL;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Copy the old entry */
		gr2.gr_name = gr->gr_name;
		gr2.gr_gid = gr->gr_gid;
		gr2.gr_passwd = gr->gr_passwd;

		/* Join the member array into a list */
		if (!(memlist = strjoinv(",", gr->gr_mem))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strjoinv", strerror(errno));
			status = errno;
			goto EXIT2;
		}

		/* If we have a list of new groups and the current group
		 * is a part of that list, add the user to it.
		 */
		if (newgroups) {
			if (is_in_list(newgroups, gr->gr_name)) {
				char *tmp = uniq_add_to_list(memlist, newname);

				free(memlist);

				if (!tmp) {
					status = errno;
					goto EXIT2;
				}
				memlist = tmp;
			}
		}

		/* Split the memlist back into an array */
		gr2.gr_mem = strsplit(memlist, ",", 0);

		free(memlist);

		if (!gr2.gr_mem) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT2;
		}

		/* Write the entry */
		status = fputgrent(&gr2, grwfp);

		strfreev(gr2.gr_mem);

		if (status) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputgrent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetgrent()",
			strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/group.write */
	if ((status = close_file(&grwfp)))
		goto EXIT2;

	/* Close /etc/group */
	if ((status = close_file(&grrfp)))
		goto EXIT2;

	/* Perform changes on /etc/shadow */
	while ((sp = fgetspent(sprfp))) {
		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Write the entry */
		if ((status = fputspent(sp, spwfp))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputspent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Write the new entry */
	if ((!errno || errno == ENOENT) && (status = fputspent(&sp2, spwfp))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fputspent", strerror(errno));
		goto EXIT2;
	} else {
		/* Set as an indication that the file has at least 1 entry */
		empty = 0;
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetspent()",
			strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/shadow.write */
	if ((status = close_file(&spwfp)))
		goto EXIT2;

	/* Close /etc/shadow */
	if ((status = close_file(&sprfp)))
		goto EXIT2;

	/* Perform changes on /etc/gshadow */
	while ((sg = fgetsgent(sgrfp))) {
		static struct sgrp sg2;
		char *admlist = NULL;
		char *memlist = NULL;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Copy the old entry */
		sg2.sg_name = sg->sg_name;
		sg2.sg_passwd = sg->sg_passwd;

		/* Join the admin array into a list */
		if (!(admlist = strjoinv(",", sg->sg_adm))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strjoinv", strerror(errno));
			status = errno;
			goto EXIT2;
		}

		/* If we have a list of new admgroups and the current group
		 * is a part of that list, add the user to it.
		 */
		if (newadmgroups) {
			if (is_in_list(newadmgroups, sg->sg_name)) {
				char *tmp = uniq_add_to_list(admlist, newname);

				free(admlist);

				if (!tmp) {
					status = errno;
					goto EXIT2;
				}

				admlist = tmp;
			}
		}

		/* Split the admlist back into an array */
		sg2.sg_adm = strsplit(admlist, ",", 0);

		free(admlist);

		if (!sg2.sg_adm) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT2;
		}

		/* Join the member array into a list */
		if (!(memlist = strjoinv(",", sg->sg_mem))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strjoinv", strerror(errno));
			strfreev(sg2.sg_adm);
			status = errno;
			goto EXIT2;
		}

		/* If we have a list of new groups and the current group
		 * is a part of that list, add the user to it.
		 */
		if (newgroups) {
			if (is_in_list(newgroups, sg->sg_name)) {
				char *tmp = uniq_add_to_list(memlist, newname);

				free(memlist);

				if (!tmp) {
					strfreev(sg2.sg_adm);
					status = errno;
					goto EXIT2;
				}
				memlist = tmp;
			}
		}

		/* Split the memlist back into an array */
		sg2.sg_mem = strsplit(memlist, ",", 0);

		free(memlist);

		if (!sg2.sg_mem) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			strfreev(sg2.sg_adm);
			status = errno;
			goto EXIT2;
		}

		/* Write the entry */
		status = fputsgent(&sg2, sgwfp);

		strfreev(sg2.sg_adm);
		strfreev(sg2.sg_mem);

		if (status) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputsgent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetsgent", strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/gshadow.write */
	if ((status = close_file(&sgwfp)))
		goto EXIT2;

	/* Close /etc/gshadow */
	if ((status = close_file(&sgrfp)))
		goto EXIT2;

	/* If nothing has changed, don't replace old files */
	if (!changed)
		goto EXIT2;

	/* Everything is in order, move the new files in place */
	if ((status = replace_file(pwwname, PASSWD_FILE)))
		goto EXIT2;

	if ((status = replace_file(grwname, GROUP_FILE)))
		goto EXIT2;

	if ((status = replace_file(spwname, SHADOW_FILE)))
		goto EXIT2;

	if ((status = replace_file(sgwname, GSHADOW_FILE)))
		goto EXIT2;

	/* Set file permissions properly */
	if ((status = copy_file_modes(pwbname, PASSWD_FILE)))
		goto EXIT2;

	if ((status = copy_file_modes(pwbname, SHADOW_FILE)))
		goto EXIT2;

	if ((status = copy_file_modes(grbname, GROUP_FILE)))
		goto EXIT2;

	if ((status = copy_file_modes(sgbname, GSHADOW_FILE)))
		goto EXIT2;

EXIT2:
	/* Restore umask */
	umask(oldmask);

	/* These files might not exist, but that's ok */
	status = unlink_file(pwwname, status);
	status = unlink_file(grwname, status);
	status = unlink_file(spwname, status);
	status = unlink_file(sgwname, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT:
	strfreev(newadmgroupsv);
	strfreev(newgroupsv);
	free(newadmgroups);
	free(newhomedir);
	free(newgroups);
	free(newhphone);
	free(newwphone);
	free(newother);
	free(newshell);
	free(newgecos);
	free(newfname);
	free(newroom);
	free(newpgrp);
	free(pwwname);
	free(pwbname);
	free(spwname);
	free(spbname);
	free(grwname);
	free(grbname);
	free(sgwname);
	free(sgbname);

	return status;
}
