/**
 * @file mkgroup.c
 * Create a new group
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <grp.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gshadow.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "mkgroup"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("ATTRIBUTE=VALUE... GROUP");

/** Program synopsis */
static char doc[] =
	N_("Create a new group, with the specified attributes.\n"
	   "\n"
	   "Valid attributes:\n"
	   "\n"
	   "adms                           List of group administrators\n"
	   "group_locked | locked          Lock/unlock group\n"
	   "id | gid                       The group's id\n"
	   "users                          List of group members\n"
	   "\n"
	   "If you do not specify a group id, the group will be assigned "
	   "the lowest available id.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "firstgid", 'F', N_("ID"), 0,
	  N_("The first id in the range from which the id is taken"), 0 },
	{ "lastgid", 'L', N_("ID"), 0,
	  N_("The last id in the range from which the id is taken"), 0 },
	{ "system", 's', 0, 0,
	  N_("The group should, if possible, "
	     "get an id in the system-range"), 0 },
	{ "force", 'f', 0, 0,
	  N_("Allow non-existing users to be specified "
	     "as group members and/or administrators"), -2 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	char *const *args;	/**< Arguments */
	int nargs;		/**< Number of arguments */
	int firstid;		/**< First GID in the range to choose from */
	int lastid;		/**< Last GID in the range to choose from */
	int usesysid;		/**< Use a GID from the system range */
	int force;		/**< Allow non-existing members/admins */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'F':
		if ((status = is_gid_t(arg))) {
			fprintf(stderr,
				_("%s: invalid value supplied to `%s'\n"),
				progname, "-F | --firstgid");
			goto EXIT;
		}

		if ((status = string_to_gid_t(arg, (void **)&args->firstid))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_gid_t",
				strerror(errno));
			goto EXIT;
		}

		break;

	case 'L':
		if ((status = is_gid_t(arg))) {
			fprintf(stderr,
				_("%s: invalid value supplied to `%s'\n"),
				progname, "-L | --lastgid");
			goto EXIT;
		}

		if ((status = string_to_gid_t(arg, (void **)&args->lastid))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_gid_t",
				strerror(errno));
			goto EXIT;
		}

		break;

	case 's':
		args->usesysid = 1;
		break;

	case 'f':
		args->force = 1;
		break;

	case ARGP_KEY_INIT:
		args->args = NULL;
		args->nargs = 0;
		args->firstid = 65535;
		args->lastid = 65535;
		args->usesysid = 0;
		args->force = 0;
		break;

	case ARGP_KEY_ARG:
		status = ARGP_ERR_UNKNOWN;
		break;

	case ARGP_KEY_ARGS:
		args->args = state->argv + state->next;
		args->nargs = state->argc - state->next;
		break;

	case ARGP_KEY_NO_ARGS:
		argp_usage(state);
		break;

	default:
		(void)arg;
		status = ARGP_ERR_UNKNOWN;
		break;
	}

EXIT:

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	FILE *grrfp = NULL;
	FILE *grwfp = NULL;
	struct group *gr;
	struct group gr2;

	FILE *sgrfp = NULL;
	FILE *sgwfp = NULL;
	struct sgrp *sg;
	struct sgrp sg2;

	int empty = 1;
	int changed = 0;

	int j;

	error_t status = 0;

	char *grwname = NULL;
	char *grbname = NULL;
	char *sgwname = NULL;
	char *sgbname = NULL;

	gid_t firstsysgid = 100;	/* Default is 100 */
	gid_t lastsysgid = 999;		/* Default is 999 */
	gid_t firstnormalgid = 1000;	/* Default is 1000 */
	gid_t lastnormalgid = 29999;	/* Default is 29999 */

	int newlocked = 1;		/* Default is to lock the group */
	gid_t newid = 65535;		/* Default is the first free gid */
	char *newname = NULL;		/* Must always be specified */
	char *newadms = NULL;		/* Default is no group administrators */
	char *newusers = NULL;		/* Default is no group members */
	char **newadmsv = NULL;
	char **newusersv = NULL;

	struct attr attributes[] = {
		{
			.attribute	= AS_ADMS,
			.value		= (void **)&newadms,
			.validator	= is_valid_namelist_empty,
			.converter	= string_to_string
		}, {
			.attribute	= AS_G_LOCKED,
			.value		= (void **)&newlocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= AS_LOCKED,
			.value		= (void **)&newlocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= AS_GID,
			.value		= (void **)&newid,
			.validator	= is_gid_t,
			.converter	= string_to_gid_t
		}, {
			.attribute	= AS_ID,
			.value		= (void **)&newid,
			.validator	= is_gid_t,
			.converter	= string_to_gid_t
		}, {
			.attribute	= AS_USERS,
			.value		= (void **)&newusers,
			.validator	= is_valid_namelist_empty,
			.converter	= string_to_string
		}, {
			.attribute	= NULL,
			.value		= NULL,
			.validator	= NULL,
			.converter	= NULL
		}
	};

	mode_t oldmask;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Make sure the caller is a user-admin */
	if ((status = is_useradmin())) {
		if (status == EPERM) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be a user-administrator to %s\n"),
				progname, _("create groups"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_useradmin", strerror(errno));
		}

		goto EXIT;
	}

	/* Make sure we get the correct number of arguments */
	if (args.nargs < 1) {
		fprintf(stderr,
			_("%s: too few arguments\n"
			  "Try `%s --help' for more information.\n"),
			progname, progname);
		status = EINVAL;
		goto EXIT;
	}

	/* Parse /etc/mkgroup.default */
	/* TODO */

	/* Parse the attribute=value pairs */
	for (j = 0; j < (args.nargs - 1); j++) {
		if ((status = parse_key_pairs(args.args[j], attributes))) {
			if (status == ENOENT) {
				fprintf(stderr,
					_("%s: invalid attributes\n"
					  "Valid attributes are:\n"
					  "- `%s'\n"
					  "- `%s' | `%s'\n"
					  "- `%s' | `%s'\n"
					  "- `%s'\n"
					  "Try `%s --help' for more "
					  "information.\n"),
					progname,
					AS_ADMS,
					AS_G_LOCKED, AS_LOCKED,
					AS_ID, AS_GID,
					AS_USERS,
					progname);
				status = EINVAL;
			}

			goto EXIT;
		}
	}

	/* The last argument is the name of the new group */

	/* Verify that the groupname is valid */
	if ((status = is_valid_name(args.args[args.nargs - 1]))) {
		fprintf(stderr,
			_("%s: the specified %s is invalid\n"),
			progname, _("groupname"));
		goto EXIT;
	}

	/* The groupname is valid; point to it */
	newname = args.args[args.nargs - 1];

	/* Verify that the requested groupname is unused */
	if ((status = is_free_groupname(newname))) {
		if (status == EEXIST) {
			fprintf(stderr,
				_("%s: the specified %s already exists\n"),
				progname, _("groupname"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_free_groupname", strerror(errno));
		}

		goto EXIT;
	}

	/* Verify that the requested gid is free */
	if (newid != 65535 && (status = is_free_gid(newid))) {
		if (status == EEXIST) {
			fprintf(stderr,
				_("%s: the specified %s already exists\n"),
				progname, _("group id"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_free_gid", strerror(errno));
		}

		goto EXIT;
	}

	/* Verify that !((lastid || firstid) && newid) */
	if ((args.lastid != 65535 || args.firstid != 65535) && newid != 65535) {
		fprintf(stderr,
			_("%s: `%s' cannot be combined with `%s' or `%s'\n"),
			progname,
			"id | gid", "-f | --firstgid", "-l | --lastgid");
		status = EINVAL;
		goto EXIT;
	}

	/* Verify that firstgid <= lastgid */
	if (args.firstid != 65535 && args.lastid != 65535
	    && args.firstid > args.lastid) {
		fprintf(stderr,
			_("%s: `%s' cannot be larger than `%s'\n"),
			progname, "firstgid", "lastgid");
		status = EINVAL;
		goto EXIT;
	}

	/* Set the ranges */
	if (args.firstid == 65535)
		args.firstid = (args.usesysid) ? firstsysgid : firstnormalgid;

	if (args.lastid == 65535)
		args.lastid = (args.usesysid) ? lastsysgid : lastnormalgid;

	if (newid != 65535)
		args.firstid = args.lastid = newid;

	/* Make sure that newid isn't unreasonably large */
	if (newid != 65535 && args.lastid > LASTGID) {
		fprintf(stderr,
			_("%s: the %s cannot be larger than %d\n"),
			progname, _("group id"), LASTGID);
		status = EINVAL;
		goto EXIT;
	}

	/* Try to obtain a gid in the specified range */
	if ((newid = get_free_gid(args.firstid, args.lastid)) == 65535) {
		if (errno) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "get_free_gid", strerror(errno));
			status = errno;
		} else {
			fprintf(stderr,
				_("%s: could not find a free %s\n"),
				progname, _("group id"));
			errno = EINVAL;
		}

		goto EXIT;
	}

	/* If newusers or newadms are empty, free and NULL them */
	if (newusers && !strlen(newusers)) {
		free(newusers);
		newusers = NULL;
	}

	if (newadms && !strlen(newadms)) {
		free(newadms);
		newadms = NULL;
	}

	/* Make sure that the list of group members is unique */
	if (!is_uniq_list(newusers)) {
		fprintf(stderr,
			_("%s: the list of %s contains duplicate entries\n"),
			progname, "group members");
		status = EINVAL;
		goto EXIT;
	}

	/* Make sure that the list of group administrators is unique */
	if (!is_uniq_list(newadms)) {
		fprintf(stderr,
			_("%s: the list of %s contains duplicate entries\n"),
			progname, "group administrators");
		status = EINVAL;
		goto EXIT;
	}

	/* Split the list of group members */
	if (newusers && !(newusersv = strsplit(newusers, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Split the list of group administrators */
	if (newadms && !(newadmsv = strsplit(newadms, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Verify that all group members exist */
	if (!args.force && newusersv) {
		uid_t i;

		for (i = 0; newusersv[i]; i++) {
			if (!(status = is_free_username(newusersv[i]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("group members"));
				status = ENOENT;
				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_username",
					strerror(errno));
				status = errno;
				goto EXIT;
			}
		}
	}

	/* Verify that all group administrators exist */
	if (!args.force && newadmsv) {
		uid_t i;

		for (i = 0; newadmsv[i]; i++) {
			if (!(status = is_free_username(newadmsv[i]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("group administrators"));
				status = ENOENT;
				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_username",
					strerror(errno));
				status = errno;
				goto EXIT;
			}
		}
	}

	/* Create the new group entry */
	gr2.gr_name = newname;
	gr2.gr_gid = newid;
	gr2.gr_passwd = (char *)"x";
	gr2.gr_mem = newusersv;

	/* Create the new shadow group entry */
	sg2.sg_name = newname;
	sg2.sg_passwd = (char *)((newlocked) ? "!" : "");
	sg2.sg_adm = newadmsv;
	sg2.sg_mem = newusersv;

	/* Create filename /etc/group.write */
	if (!(grwname = create_filename(GROUP_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/group- */
	if (!(grbname = create_filename(GROUP_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/gshadow.write */
	if (!(sgwname = create_filename(GSHADOW_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/gshadow- */
	if (!(sgbname = create_filename(GSHADOW_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT;

	/* Change umask */
	oldmask = umask(0077);

	/* Open /etc/group */
	if (!(grrfp = open_file(GROUP_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/group to /etc/group- */
	if ((status = backup_file(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Copy permissions from /etc/group to /etc/group- */
	if ((status = copy_file_modes(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Open /etc/group.write */
	if (!(grwfp = open_file(grwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Open /etc/gshadow */
	if (!(sgrfp = open_file(GSHADOW_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/gshadow to /etc/gshadow- */
	if ((status = backup_file(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Copy permissions from /etc/gshadow to /etc/gshadow- */
	if ((status = copy_file_modes(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Open /etc/gshadow.write */
	if (!(sgwfp = open_file(sgwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Perform changes on /etc/group */
	while ((gr = fgetgrent(grrfp))) {
		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Write the entry */
		if ((status = fputgrent(gr, grwfp))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputgrent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Write the new entry */
	if ((!errno || errno == ENOENT) && (status = fputgrent(&gr2, grwfp))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fputgrent", strerror(errno));
		goto EXIT2;
	} else {
		/* Set as an indication that the file has at least 1 entry */
		empty = 0;
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetgrent()", strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/group.write */
	if ((status = close_file(&grwfp)))
		goto EXIT2;

	/* Close /etc/group */
	if ((status = close_file(&grrfp)))
		goto EXIT2;

	/* Perform changes on /etc/gshadow */
	while ((sg = fgetsgent(sgrfp))) {
		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Write the entry */
		if ((status = fputsgent(sg, sgwfp))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputsgent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Write the new entry */
	if ((!errno || errno == ENOENT) && (status = fputsgent(&sg2, sgwfp))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fputsgent", strerror(errno));
		goto EXIT2;
	} else {
		/* Set as an indication that the file has at least 1 entry */
		empty = 0;
		changed = 1;
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetsgent", strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/gshadow.write */
	if ((status = close_file(&sgwfp)))
		goto EXIT2;

	/* Close /etc/gshadow */
	if ((status = close_file(&sgrfp)))
		goto EXIT2;

	/* If nothing has changed, don't replace old files */
	if (!changed)
		goto EXIT2;

	/* Everything is in order, move the new files in place */
	if ((status = replace_file(grwname, GROUP_FILE)))
		goto EXIT2;

	if ((status = replace_file(sgwname, GSHADOW_FILE)))
		goto EXIT2;

	/* Set file permissions properly */
	if ((status = copy_file_modes(grbname, GROUP_FILE)))
		goto EXIT2;

	if ((status = copy_file_modes(sgbname, GSHADOW_FILE)))
		goto EXIT2;

EXIT2:
	/* Restore umask */
	umask(oldmask);

	/* These files might not exist, but that's ok */
	status = unlink_file(grwname, status);
	status = unlink_file(sgwname, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT:
	strfreev(newusersv);
	strfreev(newadmsv);
	free(newusers);
	free(newadms);
	free(grwname);
	free(grbname);
	free(sgwname);
	free(sgbname);

	return status;
}
