/**
 * @file libpasswd.c
 * Common routines used by passwd and gpasswd
 *
 * Copyright (C) 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "misc.h"		/* Keep this first, to get config.h */

#include <grp.h>
#include <pwd.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"
#include "libpasswd.h"

/**
 * Mega function that does all the real action in passwd/gpasswd
 *
 * @todo Should be split into several functions
 * @todo The parameter user should be renamed
 * @param user The user or group to change the password for
 * @param group 0 if passwd behaviour is expected,
 *              1 if gpasswd behaviour is expected
 * @param removepass 0 to behave as normal,
 *                   1 to empty the password
 * @param showhash 0 to behave as normal
 *                 1 to show the password hash instead of changing anything
 * @return 0 on success,
 *         errno on failure
 */
int setpasswd(const char *user, int group, int removepass, int showhash)
{
	FILE *rfp = NULL;
	FILE *wfp = NULL;
	struct spwd *sp = NULL;
	struct sgrp *sg = NULL;

	int empty = 1;
	int locked = 0;

	error_t status = 0;

	int isadmin = 0;

	long today = 0;

	char *name = NULL;
	char *spassword = NULL;
	char *wname = NULL;
	char *bname = NULL;

	mode_t oldmask;

	/* Check if the caller has root privileges */
	if ((status = is_useradmin())) {
		if (status == EPERM) {
			status = 0;
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_useradmin", strerror(errno));
			return status;
		}
	} else {
		isadmin = 1;
	}

	/* If an argument is supplied, it is the name
	 * of the user or group to change the password of
	 */
	if (user) {
		if (!(name = strdup(user))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	if ((today = get_current_date()) == -1) {
		status = errno;
		goto EXIT;
	}

	/* Unless the user requests to change his own password,
	 * or calls the program as a user-administrator, deny the request
	 * If the request is for changing a group-password, we have to
	 * check if the user is an administrator for the group or a
	 * user-administrator
	 */
	if (name && !isadmin && !group) {
		int retval = is_caller(name);

		if (retval == -1) {
			status = errno;
			goto EXIT;
		} else if (!retval) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be a user-administrator to "
				  "%s %s\n"
				  "for other users than yourself.\n"),
				progname, _("change"), _("the password"));
			status = EPERM;
			goto EXIT;
		}
	} else if ((group) && (showhash == 0)) {
		int retval = is_groupadmin(name);

		if (retval == -1) {
			status = errno;
			goto EXIT;
		} else if (retval) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "Only group-administrators may %s %s.\n"),
				progname, _("change"), _("the group password"));
			status = EPERM;
			goto EXIT;
		}
	}

	if (!group) {
		/* If no username was provided, retrieve the name
		 * of the caller of the program
		 */
		if (!name) {
			if (!(name = get_username(getuid()))) {
				status = errno;
				goto EXIT;
			}
		}

		/* Verify that the user exists */
		if (!(sp = getspnam(name)) && errno) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getspnam()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* Abort on non-existing users */
		if (!sp) {
			fprintf(stderr,
				_("%s: the specified %s does not exist\n"),
				progname, _("user"));
			status = ENOENT;
			goto EXIT;
		} else {
			spassword = sp->sp_pwdp;
		}

		/* Verify that the user may change his password */
		if (!isadmin && ((today - sp->sp_lstchg) < sp->sp_min)) {
			fprintf(stderr,
				_("%s: not enough time since last password "
				  "change\n"),
				progname);
			goto EXIT;
		}
	} else if ((group) && (showhash == 0)) {
		/* Verify that the group exists */
		if (!(sg = getsgnam(name)) && errno) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getsgnam()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* Abort on non-existing groups */
		if (!sg) {
			fprintf(stderr,
				_("%s: the specified %s does not exist\n"),
				progname, _("group"));
			status = ENOENT;
			goto EXIT;
		} else {
			spassword = sg->sg_passwd;
		}
	}

	if (showhash == 0)
		if ((locked = is_password_locked(spassword)))
			spassword++;

	if (!removepass) {
		/* Note: from this point on we know that the name is valid,
		 * since it existed in the user or group database, hence we
		 * can print it without fear if we want to
		 */

		/* Prompt for a password, unless the user
		 * is a user-administrator
		 */
		if (!isadmin) {
			char *result = NULL;
			int match = 0;

			fprintf(stdout, _("(Current) password: "));
			(void)fflush(stdout);

			result = input_password(spassword, spassword, &match);

			if (!result && errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "prompt_for_password",
					strerror(errno));
				status = errno;
				goto EXIT;
			} else if (result && !match) {
				fprintf(stderr,
					_("%s: authentication failure\n"),
					progname);

				memset(result, 0, strlen(result));
				free(result);

				status = EPERM;
				goto EXIT;
			} else if (!result) {
				fprintf(stderr,
					_("Aborted.\n"));
				goto EXIT;
			}
		}

		/* If we've gotten this far, we're authenticated */
		while (1) {
			char *newpassword;
			char *newpassword2;
			int counter = 0;
			int match = 0;
			int same = 0;

			fprintf(stdout, _("Enter new password: "));
			(void)fflush(stdout);

			if (!(newpassword = input_password(NULL,
							   spassword,
							   &same))) {
				if (counter > 2) {
					fprintf(stderr,
						_("Aborted.\n"));
					goto EXIT;
				}

				fprintf(stderr,
					_("No password supplied\n"));
				counter++;
				continue;
			} else {
				/* This input was successful, reset counter */
				counter = 0;
			}

			fprintf(stdout, _("Retype new password: "));
			(void)fflush(stdout);

			if (!(newpassword2 = input_password(newpassword,
							    newpassword,
							    &match)) ||
			    !match) {
				fprintf(stderr,
					_("Sorry, passwords do not match\n"));
				memset(newpassword, 0, strlen(newpassword));
				free(newpassword);

				if (newpassword2) {
					memset(newpassword2, 0,
					       strlen(newpassword2));
					free(newpassword2);
				}

				memset(spassword, 0, strlen(spassword));
				goto EXIT;
			} else if (same) {
				fprintf(stderr,
					_("Password unchanged\n"));
				memset(newpassword, 0, strlen(newpassword));
				free(newpassword);
				memset(newpassword2, 0, strlen(newpassword2));
				free(newpassword2);
				continue;
			} else {
				/* We clean up the retype password
				 * and the old password, since we
				 * only need the new one from now on
				 */
				memset(newpassword2, 0, strlen(newpassword2));
				free(newpassword2);
				if (spassword)
					memset(spassword, 0, strlen(spassword));
				spassword = strconcat(locked ? "!" : "",
						      newpassword, NULL);
				memset(newpassword, 0, strlen(newpassword));

				if (!spassword) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "strconcat",
						strerror(errno));
					status = errno;
					goto EXIT;
				}

				break;
			}
		}
	} else {
		memset(spassword, 0, strlen(spassword));

		if (!(spassword = strdup(locked ? "!" : ""))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	if (showhash) {
		printf ("%s\n", spassword);
		goto EXIT2;
	}

	/* Create filename /etc/shadow.write or /etc/gshadow.write */
	if (!(wname = create_filename(!group ? SHADOW_FILE : GSHADOW_FILE,
				      WRITE_EXT))) {
		status = errno;
		goto EXIT2;
	}

	/* Create filename /etc/shadow- or /etc/gshadow- */
	if (!(bname = create_filename(!group ? SHADOW_FILE : GSHADOW_FILE,
				      BACKUP_EXT))) {
		status = errno;
		goto EXIT2;
	}

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT2;

	/* Change umask */
	oldmask = umask(0077);

	/* Open /etc/shadow or /etc/gshadow */
	if (!(rfp = open_file(!group ? SHADOW_FILE : GSHADOW_FILE, "r"))) {
		status = errno;
		goto EXIT3;
	}

	/* Backup /etc/shadow to /etc/shadow-
	 * or /etc/gshadow to /etc/gshadow-
	 */
	if ((status = backup_file(!group ? SHADOW_FILE : GSHADOW_FILE,
				  bname)))
		goto EXIT3;

	/* Copy permissions from /etc/shadow to /etc/shadow-
	 * or /etc/gshadow to /etc/gshadow-
	 */
	if ((status = copy_file_modes(!group ? SHADOW_FILE : GSHADOW_FILE,
				      bname)))
		goto EXIT3;

	/* Open /etc/shadow.write or /etc/gshadow.write */
	if (!(wfp = open_file(wname, "w"))) {
		status = errno;
		goto EXIT3;
	}

	if (!group) {
		/* Perform changes on /etc/shadow */
		while ((sp = fgetspent(rfp))) {
			/* Set as an indication that the file
			 * has at least 1 entry
			 */
			empty = 0;

			/* If this is the user to change, empty the old
			 * password, just for paranoia's sake, change the
			 * struct to point to the new password
			 */
			if (!strcmp(name, sp->sp_namp)) {
				memset(sp->sp_pwdp, 0, strlen(sp->sp_pwdp));
				sp->sp_pwdp = spassword;
				sp->sp_lstchg = today;
			}

			/* Write the entry */
			if ((status = fputspent(sp, wfp))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "fputspent", strerror(errno));
				goto EXIT3;
			}

		}
	} else {
		/* Perform changes on /etc/gshadow */
		while ((sg = fgetsgent(rfp))) {
			/* Set as an indication that the file
			 * has at least 1 entry
			 */
			empty = 0;

			/* If this is the user to change, empty the old
			 * password, just for paranoia's sake, change the
			 * struct to point to the new password
			 */
			if (!strcmp(name, sg->sg_name)) {
				memset(sg->sg_passwd, 0, strlen(sg->sg_passwd));
				sg->sg_passwd = spassword;
			}

			/* Write the entry */
			if ((status = fputsgent(sg, wfp))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "fputsgent", strerror(errno));
				goto EXIT3;
			}
		}
	}


	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, !group ? "fgetspent()" : "fgetsgent()",
			strerror(errno));
		status = errno;
		goto EXIT3;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/shadow.write or /etc/gshadow.write */
	if ((status = close_file(&wfp)))
		goto EXIT3;

	/* Close /etc/shadow or /etc/gshadow */
	if ((status = close_file(&rfp)))
		goto EXIT3;

	/* Everything is in order, move the new file in place */
	if ((status = replace_file(wname,
				   !group ? SHADOW_FILE : GSHADOW_FILE)))
		goto EXIT3;

	/* Set file permissions properly */
	if ((status = copy_file_modes(bname,
				      !group ? SHADOW_FILE : GSHADOW_FILE)))
		goto EXIT3;

EXIT3:
	/* Restore umask */
	umask(oldmask);

	/* This file might not exist, but that's ok */
	status = unlink_file(wname, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT2:
	memset(spassword, 0, strlen(spassword));
	free(spassword);

EXIT:
	free(wname);
	free(bname);
	free(name);

	return status;
}

/** Structure with the available command line options */
static struct argp_option setpasswd_options[] = {
	{ "remove-password", 'd', 0, 0,
	  N_("Make the password empty"), 0 },
	{ "show-hash", 's', 0, 0,
	  N_("Show the password hash on the screen instead"), 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_setpasswd_opt(int key, char *arg, struct argp_state *state)
{
	struct setpasswd_options_t *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'd':
		args->removepass = 1;
		break;

	case 's':
		args->showhash = 1;
		break;

	case ARGP_KEY_INIT:
		args->name = NULL;
		args->removepass = 0;
		args->showhash = 0;
		break;

	case ARGP_KEY_ARG:
		if (args->name)
			argp_usage(state);

		args->name = arg;
		break;

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/** argp struct used for argument passing in libpasswd */
struct argp setpasswd_argp = {
	setpasswd_options, parse_setpasswd_opt, 0, 0, 0, 0, 0
};
