/**
 * @file lastlog.c
 * Show lastlog information
 *
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <time.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <lastlog.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "lastlog"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char doc[] =
	N_("Show information about latest login dates for users.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "before", 'b', N_("DAYS"), 0,
	  N_("Only show logins that occured earlier than the last "
	     "DAYS number of days"), 0 },
	{ "has-logged-in", 'l', 0, 0,
	  N_("Only show users that have been logged in on the system"), 0},
	{ "sort", 's', 0, 0,
	  N_("Show the list sorted by uid rather than the order they occur "
	     "in the user database"), 0 },
	{ "time", 't', N_("DAYS"), 0,
	  N_("Only show logins that occured the last DAYS number of days"), 0 },
	{ "user", 'u', N_("USER"), 0,
	  N_("Only show information about USER"), 0, },
	{ "rfc-2822", 'R', 0, 0,
	  N_("Output date and time in RFC 2822 format"), 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	const char *user;	/**< Only show information about "user" */
	long days;		/**< Only logins made the last "days" days */
	int before;		/**< Only logins made before "before" days */
	int logged;		/**< Only show users that have been logged in */
	int sort;		/**< Sort the list by UID */
	int rfc2822;		/**< Display dates in RFC 2822 format */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'b':
		args->before = 1;

		if ((status = is_long(arg)))
			argp_error(state,
				   _("invalid value supplied to `-b'"));

		if ((status = string_to_long(arg,
					     (void **)&args->days))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_long", strerror(errno));
			goto EXIT;
		}

		break;

	case 'l':
		args->logged = 1;
		break;

	case 'R':
		args->rfc2822 = 1;
		break;

	case 's':
		args->sort = 1;
		break;

	case 't':
		if ((status = is_long(arg)))
			argp_error(state,
				   _("invalid value supplied to `-t'"));

		if ((status = string_to_long(arg,
					     (void **)&args->days))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_long", strerror(errno));
			goto EXIT;
		}

		break;

	case 'u':
		args->user = arg;
		break;

	case ARGP_KEY_INIT:
		args->user = NULL;
		args->days = -1;
		args->before = 0;
		args->logged = 0;
		args->sort = 0;
		args->rfc2822 = 0;
		break;

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

EXIT:

	return status;
}

/**
 * Compare function used by qsort
 *
 * @param uid1 The first uid in the comparision
 * @param uid2 The second uid in the comparision
 * @return negative when uid1 < uid2
 *         0 if uid1 == uid2
 *         positive when uid1 > uid2
 */
static int uid_compare(const void *uid1, const void *uid2)
{
	return *(uid_t *)uid1 - *(uid_t *)uid2;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	int (*compare_ptr)(const void *, const void *) = uid_compare;
	static struct lastlog llog;
	static struct stat st;
	static time_t now;
	uid_t *uidarray = NULL;

	FILE *rfp = NULL;
	int fd;

	error_t status = 0;

	time_t limit = 0;
	uid_t i; /* We're scanning <= max(nrofusers), hence uid_t */
	size_t nusers;
	int first = 1;
	const char *timefmt = NULL;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	if (args.days != -1)
		limit = 60 * 60 * 24 * args.days;

	/* Try to get the current date */
	if (time(&now) == (time_t)-1) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "time()", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Get the array of uid's */
	if (args.user) {
		struct passwd *pw;

		if (!(pw = getpwnam(args.user))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getpwnam()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* We failed to allocate sizeof (uid_t) bytes of mem,
		 * hence we have serious problems...
		 */
		if (!(uidarray = malloc(sizeof (uid_t)))) {
			status = errno;
			goto EXIT;
		}

		uidarray[0] = pw->pw_uid;
		nusers = 1;
	} else if (!(uidarray = get_all_uids(&nusers))) {
		status = errno;
		goto EXIT;
	}

	/* If sorting is requested, sort uidarray2 */
	if ((nusers != 1) && args.sort)
		qsort(uidarray, nusers, sizeof (uid_t), compare_ptr);

	/* Open /var/log/lastlog for reading */
	if (!(rfp = open_file(LASTLOG_FILE, "r"))) {
		status = errno;
		goto EXIT;
	}

	/* Get the file-descriptor for our new file-pointer */
	if ((fd = fileno(rfp)) == -1) {
		fprintf(stderr,
			_("%s: aiiik, this should never "
			  "happen. Alert the programmer!\n"),
			progname);
		status = errno;
		goto EXIT;
	}

	/* Get the size of /var/log/lastlog */
	if (fstat(fd, &st) == -1) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fstat()", strerror(errno));
		status = errno;
		goto EXIT;
	}

	for (i = 0; i < nusers; i++) {
		/* Our date will never overflow half a line with the format
		 * we have specified, and if it magically does, strftime
		 * will handle the problem for us.
		 */
		static char tbuf[40];
		struct tm ltm;
		struct passwd *pw = NULL;
		long off = (long)(uidarray[i] * sizeof (struct lastlog));
		char *llstr;

		/* Ignore entries that cannot possibly have an entry */
		if (off >= st.st_size)
			continue;

		/* Seek to the right position in the lastlog */
		if (fseek(rfp, (long)(uidarray[i] * sizeof (struct lastlog)),
			  SEEK_SET) == -1) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fseek()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* Read the entry from the log */
		if (fread((char *)&llog, sizeof llog, 1, rfp) != 1) {
			if (!errno)
				break;

			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fread()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		if (!(pw = getpwuid(uidarray[i]))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getpwuid()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		if ((args.days != -1) && (args.before == 0) &&
		    (now - limit) > (time_t)llog.ll_time) {
			continue;
		} else if ((args.days != -1) && (args.before == 1) &&
		    (now - limit) < (time_t)llog.ll_time) {
			continue;
		} else if ((time_t)llog.ll_time == 0) {
			if (args.logged)
				continue;

			llstr = _("**Never logged in**");
		} else {
			char *oldlocale = NULL;
			char *tmp;

			if (!localtime_r((time_t *)&llog.ll_time, &ltm)) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "localtime_r()",
					strerror(errno));
				status = errno;
				goto EXIT;
			}

			if (args.rfc2822)
				timefmt = "%a, %d %b %Y %H:%M:%S %z";
			else
				timefmt = "%c %z";

			if ((tmp = setlocale(LC_TIME, NULL))) {
				if (!(oldlocale = strdup(tmp))) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "strdup()",
						strerror(errno));
					status = errno;
					goto EXIT;
				}
			}

			/* We need to set LC_TIME=C here to get the
			 * the right format when using RFC 2822
			 * dates, otherwise we'll get localised
			 * day and month names
			 */
			if (args.rfc2822)
				setlocale(LC_TIME, "C");

			if (!strftime(tbuf, sizeof tbuf, timefmt, &ltm)) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "strftime()",
					strerror(errno));
				status = errno;
				goto EXIT;
			}

			if (args.rfc2822 && oldlocale) {
				setlocale(LC_TIME, oldlocale);
				free(oldlocale);
			}

			llstr = tbuf;
		}

		if (first) {
			fprintf(stdout,
				_("Username         Port     From"
				  "             Latest\n"));
			first = 0;
		}

		fprintf(stdout,
			"%-16s %-8.8s %-16.16s %s\n",
			pw->pw_name, llog.ll_line, llog.ll_host, llstr);
	}

	/* Close infile */
	if ((status = close_file(&rfp)))
		goto EXIT;

EXIT:
	/* Free all allocated memory */
	free(uidarray);

	return status;
}
