/**
 * @file chuser.c
 * Change attributes for the specified users
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <grp.h>
#include <pwd.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gshadow.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "chuser"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("ATTRIBUTE=VALUE... USERS");

/** Program synopsis */
static char doc[] =
	N_("Change attributes for the specified users.\n"
	   "\n"
	   "USERS should be a comma-separated list of users, "
	   "or the word ALL to change attributes for all users.  "
	   "Note that some attributes can only be changed for one "
	   "user at a time.\n"
	   "\n"
	   "Valid attributes:\n"
	   "\n"
	   "account_locked | locked        Lock/Unlock account\n"
	   "admgroups                      List of the groups that the "
	   "users administers\n"
	   "fname | fullname (*)           The user's full name\n"
	   "gecos (*)                      The user's gecos-field\n"
	   "groups                         The users' group "
	   "membership list\n"
	   "home (*)                       The user's home directory\n"
	   "hphone (*)                     The user's home phone\n"
	   "id | uid (*, **)               The user's id\n"
	   "other (*)                      Other information\n"
	   "pgrp                           The users' primary group\n"
	   "room (*)                       The user's room number\n"
	   "shell (**)                     The users' login shell\n"
	   "username (*, **)               The username\n"
	   "wphone (*)                     The user's work phone\n"
	   "\n"
	   "Attributes marked (*) can only be changed when a single "
	   "user is specified\n"
	   "Attributes with (**) requires use of the `--force' option\n"
	   "To set a non-listed (but existing) shell path, use `--force'\n"
	   "\n"
	   "Note, that the attribute `gecos' cannot be combined with any of "
	   "the attributes `fname', `room', `wphone', `hphone', or `other'.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "force", 'f', 0, 0,
	  N_("Force a change of username or id"), -2 },
	{ "verbose", 'v', 0, 0,
	  N_("Warn if the specified users does not exist"), -2 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	char *const *args;	/**< Arguments */
	int nargs;		/**< Number of arguments */
	int force;		/**< Force a change of username or ID */
	int verbose;		/**< Warn about non-existing users */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'f':
		args->force = 1;
		break;

	case 'v':
		args->verbose = 1;
		break;

	case ARGP_KEY_INIT:
		args->args = NULL;
		args->nargs = 0;
		args->force = 0;
		args->verbose = 0;
		break;

	case ARGP_KEY_ARG:
		status = ARGP_ERR_UNKNOWN;
		break;

	case ARGP_KEY_ARGS:
		args->args = state->argv + state->next;
		args->nargs = state->argc - state->next;
		break;

	case ARGP_KEY_NO_ARGS:
		argp_usage(state);
		break;

	default:
		(void)arg;
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	FILE *pwrfp = NULL;
	FILE *pwwfp = NULL;
	struct passwd *pw;

	FILE *sprfp = NULL;
	FILE *spwfp = NULL;
	struct spwd *sp;

	FILE *grrfp = NULL;
	FILE *grwfp = NULL;
	struct group *gr;

	FILE *sgrfp = NULL;
	FILE *sgwfp = NULL;
	struct sgrp *sg;

	int empty = 1;
	int changed = 0;

	int attr_index = 0;

	int j;

	error_t status = 0;

	int singleuser = 1;

	char **usrarray = NULL;

	char *pwwname = NULL;
	char *pwbname = NULL;
	char *spwname = NULL;
	char *spbname = NULL;
	char *grwname = NULL;
	char *grbname = NULL;
	char *sgwname = NULL;
	char *sgbname = NULL;

	int newlocked = -1;
	uid_t newid = 65535;
	char *newpgrp = NULL;
	gid_t newpgrpgid = 65535;
	char *newname = NULL;
	char *newshell = NULL;
	char *newhomedir = NULL;
	char *newgroups = NULL;
	char *newadmgroups = NULL;
	char **newgroupsv = NULL;
	char **newadmgroupsv = NULL;

	/* GECOS information */
	char *newgecos = NULL;
	char *newfname = NULL;
	char *newroom = NULL;
	char *newwphone = NULL;
	char *newhphone = NULL;
	char *newother = NULL;

	char **gecosv = NULL;

	uid_t i; /* We're scanning <= LASTUID, hence uid_t */

	mode_t oldmask;

	struct attr attributes[] = {
		{
			.attribute	= AS_A_LOCKED,
			.value		= (void **)&newlocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= AS_ADMGROUPS,
			.value		= (void **)&newadmgroups,
			.validator	= is_valid_namelist,
			.converter	= string_to_string
		}, {
			.attribute	= GS_FNAME,
			.value		= (void **)&newfname,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= GS_FULLNAME,
			.value		= (void **)&newfname,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= AS_GECOS,
			.value		= (void **)&newgecos,
			.validator	= is_valid_gecos,
			.converter	= string_to_string
		}, {
			.attribute	= AS_GROUPS,
			.value		= (void **)&newgroups,
			.validator	= is_valid_namelist,
			.converter	= string_to_string
		}, {
			.attribute	= AS_HOME,
			.value		= (void **)&newhomedir,
			.validator	= is_valid_path,
			.converter	= string_to_string
		}, {
			.attribute	= GS_HPHONE,
			.value		= (void **)&newhphone,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= AS_ID,
			.value		= (void **)&newid,
			.validator	= is_uid_t,
			.converter	= string_to_uid_t
		}, {
			.attribute	= AS_LOCKED,
			.value		= (void **)&newlocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= GS_OTHER,
			.value		= (void **)&newother,
			.validator	= is_valid_gecos_other,
			.converter	= string_to_string
		}, {
			.attribute	= AS_PGRP,
			.value		= (void **)&newpgrp,
			.validator	= is_valid_name,
			.converter	= string_to_string
		}, {
			.attribute	= GS_ROOM,
			.value		= (void **)&newroom,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= AS_SHELL,
			.value		= (void **)&newshell,
			.validator	= is_valid_shell,
			.converter	= string_to_string
		}, {
			.attribute	= AS_UID,
			.value		= (void **)&newid,
			.validator	= is_uid_t,
			.converter	= string_to_uid_t
		}, {
			.attribute	= AS_USERNAME,
			.value		= (void **)&newname,
			.validator	= is_valid_name,
			.converter	= string_to_string
		}, {
			.attribute	= GS_WPHONE,
			.value		= (void **)&newwphone,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= NULL,
			.value		= NULL,
			.validator	= NULL,
			.converter	= NULL
		}
	};

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Make sure the caller is a user-admin */
	if ((status = is_useradmin())) {
		if (status == EPERM) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be a user-administrator to %s.\n"),
				progname, _("change user attributes"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_useradmin", strerror(errno));
		}

		goto EXIT;
	}

	/* Make sure we get the correct number of arguments */
	if (args.nargs < 2) {
		fprintf(stderr,
			_("%s: too few arguments\n"
			  "Try `%s --help' for more information.\n"),
			progname, progname);
		status = EINVAL;
		goto EXIT;
	}

	/* Allow a non-listed (but existing) shell path with `--force' */
	while (attributes[attr_index].attribute != NULL) {
		if (attributes[attr_index].attribute == AS_SHELL)
			break;
		attr_index++;
       };

	if (args.force && (attributes[attr_index].attribute == AS_SHELL)) {
		attributes[attr_index].validator = is_valid_filepath;
	}

	/* Parse the attribute=value pairs */
	for (j = 0; j < (args.nargs - 1); j++) {
		if ((status = parse_key_pairs(args.args[j], attributes))) {
			if (status == ENOENT) {
				fprintf(stderr,
					_("%s: invalid attributes\n"
					  "Valid attributes are:\n"
					  "- `%s' | `%s'\n"
					  "- `%s'\n"
					  "- `%s' | `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s' | `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "Try `%s --help' for more "
					  "information.\n"),
					progname,
					AS_A_LOCKED, AS_LOCKED,
					AS_ADMGROUPS,
					GS_FNAME, GS_FULLNAME,
					AS_GECOS,
					AS_GROUPS,
					AS_HOME,
					GS_HPHONE,
					AS_ID, AS_GID,
					GS_OTHER,
					AS_PGRP,
					GS_ROOM,
					AS_SHELL,
					AS_USERNAME,
					GS_WPHONE,
					progname);
				status = EINVAL;
			}

			goto EXIT;
		}
	}

	/* `gecos' cannot be combined with any of `fname', `room',
	 * `wphone', `hphone', `other'.
	 */
	if (newgecos &&
	    (newfname || newhphone || newwphone || newother || newroom)) {
		fprintf(stderr,
			_("%s: `%s' cannot be combined with any of\n"
			  "`%s', `%s', `%s', `%s', or `%s'\n"),
			progname,
			"gecos", "fname", "room", "wphone", "hphone", "other");
		status = EINVAL;
		goto EXIT;
	}

	/* If any commands that require `--force' are specified
	 * without the use of `--force', abort.
	 */
	if (!args.force && (newid != 65535 || newname)) {
		fprintf(stderr,
			_("%s: these changes require `--force' to be "
			  "specified\n"),
			progname);
		status = EINVAL;
		goto EXIT;
	}

	/* There are two alternatives here, neither of which are really
	 * pretty; either to read the entire group file once to get
	 * all groupnames, then use them for the ALL list, or to
	 * have separate code for the ALL case and the case of separate
	 * group-entries.  Since the latter is probably the most common,
	 * the latter has been chosen.
	 */
	if (!strcmp(args.args[args.nargs - 1], "ALL")) {
		char *tmp = NULL;

		if (!args.force) {
			fprintf(stderr,
				_("%s: specifying ALL requires `--force' "
				  "to be specified\n"),
				progname);
			status = EINVAL;
			goto EXIT;
		}

		if (!(tmp = get_all_users())) {
			status = errno;
		} else if (!strlen(tmp)) {
			fprintf(stderr,
				_("%s: could not find any %s; the %s "
				  "might be corrupt\n"),
				progname, _("users"), _("user database"));
			status = ENOENT;
		} else if (!(usrarray = strsplit(tmp, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
		}

		free(tmp);

		if (status)
			goto EXIT;
	} else {
		char *tmp = NULL;
		i = 0;

		if (!(usrarray = strsplit(args.args[args.nargs - 1], ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* If verbose mode has been requested,
		 * warn about all non-existing users
		 */
		while (args.verbose && (tmp = usrarray[i++])) {
			if (!getpwnam(tmp)) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "getpwnam()",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					fprintf(stderr,
						_("%s: warning: %s `%s' "
						  "does not exist\n"),
						progname, _("user"), tmp);
				}
			}
		}
	}

	/* If more than 1 user is specified, switch to multiuser mode */
	if (usrarray[1])
		singleuser = 0;

	/* If singleuser commands are specified together
	 * with multiple users, abort.
	 */
	if (!singleuser &&
	    (newid != 65535 || newname || newgecos || newhomedir)) {
		fprintf(stderr,
			_("%s: multiple %s specified together with attributes\n"
			  "that can only be used in %s\n"),
			progname, _("users"), _("single-user mode"));
		status = EINVAL;
		goto EXIT;
	}

	/* Verify that the requested uid is free */
	if (newid != 65535 && (status = is_free_uid(newid))) {
		if (status == EEXIST) {
			fprintf(stderr,
				_("%s: the specified %s already exists\n"),
				progname, _("user id"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_free_uid", strerror(errno));
		}

		goto EXIT;
	}

	/* Make sure that newid isn't unreasonably large */
	if (newid != 65535 && newid > LASTUID) {
		fprintf(stderr,
			_("%s: the %s cannot be larger than %d\n"),
			progname, _("user id"), LASTGID);
		status = EINVAL;
		goto EXIT;
	}

	/* Verify that the requested username is unused */
	if (newname && (status = is_free_username(newname))) {
		if (status == EEXIST) {
			fprintf(stderr,
				_("%s: the specified %s already exists\n"),
				progname, _("username"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_free_username", strerror(errno));
		}

		goto EXIT;
	}

	/* If newgroups or newadmgroups are empty, free and NULL them */
	if (newgroups && !strlen(newgroups)) {
		free(newgroups);
		newgroups = NULL;
	}

	if (newadmgroups && !strlen(newadmgroups)) {
		free(newadmgroups);
		newadmgroups = NULL;
	}

	if (newpgrp) {
		/* Verify that the specified primary group exists */
		if (!(gr = getgrnam(newpgrp)) && errno) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getpwnam()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* Abort on non-existing groups */
		if (!gr) {
			fprintf(stderr,
				_("%s: the specified %s does not exist\n"),
				progname, _("primary group"));
			status = ENOENT;
			goto EXIT;
		} else {
			newpgrpgid = gr->gr_gid;
		}
	}

	/* Make sure that the list of groups is unique */
	if (!is_uniq_list(newgroups)) {
		fprintf(stderr,
			_("%s: the list of %s contains duplicate entries\n"),
			progname, "groups");
		status = EINVAL;
		goto EXIT;
	}

	/* Make sure that the list of admgroups is unique */
	if (!is_uniq_list(newadmgroups)) {
		fprintf(stderr,
			_("%s: the list of %s contains duplicate entries\n"),
			progname, "admgroups");
		status = EINVAL;
		goto EXIT;
	}

	/* Split the list of groups */
	if (newgroups && !(newgroupsv = strsplit(newgroups, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Split the list of admgroups */
	if (newadmgroups && !(newadmgroupsv = strsplit(newadmgroups, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Verify that all groups exist */
	if (newgroupsv) {
		for (i = 0; newgroupsv[i]; i++) {
			if (!(status = is_free_groupname(newgroupsv[i]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("groups"));
				status = ENOENT;
				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_groupname",
					strerror(errno));
				status = errno;
				goto EXIT;
			}
		}
	}

	/* Verify that all admgroups exist */
	if (newadmgroupsv) {
		for (i = 0; newadmgroupsv[i]; i++) {
			if (!(status = is_free_groupname(newadmgroupsv[i]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("admgroups"));
				status = ENOENT;
				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_groupname",
					strerror(errno));
				status = errno;
				goto EXIT;
			}
		}
	}

	/* Create filename /etc/passwd.write */
	if (!(pwwname = create_filename(PASSWD_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/passwd- */
	if (!(pwbname = create_filename(PASSWD_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/group.write */
	if (!(grwname = create_filename(GROUP_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/group- */
	if (!(grbname = create_filename(GROUP_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/shadow.write */
	if (!(spwname = create_filename(SHADOW_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/shadow- */
	if (!(spbname = create_filename(SHADOW_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/gshadow.write */
	if (!(sgwname = create_filename(GSHADOW_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/gshadow- */
	if (!(sgbname = create_filename(GSHADOW_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT;

	/* Change umask */
	oldmask = umask(0077);

	/* Open /etc/passwd */
	if (!(pwrfp = open_file(PASSWD_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/passwd to /etc/passwd- */
	if ((status = backup_file(PASSWD_FILE, pwbname)))
		goto EXIT2;

	/* Copy permissions from /etc/passwd to /etc/passwd- */
	if ((status = copy_file_modes(PASSWD_FILE, pwbname)))
		goto EXIT2;

	/* Open /etc/passwd.write */
	if (!(pwwfp = open_file(pwwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Open /etc/group */
	if (!(grrfp = open_file(GROUP_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/group to /etc/group- */
	if ((status = backup_file(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Copy permissions from /etc/group to /etc/group- */
	if ((status = copy_file_modes(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Open /etc/group.write */
	if (!(grwfp = open_file(grwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Open /etc/shadow */
	if (!(sprfp = open_file(SHADOW_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/shadow to /etc/shadow- */
	if ((status = backup_file(SHADOW_FILE, spbname)))
		goto EXIT2;

	/* Copy permissions from /etc/shadow to /etc/shadow- */
	if ((status = copy_file_modes(SHADOW_FILE, spbname)))
		goto EXIT2;

	/* Open /etc/shadow.write */
	if (!(spwfp = open_file(spwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Open /etc/gshadow */
	if (!(sgrfp = open_file(GSHADOW_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/gshadow to /etc/gshadow- */
	if ((status = backup_file(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Copy permissions from /etc/gshadow to /etc/gshadow- */
	if ((status = copy_file_modes(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Open /etc/gshadow.write */
	if (!(sgwfp = open_file(sgwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Perform changes on /etc/passwd */
	while ((pw = fgetpwent(pwrfp))) {
		static struct passwd pw2;
		char *tmpgecos = NULL;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Copy the old entry */
		pw2.pw_passwd = pw->pw_passwd;

		/* If the entry is a part of the array of users to modify,
		 * modify its attributes; if not, copy the old values
		 */
		if (is_in_array(usrarray, pw->pw_name)) {
			pw2.pw_name = newname ? newname : pw->pw_name;
			pw2.pw_uid = (newid != 65535) ? newid : pw->pw_uid;
			pw2.pw_gid = (newpgrpgid != 65535) ? newpgrpgid : pw->pw_gid;

			/* Split the gecos-information into an array */
			if (!(gecosv = split_gecos(pw->pw_gecos))) {
				status = errno;
				goto EXIT2;
			}

			if (!newgecos &&
			    !(tmpgecos = join_gecos(newfname ? newfname :
							       gecosv[0],
						    newroom ? newroom :
							      gecosv[1],
						    newwphone ? newwphone :
								gecosv[2],
						    newhphone ? newhphone :
								gecosv[3],
						    newother ? newother :
							       gecosv[4]))) {
				status = errno;
				goto EXIT;
			}

			pw2.pw_gecos = newgecos ? newgecos : tmpgecos;
			pw2.pw_dir = newhomedir ? newhomedir : pw->pw_dir;
			pw2.pw_shell = newshell ? newshell : pw->pw_shell;
			changed = 1;

			strfreev(gecosv);
		} else {
			pw2.pw_name = pw->pw_name;
			pw2.pw_uid = pw->pw_uid;
			pw2.pw_gid = pw->pw_gid;
			pw2.pw_gecos = pw->pw_gecos;
			pw2.pw_dir = pw->pw_dir;
			pw2.pw_shell = pw->pw_shell;
		}

		/* Write the entry */
		if ((status = fputpwent(&pw2, pwwfp))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputpwent", strerror(errno));
			free(tmpgecos);
			goto EXIT2;
		}

		free(tmpgecos);
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetpwent()", strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/passwd.write */
	if ((status = close_file(&pwwfp)))
		goto EXIT2;

	/* Close /etc/passwd */
	if ((status = close_file(&pwrfp)))
		goto EXIT2;

	/* Perform changes on /etc/group */
	while ((gr = fgetgrent(grrfp))) {
		static struct group gr2;
		char *memlist = NULL;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Copy the old entry */
		gr2.gr_name = gr->gr_name;
		gr2.gr_gid = gr->gr_gid;
		gr2.gr_passwd = gr->gr_passwd;

		/* Join the member array into a list */
		if (!(memlist = strjoinv(",", gr->gr_mem))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strjoinv", strerror(errno));
			status = errno;
			goto EXIT2;
		}

		/* If we have a list of new groups and the current group
		 * is a part of that list, add all users in usrarray to it.
		 * If we have a list of new groups, but the current group is
		 * not in that list, remove all users in usrarray from it.
		 */
		if (newgroups) {
			int add = is_in_list(newgroups, gr->gr_name);

			/* Add all entries in usrarray to list */
			for (i = 0; usrarray[i]; i++) {
				char *tmp = NULL;

				if (add)
					tmp = uniq_add_to_list(memlist,
							       usrarray[i]);
				else
					tmp = remove_from_list(memlist,
							       usrarray[i]);

				free(memlist);

				if (!tmp) {
					status = errno;
					goto EXIT2;
				}

				memlist = tmp;
			}
		}

		/* Split the memlist back into an array */
		gr2.gr_mem = strsplit(memlist, ",", 0);

		free(memlist);

		if (!gr2.gr_mem) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT2;
		}

		/* Write the entry */
		status = fputgrent(&gr2, grwfp);

		strfreev(gr2.gr_mem);

		if (status) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputgrent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetgrent()", strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/group.write */
	if ((status = close_file(&grwfp)))
		goto EXIT2;

	/* Close /etc/group */
	if ((status = close_file(&grrfp)))
		goto EXIT2;

	/* Perform changes on /etc/shadow */
	while ((sp = fgetspent(sprfp))) {
		static struct spwd sp2;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Copy the old entry */
		sp2.sp_pwdp = sp->sp_pwdp;
		sp2.sp_lstchg = sp->sp_lstchg;
		sp2.sp_min = sp->sp_min;
		sp2.sp_max = sp->sp_max;
		sp2.sp_warn = sp->sp_warn;
		sp2.sp_inact = sp->sp_inact;
		sp2.sp_expire = sp->sp_expire;
		sp2.sp_flag = sp->sp_flag;

		/* If the entry is a part of the array of users to modify,
		 * modify its attributes; if not, copy the old values
		 */
		if (is_in_array(usrarray, sp->sp_namp))
			sp2.sp_namp = newname ? newname : sp->sp_namp;
		else
			sp2.sp_namp = sp->sp_namp;

		/* Write the entry */
		if ((status = fputspent(&sp2, spwfp))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputspent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetspent()", strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/shadow.write */
	if ((status = close_file(&spwfp)))
		goto EXIT2;

	/* Close /etc/shadow */
	if ((status = close_file(&sprfp)))
		goto EXIT2;

	/* Perform changes on /etc/gshadow */
	while ((sg = fgetsgent(sgrfp))) {
		static struct sgrp sg2;
		char *admlist = NULL;
		char *memlist = NULL;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Copy the old entry */
		sg2.sg_name = sg->sg_name;
		sg2.sg_passwd = sg->sg_passwd;

		/* Join the admin array into a list */
		if (!(admlist = strjoinv(",", sg->sg_adm))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strjoinv", strerror(errno));
			status = errno;
			goto EXIT2;
		}

		/* If we have a list of new admgroups and the current group
		 * is a part of that list, add all users in usrarray to it.
		 * If we have a list of new admgroups, but the current group
		 * is not in that list, remove all users in usrarray from it.
		 */
		if (newadmgroups) {
			int add = is_in_list(newadmgroups, sg->sg_name);

			/* Add all entries in usrarray to list */
			for (i = 0; usrarray[i]; i++) {
				char *tmp = NULL;

				if (add)
					tmp = uniq_add_to_list(admlist,
							       usrarray[i]);
				else
					tmp = remove_from_list(admlist,
							       usrarray[i]);

				free(admlist);

				if (!tmp) {
					status = errno;
					goto EXIT2;
				}

				admlist = tmp;
			}
		}

		/* Split the admlist back into an array */
		sg2.sg_adm = strsplit(admlist, ",", 0);

		free(admlist);

		if (!sg2.sg_adm) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT2;
		}

		/* Join the member array into a list */
		if (!(memlist = strjoinv(",", sg->sg_mem))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strjoinv", strerror(errno));
			strfreev(sg2.sg_adm);
			status = errno;
			goto EXIT2;
		}

		/* If we have a list of new groups and the current group
		 * is a part of that list, add all users in usrarray to it.
		 * If we have a list of new groups, but the current group is
		 * not in that list, remove all users in usrarray from it.
		 */
		if (newgroups) {
			int add = is_in_list(newgroups, sg->sg_name);

			/* Add all entries in usrarray to list */
			for (i = 0; usrarray[i]; i++) {
				char *tmp = NULL;

				if (add)
					tmp = uniq_add_to_list(memlist,
							       usrarray[i]);
				else
					tmp = remove_from_list(memlist,
							       usrarray[i]);

				free(memlist);

				if (!tmp) {
					strfreev(sg2.sg_adm);
					status = errno;
					goto EXIT2;
				}

				memlist = tmp;
			}
		}

		/* Split the memlist back into an array */
		sg2.sg_mem = strsplit(memlist, ",", 0);

		free(memlist);

		if (!sg2.sg_mem) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			strfreev(sg2.sg_adm);
			status = errno;
			goto EXIT2;
		}

		/* Write the entry */
		status = fputsgent(&sg2, sgwfp);

		strfreev(sg2.sg_adm);
		strfreev(sg2.sg_mem);

		if (status) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputsgent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetsgent", strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/gshadow.write */
	if ((status = close_file(&sgwfp)))
		goto EXIT2;

	/* Close /etc/gshadow */
	if ((status = close_file(&sgrfp)))
		goto EXIT2;

	/* If nothing has changed, don't replace old files */
	if (!changed)
		goto EXIT2;

	/* Everything is in order, move the new files in place */
	if ((status = replace_file(pwwname, PASSWD_FILE)))
		goto EXIT2;

	if ((status = replace_file(grwname, GROUP_FILE)))
		goto EXIT2;

	if ((status = replace_file(spwname, SHADOW_FILE)))
		goto EXIT2;

	if ((status = replace_file(sgwname, GSHADOW_FILE)))
		goto EXIT2;

	/* Set file permissions properly */
	if ((status = copy_file_modes(pwbname, PASSWD_FILE)))
		goto EXIT2;

	if ((status = copy_file_modes(pwbname, SHADOW_FILE)))
		goto EXIT2;

	if ((status = copy_file_modes(grbname, GROUP_FILE)))
		goto EXIT2;

	if ((status = copy_file_modes(sgbname, GSHADOW_FILE)))
		goto EXIT2;

EXIT2:
	/* Restore umask */
	umask(oldmask);

	/* These files might not exist, but that's ok */
	status = unlink_file(pwwname, status);
	status = unlink_file(grwname, status);
	status = unlink_file(spwname, status);
	status = unlink_file(sgwname, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT:
	/* Free all allocated memory */
	strfreev(newadmgroupsv);
	strfreev(newgroupsv);
	strfreev(usrarray);
	free(newadmgroups);
	free(newhomedir);
	free(newgroups);
	free(newhphone);
	free(newwphone);
	free(newother);
	free(newshell);
	free(newgecos);
	free(newfname);
	free(newroom);
	free(newname);
	free(newpgrp);
	free(pwwname);
	free(pwbname);
	free(spwname);
	free(spbname);
	free(grwname);
	free(grbname);
	free(sgwname);
	free(sgbname);

	return status;
}
