/*  Synapse 0.1
 *  Copyright (C) 2006 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file		conf.c
	@brief		Synapse configuration panel
*/

#include "synapse.h"

/**
	Nome del file di configurazione generale
*/
#define DEFAULT_NAME_FOR_CONFIG_FILE		"conf"

/**
	Riassume la struttura del pannello di configurazione
*/
struct {
	GtkWidget	*ask_remove;						/**< Toggle per l'abilitazione della richiesta di conferma all'eliminazione di un file */
	GtkWidget	*oms_modes;						/**< Combo box to choose the default OnMouse-Shell mode */
	GtkWidget	*shortcuts [ S_MAX_SHORTCUTS ];				/**< Lista di shortcut configurate */
	GtkWidget	*show_tooltips;

	GtkWidget	*enable_actions;
	GtkWidget	*action_command_terminal;
	GtkTreeStore	*actions_model;
	GtkWidget	*actions_view;
	GList		*actions_list;

	gboolean	action_manipulated;
	gchar		*action_current;
	GtkWidget	*actions_conf;
	GtkWidget	*action_frame;
	GtkWidget	*action_name;
	GtkWidget	*action_command;
	GtkWidget	*action_term;
	GtkWidget	*action_metas;
} ConfPanel;

Configurations		Confs;							/**< Istanza globale del pannello di configurazione */

/**
	Verifica se una shortcut immessa in un SynapseShortcutEditor e' gia'
	contemplata nel sistema, ed in tal caso chiede conferma per la
	riassegnazione della combinazione di tasti

	@param	ttt	Il SynapseShortcutEditor il cui valore e' stato
			cambiato
	@param	data	Posizione del SynapseShortcutEditor relativamente
			all'array interno di elementi contenuti nel pannello
			di configurazione dell'applicazione

	@return		FALSE
*/
static gboolean check_if_shortcut_used ( KiazmaShortcutEditor *ttt, gpointer data ) {
	int i;
	int pos;
	guint this_key;
	guint key;
	GtkWidget *confirm;
	GdkModifierType this_mod;
	GdkModifierType mod;

	pos = GPOINTER_TO_INT ( data );
	kiazma_shortcut_editor_get_value ( ttt, &this_mod, &this_key );

	for ( i = 0; i < S_MAX_SHORTCUTS; i++ ) {
		if ( i == pos )
			continue;

		kiazma_shortcut_editor_get_value ( KIAZMA_SHORTCUT_EDITOR ( ConfPanel.shortcuts [ i ] ), &mod, &key );

		if ( this_mod == mod && this_key == key ) {
			confirm = gtk_dialog_new_with_buttons ( _( "Confirm" ), GTK_WINDOW ( Graph.main_win ), GTK_DIALOG_MODAL,
		                                       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, GTK_STOCK_OK, GTK_RESPONSE_OK, NULL );

			gtk_box_pack_start ( GTK_BOX ( GTK_DIALOG ( confirm )->vbox ),
			                     gtk_label_new ( _( "Shortcut is already used. Overwrite?" ) ), FALSE, FALSE, 0 );

			gtk_widget_show_all ( confirm );

			if ( gtk_dialog_run ( GTK_DIALOG ( confirm ) ) == GTK_RESPONSE_OK )
				kiazma_shortcut_editor_set_value ( KIAZMA_SHORTCUT_EDITOR ( ConfPanel.shortcuts [ i ] ), 0, 0 );
			else
				kiazma_shortcut_editor_set_value ( ttt, 0, 0 );

			gtk_widget_destroy ( confirm );
		}
	}

	return FALSE;
}

/**
	Check in the local list of actions the one named as specified

	@param	name		The name for the action to look for

	@return			Reference to the action named as desided, or
				NULL if no actions with that name are find
*/
static ActionDesc* get_action_in_local_list ( gchar *name ) {
	GList *iter;
	ActionDesc *a;

	for ( iter = g_list_first ( ConfPanel.actions_list ); iter; iter = g_list_next ( iter ) ) {
		a = ( ActionDesc* ) iter->data;
		if ( a->name && strcmp ( a->name, name ) == 0 )
			return a;
	}

	return NULL;
}

/**
	Build the panel used to edit or add an action

	@param	name		Will be setted with the widget where the name
				of the action will be placed from the user

	@return			The GtkDialog which include fields for the
				managment of the action
*/
static GtkWidget* edit_dialog ( GtkWidget **name ) {
	GtkWidget *dialog;
	GtkWidget *main_box;
	GtkWidget *box;
	GtkWidget *label;
	GtkSizeGroup *sg;

	dialog = gtk_dialog_new_with_buttons ( _( "Edit Action Properties" ), GTK_WINDOW ( Graph.main_win ),
	                                       GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_MODAL, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
	                                       GTK_STOCK_OK, GTK_RESPONSE_OK, NULL );

	gtk_dialog_set_default_response ( GTK_DIALOG ( dialog ), GTK_RESPONSE_OK );
	main_box = GTK_DIALOG ( dialog )->vbox;

	sg = gtk_size_group_new ( GTK_SIZE_GROUP_HORIZONTAL );

	box = gtk_hbox_new ( FALSE, 1 );
	label = gtk_label_new ( _( "Name: " ) );
	gtk_size_group_add_widget ( sg, label );
	gtk_box_pack_start ( GTK_BOX ( box ), label, FALSE, FALSE, 0 );
	*name = gtk_entry_new ();
	gtk_box_pack_start ( GTK_BOX ( box ), *name, TRUE, TRUE, 0 );
	gtk_box_pack_start ( GTK_BOX ( main_box ), box, FALSE, FALSE, 0 );

	gtk_widget_show_all ( dialog );
	return dialog;
}

/**
*/
gboolean put_action_in_list ( gchar **name ) {
	gboolean ret					= TRUE;
	GtkWidget *rename_dialog;
	GtkWidget *new_name;
	GtkTreeIter child;

	while ( ret && get_action_in_local_list ( *name ) ) {
		new_name = gtk_entry_new ();
		gtk_entry_set_text ( GTK_ENTRY ( new_name ), *name );

		rename_dialog = kiazma_input_dialog_new ( _( "Action Name Duplicate" ),
						_( "The action overlap the name of an already existing one.\n"
						"Please specify a different name for the action "
						"or cancel the operation to skip it" ),
						GTK_MESSAGE_WARNING, GTK_BUTTONS_OK_CANCEL, new_name, NULL );

		gtk_widget_show_all ( rename_dialog );

		if ( gtk_dialog_run ( GTK_DIALOG ( rename_dialog ) ) == GTK_RESPONSE_OK ) {
			g_free ( *name );
			*name = g_strdup ( gtk_entry_get_text ( GTK_ENTRY ( new_name ) ) );
		}
		else
			ret = FALSE;

		/*
			new_name is destroyed together the whole dialog
		*/
		gtk_widget_destroy ( rename_dialog );
	}

	if ( ret ) {
		gtk_tree_store_append ( ConfPanel.actions_model, &child, NULL );
		gtk_tree_store_set ( ConfPanel.actions_model, &child, 0, *name, -1 );
	}

	return ret;
}

/**
	Import a new action from a file generated with the export_action()
	routine

	@param	button		Button pressed to import an action file
	@param	useless		NULL
*/
static void import_action ( GtkButton *button, gpointer useless ) {
	int n					= 0;
	gchar *path;
	GtkWidget *chooser;
	GList *list;
	GList *iter;
	GError *error				= NULL;
	GIOChannel *file;
	ActionDesc *act;

	chooser = gtk_file_chooser_dialog_new ( _( "Choose Action Files" ), GTK_WINDOW ( Graph.main_win ), GTK_FILE_CHOOSER_ACTION_OPEN,
	                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL );
	gtk_file_chooser_set_current_folder ( GTK_FILE_CHOOSER ( chooser ), getenv ( "HOME" ) );
	gtk_file_chooser_set_local_only ( GTK_FILE_CHOOSER ( chooser ), TRUE );

	if ( gtk_dialog_run ( GTK_DIALOG ( chooser ) ) == GTK_RESPONSE_ACCEPT ) {
		path = gtk_file_chooser_get_filename ( GTK_FILE_CHOOSER ( chooser ) );
		file = g_io_channel_new_file ( path, "r", &error );
		g_free ( path );

		if ( file == NULL ) {
			xfce_err ( _( "Unable to open the actions file: %s" ), error->message );
			return;
		}

		list = read_actions_from_file ( file );
		g_io_channel_unref ( file );

		if ( g_list_length ( list ) == 0 )
			xfce_warn ( _( "No new actions have been loaded" ) );

		else {
			for ( iter = g_list_first ( list ); iter; iter = g_list_next ( iter ) ) {
				act = ( ActionDesc* ) iter->data;

				if ( put_action_in_list ( &( act->name ) ) )
					n++;

				else {
					action_destroy ( act );
					list = g_list_delete_link ( list, iter );
				}
			}

			if ( n != 0 ) {
				ConfPanel.actions_list = g_list_concat ( ConfPanel.actions_list, list );
				xfce_info ( _( "%d new actions have been loaded" ), n );
			}
			else
				xfce_info ( _( "No new actions have been loaded" ) );
		}
	}

	gtk_widget_destroy ( chooser );
}

/**
	Exports the selected action in a separated file, suitable for the
	import_action() routine, so to be copied on a different machine or
	for backup

	@param	button		Button pressed to export the selected action
	@param	useless		NULL
*/
static void export_action ( GtkButton *button, gpointer useless ) {
	FILE *file;
	gchar *name;
	gchar *path;
	GtkWidget *chooser;
	GtkTreeSelection *list_selector;
	GtkTreeIter iter;
	ActionDesc *action;

	list_selector = gtk_tree_view_get_selection ( GTK_TREE_VIEW ( ConfPanel.actions_view ) );
	if ( gtk_tree_selection_get_selected ( list_selector, ( GtkTreeModel** ) &( ConfPanel.actions_model ), &iter ) ) {
		gtk_tree_model_get ( GTK_TREE_MODEL ( ConfPanel.actions_model ), &iter, 0, &name, -1 );
		action = get_action_in_local_list ( name );
		g_free ( name );
	}
	else
		return;

	chooser = gtk_file_chooser_dialog_new ( _( "Choose Exporting Files" ), GTK_WINDOW ( Graph.main_win ), GTK_FILE_CHOOSER_ACTION_SAVE,
	                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL );
	gtk_file_chooser_set_current_folder ( GTK_FILE_CHOOSER ( chooser ), getenv ( "HOME" ) );
	gtk_file_chooser_set_local_only ( GTK_FILE_CHOOSER ( chooser ), TRUE );

	if ( gtk_dialog_run ( GTK_DIALOG ( chooser ) ) == GTK_RESPONSE_ACCEPT ) {
		path = gtk_file_chooser_get_filename ( GTK_FILE_CHOOSER ( chooser ) );
		file = fopen ( path, "a" );
		g_free ( path );

		if ( file == NULL ) {
			xfce_err ( _( "Unable to open file for writing: %s" ), strerror ( errno ) );
			return;
		}

		write_action_to_file ( action, file );
		fclose ( file );
		xfce_info ( _( "Action written to file" ) );
	}

	gtk_widget_destroy ( chooser );
}

/**
	Callback assigned to the button dedicated to the creation of a new
	action

	@param	button		Button pressed to add an action
	@param	useless		NULL
*/
static void add_action ( GtkButton *button, gpointer useless ) {
	gchar *name;
	GtkWidget *dialog;
	GtkWidget *wname;
	ActionDesc *action;

	dialog = edit_dialog ( &wname );

	while ( 1 ) {
		if ( gtk_dialog_run ( GTK_DIALOG ( dialog ) ) == GTK_RESPONSE_OK ) {
			name = g_strdup ( gtk_entry_get_text ( GTK_ENTRY ( wname ) ) );

			if ( strlen ( name ) == 0 ) {
				xfce_err ( _( "No action name provided" ) );
				continue;
			}

			else {
				if ( put_action_in_list ( &name ) ) {
					action = g_new0 ( ActionDesc, 1 );
					action->name = name;
					ConfPanel.actions_list = g_list_append ( ConfPanel.actions_list, action );
				}
				else
					g_free ( name );

				break;
			}
		}
	}

	gtk_widget_destroy ( dialog );
}

/**
	Guess a valid unique name for an action

	@return			An unique name to assign as name for an
				action. The string is dinamically allocated
*/
gchar* generate_casual_action_name () {
	int i			= 0;
	gchar *ret;

	ret = g_strdup_printf ( "action_%d", i );
	while ( get_action_in_local_list ( ret ) ) {
		g_free ( ret );
		i++;
		ret = g_strdup_printf ( "action_%d", i );
	}

	return ret;
}

/**
	Callback invoked when a different action is selected in the main
	list, updates the frame where details are
*/
static void update_selected () {
	UINT64 meta;
	gchar *name;
	gchar *command;
	ActionDesc *action;
	ActionConstraintDesc *desc;

	action = get_action_in_local_list ( ConfPanel.action_current );

	if ( action ) {
		if ( action->name )
			g_free ( action->name );

		name = ( gchar* ) gtk_entry_get_text ( GTK_ENTRY ( ConfPanel.action_name ) );

		if ( strlen ( name ) )
			action->name = g_strdup ( name );

		else {
			xfce_warn ( _( "Undefined name for previous action" ) );
			action->name = generate_casual_action_name ();
		}

		if ( action->command ) {
			g_free ( action->command );
			action->command = NULL;
		}
		command = ( gchar* ) gtk_entry_get_text ( GTK_ENTRY ( ConfPanel.action_command ) );
		if ( strlen ( command ) )
			action->command = g_strdup ( command );

		if ( ConfPanel.action_manipulated ) {
			if ( action->constraints )
				action_destroy_constraints ( action );

			if ( kiazma_metadata_collector_get_first ( KIAZMA_METADATA_COLLECTOR ( ConfPanel.action_metas ),
									&meta, &name ) ) {
				do {
					desc = g_new0 ( ActionConstraintDesc, 1 );
					desc->meta = meta;
					desc->value = name;
					action->constraints = g_list_prepend ( action->constraints, desc );
				} while ( kiazma_metadata_collector_get_next ( KIAZMA_METADATA_COLLECTOR ( ConfPanel.action_metas ),
										&meta, &name ) );

				action->constraints = g_list_reverse ( action->constraints );
			}
		}

		action->in_terminal = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( ConfPanel.action_term ) );
	}
}

/**
*/
static void edit_action ( GtkTreeSelection *select, gpointer useless ) {
	gchar *name;
	GtkTreeIter iter;
	GtkTreeSelection *list_selector;
	GList *loop;
	ActionDesc *action;
	ActionConstraintDesc *desc;

	if ( GTK_WIDGET_IS_SENSITIVE ( GTK_WIDGET ( ConfPanel.action_frame ) ) ) {
		if ( ConfPanel.action_current )
			update_selected ();
	}
	else
		gtk_widget_set_sensitive ( ConfPanel.action_frame, TRUE );

	ConfPanel.action_manipulated = FALSE;

	if ( ConfPanel.action_current ) {
		g_free ( ConfPanel.action_current );
		ConfPanel.action_current = NULL;
	}

	list_selector = gtk_tree_view_get_selection ( GTK_TREE_VIEW ( ConfPanel.actions_view ) );
	if ( gtk_tree_selection_get_selected ( list_selector, ( GtkTreeModel** ) &( ConfPanel.actions_model ), &iter ) ) {
		gtk_tree_model_get ( GTK_TREE_MODEL ( ConfPanel.actions_model ), &iter, 0, &name, -1 );

		action = get_action_in_local_list ( name );

		if ( name ) {
			ConfPanel.action_current = g_strdup ( name );
			gtk_entry_set_text ( GTK_ENTRY ( ConfPanel.action_name ), name );
			g_free ( name );
		}

		if ( action->command )
			gtk_entry_set_text ( GTK_ENTRY ( ConfPanel.action_command ), action->command );
		else
			gtk_entry_set_text ( GTK_ENTRY ( ConfPanel.action_command ), "" );

		if ( action->constraints ) {
			kiazma_metadata_collector_reset ( KIAZMA_METADATA_COLLECTOR ( ConfPanel.action_metas ) );
			for ( loop = g_list_first ( action->constraints ); loop; loop = g_list_next ( loop ) ) {
				desc = ( ActionConstraintDesc* ) loop->data;
				kiazma_metadata_collector_add_item ( KIAZMA_METADATA_COLLECTOR ( ConfPanel.action_metas ),
									desc->meta, desc->value );
			}
		}
		else
			kiazma_metadata_collector_reset ( KIAZMA_METADATA_COLLECTOR ( ConfPanel.action_metas ) );

		gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( ConfPanel.action_term ), action->in_terminal );
	}
}

/**
	Callback which remove the currently selected action

	@param	button		Button pressed to remove the action
	@param	useless		NULL
*/
static void remove_action ( GtkButton *button, gpointer useless ) {
	gchar *name;
	GtkTreeIter child;
	GtkTreeModel **mod;
	GtkTreeSelection *list_selector;
	ActionDesc *action;

	mod = ( GtkTreeModel** ) &( ConfPanel.actions_model );
	list_selector = gtk_tree_view_get_selection ( GTK_TREE_VIEW ( ConfPanel.actions_view ) );

	if ( gtk_tree_selection_get_selected ( list_selector, mod, &child ) ) {
		gtk_tree_model_get ( GTK_TREE_MODEL ( ConfPanel.actions_model ), &child, 0, &name, -1 );
		action = get_action_in_local_list ( name );
		if ( action ) {
			ConfPanel.actions_list = g_list_remove ( ConfPanel.actions_list, action );
			action_destroy ( action );
		}
		g_free ( name );

		gtk_tree_store_remove ( ConfPanel.actions_model, &child );

		if ( gtk_tree_selection_get_selected ( list_selector, mod, &child ) == FALSE )
			gtk_widget_set_sensitive ( ConfPanel.action_frame, FALSE );
	}
}

/**
	Invoked when the set of constraints assigned to an action is
	modified, raise the ConfPanel::action_manipulated flag which permits
	to skip useless operation when updating internal informations

	@param	collector	Collector where a constraint for the selected
				action were added or removed
	@param	useless		NULL
*/
static void constraints_modified ( KiazmaMetadataCollector *collector, gpointer useless ) {
	ConfPanel.action_manipulated = TRUE;
}

/**
	Enable or disable the actions' setup frame when the button to
	enable / disable this feature is toggled

	@param	toggle		Toggle button which changed the status,
				enabling or disabling actions
	@param	useless		NULL
*/
static void enable_disable_actions ( GtkToggleButton *toggle, gpointer useless ) {
	if ( gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( ConfPanel.enable_actions ) ) )
		gtk_widget_set_sensitive ( ConfPanel.actions_conf, TRUE );
	else
		gtk_widget_set_sensitive ( ConfPanel.actions_conf, FALSE );
}

/**
	Build the page for the configuration panel where to setup actions

	@return			The page to add in the main notebook
*/
static GtkWidget* do_actions_panel () {
	GtkWidget *whole;
	GtkWidget *box;
	GtkWidget *container;
	GtkWidget *label;
	GList *iter;
	GtkTreeIter child;
	GtkTreeSelection *list_selector;
	ActionDesc *act;

	whole = gtk_vbox_new ( FALSE, 2 );

	box = kiazma_frame_new ( _( "General" ) );

	ConfPanel.enable_actions = gtk_check_button_new ();
	gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( ConfPanel.enable_actions ), Confs.enable_actions );
	g_signal_connect ( G_OBJECT ( ConfPanel.enable_actions ), "toggled", G_CALLBACK ( enable_disable_actions ), NULL );
	kiazma_frame_add ( KIAZMA_FRAME ( box ), _( "Enable Action " ), ConfPanel.enable_actions );

	ConfPanel.action_command_terminal = gtk_entry_new_with_max_length ( MAX_TERMINAL_COMMAND_STRING_LEN );
	gtk_entry_set_text ( GTK_ENTRY ( ConfPanel.action_command_terminal ), Confs.action_in_terminal_command );
	kiazma_frame_add ( KIAZMA_FRAME ( box ), _( "Terminal Command " ), ConfPanel.action_command_terminal );

	gtk_box_pack_start ( GTK_BOX ( whole ), box, FALSE, FALSE, 0 );

	ConfPanel.actions_conf = gtk_hpaned_new ();
	gtk_box_pack_start ( GTK_BOX ( whole ), ConfPanel.actions_conf, TRUE, TRUE, 0 );

	/*** lista azioni ***/

	container = gtk_vbox_new ( FALSE, 3 );
	gtk_container_set_border_width ( GTK_CONTAINER ( container ), 2 );
	gtk_paned_pack1 ( GTK_PANED ( ConfPanel.actions_conf ), container, FALSE, TRUE );

	ConfPanel.actions_model = gtk_tree_store_new ( 1, G_TYPE_STRING );
	ConfPanel.actions_list = action_get_all ();

	for ( iter = g_list_first ( ConfPanel.actions_list ); iter; iter = g_list_next ( iter ) ) {
		act = iter->data;
		gtk_tree_store_append ( ConfPanel.actions_model, &child, NULL );
		gtk_tree_store_set ( ConfPanel.actions_model, &child, 0, act->name, -1 );
	}

	ConfPanel.actions_view = gtk_tree_view_new_with_model ( ( GtkTreeModel* ) ConfPanel.actions_model );
	gtk_widget_set_size_request ( ConfPanel.actions_view, 200, 300 );
	list_selector = gtk_tree_view_get_selection ( GTK_TREE_VIEW ( ConfPanel.actions_view ) );
	gtk_tree_selection_set_mode ( list_selector, GTK_SELECTION_BROWSE );
	gtk_tree_view_insert_column_with_attributes ( GTK_TREE_VIEW ( ConfPanel.actions_view ), -1, _( "Names" ),
							gtk_cell_renderer_text_new (), "text", 0, NULL );
	g_signal_connect ( G_OBJECT ( list_selector ), "changed", G_CALLBACK ( edit_action ), NULL );
	gtk_box_pack_start ( GTK_BOX ( container ), ConfPanel.actions_view, TRUE, TRUE, 0 );

	label = gtk_button_new_from_stock ( GTK_STOCK_ADD );
	g_signal_connect ( G_OBJECT ( label ), "clicked", G_CALLBACK ( add_action ), NULL );
	gtk_box_pack_start ( GTK_BOX ( container ), label, FALSE, FALSE, 0 );

	label = gtk_button_new_from_stock ( GTK_STOCK_OPEN );
	g_signal_connect ( G_OBJECT ( label ), "clicked", G_CALLBACK ( import_action ), NULL );
	gtk_box_pack_start ( GTK_BOX ( container ), label, FALSE, FALSE, 0 );

	/*** pannello edit ***/

	container = gtk_vbox_new ( FALSE, 0 );
	gtk_paned_pack2 ( GTK_PANED ( ConfPanel.actions_conf ), container, FALSE, TRUE );

	ConfPanel.action_frame = kiazma_frame_new ( _( "Action Properties" ) );
	gtk_box_pack_start ( GTK_BOX ( container ), ConfPanel.action_frame, FALSE, FALSE, 0 );

	ConfPanel.action_name = gtk_entry_new ();
	kiazma_frame_add ( KIAZMA_FRAME ( ConfPanel.action_frame ), _( "Name " ), ConfPanel.action_name );

	ConfPanel.action_metas = kiazma_metadata_collector_new ();
	g_signal_connect ( G_OBJECT ( ConfPanel.action_metas ), "item_added", G_CALLBACK ( constraints_modified ), NULL );
	g_signal_connect ( G_OBJECT ( ConfPanel.action_metas ), "item_removed", G_CALLBACK ( constraints_modified ), NULL );
	kiazma_frame_free_add ( KIAZMA_FRAME ( ConfPanel.action_frame ), ConfPanel.action_metas );

	ConfPanel.action_command = gtk_entry_new ();
	kiazma_frame_add ( KIAZMA_FRAME ( ConfPanel.action_frame ), _( "Command " ), ConfPanel.action_command );

	ConfPanel.action_term = gtk_check_button_new ();
	kiazma_frame_add ( KIAZMA_FRAME ( ConfPanel.action_frame ), _( "Run in Terminal " ), ConfPanel.action_term );

	label = kiazma_button_stock_new ( GTK_STOCK_DELETE );
	g_signal_connect ( G_OBJECT ( label ), "clicked", G_CALLBACK ( remove_action ), NULL );
	kiazma_frame_add ( KIAZMA_FRAME ( ConfPanel.action_frame ), _( "Delete this item " ), label );

	label = kiazma_button_stock_new ( GTK_STOCK_CONVERT );
	g_signal_connect ( G_OBJECT ( label ), "clicked", G_CALLBACK ( export_action ), NULL );
	kiazma_frame_add ( KIAZMA_FRAME ( ConfPanel.action_frame ), _( "Export this item " ), label );

	gtk_widget_set_sensitive ( ConfPanel.action_frame, FALSE );

	if ( Confs.enable_actions )
		gtk_widget_set_sensitive ( ConfPanel.actions_conf, TRUE );
	else
		gtk_widget_set_sensitive ( ConfPanel.actions_conf, FALSE );

	ConfPanel.action_current = NULL;
	return whole;
}

/**
	Costruisce e mostra il pannello di configurazione globale
	dell'applicazione
*/
void show_preferences () {
	int i;
	guint key;
	gchar *conf_path;
	FILE *file;
	GtkWidget *dialog;
	GtkWidget *elements_box;
	GtkWidget *frame;
	GtkWidget *notebook;
	GtkWidget *box;
	GdkModifierType mod;

	/**
		Quando vengono aggiunte nuove scorciatoie da tastiera
		aggiungere in questa funzione la dicitura che deve apparire
		nell'editor
	*/
	gchar *shortcut_labels [] = {
					_( "Add File: " ),
					_( "Remove File: " ),
					_( "File Info: " ),
					_( "Select All: " ),
					_( "Open New Tab: " ),
					_( "Close Tab: " ),
					_( "Prev Tab: " ),
					_( "Next Tab: " ),
					_( "Switch to Query Bar: " ),
					_( "Switch to Search Bar: " ),
					_( "Show History: " ),
					_( "Bookmark Current Query: " ),
					_( "Open Preferences Panel: " ),
					_( "Open Query Editor: " )
				};

	dialog = gtk_dialog_new_with_buttons ( _( "Synapse Preferences" ), GTK_WINDOW ( Graph.main_win ),
	                                       GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_MODAL, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
	                                       GTK_STOCK_OK, GTK_RESPONSE_OK, NULL );

	gtk_dialog_set_default_response ( GTK_DIALOG ( dialog ), GTK_RESPONSE_OK );

	elements_box = GTK_DIALOG ( dialog )->vbox;
	gtk_box_set_spacing ( GTK_BOX ( elements_box ), 10 );

	notebook = gtk_notebook_new ();
	gtk_box_pack_start ( GTK_BOX ( elements_box ), notebook, FALSE, FALSE, 0 );

	/******* global conf *******/

	box = gtk_vbox_new ( FALSE, 5 );
	gtk_notebook_append_page ( GTK_NOTEBOOK ( notebook ), box, gtk_label_new ( _( "Global Conf" ) ) );

	frame = kiazma_frame_new ( _( "Behaviour" ) );
	gtk_box_pack_start ( GTK_BOX ( box ), frame, FALSE, FALSE, 0 );

	ConfPanel.ask_remove = gtk_check_button_new ();
	gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( ConfPanel.ask_remove ), Confs.ask_remove_confirm );
	kiazma_frame_add ( KIAZMA_FRAME ( frame ), _( "Ask confirm for delete " ), ConfPanel.ask_remove );

	ConfPanel.show_tooltips = gtk_check_button_new ();
	gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( ConfPanel.show_tooltips ), Confs.show_tooltips );
	kiazma_frame_add ( KIAZMA_FRAME ( frame ), _( "Show tooltips " ), ConfPanel.show_tooltips );

	/******* shortcuts *******/

	box = gtk_vbox_new ( FALSE, 5 );
	gtk_notebook_append_page ( GTK_NOTEBOOK ( notebook ), box, gtk_label_new ( _( "Shortcuts" ) ) );

	frame = kiazma_frame_new ( _( "Keyboard Shortcut" ) );
	gtk_box_pack_start ( GTK_BOX ( box ), frame, FALSE, FALSE, 0 );

	for ( i = 0; i < S_MAX_SHORTCUTS; i++ ) {
		ConfPanel.shortcuts [ i ] = kiazma_shortcut_editor_new ();
		kiazma_shortcut_editor_set_value ( KIAZMA_SHORTCUT_EDITOR ( ConfPanel.shortcuts [ i ] ),
		                                    Confs.shortcuts [ i ].modifier, Confs.shortcuts [ i ].key );
		g_signal_connect ( G_OBJECT ( ConfPanel.shortcuts [ i ] ), "shortcut-changed",
		                   G_CALLBACK ( check_if_shortcut_used ), GINT_TO_POINTER ( i ) );

		kiazma_frame_add ( KIAZMA_FRAME ( frame ), shortcut_labels [ i ], ConfPanel.shortcuts [ i ] );
	}

	/******* actions *******/

	gtk_notebook_append_page ( GTK_NOTEBOOK ( notebook ), do_actions_panel (), gtk_label_new ( _( "Actions" ) ) );

	/******* fine *******/

	gtk_widget_show_all ( dialog );

	if ( gtk_dialog_run ( GTK_DIALOG ( dialog ) ) == GTK_RESPONSE_OK ) {
		memset ( &Confs, 0, sizeof ( Configurations ) );
		conf_path = get_system_path ( DEFAULT_NAME_FOR_CONFIG_FILE );

		Confs.ask_remove_confirm = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( ConfPanel.ask_remove ) );

		Confs.show_tooltips = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( ConfPanel.show_tooltips ) );
		enable_tooltips ();

		Confs.enable_actions = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( ConfPanel.enable_actions ) );

		( void ) snprintf ( Confs.action_in_terminal_command, MAX_TERMINAL_COMMAND_STRING_LEN,
					"%s", gtk_entry_get_text ( GTK_ENTRY ( ConfPanel.action_command_terminal ) ) );

		for ( i = 0; i < S_MAX_SHORTCUTS; i++ ) {
			kiazma_shortcut_editor_get_value ( KIAZMA_SHORTCUT_EDITOR ( ConfPanel.shortcuts [ i ] ), &mod, &key );
			Confs.shortcuts [ i ].modifier = mod;
			Confs.shortcuts [ i ].key = key;
		}

		file = fopen ( conf_path, "w" );
		fwrite ( &Confs, sizeof ( Configurations ), 1, file );
		fclose ( file );
		g_free ( conf_path );

		if ( ConfPanel.action_current )
			update_selected ();

		action_replace ( ConfPanel.actions_list );
	}
	else
		action_destroy_list ( ConfPanel.actions_list );

	gtk_widget_destroy ( dialog );
}

/**
	Carica la configurazione di Synapse, riempiendo l'apposita struttura.
	Attualmente i parametri vengono letti da un file (binario, per
	comodita'), ma una volta che saranno definite le modalita' avanzate di
	accesso ad Hyppocampus i parametri potrebbero essere fissati
	direttamente all'interno del filesystem stesso
*/
void load_config () {
	gchar *conf_path;
	gchar *path;
	FILE *file;
	struct stat sbuf;

	memset ( &Confs, 0, sizeof ( Configurations ) );
	conf_path = get_system_path ( DEFAULT_NAME_FOR_CONFIG_FILE );

	if ( access ( conf_path, F_OK ) ||
			stat ( conf_path, &sbuf ) ||
			sbuf.st_size != sizeof ( Configurations ) ||
			( file = fopen ( conf_path, "r" ) ) == NULL ) {

		Confs.ask_remove_confirm = TRUE;
		Confs.show_tooltips = TRUE;
		Confs.enable_actions = TRUE;
		( void ) snprintf ( Confs.action_in_terminal_command, MAX_TERMINAL_COMMAND_STRING_LEN, "%s", "xterm -e" );

		Confs.shortcuts [ S_ADD_FILE ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_ADD_FILE ].key = GDK_o;

		Confs.shortcuts [ S_REMOVE_FILE ].modifier = 0;
		Confs.shortcuts [ S_REMOVE_FILE ].key = GDK_Delete;

		Confs.shortcuts [ S_FILE_INFO ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_FILE_INFO ].key = GDK_i;

		Confs.shortcuts [ S_SELECT_ALL ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_SELECT_ALL ].key = GDK_a;

		Confs.shortcuts [ S_OPEN_NEW_TAB ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_OPEN_NEW_TAB ].key = GDK_t;

		Confs.shortcuts [ S_CLOSE_CURRENT_TAB ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_CLOSE_CURRENT_TAB ].key = GDK_w;

		Confs.shortcuts [ S_PREV_TAB ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_PREV_TAB ].key = GDK_Page_Up;

		Confs.shortcuts [ S_NEXT_TAB ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_NEXT_TAB ].key = GDK_Page_Down;

		Confs.shortcuts [ S_SWITCH_ADDRESS_BAR ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_SWITCH_ADDRESS_BAR ].key = GDK_l;

		Confs.shortcuts [ S_SWITCH_KEYWORD_SEARCH ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_SWITCH_KEYWORD_SEARCH ].key = GDK_f;

		Confs.shortcuts [ S_HISTORY ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_HISTORY ].key = GDK_h;

		Confs.shortcuts [ S_BOOKMARK_THIS ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_BOOKMARK_THIS ].key = GDK_d;

		Confs.shortcuts [ S_PREFERENCES_PANEL ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_PREFERENCES_PANEL ].key = GDK_p;

		Confs.shortcuts [ S_QUERY_EDITOR ].modifier = GDK_CONTROL_MASK;
		Confs.shortcuts [ S_QUERY_EDITOR ].key = GDK_e;

		path = g_strdup_printf ( "%s/%s", g_get_home_dir (), DEFAULT_PATH_FOR_CONFIG_FILE );
		mkdir ( path, 0777 );
		g_free ( path );
		file = fopen ( conf_path, "w" );
		fwrite ( &Confs, sizeof ( Configurations ), 1, file );
	}

	else
		fread ( &Confs, sizeof ( Configurations ), 1, file );

	fclose ( file );
	g_free ( conf_path );

	read_bookmarks ();
	g_random_set_seed ( time ( NULL ) );
}
