/*  LibKiazma 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file	kiazma_time.c
*/

#include "libkiazma.h"
#include "kiazma_time.h"

/**
	@internal

	List of signals emitted by the KiazmaPermissions
*/
enum {
	KIAZMA_TIME_CHANGED,									/**< Emitted when the value is changed */
	KIAZMA_TIME_END_SIGNALS									/**< End of list */
};

static guint KiazmaTimeSignals [ KIAZMA_TIME_END_SIGNALS ];					/**< Array filled with the identifier of signals registered by the widget */

/**
	Init the GTK class rappresenting the KiazmaTime

	@param	klass	The KiazmaTime to init from beginning
*/
static void kiazma_time_class_init ( KiazmaTimeClass *klass ) {
	GObjectClass *gobject_class;

	gobject_class = ( GObjectClass* ) klass;

	KiazmaTimeSignals [ KIAZMA_TIME_CHANGED ] = g_signal_new ( g_intern_static_string ( "changed" ),
						G_TYPE_FROM_CLASS ( gobject_class ), G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
						g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0 );
}

/**
	Update the graphical rappresentation of the date from the saved date
	inside the widget

	@param	ttt	The KiazmaTime to upgrade
*/
static void update_graphic_from_time ( KiazmaTime *ttt ) {
	char date_string [ 100 ];
	struct tm now;

	localtime_r ( &( ttt->date ), &now );
	strftime ( date_string, 100, "%x %X", &now );
	gtk_expander_set_label ( GTK_EXPANDER ( ttt ), date_string );

	gtk_calendar_select_month ( GTK_CALENDAR ( ttt->calendar ), now.tm_mon, now.tm_year + 1900 );
	gtk_calendar_select_day ( GTK_CALENDAR ( ttt->calendar ), now.tm_mday );

	kiazma_hour_set_value ( KIAZMA_HOUR ( ttt->hour ), ttt->date );
}

/**
	Update the date saved from the widget, getting her from the graphical
	elements manipulated by the user

	@param	ttt	The KiazmaTime to upgrade
*/
static void update_time_from_graphic ( KiazmaTime *ttt ) {
	char date_string [ 100 ];
	struct tm now;
	time_t tim;
	guint y;
	guint m;
	guint d;

	tim = kiazma_hour_get_value ( KIAZMA_HOUR ( ttt->hour ) );
	localtime_r ( &tim, &now );

	gtk_calendar_get_date ( GTK_CALENDAR ( ttt->calendar ), &y, &m, &d );
	now.tm_year = y - 1900;
	now.tm_mon = m;
	now.tm_mday = d;

	ttt->date = mktime ( &now );

	strftime ( date_string, 100, "%x %X", &now );
	gtk_expander_set_label ( GTK_EXPANDER ( ttt ), date_string );

	g_signal_emit ( ttt, KiazmaTimeSignals [ KIAZMA_TIME_CHANGED ], 0 );
}

/**
	Callback invoked when the expander is opened or closed

	@param	ttt	The expander activated
	@param	unused	Useless
	@param	useless	Useless
*/
static void expander_action ( GObject *ttt, GParamSpec *unused, gpointer useless ) {
	if ( gtk_expander_get_expanded ( GTK_EXPANDER ( ttt ) ) ) {
		update_graphic_from_time ( KIAZMA_TIME ( ttt ) );
		gtk_widget_show_all ( GTK_WIDGET ( ttt ) );
	}

	else
		update_time_from_graphic ( KIAZMA_TIME ( ttt ) );
}

/**
	Init the GTK widgets that compose the KiazmaTime

	@param	ttt	The KiazmaTime to init from beginning
*/
static void kiazma_time_init ( KiazmaTime *ttt ) {
	GtkWidget *box;

	ttt->date = time ( NULL );

	gtk_expander_set_expanded ( GTK_EXPANDER ( ttt ), FALSE );
	g_signal_connect ( G_OBJECT ( ttt ), "notify::expanded", G_CALLBACK ( expander_action ), NULL );

	box = gtk_vbox_new ( FALSE, 2 );
	gtk_container_add ( GTK_CONTAINER ( ttt ), box );

	ttt->calendar = gtk_calendar_new ();
	gtk_box_pack_start ( GTK_BOX ( box ), ttt->calendar, TRUE, TRUE, 0 );

	ttt->hour = kiazma_hour_new ();
	gtk_box_pack_start ( GTK_BOX ( box ), ttt->hour, TRUE, TRUE, 0 );

	update_graphic_from_time ( ttt );
}

/**
	Return the GType of the widget KiazmaTime

	@return		The type of widget for KiazmaTime
*/
GType kiazma_time_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaTimeClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_time_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaTime ),
			0,
			( GInstanceInitFunc ) kiazma_time_init,
		};

		ttt_type = g_type_register_static ( GTK_TYPE_EXPANDER, "KiazmaTime", &ttt_info, 0 );
	}

	return ttt_type;
}

/**
	Create a new KiazmaTime

	@return		A new KiazmaTime
*/
GtkWidget* kiazma_time_new () {
	return GTK_WIDGET ( g_object_new ( kiazma_time_get_type (), NULL ) );
}

/**
	Reset the value into a KiazmaTime to the default

	@param	synp	The KiazmaTime to reset
*/
void kiazma_time_clear ( KiazmaTime *synp ) {
	kiazma_time_set_value ( synp, time ( NULL ) );
}

/**
	Used to retrive the value manipulated with a KiazmaTime

	@param	synp	The KiazmaTime from which get the value

	@return		The time expressed from the value inside the widget
*/
time_t kiazma_time_get_value ( KiazmaTime *synp ) {
	return synp->date;
}

/**
	To init the value inside a KiazmaTime

	@param	synp	The KiazmaTime where to set the value
	@param	value	The value to set into the widget, expressed as Unix
			time
*/
void kiazma_time_set_value ( KiazmaTime *synp, time_t value ) {
	synp->date = value;
}
