/*  LibKiazma 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file	kiazma_permissions.c

	The KiazmaPermissions is a widget which permits to define the
	permissions mask for a file: is a 3x3 matrix of check boxes, to define
	read/write/execute permissions for owner, group and others
*/

#include "libkiazma.h"
#include "kiazma_permissions.h"

#include <sys/stat.h>

/**
	@internal

	List of signals emitted by the KiazmaPermissions
*/
enum {
	KIAZMA_PERMISSIONS_CHANGED,								/**< Emitted when the value is changed */
	KIAZMA_PERMISSIONS_END_SIGNALS								/**< End of list */
};

static guint KiazmaPermissionsSignals [ KIAZMA_PERMISSIONS_END_SIGNALS ];			/**< Array filled with the identifier of signals registered by the widget */

/**
	@internal

	Init the GTK class rappresenting the KiazmaPermissions

	@param	klass	The KiazmaPermissionsClass to init from beginning
*/
static void kiazma_permissions_class_init ( KiazmaPermissionsClass *klass ) {
	GObjectClass *gobject_class;

	gobject_class = ( GObjectClass* ) klass;

	KiazmaPermissionsSignals [ KIAZMA_PERMISSIONS_CHANGED ] = g_signal_new ( g_intern_static_string ( "changed" ),
						G_TYPE_FROM_CLASS ( gobject_class ), G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
						g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0 );
}

/**
	@internal

	Emits the signal "changed" when the value inside the widget is changed

	@param	check	The check button toggled
	@param	ttt	The KiazmaPermissions which emits the signal
*/
static void value_changed ( GtkToggleButton *check, KiazmaPermissions *ttt ) {
	g_signal_emit ( ttt, KiazmaPermissionsSignals [ KIAZMA_PERMISSIONS_CHANGED ], 0 );
}

/**
	@internal

	Init the GTK widgets that compose the KiazmaPermissions

	@param	ttt	The KiazmaPermissions to init from beginning
*/
static void kiazma_permissions_init ( KiazmaPermissions *ttt ) {
	gint i,j;

	gtk_table_resize ( GTK_TABLE ( ttt ), 4, 4 );
	gtk_table_set_homogeneous ( GTK_TABLE ( ttt ), FALSE );

	gtk_table_attach_defaults ( GTK_TABLE ( ttt ), gtk_label_new ( _( "Owner" ) ), 1, 2, 0, 1 );
	gtk_table_attach_defaults ( GTK_TABLE ( ttt ), gtk_label_new ( _( "Group" ) ), 2, 3, 0, 1 );
	gtk_table_attach_defaults ( GTK_TABLE ( ttt ), gtk_label_new ( _( "Others" ) ), 3, 4, 0, 1 );

	gtk_table_attach_defaults ( GTK_TABLE ( ttt ), gtk_label_new ( _( "Read" ) ), 0, 1, 1, 2 );
	gtk_table_attach_defaults ( GTK_TABLE ( ttt ), gtk_label_new ( _( "Write" ) ), 0, 1, 2, 3 );
	gtk_table_attach_defaults ( GTK_TABLE ( ttt ), gtk_label_new ( _( "Exec" ) ), 0, 1, 3, 4 );

	for ( i = 0; i < 3; i++ )
		for ( j = 0; j < 3; j++ ) {
			ttt->checks [ i ] [ j ] = gtk_check_button_new ();
			g_signal_connect ( G_OBJECT ( ttt->checks [ i ] [ j ] ), "toggled", G_CALLBACK ( value_changed ), ttt );
			gtk_table_attach_defaults ( GTK_TABLE ( ttt ), ttt->checks [ i ] [ j ],
			                            i + 1, i + 2, j + 1, j + 2 );
		}
}

/**
	Return the GType of the widget KiazmaPermissions

	@return		The type of widget for KiazmaPermissions
*/
GType kiazma_permissions_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaPermissionsClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_permissions_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaPermissions ),
			0,
			( GInstanceInitFunc ) kiazma_permissions_init,
		};

		ttt_type = g_type_register_static ( GTK_TYPE_TABLE, "KiazmaPermissions", &ttt_info, 0 );
	}

	return ttt_type;
}

/**
	Create a new KiazmaPermissions

	@return		A new KiazmaPermissions, rappresenting no permissions
*/
GtkWidget* kiazma_permissions_new () {
	return GTK_WIDGET ( g_object_new ( kiazma_permissions_get_type (), NULL ) );
}

/**
	Reset the value into a KiazmaPermissions to the default

	@param	synp	The KiazmaPermissions to reset
*/
void kiazma_permissions_clear ( KiazmaPermissions *synp ) {
	int i, j;

	for ( i = 0; i < 3; i++ )
		for ( j = 0; j < 3; j++ )
			gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( synp->checks [ i ] [ j ] ), FALSE );
}

/**
	Used to retrive the value manipulated with a KiazmaPermissions

	@param	synp	The KiazmaPermissions from which get the value

	@return		The permissions mask expressed from the value inside
			the widget
*/
mode_t kiazma_permissions_get_value ( KiazmaPermissions *synp ) {
	mode_t ret	= 0;

	ret |= S_IFREG;

	if ( gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 0 ] [ 0 ] ) ) )
		ret |= S_IRUSR;
	if ( gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 0 ] [ 1 ] ) ) )
		ret |= S_IRGRP;
	if ( gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 0 ] [ 2 ] ) ) )
		ret |= S_IROTH;

	if ( gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 1 ] [ 0 ] ) ) )
		ret |= S_IWUSR;
	if ( gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 1 ] [ 1 ] ) ) )
		ret |= S_IWGRP;
	if ( gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 1 ] [ 2 ] ) ) )
		ret |= S_IWOTH;

	if ( gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 2 ] [ 0 ] ) ) )
		ret |= S_IXUSR;
	if ( gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 2 ] [ 1 ] ) ) )
		ret |= S_IXGRP;
	if ( gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 2 ] [ 2 ] ) ) )
		ret |= S_IXOTH;

	return ret;
}

/**
	To init the value inside a KiazmaPermissions

	@param	synp	The KiazmaPermissions where to set the value
	@param	value	The value to set into the widget, expressed as
			permissions mask
*/
void kiazma_permissions_set_value ( KiazmaPermissions *synp, mode_t value ) {
	kiazma_permissions_clear ( synp );

	if ( value & S_IRUSR )
		gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 0 ] [ 0 ] ), TRUE );
	if ( value & S_IRGRP )
		gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 0 ] [ 1 ] ), TRUE );
	if ( value & S_IROTH )
		gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 0 ] [ 2 ] ), TRUE );

	if ( value & S_IWUSR )
		gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 1 ] [ 0 ] ), TRUE );
	if ( value & S_IWGRP )
		gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 1 ] [ 1 ] ), TRUE );
	if ( value & S_IWOTH )
		gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 1 ] [ 2 ] ), TRUE );

	if ( value & S_IXUSR )
		gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 2 ] [ 0 ] ), TRUE );
	if ( value & S_IXGRP )
		gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 2 ] [ 1 ] ), TRUE );
	if ( value & S_IXOTH )
		gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( synp->checks [ 2 ] [ 2 ] ), TRUE );
}
