/*  LibKiazma 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file	kiazma_oms.c
*/

#include "kiazma_oms.h"
#include <vte/vte.h>
#include <pty.h>

/**
	@internal

	Minimal dimension of the selected area to be considered a new
	OnMouse-Shell area. If the area is smaller, the event is ignored
*/
#define MIN_OMS_AREA_SIZE			20

/**
	@internal

	List of signals registered for the OnMouse-Shell
*/
enum {
	KIAZMA_OMS_STARTING,						/**< Emitted when the main window of OnMouse-Shell is created: to the signal is attached a copy of the widget inside the window */
	KIAZMA_OMS_END_TEXT,						/**< Emitted when the main window of OnMouse-Shell is closed from textual mode: to the signal is attached a GString with contents edited by the user */
	KIAZMA_OMS_END_DRAW,						/**< Emitted when the main window of OnMouse-Shell is closed from drawing mode: to the signal is attached a GdkPixmap with contents edited by the user */
	KIAZMA_OMS_END_SHELL,						/**< Emitted when the main window of OnMouse-Shell is closed from shell mode: to the signal is attached none */
	KIAZMA_OMS_LAST_SIGNAL						/**< End of list */
};

typedef enum {
	KIAZMA_OMS_TEXT,
	KIAZMA_OMS_SHELL,
	KIAZMA_OMS_DRAW
} KIAZMA_OMS_MODE;

gboolean		DoingOMS;
gint			Rubberband_X1;
gint			Rubberband_X2;
gint			Rubberband_Y1;
gint			Rubberband_Y2;
gulong			ClosureHandle;

KIAZMA_OMS_MODE		OMSMode;
GtkWidget		*ParentWidget;
GtkWidget		*OMSInput;
static GString		*OMSTextModeContext				= NULL;
static pid_t		OMSShellModePID					= 0;
static guint		OMSSignals [ KIAZMA_OMS_LAST_SIGNAL ]		= { 0 };

/**
	Destroy the OnMouse-Shell main window, and emit a
	"oms_selection_closed" signal with, in attachement, the data
	elaborated in the input widget. Dependly from the OMS mode, is
	returned:
	- KIAZMA_OMS_SHELL : NULL
	- KIAZMA_OMS_TEXT : the text inserted in the widget, wrapped in a
		GString structure
	- KIAZMA_OMS_DRAW : the pixmap traced by the user
	To avoid data overlapping, it is suggested to increase the
	reference number of data catched with the signal
*/
void kiazma_oms_destroy () {
	gchar *text;
	GtkTextBuffer *buffer;
	GtkTextIter start;
	GtkTextIter end;

	DoingOMS = FALSE;
	g_return_if_fail ( OMSInput );
	gtk_grab_remove ( OMSInput );
	gtk_widget_hide ( OMSInput );

	switch ( OMSMode ) {
		case KIAZMA_OMS_SHELL:
			if ( OMSShellModePID )
				kill ( OMSShellModePID, SIGKILL );

			OMSShellModePID = 0;
			g_signal_emit ( ParentWidget, OMSSignals [ KIAZMA_OMS_END_SHELL ], 0, NULL );
			break;

		case KIAZMA_OMS_TEXT:
			buffer = gtk_text_view_get_buffer ( GTK_TEXT_VIEW ( OMSInput ) );
			gtk_text_buffer_get_start_iter ( buffer, &start );
			gtk_text_buffer_get_end_iter ( buffer, &end );
			text = gtk_text_buffer_get_text ( GTK_TEXT_BUFFER ( buffer ), &start, &end, TRUE );

			if ( OMSTextModeContext )
				g_object_unref ( OMSTextModeContext );

			OMSTextModeContext = g_string_new ( text );
			g_signal_emit ( ParentWidget, OMSSignals [ KIAZMA_OMS_END_TEXT ], 0, OMSTextModeContext );
			break;

		case KIAZMA_OMS_DRAW:
			g_signal_emit ( ParentWidget, OMSSignals [ KIAZMA_OMS_END_DRAW ], 0,
					kiazma_drawing_area_get_draw ( KIAZMA_DRAWING_AREA ( OMSInput ) ) );
			break;

		default:
			break;
	}

	gtk_widget_destroy ( OMSInput );
	OMSInput = NULL;
	gtk_widget_queue_draw ( ParentWidget );
}

/**
	@internal

	@param	oms_widget
	@param	event
	@param	useless

	@return
*/
static gboolean kiazma_oms_remove_by_button ( GtkWidget *oms_widget, GdkEventButton *event, gpointer useless ) {
	if ( DoingOMS ) {
		kiazma_oms_destroy ();
		DoingOMS = FALSE;
		return FALSE;
	}
	else
		return TRUE;
}

/**
	@internal

	@param	widget
	@param	event

	@return
*/
static gboolean kiazma_icon_stack_view_oms_button_press_while_drawing ( GtkWidget *widget, GdkEventButton *event ) {
	if ( OMSInput && event->window != OMSInput->window ) {
		kiazma_oms_remove_by_button ( ParentWidget, event, NULL );
		g_signal_handler_disconnect ( G_OBJECT ( ParentWidget ), ClosureHandle );
	}

	return FALSE;
}

/**
	@internal
*/
static void shell_mode_exited ( VteTerminal *vteterminal, gpointer user_data ) {
	kiazma_oms_destroy ();
	g_signal_handler_disconnect ( G_OBJECT ( ParentWidget ), ClosureHandle );
}

/**
	@internal

	Create the interactive shell in the KIAZMA_OMS_SHELL mode

	@return			A VTE_TERMINAL running an instance for bash
*/
static GtkWidget* create_shell () {
	gchar *path;
	int fdm;
	int fds;
	char *selection			= NULL;
	gchar *vte_args []		= { "bash", "-r", NULL };
	gchar *vte_env []		= { "PS1='# '", NULL };
	GtkWidget *shell;

	if ( IS_KIAZMA_ICON_STACKED_VIEW ( ParentWidget ) ) {
		gint id_col;
		UINT64 id;
		GtkTreeIter element;
		GtkTreePath *path;
		GtkTreeModel *model;
		GList *list;
		GList *iter;
		GString *str;
		KiazmaIconStackedView *icon_view;

		icon_view = KIAZMA_ICON_STACKED_VIEW ( ParentWidget );
		list = kiazma_icon_stacked_view_get_selected_items ( icon_view );

		if ( list ) {
			id_col = kiazma_icon_stacked_view_get_id_column ( icon_view );
			model = kiazma_icon_stacked_view_get_model ( icon_view );
			str = g_string_new ( "" );

			for ( iter = g_list_first ( list ); iter; iter = g_list_next ( iter ) ) {
				path = ( GtkTreePath* ) iter->data;
				gtk_tree_model_get_iter ( model, &element, path );
				gtk_tree_model_get ( model, &element, id_col, &id, -1 );
				g_string_append_printf ( str, "%llu ", id );
			}

			g_list_foreach ( list, gtk_tree_path_free, NULL );
			g_list_free ( list );

			selection = g_string_free ( str, FALSE );
		}
	}

	shell = vte_terminal_new ();

	/*
		Inspired by a post published Jan 18, 2007 on Nabble Forums
		Thanks to John W. Eaton
	*/

	/**
		@warning	Here is adopted a workaround trick to reduce the size of the prompt of the shell.
				The conventional way is to change value of the environmental variable "PS1", but
				when executing bash she read the real configuration, which override the variable
				with value written in bashrc . So here I run the command to export the variable
				*after* executing bash, writing the PTY file descriptor
	*/

	openpty ( &fdm, &fds, 0, 0, 0 );
	dup2 ( fds, 0 );
	dup2 ( fds, 1 );
	dup2 ( fds, 2 );
	vte_terminal_set_pty ( VTE_TERMINAL ( shell ), fdm );

	path = g_find_program_in_path ( "bash" );
	OMSShellModePID = vte_terminal_fork_command ( VTE_TERMINAL ( shell ), path, vte_args, vte_env,
							hyppo_vfs_get_mounted_path (), FALSE, FALSE, FALSE );

	dprintf ( fdm, "%s", "export PS1='# '\n" );

	if ( selection ) {
		dprintf ( fdm, "export SELECTION=\"%s\"\n", selection );
		g_free ( selection );
	}

	dprintf ( fdm, "%s", "clear\n" );

	g_signal_connect ( G_OBJECT ( shell ), "child-exited", G_CALLBACK ( shell_mode_exited ), NULL );
	g_free ( path );
	return shell;
}

static void do_oms_common () {
	gint start_x;
	gint start_y;
	gint width;
	gint height;
	GtkAllocation allocation;

	start_x = MIN ( Rubberband_X1, Rubberband_X2 );
	start_y = MIN ( Rubberband_Y1, Rubberband_Y2 );
	width = ABS ( Rubberband_X1 - Rubberband_X2 );
	height = ABS ( Rubberband_Y1 - Rubberband_Y2 );

	ClosureHandle = g_signal_connect ( G_OBJECT ( ParentWidget ), "button_press_event",
				G_CALLBACK ( kiazma_icon_stack_view_oms_button_press_while_drawing ), NULL );

	/**
		@todo	Make the background transparent (perhaps with cairo?)
	*/

	allocation.x = start_x;
	allocation.y = start_y;
	allocation.width = width;
	allocation.height = height;
	gtk_widget_set_parent ( OMSInput, GTK_WIDGET ( ParentWidget ) );
	gtk_widget_size_allocate ( OMSInput, &allocation );
	gtk_widget_show_all ( OMSInput );

	gtk_grab_remove ( ParentWidget );
	gtk_grab_add ( OMSInput );

	g_signal_emit ( ParentWidget, OMSSignals [ KIAZMA_OMS_STARTING ], 0, OMSInput );
}

/**
*/
static gboolean do_oms_textual ( GtkMenuItem *item, gpointer data ) {
	OMSInput = gtk_text_view_new ();
	OMSMode = KIAZMA_OMS_TEXT;
	gtk_text_view_set_wrap_mode ( GTK_TEXT_VIEW ( OMSInput ), GTK_WRAP_CHAR );
	do_oms_common ();
	return FALSE;
}

/**
*/
static gboolean do_oms_graphic ( GtkMenuItem *item, gpointer data ) {
	OMSInput = kiazma_drawing_area_new ();
	OMSMode = KIAZMA_OMS_DRAW;
	do_oms_common ();
	return FALSE;
}

/**
*/
static gboolean do_oms_shell ( GtkMenuItem *item, gpointer data ) {
	OMSInput = create_shell ();
	OMSMode = KIAZMA_OMS_SHELL;
	do_oms_common ();
	return FALSE;
}

/**
*/
static void create_chooice_menu () {
	GtkWidget *menu;
	GtkWidget *item;

	menu = gtk_menu_new ();

	item = gtk_menu_item_new_with_label ( _( "Textual Mode" ) );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( do_oms_textual ), NULL );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	item = gtk_menu_item_new_with_label ( _( "Drawing Mode" ) );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( do_oms_graphic ), NULL );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	item = gtk_menu_item_new_with_label ( _( "Shell Mode" ) );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( do_oms_shell ), NULL );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	gtk_menu_popup ( GTK_MENU ( menu ), NULL, NULL, NULL, NULL, 0, GDK_CURRENT_TIME );
}

/**
	Stops the OnMouse-Shell rubberband and substitutes the rectangle
	drawed during the selection with an appropriate input window

	@param	x2		Final X coordinate of the selection rubberband
	@param	y2		Final Y coordinate of the selection rubberband
*/
void kiazma_oms_stop_rubberbanding ( gint x2, gint y2 ) {
	gint width;
	gint height;

	g_return_if_fail ( DoingOMS );
	Rubberband_X2 = x2;
	Rubberband_Y2 = y2;

	width = ABS ( Rubberband_X1 - Rubberband_X2 );
	height = ABS ( Rubberband_Y1 - Rubberband_Y2 );

	if ( width < MIN_OMS_AREA_SIZE || height < MIN_OMS_AREA_SIZE )
		DoingOMS = FALSE;
	else
		create_chooice_menu ();
}

/**
	Starts an OnMouse-Shell selection

	@param	x		Starting X of the selection
	@param	y		Starting Y of the selection
*/
void kiazma_oms_start_rubberbanding ( gint x, gint y ) {
	if ( DoingOMS )
		kiazma_oms_destroy ();

	DoingOMS = TRUE;

	Rubberband_X1 = x;
	Rubberband_Y1 = y;
	Rubberband_X2 = x;
	Rubberband_Y2 = y;
	gtk_grab_add ( ParentWidget );
}

/**
	Used to know if the KiazmaOMS is active or not

	@return			TRUE if the KiazmaOMS is activated, so if it
				is in the rubberbanding state or its main
				window have been already opened, FALSE
				otherwise
*/
gboolean kiazma_oms_doing () {
	return DoingOMS;
}

/**
	@internal
*/
void _gtk_marshal_BOOLEAN__OBJECT ( GClosure *closure, GValue *return_value, guint n_param_values,
					const GValue *param_values, gpointer invocation_hint,
					gpointer marshal_data ) {

	typedef gboolean ( *GMarshalFunc_BOOLEAN__OBJECT ) ( gpointer data1, GObject *arg_1, gpointer data2 );

	register GMarshalFunc_BOOLEAN__OBJECT callback;
	register GCClosure *cc = ( GCClosure* ) closure;
	register gpointer data1, data2;
	gboolean v_return;

	g_return_if_fail ( return_value != NULL );
	g_return_if_fail ( n_param_values == 2 );

	if ( G_CCLOSURE_SWAP_DATA ( closure ) ) {
		data1 = closure->data;
		data2 = g_value_peek_pointer ( param_values + 0 );
	}
	else {
		data1 = g_value_peek_pointer ( param_values + 0 );
		data2 = closure->data;
	}

	callback = ( GMarshalFunc_BOOLEAN__OBJECT ) ( marshal_data ? marshal_data : cc->callback );

	v_return = callback ( data1, g_value_get_object ( param_values + 1 ), data2 );
	g_value_set_boolean ( return_value, v_return );
}

/**
	Init the interface to KiazmaOMS: must be called before all other
	functions

	@param	parent		Parent widget which will contains the OMS
				window
*/
void kiazma_oms_init ( GtkWidget *parent ) {
	ParentWidget = parent;
}

/**
	Init internal signals used by KiazmaOMS to operate with his parent
	widget

	@param	class		Parent class which implements the OMS
*/
void kiazma_oms_init_signals ( GObjectClass *class ) {
	OMSSignals [ KIAZMA_OMS_STARTING ] = g_signal_new ( g_intern_static_string ( "oms_selection_opened" ),
								G_TYPE_FROM_CLASS ( class ),
								G_SIGNAL_RUN_LAST, 0, NULL, NULL,
								_gtk_marshal_BOOLEAN__OBJECT,
								G_TYPE_BOOLEAN, 1, G_TYPE_POINTER );

	OMSSignals [ KIAZMA_OMS_END_TEXT ] = g_signal_new ( g_intern_static_string ( "oms_selection_text_closed" ),
								G_TYPE_FROM_CLASS ( class ),
								G_SIGNAL_RUN_LAST, 0, NULL, NULL,
								g_cclosure_marshal_VOID__OBJECT,
								G_TYPE_NONE, 1, G_TYPE_POINTER );

	OMSSignals [ KIAZMA_OMS_END_DRAW ] = g_signal_new ( g_intern_static_string ( "oms_selection_draw_closed" ),
								G_TYPE_FROM_CLASS ( class ),
								G_SIGNAL_RUN_LAST, 0, NULL, NULL,
								g_cclosure_marshal_VOID__OBJECT,
								G_TYPE_NONE, 1, G_TYPE_POINTER );

	OMSSignals [ KIAZMA_OMS_END_SHELL ] = g_signal_new ( g_intern_static_string ( "oms_selection_shell_closed" ),
								G_TYPE_FROM_CLASS ( class ),
								G_SIGNAL_RUN_LAST, 0, NULL, NULL,
								g_cclosure_marshal_VOID__OBJECT,
								G_TYPE_NONE, 1, G_TYPE_POINTER );
}
