/*  LibKiazma 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file	kiazma_metadata_set_chooser.c

	The KiazmaMetadataSetChooser is really similar to
	KiazmaMetadataChooser, with an addictional combo box which permits to
	specify the relation between the selected metadata and the specified
	value. This is used, for example, to identify an entire set (in the
	SQL meaning) of elements having a particular metadata assigned
*/

#include "libkiazma.h"
#include "kiazma_metadata_set_chooser.h"

/**
	@internal

	Init the class for KiazmaMetadataSetChooser

	@param	klass		Instance of the class to init
*/
static void kiazma_metadata_set_chooser_class_init ( KiazmaMetadataSetChooserClass *klass ) {
	return;
}

/**
	@internal

	Build the more appropriated widget to input the value to assign to
	the selected metadata

	@param	ttt		The KiazmaMetadataSetChooser where a metadata
				have been choosed
	@param	meta		Metadata choosed
*/
static void kiazma_metadata_set_chooser_set_value_widget ( KiazmaMetadataSetChooser *ttt, UINT64 meta ) {
	GtkWidget *value;

	value = gtk_bin_get_child ( GTK_BIN ( ttt->value ) );
	gtk_widget_destroy ( value );

	if ( meta == 0 )
		value = gtk_label_new ( _( "Choose a metadata" ) );
	else
		value = kiazma_metadata_wrapper_new ( meta );

	gtk_container_add ( GTK_CONTAINER ( ttt->value ), value );
	gtk_widget_show_all ( GTK_WIDGET ( ttt ) );
}

/**
	@internal

	Build the more appropriated combo box which permits to select the
	relation between the choosed metadata and the value specified

	@param	ttt		The KiazmaMetadataSetChooser where a metadata
				have been choosed
	@param	meta		Metadata choosed
*/
static void kiazma_metadata_set_chooser_set_types_set ( KiazmaMetadataSetChooser *ttt, UINT64 meta ) {
	GtkListStore *mod;
	GtkTreeIter iter;

	mod = ( GtkListStore* ) gtk_combo_box_get_model ( GTK_COMBO_BOX ( ttt->set_type ) );
	gtk_list_store_clear ( mod );

	if ( AllMetadata [ meta ].type & META_TYPE_CHAR ||
		AllMetadata [ meta ].type & META_TYPE_SHORT ||
		AllMetadata [ meta ].type & META_TYPE_INT ||
		AllMetadata [ meta ].type & META_TYPE_DOUBLE ||
		AllMetadata [ meta ].type & META_TYPE_FLOAT ||
		AllMetadata [ meta ].type & META_TYPE_LONG ||
		AllMetadata [ meta ].type & META_TYPE_LLONG ||
		AllMetadata [ meta ].type & META_TYPE_WCHAR ||
		AllMetadata [ meta ].type & META_TYPE_TIME ||
		AllMetadata [ meta ].type & META_TYPE_UID ||
		AllMetadata [ meta ].type & META_TYPE_GID ||
		AllMetadata [ meta ].type & META_TYPE_MODE ||
		AllMetadata [ meta ].type & META_TYPE_INTERNAL ) {

		gtk_list_store_append ( mod, &iter );
		gtk_list_store_set ( mod, &iter, 0, "=", 1, EQUAL_OP, -1 );

		gtk_list_store_append ( mod, &iter );
		gtk_list_store_set ( mod, &iter, 0, "<", 1, MINOR_OP, -1 );

		gtk_list_store_append ( mod, &iter );
		gtk_list_store_set ( mod, &iter, 0, ">", 1, MAJOR_OP, -1 );

		gtk_list_store_append ( mod, &iter );
		gtk_list_store_set ( mod, &iter, 0, "<=", 1, MINOR_EQUAL_OP, -1 );

		gtk_list_store_append ( mod, &iter );
		gtk_list_store_set ( mod, &iter, 0, ">=", 1, MAJOR_EQUAL_OP, -1 );

		gtk_list_store_append ( mod, &iter );
		gtk_list_store_set ( mod, &iter, 0, "!=", 1, DIFFER_OP, -1 );
	}
	else if ( AllMetadata [ meta ].type & META_TYPE_STRING ||
		AllMetadata [ meta ].type & META_TYPE_WSTRING ) {

		gtk_list_store_append ( mod, &iter );
		gtk_list_store_set ( mod, &iter, 0, "=", 1, MATCH_OP, -1 );

		gtk_list_store_append ( mod, &iter );
		gtk_list_store_set ( mod, &iter, 0, "?=", 1, CONTAIN_OP, -1 );

		gtk_list_store_append ( mod, &iter );
		gtk_list_store_set ( mod, &iter, 0, "=~", 1, REGEXP_OP, -1 );
	}
}

/**
	@internal

	Callback activated when a metadata is choosen from the first combo
	box into the widget

	@param	combo		The GtkComboBox which permits the selection
				of a metadata
	@param	ttt		Main KiazmaMetadataSetChooser

	@return			FALSE
*/
gboolean kiazma_metadata_set_chooser_choose ( GtkComboBox *combo, KiazmaMetadataSetChooser *ttt ) {
	UINT64 meta;

	meta = kiazma_metadata_selector_get ( KIAZMA_METADATA_SELECTOR ( ttt->meta ) );
	if ( meta ) {
		kiazma_metadata_set_chooser_set_types_set ( ttt, meta );
		kiazma_metadata_set_chooser_set_value_widget ( ttt, meta );
	}

	return FALSE;
}

/**
	@internal

	Builds and inits a new instance of KiazmaMetadataSetChooser

	@param	ttt		The instance to init
*/
static void kiazma_metadata_set_chooser_init ( KiazmaMetadataSetChooser *ttt ) {
	GtkCellRenderer *render_text;
	GtkListStore *x_set_type;

	gtk_container_set_border_width ( GTK_CONTAINER ( ttt ), 2 );
	gtk_box_set_homogeneous ( GTK_BOX ( ttt ), FALSE );

	ttt->meta = kiazma_metadata_selector_new ();
	g_signal_connect ( G_OBJECT ( ttt->meta ), "changed", G_CALLBACK ( kiazma_metadata_set_chooser_choose ), ttt );

	x_set_type = gtk_list_store_new ( 2, G_TYPE_STRING, G_TYPE_INT );

	ttt->set_type = gtk_combo_box_new_with_model ( ( GtkTreeModel* ) x_set_type );
	render_text = gtk_cell_renderer_text_new ();
	gtk_cell_layout_pack_start ( GTK_CELL_LAYOUT ( ttt->set_type ), render_text, TRUE );
	gtk_cell_layout_add_attribute ( GTK_CELL_LAYOUT ( ttt->set_type ), render_text, "text", 0 );

	ttt->value = gtk_frame_new ( NULL );
	gtk_container_set_border_width ( GTK_CONTAINER ( ttt->value ), 0 );
	gtk_frame_set_shadow_type ( GTK_FRAME ( ttt->value ), GTK_SHADOW_NONE );
	gtk_container_add ( GTK_CONTAINER ( ttt->value ), gtk_label_new ( _( "Choose a metadata" ) ) );

	gtk_box_pack_start ( GTK_BOX ( ttt ), ttt->meta, FALSE, FALSE, 0 );
	gtk_box_pack_start ( GTK_BOX ( ttt ), ttt->set_type, FALSE, FALSE, 0 );
	gtk_box_pack_start ( GTK_BOX ( ttt ), ttt->value, FALSE, FALSE, 0 );
}

/**
	Register and return the type for KiazmaMetadataSetChooser

	@return			The GType registered for
				KiazmaMetadataSetChooser
*/
GType kiazma_metadata_set_chooser_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaMetadataSetChooserClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_metadata_set_chooser_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaMetadataSetChooser ),
			0,
			( GInstanceInitFunc ) kiazma_metadata_set_chooser_init,
		};

		ttt_type = g_type_register_static ( GTK_TYPE_HBOX, "KiazmaMetadataSetChooser", &ttt_info, 0 );
	}

	return ttt_type;
}

/**
	Builds a new KiazmaMetadataSetChooser

	@return			A new KiazmaMetadataSetChooser allocated and
				inited
*/
GtkWidget* kiazma_metadata_set_chooser_new () {
	return GTK_WIDGET ( g_object_new ( kiazma_metadata_set_chooser_get_type (), NULL ) );
}

/**
	As kiazma_metadata_set_chooser_new(), but fills the widget with the
	whole set of available metadata identifiers

	@return			A new KiazmaMetadataSetChooser allocated and
				inited with all available metadata
*/
GtkWidget* kiazma_metadata_set_chooser_new_full () {
	UINT64 i;
	GtkWidget *item;

	item = kiazma_metadata_set_chooser_new ();

	for ( i = 0; i < METADATA_END; i++ )
		kiazma_metadata_set_chooser_add_meta ( KIAZMA_METADATA_SET_CHOOSER ( item ), i );

	return item;
}

/**
	Add a metadata to the internal list of the KiazmaMetadataSetChooser

	@param	chooser		The KiazmaMetadataSetChooser in which add the
				metadata
	@param	meta		The identifier of the new metadata
*/
void kiazma_metadata_set_chooser_add_meta ( KiazmaMetadataSetChooser *chooser, UINT64 meta ) {
	kiazma_metadata_selector_add_meta ( KIAZMA_METADATA_SELECTOR ( chooser->meta ), meta );
}

/**
	Remove a metadata from the internal list of the
	KiazmaMetadataSetChooser

	@param	chooser		The KiazmaMetadataSetChooser from which
				remove a metadata
	@param	meta		Metadata identifier to remove from the list
*/
void kiazma_metadata_set_chooser_remove_meta ( KiazmaMetadataSetChooser *chooser, UINT64 meta ) {
	kiazma_metadata_selector_remove_meta ( KIAZMA_METADATA_SELECTOR ( chooser->meta ), meta );
}

/**
	Init a KiazmaMetadataSetChooser with the specified values

	@param	chooser		KiazmaMetadataSetChooser to init
	@param	meta		Metadata identifier to select
	@param	op		Identifier of the relation between the
				selected metadata and the specified value
	@param	value		Value to assign to the metadata
*/
void kiazma_metadata_set_chooser_set ( KiazmaMetadataSetChooser *chooser, UINT64 meta,
					HYPPO_COMPARISON_OPERATOR op, gchar *value ) {

	HYPPO_COMPARISON_OPERATOR i;
	GtkWidget *value_widget;
	GtkTreeIter iter;
	GtkTreeModel *mod;

	kiazma_metadata_selector_set ( KIAZMA_METADATA_SELECTOR ( chooser->meta ), meta );
	kiazma_metadata_set_chooser_set_types_set ( chooser, meta );

	mod = gtk_combo_box_get_model ( GTK_COMBO_BOX ( chooser->set_type ) );
	gtk_tree_model_get_iter_first ( mod, &iter );

	do {
		gtk_tree_model_get ( mod, &iter, 1, &i, -1 );
		if ( op == i ) {
			gtk_combo_box_set_active_iter ( GTK_COMBO_BOX ( chooser->set_type ), &iter );
			break;
		}
	} while ( gtk_tree_model_iter_next ( mod, &iter ) );

	value_widget = gtk_bin_get_child ( GTK_BIN ( chooser->value ) );
	kiazma_metadata_wrapper_set_value ( KIAZMA_METADATA_WRAPPER ( value_widget ), value );
}

/**
	Retrive the selected information from a KiazmaMetadataSetChooser

	@param	chooser		KiazmaMetadataSetChooser from which get
				informations
	@param	meta		Will be filled with the identifier of the
				selected metadata
	@param	op		Will be filled with the identifier of the
				selected relational operator
	@param	value		Will be filled with the value assigned to the
				metadata

	@return			TRUE if the operation goes right, FALSE in
				case of error or if no metadata have been
				choosed into the KiazmaMetadataSetChooser
*/
gboolean kiazma_metadata_set_chooser_get ( KiazmaMetadataSetChooser *chooser, UINT64 *meta,
						HYPPO_COMPARISON_OPERATOR *op, gchar **value ) {

	UINT64 i_meta;
	GtkWidget *i_value;
	GtkTreeModel *mod;
	GtkTreeIter iter;

	i_meta = kiazma_metadata_selector_get ( KIAZMA_METADATA_SELECTOR ( chooser->meta ) );
	if ( i_meta == 0 )
		return FALSE;
	else
		*meta = i_meta;

	mod = gtk_combo_box_get_model ( GTK_COMBO_BOX ( chooser->set_type ) );
	if ( gtk_combo_box_get_active_iter ( GTK_COMBO_BOX ( chooser->set_type ), &iter ) )
		gtk_tree_model_get ( mod, &iter, 1, op, -1 );
	else
		return FALSE;

	i_value = gtk_bin_get_child ( GTK_BIN ( chooser->value ) );
	*value = kiazma_metadata_wrapper_get_value ( KIAZMA_METADATA_WRAPPER ( i_value ) );

	return TRUE;
}

/**
	Builds a textual rappresentation of the values inside the
	KiazmaMetadataSetChooser, e.g. "1 = 'foo'"

	@param	chooser		The KiazmaMetadataSetChooser from which
				build the string

	@return			A newly allocated string with the textual
				rappresentation
*/
gchar* kiazma_metadata_set_chooser_to_string ( KiazmaMetadataSetChooser *chooser ) {
	UINT64 meta;
	gchar *string;
	gchar *op;
	gchar *value;
	GtkWidget *i_value;
	GtkTreeModel *mod;
	GtkTreeIter iter;

	meta = kiazma_metadata_selector_get ( KIAZMA_METADATA_SELECTOR ( chooser->meta ) );
	if ( meta == 0 )
		return NULL;

	mod = gtk_combo_box_get_model ( GTK_COMBO_BOX ( chooser->set_type ) );
	if ( gtk_combo_box_get_active_iter ( GTK_COMBO_BOX ( chooser->set_type ), &iter ) )
		gtk_tree_model_get ( mod, &iter, 0, &op, -1 );
	else
		return NULL;

	i_value = gtk_bin_get_child ( GTK_BIN ( chooser->value ) );
	value = kiazma_metadata_wrapper_get_value ( KIAZMA_METADATA_WRAPPER ( i_value ) );
	if ( value == NULL )
		return NULL;

	if ( AllMetadata [ meta ].type & META_TYPE_STRING || AllMetadata [ meta ].type & META_TYPE_WSTRING )
		string = g_strdup_printf ( "%llu %s '%s'", meta, op, value );
	else
		string = g_strdup_printf ( "%llu %s %s", meta, op, value );

	return string;
}

/**
	Duplicate a KiazmaMetadataSetChooser

	@param	item		KiazmaMetadataSetChooser to duplicate

	@return			A new KiazmaMetadataSetChooser, copy of the
				passed one
*/
GtkWidget* kiazma_metadata_set_chooser_copy ( KiazmaMetadataSetChooser *item ) {
	UINT64 meta;
	UINT64 a_meta;
	gchar *value;
	HYPPO_COMPARISON_OPERATOR op;
	GtkWidget *copy;
	GtkTreeIter iter;
	GtkTreeModel *mod;
	KiazmaMetadataSetChooser *copia;

	if ( kiazma_metadata_set_chooser_get ( item, &meta, &op, &value ) ) {
		copy = kiazma_metadata_set_chooser_new ();
		copia = KIAZMA_METADATA_SET_CHOOSER ( copy );

		/**
			@todo	Is there a faster method to duplicate a GtkListStore
				without iter each single element?
		*/

		mod = gtk_combo_box_get_model ( GTK_COMBO_BOX ( KIAZMA_METADATA_SET_CHOOSER ( item )->meta ) );
		gtk_tree_model_get_iter_first ( mod, &iter );

		do {
			gtk_tree_model_get ( mod, &iter, 1, &a_meta, -1 );
			kiazma_metadata_set_chooser_add_meta ( copia, a_meta );
		} while ( gtk_tree_model_iter_next ( mod, &iter ) );

		kiazma_metadata_set_chooser_set ( copia, meta, op, value );
	}
	else
		copy = NULL;

	return copy;
}

/**
	Reset the values into a KiazmaMetadataSetChooser. The metadata list
	is untouched, only indexes of various combo box are manipulated

	@param	chooser		The KiazmaMetadataSetChooser to reset
*/
void kiazma_metadata_set_chooser_reset ( KiazmaMetadataSetChooser *chooser ) {
	GtkListStore *mod;

	gtk_combo_box_set_active ( GTK_COMBO_BOX ( chooser->meta ), 0 );
	kiazma_metadata_set_chooser_set_value_widget ( chooser, 0 );

	mod = ( GtkListStore* ) gtk_combo_box_get_model ( GTK_COMBO_BOX ( chooser->set_type ) );
	gtk_list_store_clear ( mod );
}
