/*
 * bltWinUtil.c --
 *
 *	This module contains WIN32 routines not included in the Tcl/Tk
 *	libraries.
 *
 * Copyright 1998 by Bell Labs Innovations for Lucent Technologies.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that the
 * copyright notice and warranty disclaimer appear in supporting documentation,
 * and that the names of Lucent Technologies any of their entities not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this software,
 * including all implied warranties of merchantability and fitness.  In no event
 * shall Lucent Technologies be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of use,
 * data or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or performance
 * of this software.  
 *
 */

#ifdef __CYGWIN32__
#define XFree X11XFree
#endif

#include <bltInt.h>
#include <X11/Xutil.h>
#include <X11/Xlib.h>

#include <windowsx.h>

#ifdef __CYGWIN32__
HPEN SelectPen (HDC dc, HPEN pen)
{
  return SelectObject (dc, pen);
}
BOOL DeletePen (HPEN pen)
{
  return DeleteObject (pen);
}

HBRUSH SelectBrush (HDC dc, HBRUSH brush)
{
  return SelectObject (dc, brush);
}

BOOL DeleteBrush (HBRUSH brush)
{
  return DeleteObject (brush);
}
#endif

static int tkpWinRopModes[] = {
    R2_BLACK,			/* GXclear */
    R2_MASKPEN,			/* GXand */
    R2_MASKPENNOT,		/* GXandReverse */
    R2_COPYPEN,			/* GXcopy */
    R2_MASKNOTPEN,		/* GXandInverted */
    R2_NOT,			/* GXnoop */
    R2_XORPEN,			/* GXxor */
    R2_MERGEPEN,		/* GXor */
    R2_NOTMERGEPEN,		/* GXnor */
    R2_NOTXORPEN,		/* GXequiv */
    R2_NOT,			/* GXinvert */
    R2_MERGEPENNOT,		/* GXorReverse */
    R2_NOTCOPYPEN,		/* GXcopyInverted */
    R2_MERGENOTPEN,		/* GXorInverted */
    R2_NOTMASKPEN,		/* GXnand */
    R2_WHITE			/* GXset */
};

static HPEN GCToPen _ANSI_ARGS_((HDC dc, GC gc));

void *
Blt_Calloc(unsigned int numElems, unsigned int sizeOfElem)
{
    void *array;
    unsigned int numBytes;

    numBytes = numElems * sizeOfElem;
    array = Tcl_Alloc(numBytes);
    memset(array, 0, numBytes);
    return array;
}

char *
Blt_StrDup(const char *string)
{
    unsigned int numBytes;
    char *array;

    numBytes = strlen(string) + 1;
    array = (char *)Tcl_Alloc(numBytes * sizeof(char));
    strcpy(array, string);
    return array;
}


double 
drand48() 
{
  return (double)rand() / (double)RAND_MAX;
}

void
srand48(seed)	
    long seed;
{ 
    srand(seed);
}

static int
GetPlatformId(void)
{
    static int platformId = 0;

    if (platformId == 0) {
	OSVERSIONINFO opsysInfo;

	opsysInfo.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	if (GetVersionEx(&opsysInfo)) {
	    platformId = opsysInfo.dwPlatformId;
	}
    }
    return platformId;
}


/*
 *----------------------------------------------------------------------
 *
 * XFree --
 * 
 *----------------------------------------------------------------------
 */
#undef XFree
void
XFree(char *ptr)
{
    free(ptr);
}

/*
 *----------------------------------------------------------------------
 *
 * XMaxRequestSize --
 * 
 *----------------------------------------------------------------------
 */
long
XMaxRequestSize(display)
    Display *display;
{
    return USHRT_MAX;
}

/*
 *----------------------------------------------------------------------
 *
 * XGetWindowAttributes --
 * 
 *----------------------------------------------------------------------
 */
int
XGetWindowAttributes(display, window, attrPtr)
    Display *display;
    Window window;
    XWindowAttributes *attrPtr;
{
    HWND hWnd;

    hWnd = (HWND)(window);
    attrPtr->map_state = IsUnviewable;
    if (IsWindowVisible(hWnd)) {
	RECT region;

	attrPtr->map_state = IsViewable;
	if (GetWindowRect(hWnd, &region)) {
	    attrPtr->x = region.left;
	    attrPtr->y = region.top;
	    attrPtr->width = region.right - region.left;
	    attrPtr->height= region.bottom - region.top;
	    return TRUE;
	}
    }
    return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * XLowerWindow --
 * 
 *----------------------------------------------------------------------
 */
void
XLowerWindow(display, window)
    Display* display;
    Window window;
{
    HWND hWnd;

    hWnd = Tk_GetHWND(window);
    display->request++;
    SetWindowPos(hWnd, HWND_BOTTOM, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
}
/*
 *----------------------------------------------------------------------
 *
 * XWarpPointer --
 * 
 *	If destWindow is None, moves the pointer by the offsets (destX, 
 *	destY) relative to the current position of the pointer.  
 *	If destWindow is a window, moves the pointer to the offsets 
 *	(destX, destY) relative to the origin of destWindow.  However, 
 *	if srcWindow is a window, the move only takes place if the window 
 *	srcWindow contains the pointer and if the specified rectangle of 
 *	srcWindow contains the pointer.
 *
 *	The srcX and srcY coordinates are relative to the origin of
 *	srcWindow.  If srcHeight is zero, it is replaced with the current
 *	height of srcWindow minus srcY.  If srcWidth is zero, it is
 *	replaced with the current width of srcWindow minus srcX.
 *
 *----------------------------------------------------------------------
 */
void
XWarpPointer(display, srcWindow, destWindow, srcX, srcY, srcWidth, srcHeight, 
	destX, destY)
    Display *display;
    Window srcWindow, destWindow;
    int srcX, srcY;
    unsigned int srcWidth, srcHeight;
    int destX, destY;
{
    HWND hWnd;
    POINT point;

    hWnd = Tk_GetHWND(destWindow);
    point.x = destX, point.y = destY;
    if (ClientToScreen(hWnd, &point)) {
        SetCursorPos(point.x, point.y);
    }
}


/*
 *----------------------------------------------------------------------
 *
 * GCToPen --
 *
 *	Set up the graphics port from the given GC.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The current port is adjusted.
 *
 *----------------------------------------------------------------------
 */
static HPEN
GCToPen(dc, gc)
    HDC dc;
    GC gc;
{
    DWORD lineAttrs, lineStyle;
    DWORD dashArr[12];
    DWORD *dashPtr;
    int numValues, lineWidth;
    LOGBRUSH lb;
    HPEN pen;

    numValues = 0;
    lineWidth = (gc->line_width < 1) ? 1 : gc->line_width;
    if ((gc->line_style == LineOnOffDash) || 
	(gc->line_style == LineDoubleDash)) {
	unsigned char *valueArr = (unsigned char *)(gc->dash_offset);
	
	numValues = (int)gc->dashes;
	if (valueArr == NULL) {
	    dashArr[0] = numValues;
	    numValues = 1;
	} else {
	    register int i;

	    for (i = 0; i < numValues; i++) {
		dashArr[i] = (DWORD)valueArr[i];
	    }
	}
    }
    switch(numValues) {
    case 0:
	lineStyle = PS_SOLID;
	break;
    case 3:
	lineStyle = PS_DASHDOT;
	break;
    case 4:
	lineStyle = PS_DASHDOTDOT;
	break;
    case 2:
    default:
	/* PS_DASH style dash length is too long. */ 
	lineStyle = PS_DOT;
	break;
    }

    lb.lbStyle = BS_SOLID;
    lb.lbColor = gc->foreground;
    lb.lbHatch = 0;		/* Value is ignored with style is BS_SOLID */
    
    lineAttrs = 0;
    switch (gc->cap_style) {
    case CapNotLast:
    case CapButt:
	lineAttrs |= PS_ENDCAP_FLAT; 
	break;
    case CapRound:
	lineAttrs |= PS_ENDCAP_ROUND; 
	break;
    default:
	lineAttrs |= PS_ENDCAP_SQUARE; 
	break;
    }
    switch (gc->join_style) {
    case JoinMiter: 
	lineAttrs |= PS_JOIN_MITER; 
	break;
    case JoinRound:
	lineAttrs |= PS_JOIN_ROUND; 
	break;
    default:
	lineAttrs |= PS_JOIN_BEVEL; 
	break;
    }
    pen = NULL;
    SetBkMode(dc, TRANSPARENT);
    if ((lineStyle == PS_SOLID) ||
	(GetPlatformId() == VER_PLATFORM_WIN32_NT)) {
	/* 
	 * If the line style is solid or we're running on NT, first
	 * try to use a geometric pen.  
	 */
	if (numValues > 0) {
	    lineStyle = PS_USERSTYLE;
	    dashPtr = dashArr;
	} else {
	    dashPtr = NULL;
	}
	pen = ExtCreatePen(PS_GEOMETRIC | lineAttrs | lineStyle, lineWidth, 
	   &lb, numValues, dashPtr);
    } else {
	/* Windows 95: we'll sacrifice thick lines for dashes. */
	pen = ExtCreatePen(PS_COSMETIC | lineAttrs |lineStyle, 1, &lb, 0, NULL);
    }
    if (pen == NULL) {
	OutputDebugString("Can't create pen: trying solid");
	pen = CreatePen(PS_COSMETIC | PS_SOLID, 1, gc->foreground);
    }
    assert(pen != NULL); 
    return pen;
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawRectangles --
 *
 *       Draws the outlines of the specified rectangles as if a
 *       five-point PolyLine protocol request were specified for each
 *       rectangle:
 *
 *             [x,y] [x+width,y] [x+width,y+height] [x,y+height]
 *             [x,y]
 *
 *      For the specified rectangles, these functions do not draw a
 *      pixel more than once.  XDrawRectangles draws the rectangles in
 *      the order listed in the array.  If rectangles intersect, the
 *      intersecting pixels are drawn multiple times.  Draws a
 *      rectangle.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws rectangles on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
XDrawRectangles(display, drawable, gc, rectArr, numRects)
    Display *display;
    Drawable drawable;
    GC gc;
    XRectangle *rectArr;
    int numRects;
{
    HPEN pen, oldPen;
    TkWinDCState state;
    HBRUSH brush, oldBrush;
    HDC dc;
    register XRectangle *rectPtr;
    register int i;

    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    pen = GCToPen(dc, gc);
    brush = GetStockObject(NULL_BRUSH);
    oldPen = SelectPen(dc, pen);
    oldBrush = SelectBrush(dc, brush);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    rectPtr = rectArr;
    for (i = 0; i < numRects; i++, rectPtr++) {
	Rectangle(dc, (int)rectPtr->x, (int)rectPtr->y,
	    (int)(rectPtr->x + rectPtr->width + 1),
	    (int)(rectPtr->y + rectPtr->height + 1));
    }
    DeletePen(SelectPen(dc, oldPen));
    DeleteBrush(SelectBrush(dc, oldBrush));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}


/*
 *----------------------------------------------------------------------
 *
 * DrawArc --
 *
 *	This procedure handles the rendering of drawn or filled
 *	arcs and chords.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Renders the requested arcs.
 *
 *----------------------------------------------------------------------
 */
static void
DrawArc(dc, arcMode, arcPtr, pen, brush)
    HDC dc;
    int arcMode;		/* Mode: either ArcChord or ArcPieSlice */
    XArc *arcPtr;
    HPEN pen;
    HBRUSH brush;
{
    int start, extent, clockwise;
    int xstart, ystart, xend, yend;
    double radian_start, radian_end, xr, yr;
    double dx, dy;

    start = arcPtr->angle1, extent = arcPtr->angle2;
    clockwise = (extent < 0);	/* Non-zero if clockwise */

    /*
     * Compute the absolute starting and ending angles in normalized radians.
     * Swap the start and end if drawing clockwise.
     */
    start = start % (64 * 360);
    if (start < 0) {
	start += (64 * 360);
    }
    extent = (start + extent) % (64 * 360);
    if (extent < 0) {
	extent += (64 * 360);
    }
    if (clockwise) {
	int tmp = start;
	start = extent;
	extent = tmp;
    }

#define XAngleToRadians(a) ((double)(a) / 64 * M_PI / 180);
    radian_start = XAngleToRadians(start);
    radian_end = XAngleToRadians(extent);

    /*
     * Now compute points on the radial lines that define the starting and
     * ending angles.  Be sure to take into account that the y-coordinate
     * system is inverted.
     */
    dx = arcPtr->width * 0.5;
    dy = arcPtr->height * 0.5;

    xr = arcPtr->x + dx;
    yr = arcPtr->y + dy;
    xstart = (int)((xr + cos(radian_start) * dx) + 0.5);
    ystart = (int)((yr + sin(-radian_start) * dy) + 0.5);
    xend = (int)((xr + cos(radian_end) * dx) + 0.5);
    yend = (int)((yr + sin(-radian_end) * dy) + 0.5);

    /*
     * Now draw a filled or open figure.  Note that we have to
     * increase the size of the bounding box by one to account for the
     * difference in pixel definitions between X and Windows.
     */

    if (brush == 0) {
	/*
	 * Note that this call will leave a gap of one pixel at the
	 * end of the arc for thin arcs.  We can't use ArcTo because
	 * it's only supported under Windows NT.
	 */
	Arc(dc, arcPtr->x, arcPtr->y, arcPtr->x + arcPtr->width + 1,
	    arcPtr->y + arcPtr->height + 1, xstart, ystart, xend, yend);
	/* FIXME: */
    } else {
	if (arcMode == ArcChord) {
	    Chord(dc, arcPtr->x, arcPtr->y, arcPtr->x + arcPtr->width + 1,
		arcPtr->y + arcPtr->height + 1, xstart, ystart, xend, yend);
	} else if (arcMode == ArcPieSlice) {
	    Pie(dc, arcPtr->x, arcPtr->y, arcPtr->x + arcPtr->width + 1,
		arcPtr->y + arcPtr->height + 1, xstart, ystart, xend, yend);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawArcs --
 *
 *	Draws multiple circular or elliptical arcs.  Each arc is
 *	specified by a rectangle and two angles.  The center of the
 *	circle or ellipse is the center of the rect- angle, and the
 *	major and minor axes are specified by the width and height.
 *	Positive angles indicate counterclock- wise motion, and
 *	negative angles indicate clockwise motion.  If the magnitude
 *	of angle2 is greater than 360 degrees, XDrawArcs truncates it
 *	to 360 degrees.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws an arc for each array element on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
XDrawArcs(display, drawable, gc, arcArr, numArcs)
    Display *display;
    Drawable drawable;
    GC gc;
    XArc *arcArr;
    int numArcs;
{
    HPEN pen, oldPen;
    register int i;
    HDC dc;
    TkWinDCState state;
    register XArc *arcPtr;

    display->request++;
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    pen = GCToPen(dc, gc);
    oldPen = SelectPen(dc, pen);
    for (arcPtr = arcArr, i = 0; i < numArcs; i++, arcPtr++) {
	DrawArc(dc, gc->arc_mode, arcPtr, pen, 0);
    }
    DeletePen(SelectPen(dc, oldPen));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

/*
 *----------------------------------------------------------------------
 *
 * XFillArcs --
 *
 *	Draw a filled arc.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws a filled arc for each array element on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
XFillArcs(display, drawable, gc, arcArr, numArcs)
    Display *display;
    Drawable drawable;
    GC gc;
    XArc *arcArr;
    int numArcs;
{
    HBRUSH brush, oldBrush;
    HPEN pen, oldPen;
    register int i;
    HDC dc;
    register XArc *arcPtr;
    TkWinDCState state;

    display->request++;
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    pen = GCToPen(dc, gc);
    oldPen = SelectPen(dc, pen);
    brush = CreateSolidBrush(gc->foreground);
    oldBrush = SelectBrush(dc, brush);
    for (arcPtr = arcArr, i = 0; i < numArcs; i++, arcPtr++) {
	DrawArc(dc, gc->arc_mode, arcPtr, pen, brush);
    }
    DeleteBrush(SelectBrush(dc, oldBrush));
    DeletePen(SelectPen(dc, oldPen));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}



/*
 *----------------------------------------------------------------------
 *
 * XDrawLines --
 *
 *	Draw connected lines.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Renders a series of connected lines.
 *
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 *
 * ConvertPoints --
 *
 *	Convert an array of X points to an array of Win32 points.
 *
 * Results:
 *	Returns the converted array of POINTs.
 *
 * Side effects:
 *	Allocates a block of memory that should not be freed.
 *
 *----------------------------------------------------------------------
 */

static POINT *
ConvertPoints(points, npoints, mode, bbox)
    XPoint *points;
    int npoints;
    int mode;			/* CoordModeOrigin or CoordModePrevious. */
    RECT *bbox;			/* Bounding box of points. */
{
    static POINT *winPoints = NULL; /* Array of points that is reused. */
    static int nWinPoints = -1;	    /* Current size of point array. */
    int i;

    /*
     * To avoid paying the cost of a malloc on every drawing routine,
     * we reuse the last array if it is large enough.
     */

    if (npoints > nWinPoints) {
	if (winPoints != NULL) {
	    ckfree((char *) winPoints);
	}
	winPoints = (POINT *) ckalloc(sizeof(POINT) * npoints);
	if (winPoints == NULL) {
	    nWinPoints = -1;
	    return NULL;
	}
	nWinPoints = npoints;
    }

    bbox->left = bbox->right = points[0].x;
    bbox->top = bbox->bottom = points[0].y;
    
    if (mode == CoordModeOrigin) {
	for (i = 0; i < npoints; i++) {
	    winPoints[i].x = points[i].x;
	    winPoints[i].y = points[i].y;
	    bbox->left = MIN(bbox->left, winPoints[i].x);
	    bbox->right = MAX(bbox->right, winPoints[i].x);
	    bbox->top = MIN(bbox->top, winPoints[i].y);
	    bbox->bottom = MAX(bbox->bottom, winPoints[i].y);
	}
    } else {
	winPoints[0].x = points[0].x;
	winPoints[0].y = points[0].y;
	for (i = 1; i < npoints; i++) {
	    winPoints[i].x = winPoints[i-1].x + points[i].x;
	    winPoints[i].y = winPoints[i-1].y + points[i].y;
	    bbox->left = MIN(bbox->left, winPoints[i].x);
	    bbox->right = MAX(bbox->right, winPoints[i].x);
	    bbox->top = MIN(bbox->top, winPoints[i].y);
	    bbox->bottom = MAX(bbox->bottom, winPoints[i].y);
	}
    }
    return winPoints;
}

void
XDrawLines(display, drawable, gc, points, npoints, mode)
    Display* display;
    Drawable drawable;
    GC gc;
    XPoint* points;
    int npoints;
    int mode;
{
    HPEN pen, oldPen;
    HBRUSH brush, oldBrush;
    TkWinDCState state;
    HDC dc;
    RECT rect;
    
    POINT *pointArr = ConvertPoints(points, npoints, mode, &rect);
    
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    pen = GCToPen(dc, gc);
    oldPen = SelectPen(dc, pen);
    brush = CreateSolidBrush(gc->foreground);
    oldBrush = SelectBrush(dc, brush);
    SetROP2(dc, tkpWinRopModes[gc->function]);

    Polyline (dc, pointArr, npoints);
    DeletePen(SelectPen(dc, oldPen));
    DeleteBrush(SelectBrush(dc, oldBrush));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}
typedef struct DrawInfo {
    HDC dc;
    int count;
    COLORREF color;
} DrawInfo;

static 
void CALLBACK
DrawDot(int x, int y,		/* Coordinates of point */
	LPARAM clientData)	/* Line information */
{
    DrawInfo *infoPtr = (DrawInfo *)clientData;

    infoPtr->count++;
    if (infoPtr->count & 0x1) {
	SetPixelV(infoPtr->dc, x, y, infoPtr->color);
    }
}

void
XDrawLine(Display *display, 
	  Drawable drawable,	
	  GC gc,		
	  int x1, int y1, 
	  int x2, int y2)	
{
    TkWinDCState state;
    HDC dc;
    
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    if ((gc->line_style != LineSolid) && (gc->dashes == 1)) {
	/* Handle dotted lines specially */ 
	DrawInfo drawInfo;

	drawInfo.dc = dc;
	drawInfo.count = 0;
	drawInfo.color = gc->foreground;
	LineDDA(x1, y1, x2, y2, DrawDot, (LPARAM)&drawInfo);
    } else {
	HPEN pen, oldPen;
	HBRUSH brush, oldBrush;

	pen = GCToPen(dc, gc);
	oldPen = SelectPen(dc, pen);
	brush = CreateSolidBrush(gc->foreground);
	oldBrush = SelectBrush(dc, brush);
	SetROP2(dc, tkpWinRopModes[gc->function]);
	MoveToEx(dc, x1, y1, (LPPOINT)NULL);
	LineTo(dc, x2, y2);
	DeletePen(SelectPen(dc, oldPen));
	DeleteBrush(SelectBrush(dc, oldBrush));
    }
    TkWinReleaseDrawableDC(drawable, dc, &state);
}


/*
 *----------------------------------------------------------------------
 *
 * XDrawSegments --
 *
 *	Draws multiple, unconnected lines. For each segment, draws a
 *	line between (x1, y1) and (x2, y2).  It draws the lines in the
 *	order listed in the array of XSegment structures and does not
 *	perform joining at coincident endpoints.  For any given line,
 *	does not draw a pixel more than once. If lines intersect, the
 *	intersecting pixels are drawn multiple times.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws unconnected line segments on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
XDrawSegments(display, drawable, gc, segArr, numSegments)
    Display *display;
    Drawable drawable;
    GC gc;
    XSegment *segArr;
    int numSegments;
{
    register int i;
    HDC dc;
    register XSegment *segPtr;
    TkWinDCState state;

    display->request++;
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    if ((gc->line_style != LineSolid) && (gc->dashes == 1)) {
	/* Handle dotted lines specially */ 
	DrawInfo drawInfo;

	drawInfo.dc = dc;
	drawInfo.color = gc->foreground;
	for (segPtr = segArr, i = 0; i < numSegments; i++, segPtr++) {
	    drawInfo.count = 0;
	    LineDDA(segPtr->x1, segPtr->y1, segPtr->x2, segPtr->y2, 
		DrawDot, (LPARAM)&drawInfo);
	}
    } else {
	HPEN pen, oldPen;

	pen = GCToPen(dc, gc);
	oldPen = SelectPen(dc, pen);
	for (segPtr = segArr, i = 0; i < numSegments; i++, segPtr++) {
	    MoveToEx(dc, segPtr->x1, segPtr->y1, (LPPOINT)NULL);
	    LineTo(dc, segPtr->x2, segPtr->y2);
	}
	DeletePen(SelectPen(dc, oldPen));
    }
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawRectangle --
 *
 *       Draws the outlines of the specified rectangle as if a
 *       five-point PolyLine protocol request were specified for each
 *       rectangle:
 *
 *             [x,y] [x+width,y] [x+width,y+height] [x,y+height]
 *             [x,y]
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws a rectangle on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
XDrawRectangle(Display *display, 
	       Drawable drawable, 
	       GC gc, 
	       int x, int y,
	       unsigned int width, 
	       unsigned int height)
{
    TkWinDCState state;
    HPEN pen, oldPen;
    HBRUSH brush, oldBrush;
    HDC dc;

    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    pen = GCToPen(dc, gc);
    brush = GetStockObject(NULL_BRUSH);
    oldPen = SelectPen(dc, pen);
    oldBrush = SelectBrush(dc, brush);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    Rectangle(dc, x, y, x + width + 1, y + height + 1);
    DeletePen(SelectPen(dc, oldPen));
    DeleteBrush(SelectBrush(dc, oldBrush));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawPoints --
 *
 *	Uses the foreground pixel and function components of the GC to 
 *	draw a multiple points into the specified drawable.
 *      CoordModeOrigin treats all coordinates as relative to the
 *	origin, and CoordModePrevious treats all coordinates after
 *	the first as relative to the previous point. 
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws points on the specified drawable.
 *
 *---------------------------------------------------------------------- 
 */
void
XDrawPoints(display, drawable, gc, pointArr, numPoints, mode)
    Display *display;
    Drawable drawable;
    GC gc;
    XPoint *pointArr;
    int numPoints;
    int mode;			/* Ignored. CoordModeOrigin is assumed. */
{
    register int i;
    HDC dc;
    register XPoint *pointPtr;
    TkWinDCState state;

    display->request++;
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    for (pointPtr = pointArr, i = 0; i < numPoints; i++, pointPtr++) {
	SetPixelV(dc, pointPtr->x, pointPtr->y, gc->foreground);
    }
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

/*
 *----------------------------------------------------------------------
 *
 * XReparentWindow --
 *
 *	If the specified window is mapped, automatically performs an
 *	UnmapWindow request on it, removes it from its current
 *	position in the hierarchy, and inserts it as the child of the
 *	specified parent.  The window is placed in the stacking order
 *	on top with respect to sibling windows.  
 *
 *	Note: In WIN32 you can't reparent to/from another application.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Reparents the specified window.
 *
 *---------------------------------------------------------------------- 
 */
void
XReparentWindow(display, window, parent, x, y)
    Display *display;
    Window window, parent;
    int x, y;
{
    HWND child, newParent;

    child = Tk_GetHWND(window);
    newParent = Tk_GetHWND(parent);
    SetParent(child, newParent);
    SetWindowLong(child, GWL_STYLE, WS_CHILD | WS_CLIPCHILDREN | 
	  WS_CLIPSIBLINGS);

    XMoveWindow(display, window, x, y);
    XRaiseWindow(display, window);
    XMapWindow(display, window);
}

void
XSetDashes(display, gc, dashOffset, dashList, n)
    Display *display;
    GC gc;
    int dashOffset;
    _Xconst char *dashList;
    int n;
{
  gc->dashes = (unsigned char)strlen(dashList);
  gc->dash_offset = (int)dashList;
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawString --
 *
 *	Draw a single string in the current font.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Renders the specified string in the drawable.
 *
 *----------------------------------------------------------------------
 */

void
XDrawString(display, drawable, gc, x, y, string, numChars)
    Display* display;
    Drawable drawable;
    GC gc;
    int x, y;
    _Xconst char* string;
    int numChars;
{
    if (drawable == None) {
	return;			
    }
    Tk_DrawChars(display, drawable, gc, (Tk_Font)gc->font, string, 
	 numChars, x, y);
}


#ifdef notdef
/*
 *----------------------------------------------------------------------
 *
 * XFillRectangles --
 *
 *	Fill multiple rectangular areas in the given drawable.  
 *	Handles tiling.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws onto the specified drawable.
 *
 *----------------------------------------------------------------------
 */

void
XFillRectangles(display, d, gc, rectangles, nrectangles)
    Display* display;
    Drawable d;
    GC gc;
    XRectangle* rectangles;
    int nrectangles;
{
    HDC dc;
    int i;
    RECT rect;
    TkWinDCState state;
    HBRUSH brush;

    if (d == None) {
	return;
    }

    dc = TkWinGetDrawableDC(display, d, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    brush = CreateSolidBrush(gc->foreground);

    if (((gc->fill_style == FillStippled) || 
	 (gc->fill_style == FillOpaqueStippled)) && 
	(gc->stipple != None)) {
	TkWinDrawable *twdPtr = (TkWinDrawable *)gc->stipple;
	HBRUSH oldBrush, stipple;
	HBITMAP oldBitmap, bitmap;
	HDC dcMem;
	HBRUSH bgBrush = CreateSolidBrush(gc->background);
	
	if (twdPtr->type != TWD_BITMAP) {
	    panic("unexpected drawable type in stipple");
	}
	/*
	 * Select stipple pattern into destination dc.
	 */
	stipple = CreatePatternBrush(twdPtr->bitmap.handle);
	SetBrushOrgEx(dc, gc->ts_x_origin, gc->ts_y_origin, NULL);
	oldBrush = SelectBrush(dc, stipple);
	dcMem = CreateCompatibleDC(dc);

	/*
	 * For each rectangle, create a drawing surface which is the size of
	 * the rectangle and fill it with the background color.  Then merge the
	 * result with the stipple pattern.
	 */
/*
 * The following two raster ops are used to copy the foreground and background
 * bits of a source pattern as defined by a stipple used as the pattern.
 */

#define COPYFG		0x00CA0749 /* dest = (pat & src) | (!pat & dst) */
#define COPYBG		0x00AC0744 /* dest = (!pat & src) | (pat & dst) */

	for (i = 0; i < nrectangles; i++) {
	    bitmap = CreateCompatibleBitmap(dc, rectangles[i].width,
		    rectangles[i].height);
	    oldBitmap = SelectBitmap(dcMem, bitmap);
	    rect.left = 0;
	    rect.top = 0;
	    rect.right = rectangles[i].width;
	    rect.bottom = rectangles[i].height;
	    FillRect(dcMem, &rect, brush);
	    BitBlt(dc, rectangles[i].x, rectangles[i].y, rectangles[i].width,
		    rectangles[i].height, dcMem, 0, 0, COPYFG);
	    if (gc->fill_style == FillOpaqueStippled) {
		FillRect(dcMem, &rect, bgBrush);
		BitBlt(dc, rectangles[i].x, rectangles[i].y,
			rectangles[i].width, rectangles[i].height, dcMem,
			0, 0, COPYBG);
	    }
	    DeleteBitmap(SelectBitmap(dcMem, oldBitmap));
	}
    } else if (gc->fill_style == FillTiled) {
	TkWinDrawable *twdPtr = (TkWinDrawable *)gc->tile;
	HBRUSH oldBrush, tile;
	HBITMAP oldBitmap, bitmap;
	HDC dcMem;
	HBRUSH bgBrush = CreateSolidBrush(gc->background);

	/*
	 * Select stipple pattern into destination dc.
	 */
	tile = CreatePatternBrush(twdPtr->bitmap.handle);
	SetBrushOrgEx(dc, gc->ts_x_origin, gc->ts_y_origin, NULL);
	oldBrush = SelectBrush(dc, tile);
	dcMem = CreateCompatibleDC(dc);

	/*
	 * For each rectangle, create a drawing surface which is the size of
	 * the rectangle and fill it with the background color.  Then merge the
	 * result with the stipple pattern.
	 */
/*
 * The following two raster ops are used to copy the foreground and background
 * bits of a source pattern as defined by a stipple used as the pattern.
 */
#define COPYFG		0x00CA0749 /* dest = (pat & src) | (!pat & dst) */
#define COPYBG		0x00AC0744 /* dest = (!pat & src) | (pat & dst) */

	for (i = 0; i < nrectangles; i++) {
	    bitmap = CreateCompatibleBitmap(dc, rectangles[i].width,
		    rectangles[i].height);
	    oldBitmap = SelectBitmap(dcMem, bitmap);
	    rect.left = 0;
	    rect.top = 0;
	    rect.right = rectangles[i].width;
	    rect.bottom = rectangles[i].height;
	    FillRect(dcMem, &rect, tile);
	    BitBlt(dc, rectangles[i].x, rectangles[i].y, rectangles[i].width,
		    rectangles[i].height, dcMem, 0, 0, R2_WHITE);
	    DeleteBitmap(SelectBitmap(dcMem, oldBitmap));
	}
	DeleteBrush(SelectBrush(dc, oldBrush));
	DeleteBitmap(tile);
	DeleteBrush(bgBrush);
    } else {
	for (i = 0; i < nrectangles; i++) {
	    TkWinFillRect(dc, rectangles[i].x, rectangles[i].y,
		    rectangles[i].width, rectangles[i].height, gc->foreground);
	}
    }
    DeleteBrush(brush);
    TkWinReleaseDrawableDC(d, dc, &state);
}
#endif

Bool XGetFontProperty(XFontStruct *fs, Atom atom, unsigned long *value)
{
  abort ();
}

