#!/bin/sh -e
#
# unison package build script
#
# Adapted from:
# $Id: generic-build-script,v 1.27 2004/09/23 13:32:53 igor Exp $
#

# find out where the build script is located
tdir=`echo "$0" | sed 's%[\\/][^\\/][^\\/]*$%%'`
test "x$tdir" = "x$0" && tdir=.
scriptdir=`cd $tdir; pwd`
# find src directory.
# If scriptdir ends in SPECS, then topdir is $scriptdir/..
# If scriptdir ends in CYGWIN-PATCHES, then topdir is $scriptdir/../..
# Otherwise, we assume that topdir = scriptdir
topdir1=`echo ${scriptdir} | sed 's%/SPECS$%%'`
topdir2=`echo ${scriptdir} | sed 's%/CYGWIN-PATCHES$%%'`
if [ "x$topdir1" != "x$scriptdir" ] ; then # SPECS
  topdir=`cd ${scriptdir}/..; pwd`
else
  if [ "x$topdir2" != "x$scriptdir" ] ; then # CYGWIN-PATCHES
    topdir=`cd ${scriptdir}/../..; pwd`
  else
    topdir=`cd ${scriptdir}; pwd`
  fi
fi

scriptname=`basename $0`
tscriptname=`basename $0 .sh`
export PKG=`echo $tscriptname | sed -e 's/\-[^\-]*\-[^\-]*$//'`
export VER=`echo $tscriptname | sed -e "s/${PKG}\-//" -e 's/\-[^\-]*$//'`
export REL=`echo $tscriptname | sed -e "s/${PKG}\-${VER}\-//"`
export BASEPKG=${PKG}-${VER}
export FULLPKG=${BASEPKG}-${REL}

# determine correct decompression option and tarball filename
export src_orig_pkg_name=unison-${VER}.tar.gz
export doc_orig_pkg_name=unison-${VER}-doc.tar.gz

export src_orig_pkg=${topdir}/${src_orig_pkg_name}
export doc_orig_pkg=${topdir}/${doc_orig_pkg_name}

# determine correct names for generated files
export src_pkg_name=${FULLPKG}-src.tar.bz2
export src_patch_name=${FULLPKG}.patch
export bin_pkg_name=${FULLPKG}.tar.bz2

export src_pkg=${topdir}/${src_pkg_name}
export src_patch=${topdir}/${src_patch_name}
export bin_pkg=${topdir}/${bin_pkg_name}
export srcdir=${topdir}/${BASEPKG}
export objdir=${srcdir}/.build
export instdir=${srcdir}/.inst
export srcinstdir=${srcdir}/.sinst
export checkfile=${topdir}/${FULLPKG}.check

prefix=/usr
sysconfdir=/etc
localstatedir=/var

export install_docs="\
	ABOUT-NLS \
	ANNOUNCE \
	AUTHORS \
	BUG-REPORTS \
	CHANGES \
	CONTRIB* \
	COPYING \
	COPYRIGHT \
	CREDITS \
	CHANGELOG \
	ChangeLog* \
	FAQ \
	HOW-TO-CONTRIBUTE \
	INSTALL* \
	KNOWNBUG \
	LEGAL \
	LICENSE \
	NEWS \
	NOTES \
	PROGLIST \
	README \
	RELEASE_NOTES \
	THANKS \
	TODO \
	Bugs.txt \
	ROADMAP.txt
	TODO.txt \
	doc/* \
"
export install_docs="`for i in ${install_docs}; do echo $i; done | sort -u`"
export test_rule=test
if [ -z "$SIG" ]; then
  export SIG=0	# set to 1 to turn on signing by default
fi

unpack() {
  tar -xzvf ${src_orig_pkg}
  tar -xzvf ${doc_orig_pkg}
}
mkdirs() {
  (cd ${topdir} 
  rmdirs 
  mkdir -p ${objdir} 
  mkdir -p ${instdir} 
  mkdir -p ${srcinstdir} 
  stow ${srcdir} ${objdir} )
}
stow() {
  # create in $2 a symlinked copy of the file tree in $1
  (srcdir=$1 tgtdir=$2
  for sd in $(find ${srcdir} \
    \( -path ${objdir} -o -path ${instdir} -o -path ${srcinstdir} \) \
    -prune -o -type d -print ) ; do
    td=${tgtdir}${sd#${srcdir}}
    echo "stowing ${td#${topdir}/}"
    mkdir -p ${td}
    find ${sd} -maxdepth 1 \! -type d |
    xargs -r ln -fs --target-dir=${td} 2>/dev/null || true
  done )
}
rmdirs() {
  (cd ${topdir}
  rm -fr ${objdir} ${instdir} ${srcinstdir} )
}
prep() {
  (cd ${topdir}
  unpack
  if [ -f ${src_patch} ] ; then
    patch -Z -p0 < ${src_patch}
  fi
  mkdirs )
}
conf() {
  :
}
reconf() {
  (cd ${topdir}
  rm -fr ${objdir}
  mkdir -p ${objdir}
  stow ${srcdir} ${objdir}
  conf )
}
build() {
  (cd ${objdir}
  make OSCOMP=cygwingnuc UISTYLE=text )
}
check() {
  :
}
clean() {
  (cd ${objdir}
  make clean )
}
install() {
  (cd ${objdir}
  rm -fr ${instdir}/*
  mkdir -p ${instdir}${prefix}/bin ${instdir}${prefix}/share/doc/${BASEPKG} ${instdir}${prefix}/share/doc/Cygwin
  /usr/bin/install -m 755 unison.exe ${instdir}${prefix}/bin
  templist=""
  for f in ${install_docs} ; do
    if [ -f ${srcdir}/$f ] ; then
      templist="$templist ${srcdir}/$f"
    fi
  done
  if [ ! "x$templist" = "x" ]; then
    /usr/bin/install -m 644 $templist \
         ${instdir}${prefix}/share/doc/${BASEPKG}
  fi
  if [ -f ${srcdir}/CYGWIN-PATCHES/${PKG}.README ]; then
    /usr/bin/install -m 644 ${srcdir}/CYGWIN-PATCHES/${PKG}.README \
      ${instdir}${prefix}/share/doc/Cygwin/${BASEPKG}.README
  elif [ -f ${srcdir}/CYGWIN-PATCHES/README ] ; then
    /usr/bin/install -m 644 ${srcdir}/CYGWIN-PATCHES/README \
      ${instdir}${prefix}/share/doc/Cygwin/${BASEPKG}.README
  fi
  ln -fs ../Cygwin/${BASEPKG}.README \
         ${instdir}${prefix}/share/doc/${BASEPKG}/README.Cygwin
  if [ -f ${srcdir}/CYGWIN-PATCHES/${PKG}.sh ] ; then
    mkdir -p ${instdir}${sysconfdir}/postinstall
    /usr/bin/install -m 755 ${srcdir}/CYGWIN-PATCHES/${PKG}.sh \
      ${instdir}${sysconfdir}/postinstall/${PKG}.sh
  elif [ -f ${srcdir}/CYGWIN-PATCHES/postinstall.sh ] ; then
    mkdir -p ${instdir}${sysconfdir}/postinstall
    /usr/bin/install -m 755 ${srcdir}/CYGWIN-PATCHES/postinstall.sh \
      ${instdir}${sysconfdir}/postinstall/${PKG}.sh
  fi
  if [ -f ${srcdir}/CYGWIN-PATCHES/preremove.sh ] ; then
    mkdir -p ${instdir}${sysconfdir}/preremove
    /usr/bin/install -m 755 ${srcdir}/CYGWIN-PATCHES/preremove.sh \
      ${instdir}${sysconfdir}/preremove/${PKG}.sh
  fi )
}
strip() {
  find ${instdir} -name "*.dll" -o -name "*.exe" | xargs -r strip
}
list() {
  (cd ${instdir}
  find \! -type d | sed 's%^\.%  %' | sort )
}
depend() {
  (cd ${instdir}
  find -name "*.exe" -o -name "*.dll" | xargs -r cygcheck \
  | sed -e '/\.exe/d' -e 's,\\,/,g' | sort -bu | xargs -r -n1 cygpath -u \
  | xargs -r cygcheck -f | sed 's%^%  %' | sort -u )
}
pkg() {
  (cd ${instdir}
  tar cvjf ${bin_pkg} * )
}
mkpatch() {
  (cd ${srcdir}
  find -name "autom4te.cache" | xargs -r rm -rf
  unpack
  mv ${BASEPKG} ../${BASEPKG}-orig
  cd ${topdir}
  diff -urN -x '.build' -x '.inst' -x '.sinst' \
    ${BASEPKG}-orig ${BASEPKG} > ${srcinstdir}/${src_patch_name} \
	|| [ $? -eq 1 ]
  rm -rf ${BASEPKG}-orig
  cmp -s ${topdir}/${src_patch_name} ${srcinstdir}/${src_patch_name} \
  || cp --backup=numbered ${srcinstdir}/${src_patch_name} ${topdir} )
}
spkg() {
  (mkpatch
  if [ "${SIG}" -eq 1 ] ; then
    name=${srcinstdir}/${src_patch_name} text="PATCH" sigfile
  fi
  cp ${src_orig_pkg} ${srcinstdir}/${src_orig_pkg_name}
  cp ${doc_orig_pkg} ${srcinstdir}/${doc_orig_pkg_name}
  if [ -e ${src_orig_pkg}.sig ] ; then
    cp ${src_orig_pkg}.sig ${srcinstdir}
  fi
  cp $0 ${srcinstdir}/`basename $0`
  name=$0 text="SCRIPT" sigfile
  if [ "${SIG}" -eq 1 ] ; then
    cp $0.sig ${srcinstdir}
  fi
  cd ${srcinstdir}
  tar cvjf ${src_pkg} * )
}
finish() {
  rm -rf ${srcdir}
}
sigfile() {
  if [ \( "${SIG}" -eq 1 \) -a \( -e $name \) -a \( \( ! -e $name.sig \) -o \( $name -nt $name.sig \) \) ]; then
    if [ -x /usr/bin/gpg ]; then
      echo "$text signature need to be updated"
      rm -f $name.sig
      /usr/bin/gpg --detach-sign $name
    else
      echo "You need the gnupg package installed in order to make signatures."
    fi
  fi
}
checksig() {
  if [ -x /usr/bin/gpg ]; then
    if [ -e ${src_orig_pkg}.sig ]; then
      echo "ORIGINAL PACKAGE signature follows:"
      /usr/bin/gpg --verify ${src_orig_pkg}.sig ${src_orig_pkg}
    else
      echo "ORIGINAL PACKAGE signature missing."
    fi
    if [ -e $0.sig ]; then
      echo "SCRIPT signature follows:"
      /usr/bin/gpg --verify $0.sig $0
    else
      echo "SCRIPT signature missing."
    fi
    if [ -e ${src_patch}.sig ]; then
      echo "PATCH signature follows:"
      /usr/bin/gpg --verify ${src_patch}.sig ${src_patch}
    else
      echo "PATCH signature missing."
    fi
  else
    echo "You need the gnupg package installed in order to check signatures."
  fi
}
  
usage() {
  echo "Build script for the Cygwin ${PKG} package.
Adapted from the Cygwin generic-build-script;
see http://cygwin.com/setup.html#srcpackage_contents 
for detailed usage instructions.
Usage: ${scriptname} operation...
build operations:
  all          build source and binary packages (same as:
                 checksig prep conf build check install strip pkg spkg finish)
  build        run 'make' on patched source
  conf         run ./configure
  configure    run ./configure
  check        run test suite
  make         run 'make' on patched source
  prep         unpack and patch package source; run mkdirs
  strip        strip built executables
  test         run test suite
packaging operations:
  first        build source package first time, from existing patched 
                 source directory (same as: mkdirs spkg finish)
  install      install package into installation staging subdir 
  package      build binary package
  pkg          build binary package
  spkg         build source package
  src-package  build source package
information:
  depend       list packages on which the binary package depends
  help         display this message
  list         list files in the binary package
  usage        display this message
minor operations:
  checksig     check package, patch, and script signatures
  clean        run 'make clean'
  finish       remove source directory
  mkdirs       create package staging directories
  mkpatch      create Cygwin source patch
  reconf       remove build staging dir and rerun ./configure
  rmdirs       remove package staging directories      
"
}

test -n "$1" || set "usage"
while test -n "$1" ; do
  case $1 in
    prep)		prep ;;
    mkdirs)		mkdirs ;;
    conf|configure)		conf ;;
    reconf)		reconf ;;
    build|make)	build ;;
    check|test)	check ;;
    clean)		clean ;;
    install)	install ;;
    list)		list ;;
    depend)		depend ;;
    strip)		strip ;;
    pkg|package)	pkg ;;
    mkpatch)	mkpatch ;;
    rmdirs)		rmdirs ;;
    spkg|src-package)	spkg ;;
    finish)		finish ;;
    checksig)	checksig ;;
    first)		mkdirs ; spkg ; finish ;;
    all)		checksig ; prep ; conf ; build ; install ;
                strip ; pkg ; spkg ; finish ;;
    help|-help|--help|usage)	usage ;;
    *) echo "${scriptname}: error: unknown argument \"$1\"" >&2
	   echo "${scriptname}: run \"${scriptname} help\" to get a usage summary" >&2
	   exit 1 ;;
  esac
  shift
done
