#define _GNU_SOURCE
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>

#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>

#include "server.h"
#include "connections.h"
#include "response.h"
#include "connections.h"
#include "log.h"

#include "plugin.h"


typedef struct {
	PLUGIN_DATA;
	
	double requests;
	double bytes_written;
	double bytes_read;
	
	buffer *path_rrdtool_bin;
	buffer *path_rrd;
	
	buffer *cmd;
	buffer *resp;
	
	int read_fd, write_fd;
	pid_t rrdtool_pid;
	
	int rrdtool_running;
} plugin_data;

INIT_FUNC(mod_rrd_init) {
	plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	p->requests = 0;
	p->bytes_written = 0;
	p->path_rrdtool_bin = buffer_init();
	p->path_rrd = buffer_init();
	p->resp = buffer_init();
	p->cmd = buffer_init();
	
	return p;
}

FREE_FUNC(mod_rrd_free) {
	plugin_data *p = p_d;
	
	UNUSED(srv);

	if (!p) return HANDLER_GO_ON;
	
	buffer_free(p->path_rrdtool_bin);
	buffer_free(p->path_rrd);
	buffer_free(p->cmd);
	buffer_free(p->resp);
	
	if (p->rrdtool_pid) {
		int status;
		close(p->read_fd);
		close(p->write_fd);
		
		/* collect status */
		waitpid(p->rrdtool_pid, &status, 0);
	}
	
	free(p);
	
	return HANDLER_GO_ON;
}

int mod_rrd_create_pipe(server *srv, plugin_data *p) {
	pid_t pid;
	
	int to_rrdtool_fds[2];
	int from_rrdtool_fds[2];
	
	if (pipe(to_rrdtool_fds)) {
		log_error_write(srv, __FILE__, __LINE__, "ss", 
				"pipe failed: ", strerror(errno));
		return -1;
	}
	
	if (pipe(from_rrdtool_fds)) {
		log_error_write(srv, __FILE__, __LINE__, "ss", 
				"pipe failed: ", strerror(errno));
		return -1;
	}
	
	/* fork, execve */
	switch (pid = fork()) {
	case 0: {
		/* child */
		char **args;
		int argc;
		int i = 0;
		char *dash = "-";
		
		/* move stdout to from_rrdtool_fd[1] */
		close(STDOUT_FILENO);
		dup2(from_rrdtool_fds[1], STDOUT_FILENO);
		close(from_rrdtool_fds[1]);
		/* not needed */
		close(from_rrdtool_fds[0]);
		
		/* move the stdin to to_rrdtool_fd[0] */
		close(STDIN_FILENO);
		dup2(to_rrdtool_fds[0], STDIN_FILENO);
		close(to_rrdtool_fds[0]);
		/* not needed */
		close(to_rrdtool_fds[1]);
		
		/* set up args */
		argc = 3;
		args = malloc(sizeof(*args) * argc);
		i = 0;
		
		args[i++] = p->path_rrdtool_bin->ptr;
		args[i++] = dash;
		args[i++] = NULL;

		/* we don't need the client socket */
		for (i = 3; i < 256; i++) {
			if (i != srv->log_error_fd) close(i);
		}
		
		/* exec the cgi */
		execv(args[0], args);
		
		log_error_write(srv, __FILE__, __LINE__, "sss", "spawing rrdtool failed: ", strerror(errno), args[0]);
		
		/* */
		SEGFAULT();
		break;
	}
	case -1:
		/* error */
		log_error_write(srv, __FILE__, __LINE__, "ss", "fork failed: ", strerror(errno));
		break;
	default: {
		/* father */
		
		close(from_rrdtool_fds[1]);
		close(to_rrdtool_fds[0]);
		
		/* register PID and wait for them asyncronously */
		p->write_fd = to_rrdtool_fds[1];
		p->read_fd = from_rrdtool_fds[0];
		p->rrdtool_pid = pid;
		
		break;
	}
	}
	
	return 0;
}

SETDEFAULTS_FUNC(mod_rrd_set_defaults) {
	plugin_data *p = p_d;
	struct stat st;
	
	config_values_t cv[] = { 
		{ "rrdtool.binary",              NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ "rrdtool.db-name",             NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ NULL,                          NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
	};
	
	if (!p) return HANDLER_ERROR;
	cv[0].destination = p->path_rrdtool_bin;
	cv[1].destination = p->path_rrd;
	
	if (0 != config_insert_values(srv, cv)) {
		return HANDLER_ERROR;
	}
	
	p->rrdtool_running = 0;
	
	/* check for dir */
	
	if (buffer_is_empty(p->path_rrdtool_bin)) {
		log_error_write(srv, __FILE__, __LINE__, "s", 
				"rrdtool.binary has to be set");
		return HANDLER_ERROR;
	}
	
	if (buffer_is_empty(p->path_rrd)) {
		log_error_write(srv, __FILE__, __LINE__, "s", 
				"rrdtool.db-name has to be set");
		return HANDLER_ERROR;
	}
	
	/* open the pipe to rrdtool */
	if (mod_rrd_create_pipe(srv, p)) {
		return HANDLER_ERROR;
	}
	
	/* check if DB already exists */
	if (0 == stat(p->path_rrd->ptr, &st)) {
		/* check if it is plain file */
		if (!S_ISREG(st.st_mode)) {
			log_error_write(srv, __FILE__, __LINE__, "s", 
					"not a regular file:", p->path_rrd);
			return HANDLER_ERROR;
		}
	} else {
		int r ;
		/* create a new one */
		
		BUFFER_COPY_STRING_CONST(p->cmd, "create ");
		buffer_append_string_buffer(p->cmd, p->path_rrd);
		buffer_append_string(p->cmd, " --step 60 ");
		buffer_append_string(p->cmd, "DS:InOctets:ABSOLUTE:600:U:U ");
		buffer_append_string(p->cmd, "DS:OutOctets:ABSOLUTE:600:U:U ");
		buffer_append_string(p->cmd, "DS:Requests:ABSOLUTE:600:U:U ");
		buffer_append_string(p->cmd, "RRA:AVERAGE:0.5:1:600 ");
		buffer_append_string(p->cmd, "RRA:AVERAGE:0.5:6:700 ");
		buffer_append_string(p->cmd, "RRA:AVERAGE:0.5:24:775 ");
		buffer_append_string(p->cmd, "RRA:AVERAGE:0.5:288:797 ");
		buffer_append_string(p->cmd, "RRA:MAX:0.5:1:600 ");
		buffer_append_string(p->cmd, "RRA:MAX:0.5:6:700 ");
		buffer_append_string(p->cmd, "RRA:MAX:0.5:24:775 ");
		buffer_append_string(p->cmd, "RRA:MAX:0.5:288:797 ");
		buffer_append_string(p->cmd, "RRA:MIN:0.5:1:600 ");
		buffer_append_string(p->cmd, "RRA:MIN:0.5:6:700 ");
		buffer_append_string(p->cmd, "RRA:MIN:0.5:24:775 ");
		buffer_append_string(p->cmd, "RRA:MIN:0.5:288:797\n");
		
		if (-1 == (r != write(p->write_fd, p->cmd->ptr, p->cmd->used - 1))) {
			log_error_write(srv, __FILE__, __LINE__, "ss", 
				"rrdtool-write: failed", strerror(errno));
			
			return HANDLER_ERROR;
		}
		
		buffer_prepare_copy(p->resp, 4096);
		if (-1 == (r = read(p->read_fd, p->resp->ptr, p->resp->size))) {
			log_error_write(srv, __FILE__, __LINE__, "ss", 
				"rrdtool-read: failed", strerror(errno));
			
			return HANDLER_ERROR;
		}
		
		p->resp->used = r;
		
		if (p->resp->ptr[0] != 'O' ||
		    p->resp->ptr[1] != 'K') {
			log_error_write(srv, __FILE__, __LINE__, "sbb", 
				"rrdtool-response:", p->cmd, p->resp);
			
			return HANDLER_ERROR;
		}
	}
	
	p->rrdtool_running = 1;
		
	return HANDLER_GO_ON;
}

TRIGGER_FUNC(mod_rrd_trigger) {
	plugin_data *p = p_d;
	
	if (!p->rrdtool_running) return HANDLER_GO_ON;
	
	/* write the data down every 5 minutes */
	if ((srv->cur_ts % 60) == 0) {
		int r;
		
		BUFFER_COPY_STRING_CONST(p->cmd, "update ");
		buffer_append_string_buffer(p->cmd, p->path_rrd);
		BUFFER_APPEND_STRING_CONST(p->cmd, " N:");
		buffer_append_off_t(p->cmd, p->bytes_read);
		BUFFER_APPEND_STRING_CONST(p->cmd, ":");
		buffer_append_off_t(p->cmd, p->bytes_written);
		BUFFER_APPEND_STRING_CONST(p->cmd, ":");
		buffer_append_long(p->cmd, p->requests);
		BUFFER_APPEND_STRING_CONST(p->cmd, "\n");
		
		if (-1 == (r != write(p->write_fd, p->cmd->ptr, p->cmd->used - 1))) {
			p->rrdtool_running = 0;
			
			log_error_write(srv, __FILE__, __LINE__, "ss", 
				"rrdtool-write: failed", strerror(errno));
			
			return HANDLER_ERROR;
		}
		
		buffer_prepare_copy(p->resp, 4096);
		if (-1 == (r = read(p->read_fd, p->resp->ptr, p->resp->size))) {
			p->rrdtool_running = 0;
			
			log_error_write(srv, __FILE__, __LINE__, "ss", 
				"rrdtool-read: failed", strerror(errno));
			
			return HANDLER_ERROR;
		}
		
		p->resp->used = r;
		    
		if (p->resp->ptr[0] != 'O' ||
		    p->resp->ptr[1] != 'K') {
			p->rrdtool_running = 0;
			
			log_error_write(srv, __FILE__, __LINE__, "sbb", 
				"rrdtool-response:", p->cmd, p->resp);
			
			return HANDLER_ERROR;
		}
		p->requests = 0;
		p->bytes_written = 0;
		p->bytes_read = 0;
	}
	
	return HANDLER_GO_ON;
}

REQUESTDONE_FUNC(mod_rrd_account) {
	plugin_data *p = p_d;
	
	UNUSED(srv);

	p->requests++;
	p->bytes_written += con->bytes_written;
	p->bytes_read += con->bytes_read;
	
	return HANDLER_GO_ON;
}

int mod_rrdtool_plugin_init(plugin *p) {
	p->name        = buffer_init_string("rrd");
	
	p->init        = mod_rrd_init;
	p->cleanup     = mod_rrd_free;
	p->set_defaults= mod_rrd_set_defaults;
	
	p->handle_trigger      = mod_rrd_trigger;
	p->handle_request_done = mod_rrd_account;
	
	p->data        = NULL;
	
	return 0;
}
