#! /bin/bash
# Author: Brian Gallew <geek+@cmu.edu>

# This next line defines TOPFOLDER and X11proggies
. /etc/X11/X-start-menu-icons-list

#Ensure that a given program is somewhere in the current $PATH
function checkpath () {
    c=$(which $1 2> /dev/null)
    test ! -z "$c"
}

#Make sure the program we *think* we're pointing to is the real deal.
function canonicalize () {
    a=$(which "$1")
    while test -h "$a"
    do
	set $(ls -l "$a")
	while test ! -z "$2";do shift;done
	a=$(dirname "$a")/"$1"
    done
    test -f "$a".exe && echo "$a".exe || echo "$a"
}

# Create a shortcut for an application that draws its own windows
# Takes four or five arguments: appname, description folder, the
# offset of the icon , and any arguments

function xapp () {
    app="$1"
    shift
    description="$1"
    shift
    folder="${TOPFOLDER}/$1"
    shift
    if [ -z "$1" ]
    then
	icon="--icon=/usr/X11R6/bin/run.exe"
	ioff="--iconoffset=2"
    else
	icon=--icon=$(canonicalize "$app")
	ioff="--iconoffset=$1"
    fi
    shift
    args="$1"
    test -d "${folder}" || mkdir -p "${folder}"
    mkshortcut "${icon}" "${ioff}" --arguments=-p\ /usr/X11R6/bin\ ${app}\ -display\ 127.0.0.1:0.0\ "${args}" \
	--name="${folder}/${app}" \
        --desc="${description}" \
	--workingdir=/bin /usr/X11R6/bin/run.exe
}

# Create a shortcut for a text-output-only application
function tapp () {
    app="$1"
    shift
    description="$1"
    shift
    folder="${TOPFOLDER}/$1"
    shift
    args="$1"
    test -d "${folder}" || mkdir -p "${folder}"
    mkshortcut --arguments=-p\ /usr/X11R6/bin\ xterm\ -display\ 127.0.0.1:0.0\ -e\ bash\ -c\ \"/usr/X11R6/bin/${app}\ -display\ 127.0.0.1:0.0\ "${args}"\ "| /bin/less"\" \
	--name="${folder}/${app}" \
        --desc="${description}" \
	--icon=/cygwin.ico \
	--workingdir=/bin /usr/X11R6/bin/run.exe
    test $? -eq 0 && return
    echo $app:$description:$folder::
}

function create_icons () {
    for index in $(seq 0 ${#X11proggies[*]})
    do
	a=${X11proggies[$index]}
	test -z "$a" && continue
	# This next line breaks up the elements and set $1..$6 appropriately
	IFS=":";set $a;unset IFS
	checkpath $1
	test $? -eq 0 || continue
	case $4 in
	xapp)
	    xapp "$1" "$2" "$3" "$5" "$6"
	    ;;
	tapp)
	    tapp "$1" "$2" "$3" "$6"
	    ;;
	*)
	    echo `hosed!`
	    echo $*
	esac
    done
}

function destroy_icons () {
    for index in $(seq 0 ${#X11proggies[*]})
    do
	a=${X11proggies[$index]}
	test -z "$a" && continue
	# This next line breaks up the elements and set $1..$6 appropriately
	IFS=":";set $a;unset IFS
	icon="${TOPFOLDER}/${3}/${1}.lnk"
	test -f "${icon}" && rm -f "${icon}"
    done
    # Removal of TOPFOLDER will fail because find has a lock on it
    # Thus, we add -mindepth 1 to prevent an attempt to remove TOPFOLDER
    find "${TOPFOLDER}" -depth -mindepth 1 -type d -exec rmdir -v --ignore-fail-on-non-empty "{}" \;
    # TOPFOLDER is removed now, since find has released its lock
    rmdir -v --ignore-fail-on-non-empty "${TOPFOLDER}"
}

function usage () {
    cat >&2 <<EOF
Usage: $0 [-d]

Without any switches, this will recreate the defined icon set.  The
"-d" switch indicates that all the icons should be removed.  If, after
the icons are removed, any of the folders are empty, they, too, will
be removed.
EOF
    exit -1
}

delete=${1}
shift
test -z "$*" || usage
if [ -z "$delete" ]
then
    create_icons
else
    test "$delete" = "-d" || usage
    destroy_icons
fi
