/*
 * filterdiff - extract (or exclude) a diff from a diff file
 * lsdiff - show which files are modified by a patch
 * grepdiff - show files modified by a patch containing a regexp
 * Copyright (C) 2001, 2002, 2003 Tim Waugh <twaugh@redhat.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_ERROR_H
# include <error.h>
#endif /* HAVE_ERROR_H */
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h> // for ssize_t
#endif /* HAVE_SYS_TYPES_H */
#include <fnmatch.h>
#include <getopt.h>
#include <locale.h>
#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "util.h"
#include "diff.h"

struct range {
	struct range *next;
	unsigned long start;
	unsigned long end;
};

enum line_numbering {
	None = 0,
	Before,
	After
};

enum {
	output_none = 0,
	output_hunk,
	output_file
} output_matching = output_none;

static struct patlist *pat_include = NULL;
static struct patlist *pat_exclude = NULL;
static struct range *hunks = NULL;
static struct range *lines = NULL;
static enum line_numbering number_lines = None;

static int unzip = 0;
static enum {
	mode_filter,
	mode_list,
	mode_grep,
} mode;
static regex_t regex;
static int numbering = 0;
static int annotating = 0;
static int ignore_components = 0;
static int strip_components = 0;
static const char *prefix_to_add = NULL;
static int show_status = 0;
static int verbose = 0;

static int file_exists (const char *timestamp)
{
	struct tm t;
	long zone = -1;
	char *end;

	/* First try ISO 8601-style timestamp */
	end = strptime (timestamp, "%Y-%m-%d %H:%M:%S", &t);
	if (end) {
		/* Skip nanoseconds. */
		char *endptr;
		if (*end == '.') {
			end++;
			end += strspn (end, "0123456789");
		}

		end += strspn (end, " ");
		zone = strtol (end, &endptr, 10);
		if (end == endptr)
			zone = -1;
	} else
		/* If that fails try a traditional format */
		if (!strptime (timestamp, "%a %b %e %T %Y", &t))
			return 1;

	/* If the time is less that fifteen hours either side of the
	 * start of 1970, and it's an exact multiple of 15 minutes, it's
	 * very likely to be the result of ctime(&zero). */
	if (t.tm_sec == 0 &&
	    ((t.tm_year == 69 && t.tm_mon == 11 && t.tm_mday == 31 &&
	      t.tm_hour >= 9) ||
	     (t.tm_year == 70 && t.tm_mon == 0 && t.tm_mday == 1 &&
	      t.tm_hour <= 15)) &&
	    (t.tm_min % 15) == 0) {
		if (zone != -1) {
			/* Extra checking, since we know the timezone. */
			long offset = 0;
			if (t.tm_year == 69) {
				offset = 100 * (t.tm_hour - 24);
				if (t.tm_min)
					offset += 100 + t.tm_min - 60;
			} else {
				offset = 100 * t.tm_hour;
				offset += t.tm_min;
			}

			if (offset != zone)
				return 1;
		}

		return 0;
	}

	/* Otherwise, it's a real file timestamp. */
	return 1;
}

static int output_header_line (const char *line)
{
	char *fn;
	int h = strcspn (line + 4, "\t\n");

	fwrite (line, 1, 4, stdout);

	if (prefix_to_add)
		fputs (prefix_to_add, stdout);

	fn = xstrndup (line + 4, h);
	fputs (stripped (fn, strip_components), stdout);
	fputs (line + 4 + h, stdout);
	free (fn);
	return 0;
}

static void display_filename (unsigned long linenum, char status,
			      const char *filename)
{
	if (numbering)
		printf ("%lu\t", linenum);
	if (show_status)
		printf ("%c ", status);
	if (prefix_to_add)
		fputs (prefix_to_add, stdout);
	puts (stripped (filename, strip_components));
}

static int
hunk_matches (unsigned long orig_offset, unsigned long orig_count,
	      unsigned long hunknum)
{
	int h = 0, l = 0;
	struct range *r;

	// For the purposes of matching, zero lines at offset n counts
	// as line n.
	if (!orig_count)
		orig_count = 1;

	// See if the hunk range list includes this hunk.  -1UL is a
	// wildcard.
	for (r = hunks; r; r = r->next)
		if ((r->start == -1UL ||
		     r->start <= hunknum) &&
		    (r->end == -1UL ||
		     hunknum <= r->end)) {
			h = 1;
			break;
		}

	// See if the lines range list includes this hunk.  -1UL is a
	// wildcard.
	for (r = lines; r; r = r->next)
		if ((r->start == -1UL ||
		     r->start < (orig_offset + orig_count)) &&
		    (r->end == -1UL ||
		     r->end >= orig_offset)) {
			l = 1;
			break;
		}

	if (hunks && !h)
		return 0;

	if (lines && !l)
		return 0;

	return 1;
}

static int do_unified (FILE *f, char *header[2], int match, char **line,
		       ssize_t *linelen, unsigned long *linenum,
		       unsigned long start_linenum, char status,
		       const char *bestname)
{
	/* Skip hunk. */
	unsigned long orig_count = 0, new_count = 0;
	unsigned long orig_offset, new_offset;
	unsigned long hunknum = 0;
	unsigned long track_linenum = 0;
	int header_displayed = 0;
	int hunk_match = match;
	long munge_offset = 0;
	int displayed_filename = 0;
	unsigned long last_hunkmatch = 0;
	unsigned long hunk_linenum = *linenum;
	FILE *match_tmpf = NULL;
	int grepmatch = 0;
	long delayed_munge = 0;

	if (output_matching == output_file)
		match_tmpf = tmpfile ();

	for (;;) {
		if (getline (line, linelen, f) == -1)
			return EOF;
		++*linenum;

		if (!orig_count && !new_count && **line != '\\') {
			char *trailing;

			if (strncmp (*line, "@@ ", 3))
				break;

			/* Next chunk. */
			hunknum++;
			hunk_linenum = *linenum;

			if (output_matching == output_hunk && !grepmatch)
				// We are missing this hunk out, but
				// we need to remember how many lines
				// it would have added or removed.
				munge_offset += delayed_munge;

			if (output_matching != output_file)
				grepmatch = 0;
			if (output_matching == output_hunk) {
				if (match_tmpf)
					fclose (match_tmpf);
				match_tmpf = tmpfile ();
			}

			if (read_atatline (*line, &orig_offset, &orig_count,
					   &new_offset, &new_count))
				error (EXIT_FAILURE, 0,
				      "line not understood: %s", *line);

			// Decide if this hunk matches.
			if (match)
				hunk_match = hunk_matches (orig_offset,
							   orig_count,
							   hunknum);
			else hunk_match = 0;

			if (hunk_match && numbering && verbose &&
			    mode != mode_grep)
				printf ("\t%lu\tHunk #%lu\n",
					hunk_linenum, hunknum);

			if (hunk_match &&
			    (mode == mode_filter ||
			     output_matching != output_none)) {
				int first_hunk = !header_displayed;
				FILE *output_to = stdout;

				if (mode == mode_grep) {
					delayed_munge = orig_count - new_count;
					output_to = match_tmpf;
				}

				if (!header_displayed &&
				    mode != mode_grep) {
					// Display the header.
					if (number_lines != After)
						output_header_line (header[0]);
					if (number_lines != Before)
						output_header_line (header[1]);
					header_displayed = 1;
				}
				switch (number_lines) {
				case None:
					// Display the offsets and
					// counts, adjusting for any
					// hunks we've previously
					// missed out.
					fprintf (output_to,
						 "@@ -%lu", orig_offset);
					if (orig_count != 1)
						fprintf (output_to,
							 ",%lu", orig_count);
					fprintf (output_to, " +%lu",
						 new_offset + munge_offset);
					if (new_count != 1)
						fprintf (output_to,
							 ",%lu", new_count);
					fprintf (output_to, " @@");

					if (annotating)
						fprintf (output_to,
							 " Hunk #%lu, %s",
							 hunknum, bestname);

					trailing = strchr (*line, '+');
					trailing += strcspn (trailing, " ");
					if (*trailing == ' ')
						trailing++;
					trailing += strspn (trailing, "@");
					fputs (trailing, output_to);
					break;
				case Before:
					// Note the initial line number
					track_linenum = orig_offset;
					if (!first_hunk ||
					    (output_matching == output_file &&
					     hunknum > 1))
						fputs ("...\n", output_to);
					break;
				case After:
					// Note the initial line number
					track_linenum = (new_offset +
							 munge_offset);
					if (!first_hunk ||
					    (output_matching == output_file &&
					     hunknum > 1))
						fputs ("...\n", output_to);
					break;
				}
			} else if (mode == mode_filter)
				// We are missing this hunk out, but
				// we need to remember how many lines
				// it would have added or removed.
				munge_offset += orig_count - new_count;

			continue;
		}

		if (orig_count && **line != '+')
			orig_count--;
		if (new_count && **line != '-')
			new_count--;

		if (hunk_match && mode == mode_grep &&
		    !regexec (&regex, *line + 1, 0, NULL, 0)) {
			if (output_matching == output_none) {
				if (!displayed_filename) {
					displayed_filename = 1;
					display_filename (start_linenum,
							  status, bestname);
				}

				if (numbering && verbose &&
				    hunknum > last_hunkmatch) {
					last_hunkmatch = hunknum;
					printf ("\t%lu\tHunk #%lu\n",
						hunk_linenum, hunknum);
				}
			} else {
				if (match_tmpf) {
					if (!header_displayed &&
					    number_lines != After)
						output_header_line (header[0]);

					if (!header_displayed &&
					    number_lines != Before)
						output_header_line (header[1]);

					if (!header_displayed)
						header_displayed = 1;

					rewind (match_tmpf);
					while (!feof (match_tmpf)) {
						int ch = fgetc (match_tmpf);
						if (ch == EOF)
							break;
						putchar (ch);
					}
					fclose (match_tmpf);
					match_tmpf = NULL;
				}
				grepmatch = 1;
			}
		}

		if (hunk_match &&
		    (mode == mode_filter ||
		     output_matching != output_none)) {
			FILE *output_to = stdout;
			if (mode == mode_grep && !grepmatch)
				output_to = match_tmpf;
			if (number_lines == None)
				// Just display each line.
				fputs (*line, output_to);
			else if ((number_lines == Before && **line != '+') ||
				 (number_lines == After && **line != '-'))
				// Numbered line.
				fprintf (output_to, "%lu\t:%s",
					 track_linenum++, 1 + *line);
		}
	}

	if (match_tmpf)
		fclose (match_tmpf);

	return 0;
}

static int do_context (FILE *f, char *header[2], int match, char **line,
		       ssize_t *linelen, unsigned long *linenum,
		       unsigned long start_linenum, char status,
		       const char *bestname)
{
	/* Skip before and after segments. */
	unsigned long line_start, line_end, line_count;
	unsigned long hunknum = 0;
	unsigned long track_linenum = 0;
	unsigned long changed[2];
	long munge_offset = 0;
	int header_displayed = 0;
	char *n, *end;
	int i;
	int hunk_match = 0;
	int displayed_filename = 0;
	unsigned long last_hunkmatch = 0;
	unsigned long hunk_linenum = *linenum;
	FILE *match_tmpf = NULL;
	int grepmatch = 0;
	int ret = 0;
	unsigned long unchanged;
	int first_hunk = 0;

	/* Context diff hunks are like this:
	 *
	 * *************** [GNU diff can put stuff here]
	 * *** start[,end] **** [we sometimes put stuff here]
	 *   from lines... (omitted if there are only insertions)
	 * --- start[,end] ----
	 *   to lines... (omitted if there are only deletions)
	 *[*** start[,end] ****
	 * ...]
	 */

	if (getline (line, linelen, f) == -1)
		return EOF;
	++*linenum;

	if (strncmp (*line, "***************", 15))
		return 1;

	if (output_matching == output_file)
		match_tmpf = tmpfile ();

 next_hunk:
	unchanged = 0;
	changed[0] = changed[1] = 0; // for munge calculation

	for (i = 0; i < 2; i++) {
		int first = 1;

		if (i == 0)
			first_hunk = !header_displayed;

		if (getline (line, linelen, f) == -1) {
			ret = EOF;
			goto out;
		}

		++*linenum;

		if (!i && !strncmp (*line, "***************", 15)) {
			/* Some diffs seem to have this for every
			 * set of changes.  SuSV2 says not to,
			 * but the GNU diff info page disagrees. */
			i--;
			continue;
		}

		if (strncmp (*line, i ? "--- " : "*** ", 4)) {
			ret = 1;
			goto out;
		}

		if (!i) {
			hunknum++;
			hunk_linenum = *linenum;
			if (output_matching != output_file)
				grepmatch = 0;
			if (output_matching == output_hunk) {
				if (match_tmpf)
					fclose (match_tmpf);
				match_tmpf = tmpfile ();
			}
		}

	do_line_counts:
		n = *line + 4;
		line_start = strtoul (n, &end, 10);
		if (n == end) {
			ret = 1;
			goto out;
		}

		if (*end == ',') {
			n = end + 1;
			line_end = strtoul (n, &end, 10);
			if (n == end) {
				ret = 1;
				goto out;
			}

			if (line_start > line_end) {
				ret = 1;
				goto out;
			}

			line_count = line_end - line_start + 1;
		} else {
			line_end = line_start;
			line_count = line_start ? 1 : 0;
		}

		n = strchr (n, '*');
		if (n)
			n += 4;

		if (!i) {
			if (match)
				hunk_match = hunk_matches (line_start,
							   line_count,
							   hunknum);
			else hunk_match = 0;

			if (hunk_match && numbering && verbose &&
			    mode != mode_grep)
				printf ("\t%lu\tHunk #%lu\n",
					hunk_linenum, hunknum);
		}

		if (hunk_match &&
		    (mode == mode_filter || output_matching != output_none)) {
			FILE *output_to= stdout;

			if (mode == mode_grep && !grepmatch)
				output_to = match_tmpf;

			// Display the line counts.
			if (!header_displayed && mode == mode_filter) {
				if (number_lines != After)
					output_header_line (header[0]);
				if (number_lines != Before)
					output_header_line (header[1]);
				if (number_lines == None)
					fputs ("***************\n", stdout);
				header_displayed = 1;
			}

			if (number_lines == None) switch (i) {
			case 0:
				fprintf (output_to, "*** %lu", line_start);
				if (line_end != line_start)
					fprintf (output_to, ",%lu", line_end);
				fputs (" ****", output_to);

				if (annotating)
					fprintf (output_to,
						 " Hunk #%lu, %s\n",
						 hunknum, bestname);
				else if (n)
					fputs (n, output_to);
				else
					fputc ('\n', output_to);

				break;
			case 1:
				fprintf (output_to, "--- %lu",
					 line_start + munge_offset);
				if (line_end != line_start)
					fprintf (output_to, ",%lu",
						 line_end + munge_offset);
				fputs (" ----\n", output_to);
				break;
			}

			switch (number_lines) {
			case None:
				break;

			case Before:
				if (i != 0)
					break;

				// Note the initial line number.
				track_linenum = line_start;
				if (!first_hunk ||
				    (output_matching == output_file &&
				     hunknum > 1))
					fputs ("...\n", output_to);
				break;

			case After:
				if (i != 1)
					break;

				track_linenum = line_start + munge_offset;
				if (!first_hunk ||
				    (output_matching == output_file &&
				     hunknum > 1))
					fputs ("...\n", output_to);
				break;
			}
		}

		if (i && line_count == unchanged)
			break;

		while (line_count--) {
			if (getline (line, linelen, f) == -1) {
				ret = EOF;
				goto out;
			}

			++*linenum;

			if (hunk_match && mode == mode_grep &&
			    !regexec (&regex, *line + 2, 0, NULL, 0)) {
				if (output_matching == output_none) {
					if (!displayed_filename) {
						displayed_filename = 1;
						display_filename(start_linenum,
								 status,
								 bestname);
					}

					if (numbering && verbose &&
					    hunknum > last_hunkmatch) {
						last_hunkmatch = hunknum;
						printf ("\t%lu\tHunk #%lu\n",
							hunk_linenum, hunknum);
					}
				} else {
					if (!header_displayed) {
						if (number_lines != After)
							output_header_line (header[0]);
						if (number_lines != Before)
							output_header_line (header[1]);
						if (number_lines == None)
							fputs ("***************\n",
							       stdout);
						header_displayed = 1;
					}

					if (match_tmpf) {
						rewind (match_tmpf);
						while (!feof (match_tmpf)) {
							int ch;
							ch = fgetc(match_tmpf);
							if (ch == EOF)
								break;
							putchar (ch);
						}
						fclose (match_tmpf);
						match_tmpf = NULL;
					}

					grepmatch = 1;
				}
			}

			if (!i && first) {
				first = 0;
				if (!strncmp (*line, "--- ", 4)) {
					/* From lines were
					 * omitted. */
					i++;
					goto do_line_counts;
				}
			}

			if (**line == ' ')
				unchanged++;

			if (hunk_match &&
			    (mode == mode_filter ||
			     output_matching != output_none)) {
				FILE *output_to = stdout;
				if (mode == mode_grep && !grepmatch)
					output_to = match_tmpf;

				if (number_lines == None)
					fputs (*line, output_to);
				else if ((number_lines == Before && !i) ||
					 (number_lines == After && i)) {
					fprintf (output_to, "%lu\t:%s",
						 track_linenum++, 2 + *line);
				}
			}

			if ((mode == mode_filter && !hunk_match) ||
			    output_matching == output_hunk)
				switch (**line) {
				case '!':
					changed[i]++;
					break;
				case '+':
					changed[1]++;
					break;
				case '-':
					changed[0]++;
					break;
				}
		}
	}

	if (output_matching != output_hunk || !grepmatch)
		munge_offset += changed[0] - changed[1];
	goto next_hunk;

out:
	if (match_tmpf)
		fclose (match_tmpf);

	return ret;
}

static int filterdiff (FILE *f)
{
	static unsigned long linenum = 1;
	char *names[2];
	char *header[2] = { NULL, NULL };
	char *line = NULL;
	size_t linelen = 0;
	char *p;
	const char *p_stripped;
	int match;
	int i;

	if (getline (&line, &linelen, f) == -1)
		return 0;

	for (;;) {
		int h;
		char status = '!';
		unsigned long start_linenum;
		int orig_file_exists;
		int is_context = 0;
		int (*do_diff) (FILE *, char *[2], int, char **, ssize_t *,
				unsigned long *, unsigned long,
				char, const char *);

		orig_file_exists = 0; // shut gcc up

		// Search for start of patch ("--- " for unified diff,
		// "*** " for context).
		for (;;) {
			if (!strncmp (line, "--- ", 4)) {
				is_context = 0;
				break;
			}

			if (!strncmp (line, "*** ", 4)) {
				is_context = 1;
				break;
			}

			/* Show non-diff lines if excluding, or if
			 * in verbose mode. */
			if (mode == mode_filter && (pat_exclude || verbose))
				fputs (line, stdout);

			if (getline (&line, &linelen, f) == -1)
				goto eof;
			linenum++;
		}

		start_linenum = linenum;
		header[0] = xstrdup (line);
		h = strcspn (line + 4, "\t\n");
		names[0] = xstrndup (line + 4, h);
		h = 4 + h + strspn (line + 4 + h, "\t\n");
		if (mode != mode_filter && show_status)
			orig_file_exists = file_exists (line + h);

		if (getline (&line, &linelen, f) == -1) {
			free (names[0]);
			goto eof;
		}
		linenum++;

		if (strncmp (line, is_context ? "--- " : "+++ ", 4)) {
			free (names[0]);
			free (header[0]);
			header[0] = NULL;
			continue;
		}

		header[1] = xstrdup (line);

		h = strcspn (line + 4, "\t\n");
		names[1] = xstrndup (line + 4, h);

		// Decide whether this matches this pattern.
		p = best_name (2, names);
		p_stripped = stripped (p, ignore_components);

		match = !patlist_match(pat_exclude, p_stripped);
		if (match && pat_include != NULL)
			match = patlist_match(pat_include, p_stripped);

		// print if it matches.
		if (match) {
			if (mode != mode_filter) {
				h = 4 + h + strspn (line + 4 + h, " \t\n");
				if (!orig_file_exists)
					status = '+';
				else if (!file_exists (line + h))
					status = '-';
			}

			if (mode == mode_list)
				display_filename (start_linenum, status, p);
		}

		if (is_context)
			do_diff = do_context;
		else
			do_diff = do_unified;

		switch (do_diff (f, header, match, &line,
				 &linelen, &linenum,
				 start_linenum, status, p)) {
		case EOF:
			free (names[0]);
			free (names[1]);
			goto eof;
		case 1:
			goto next_diff;
		}

	next_diff:
		for (i = 0; i < 2; i++) {
			free (names[i]);
			free (header[i]);
			header[i] = NULL;
		}
	}

 eof:
	for (i = 0; i < 2; i++)
		if (header[i])
			free (header[i]);

	if (line)
		free (line);

	return 0;
}

static char * syntax_str =
"Options:\n"
"  -x PAT    exclude files matching PAT\n"
"  -i PAT    include only files matching PAT\n"
"  --hunks=H, -# H\n"
"            include only hunks in range H\n"
"  --lines=L include only hunks with (original) lines in range L\n"
"  --annotate\n"
"            annotate each hunk with the filename and hunk number\n"
"  --as-numbered-lines=before|after\n"
"            display lines as they would look before, or after, the\n"
"            patch is applied\n"
"  --format=context|unified\n"
"            set output format\n"
"  --output-matching=hunk|file\n"
"            show matching hunks or file-level diffs (grepdiff)\n"
"  -z        decompress .gz and .bz2 files\n"
"  -n        show line numbers (lsdiff)\n"
"  -p N      initial pathname components to ignore\n"
"  --strip=N initial pathname components to strip\n"
"  --addprefix=PREFIX\n"
"            prefix pathnames with PREFIX\n"
"  -s        show file additions and removals (lsdiff)\n"
"  -v        verbose output\n"
"  --filter  run as 'filterdiff'\n"
"  --list    run as 'lsdiff'\n"
"  --grep    run as 'grepdiff'\n"
;

NORETURN
static void syntax (int err)
{
	const char *usage = "usage: %s [OPTION]... [files ...]\n";
	if (mode == mode_grep)
		usage = "usage: %s [OPTION]... REGEX [files ...]\n";
	fprintf (err ? stderr : stdout, usage, progname);
	fprintf (err ? stderr : stdout, syntax_str);
	exit (err);
}

static void
parse_range (struct range **r, const char *rstr)
{
	unsigned long n;
	char *end;

	if (*rstr == '-')
		n = -1UL;
	else {
		n = strtoul (rstr, &end, 0);
		if (rstr == end) {
			if (*end)
				error (EXIT_FAILURE, 0,
				       "not understood: '%s'", end);
			else
				error (EXIT_FAILURE, 0,
				       "missing number in range list");

			*r = NULL;
			return;
		}

		rstr = end;
	}

	*r = xmalloc (sizeof **r);
	(*r)->start = (*r)->end = n;
	(*r)->next = NULL;
	if (*rstr == '-') {
		rstr++;
		n = strtoul (rstr, &end, 0);
		if (rstr == end)
			n = -1UL;

		(*r)->end = n;
		rstr = end;

		if ((*r)->start != -1UL && (*r)->start > (*r)->end)
			error (EXIT_FAILURE, 0, "invalid range: %lu-%lu",
			       (*r)->start, (*r)->end);
	}

	if (*rstr == ',')
		parse_range (&(*r)->next, rstr + 1);
	else if (*rstr != '\0')
		error (EXIT_FAILURE, 0, "not understood: '%s'", rstr);
}

static void set_list (void)
{
	/* This is lsdiff. */
	set_progname ("lsdiff");
	mode = mode_list;
}

static void set_filter (void)
{
	/* This is filterdiff. */
	set_progname ("filterdiff");
	mode = mode_filter;
}

static void set_grep (void)
{
	/* This is grepdiff. */
	set_progname ("grepdiff");
	mode = mode_grep;
}

static void determine_mode_from_name (const char *argv0)
{
	/* This is filterdiff, unless it is named 'lsdiff' or 'grepdiff'. */
	const char *p = strrchr (argv0, '/');
	if (!p++)
		p = argv0;
	if (strstr (p, "lsdiff"))
		set_list ();
	else if (strstr (p, "grepdiff"))
		set_grep ();
	else
		set_filter ();
}

static FILE *convert_format (FILE *f, char format)
{
	switch (format) {
	default:
		break;
	case 'c':
		f = convert_to_context (f, "rb", 0);
		break;
	case 'u':
		f = convert_to_unified (f, "rb", 0);
		break;
	}

	return f;
}

int main (int argc, char *argv[])
{
	int i;
	FILE *f = stdin;
	char format = '\0';

	setlocale (LC_TIME, "C");
	determine_mode_from_name (argv[0]);
	while (1) {
		static struct option long_options[] = {
	       		{"help", 0, 0, 1000 + 'H'},
			{"version", 0, 0, 1000 + 'V'},
			{"verbose", 0, 0, 'v'},
			{"list", 0, 0, 'l'},
			{"filter", 0, 0, 'f'},
			{"grep", 0, 0, 'g'},
			{"strip", 1, 0, 1000 + 'S'},
			{"addprefix", 1, 0, 1000 + 'A'},
			{"hunks", 1, 0, '#'},
			{"lines", 1, 0, 1000 + ':'},
			{"as-numbered-lines", 1, 0, 1000 + 'L'},
			{"annotate", 0, 0, 1000 + 'a'},
			{"format", 1, 0, 1000 + 'f'},
			{"output-matching", 1, 0, 1000 + 'o'},
			{0, 0, 0, 0}
		};
		char *end;
		int c = getopt_long (argc, argv, "vp:i:I:x:X:zns#:",
				long_options, NULL);
		if (c == -1)
			break;
		
		switch (c) {
		case 'g':
			set_grep ();
			break;
		case 'f':
			set_filter ();
			break;
		case 'l':
			set_list ();
			break;
		case 1000 + 'V':
			printf("%s - patchutils version %s\n", progname,
			       VERSION);
			exit(0);
		case 1000 + 'H':
			syntax (0);
			break;
		case 1000 + 'S':
			strip_components = strtoul (optarg, &end, 0);
			if (optarg == end)
				syntax (1);
			break;
		case 1000 + 'A':
			prefix_to_add = optarg;
			break;
		case 'p':
			ignore_components = strtoul (optarg, &end, 0);
			if (optarg == end)
				syntax (1);
			break;
		case 'x':
			patlist_add (&pat_exclude, optarg);
			break;
		case 'i':
			patlist_add (&pat_include, optarg);
			break;
		case 'z':
			unzip = 1;
			break;
		case 'n':
			numbering = 1;
			break;
		case 's':
			show_status = 1;
			break;
		case 'v':
			verbose++;
			break;
		case '#':
			if (hunks)
				syntax (1);
			parse_range (&hunks, optarg);
			break;
		case 1000 + ':':
			if (lines)
				syntax (1);
			parse_range (&lines, optarg);
			break;
		case 1000 + 'L':
			if (!strcmp (optarg, "before"))
				number_lines = Before;
			else if (!strcmp (optarg, "after"))
				number_lines = After;
			else syntax (1);
			break;
		case 1000 + 'a':
			annotating = 1;
			break;
		case 1000 + 'f':
			if (!strcmp (optarg, "context") && !format)
				format = 'c';
			else if (!strcmp (optarg, "unified") && !format)
				format = 'u';
			else syntax (1);
			break;
		case 1000 + 'o':
			if (!strncmp (optarg, "hunk", 4))
				output_matching = output_hunk;
			else if (!strncmp (optarg, "file", 4))
				output_matching = output_file;
			else syntax (1);
			break;
		default:
			syntax(1);
		}
				
	}

	/* Preserve the old semantics of -p. */
	if (mode != mode_filter && ignore_components && !strip_components &&
	    !pat_include && !pat_exclude) {
		fprintf (stderr,
			 "-p given without -i or -x; guessing that you "
			 "meant --strip instead.\n");
		strip_components = ignore_components;
		ignore_components = 0;
	}

	if (mode != mode_grep && output_matching != output_none)
		error (EXIT_FAILURE, 0, "--output-matching only applies to "
		       "grep mode");

	if (numbering &&
	    !(mode == mode_list ||
	      output_matching == output_none))
		error (EXIT_FAILURE, 0, "-n only applies to list mode");

	if (mode != mode_filter &&
	    output_matching == output_none &&
	    number_lines != None)
		error (EXIT_FAILURE, 0, "--as-numbered-lines is "
		       "inappropriate in this context");

	if (mode == mode_grep) {
		int err;

		if (optind == argc)
			syntax (1);

		err = regcomp (&regex, argv[optind++], REG_NOSUB);
		if (err) {
			char errstr[300];
			regerror (err, &regex, errstr, sizeof (errstr));
			error (EXIT_FAILURE, 0, errstr);
			exit (1);
		}
	}
	
	if (optind == argc) {
		f = convert_format (stdin, format);
		filterdiff (f);
		fclose (f);
	} else {
		for (i = optind; i < argc; i++) {
			if (unzip) {
				f = xopen_unzip (argv[i], "rb");
			} else {
				f = xopen(argv[i], "rbm");
			}

			f = convert_format (f, format);
			filterdiff(f);
			fclose (f);
		}
	}

	return 0;
}

