/*
  music-iterator.hh -- declare Music_iterator

  source file of the GNU LilyPond music typesetter

  (c) 1997--2004 Han-Wen Nienhuys <hanwen@cs.uu.nl>
*/


#ifndef MUSIC_ITERATOR_HH
#define MUSIC_ITERATOR_HH

#include "lily-proto.hh"
#include "array.hh"
#include "moment.hh"
#include "virtual-methods.hh"
#include "interpretation-context-handle.hh"
#include "smobs.hh"

/**
   ---

   Music_iterator is an object type that traverses the Music structure and
   reports the events it finds to interpretation contexts. It is not yet
   user-serviceable.


   ---
	
  Conceptually a music-iterator operates on a queue of musical events
  that are pending. This queue does not actually exist, but it is a
  way of viewing and traversing music-expressions.

  
  ok () -- events left ?

  pending_mom () -- time tag of the next event to be processed.
    PRECONDITION: this->ok () holds.
  
  process (M) -- process all at M (Precondition: no events exist
    before M, this->ok () holds).  Side-effects:
    
    * This removes all events at M from the pending queue.

    * Typically this reports the music to an interpretation context,
    thus changing the state of the interpretation context.


  TODO:

  merge pending_moment and process?
  
*/
class Music_iterator
{
protected:
  Moment music_length_;
  Moment start_mom_;

  DECLARE_SMOBS (Music_iterator,dummy);
  Music_iterator (Music_iterator const&);
public:
  
  Moment music_get_length () const;
  virtual Moment music_start_mom () const;
  Music_iterator ();

  /**
     Do the reporting.  Will try MUSIC_L_ in its own translator first,
     then its children. Returns the iterator that succeeded
  */
  Music_iterator *  try_music (Music  *) const;
  
  /**
    The translation unit that we this iterator is reporting  to now.
   */
  Context * get_outlet () const;

  void set_translator (Context *);
  
  /** Get an iterator matching the type of MUS, and use TRANS to find
    an accompanying translation unit
   */
  static SCM get_static_get_iterator (Music * mus);
  void init_translator (Music  *, Context *); 
  void quit ();
  void substitute_outlet (Context * from, Context *to);
  virtual void derived_substitute (Context *, Context *);
  virtual Moment pending_moment () const;
  virtual bool ok () const;

  virtual bool run_always () const; 
  virtual void process (Moment until);
  virtual void derived_mark ()const;
  virtual void construct_children ();
  
  /**
    Construct sub-iterators, and set the translator to 
    report to.
   */

  DECLARE_SCHEME_CALLBACK(constructor, ());
  
  /**
    Get an iterator for MUS, inheriting the translation unit from THIS.
   */
  SCM get_iterator (Music *) const;

  virtual Music_iterator* try_music_in_children (Music *) const;

  Music * get_music () const;
protected:
  virtual void do_quit();
  void descend_to_child (Context*);
private:
  Interpretation_context_handle handle_;
  Music  * music_;
};

bool is_child_context (Context *me, Context *child);

#define IMPLEMENT_CTOR_CALLBACK(Class)		\
LY_DEFINE_MEMBER_FUNCTION(Class,constructor, #Class "::constructor",\
	  0,0,0,\
	  (),\
	  "")\
{						\
  SCM val = (new Class)->self_scm();   \
  scm_gc_unprotect_object (val);\
  return val ;				\
}						\

DECLARE_UNSMOB(Music_iterator, iterator);

#endif // MUSIC_ITERATOR_HH
