#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "base.h"
#include "log.h"
#include "buffer.h"

#include "plugin.h"

#ifdef USE_OPENSSL
# include <openssl/md5.h>
#else
# include "md5_global.h"
# include "md5.h"
#endif

/* plugin config for all request/connections */

typedef struct {
	PLUGIN_DATA;
	buffer *conf_cookie_name;
} plugin_data;

/* init the plugin data */
INIT_FUNC(mod_usertrack_init) {
	plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	p->conf_cookie_name = buffer_init();
	
	return p;
}

/* detroy the plugin data */
FREE_FUNC(mod_usertrack_free) {
	plugin_data *p = p_d;
	
	UNUSED(srv);
	
	if (!p) return HANDLER_GO_ON;
	
	buffer_free(p->conf_cookie_name);
	
	free(p);
	
	return HANDLER_GO_ON;
}

/* handle plugin config and check values */

SETDEFAULTS_FUNC(mod_usertrack_set_defaults) {
	plugin_data *p = p_d;
	size_t i = 0;
	
	config_values_t cv[] = { 
		{ "usertrack.cookiename",       NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },       /* 0 */
		{ NULL,                         NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
	};
	
	if (!p) return HANDLER_ERROR;
	
	/* 0 */
	cv[i++].destination = p->conf_cookie_name;
	
	if (0 != config_insert_values(srv, cv)) {
		return HANDLER_ERROR;
	}
	
	if (p->conf_cookie_name->used == 0) {
		buffer_copy_string(p->conf_cookie_name, "TRACKID");
	} else {
		for (i = 0; i < p->conf_cookie_name->used - 1; i++) {
			char c = p->conf_cookie_name->ptr[i] | 32;
			if (c < 'a' || c > 'z') {
				log_error_write(srv, __FILE__, __LINE__, "sb", 
						"invalid character in usertrack.cookiename:", 
						p->conf_cookie_name);
				
				return HANDLER_ERROR;
			}
		}
	}
	
	return HANDLER_GO_ON;
}

URIHANDLER_FUNC(mod_usertrack_uri_handler) {
	plugin_data *p = p_d;
	data_string *ds;
	unsigned char h[16];
	MD5_CTX Md5Ctx;
	char hh[32];
	
	if (con->uri.path->used == 0) return HANDLER_GO_ON;
	
	if (NULL != (ds = (data_string *)array_get_element(con->request.headers, "Cookie"))) {
		char *g;
		/* we have a cookie, does it contain a valid name ? */
		
		/* parse the cookie 
		 * 
		 * check for cookiename + (WS | '=')
		 * 
		 */
		
		if (NULL != (g = strstr(ds->value->ptr, p->conf_cookie_name->ptr))) {
			char *nc;
			
			/* skip WS */
			for (nc = g + p->conf_cookie_name->used-1; *nc == ' ' || *nc == '\t'; nc++);
			
			if (*nc == '=') {
				/* ok, found the key of our own cookie */
				
				if (strlen(nc) > 32) {
					/* i'm lazy */
					return HANDLER_GO_ON;
				}
			}
		}
	} 
	
	/* set a cookie */
	if (NULL == (ds = (data_string *)array_get_unused_element(con->response.headers, TYPE_STRING))) {
		ds = data_response_init();
	}
	buffer_copy_string(ds->key, "Set-Cookie");
	buffer_copy_string_buffer(ds->value, p->conf_cookie_name);
	buffer_append_string(ds->value, "=");
	

	/* taken from mod_auth.c */
	
	/* generate shared-secret */
	MD5_Init(&Md5Ctx);
	MD5_Update(&Md5Ctx, (unsigned char *)con->uri.path->ptr, con->uri.path->used - 1);
	MD5_Update(&Md5Ctx, (unsigned char *)"+", 1);
	
	/* we assume sizeof(time_t) == 4 here, but if not it ain't a problem at all */
	ltostr(hh, srv->cur_ts);
	MD5_Update(&Md5Ctx, (unsigned char *)hh, strlen(hh));
	ltostr(hh, rand());
	MD5_Update(&Md5Ctx, (unsigned char *)hh, strlen(hh));
	
	MD5_Final(h, &Md5Ctx);
	
	buffer_append_string_hex(ds->value, (char *)h, 16);
	buffer_append_string(ds->value, "; path=/");
	
	array_insert_unique(con->response.headers, (data_unset *)ds);
	
	return HANDLER_GO_ON;
}

/* this function is called at dlopen() time and inits the callbacks */

int mod_usertrack_plugin_init(plugin *p) {
	p->name        = buffer_init_string("usertrack");
	
	p->init        = mod_usertrack_init;
	p->handle_uri_clean  = mod_usertrack_uri_handler;
	p->set_defaults  = mod_usertrack_set_defaults;
	p->cleanup     = mod_usertrack_free;
	
	p->data        = NULL;
	
	return 0;
}
