#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "base.h"
#include "log.h"
#include "buffer.h"
#include "response.h"

#include "plugin.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/**
 * this is a expire module for a lighttpd 
 * 
 * set 'Expires:' HTTP Headers on demand
 */



/* plugin config for all request/connections */

typedef struct {
	PLUGIN_DATA;
	buffer *expire_tstmp;
	array *conf_expire_url;
} plugin_data;

/* init the plugin data */
INIT_FUNC(mod_expire_init) {
	plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	p->expire_tstmp = buffer_init();
	p->conf_expire_url = array_init();
	
	buffer_prepare_copy(p->expire_tstmp, 255);
	
	return p;
}

/* detroy the plugin data */
FREE_FUNC(mod_expire_free) {
	plugin_data *p = p_d;
	
	UNUSED(srv);

	if (!p) return HANDLER_GO_ON;
	
	buffer_free(p->expire_tstmp);
	array_free(p->conf_expire_url);
	
	free(p);
	
	return HANDLER_GO_ON;
}

static int mod_expire_get_offset(server *srv, plugin_data *p, buffer *expire, int *offset) {
	char *ts;
	int type = -1;
	int retts = 0;
		
	UNUSED(p);

	/* 
	 * parse
	 * 
	 * '(access|modification) [plus] {<num> <type>}*'
	 * 
	 * e.g. 'access 1 years'
	 */
	
	if (expire->used == 0) {
		log_error_write(srv, __FILE__, __LINE__, "s", 
				"empty:");
		return -1;
	}
	
	ts = expire->ptr;
	
	if (0 == strncmp(ts, "access ", 7)) {
		type  = 0;
		ts   += 7;
	} else if (0 == strncmp(ts, "modification ", 13)) {
		type  = 0;
		ts   += 13;
	} else {
		/* invalid type-prefix */
		log_error_write(srv, __FILE__, __LINE__, "ss",
				"invalid <base>:", ts);
		return -1;
	}
	
	if (0 == strncmp(ts, "plus ", 5)) {
		/* skip the optional plus */
		ts   += 5;
	}
	
	/* the rest is just <number> (years|months|days|hours|minutes|seconds) */
	while (1) {
		char *space, *err;
		int num;
		
		if (NULL == (space = strchr(ts, ' '))) {
			log_error_write(srv, __FILE__, __LINE__, "ss", 
					"missing space after <num>:", ts);
			return -1;
		}
		
		num = strtol(ts, &err, 10);
		if (*err != ' ') {
			log_error_write(srv, __FILE__, __LINE__, "ss", 
					"missing <type> after <num>:", ts);
			return -1;
		}
		
		ts = space + 1;
		
		if (NULL != (space = strchr(ts, ' '))) {
			int slen;
			/* */
			
			slen = space - ts;
			
			if (slen == 5 && 
			    0 == strncmp(ts, "years", slen)) {
				num *= 60 * 60 * 24 * 30 * 12;
			} else if (slen == 6 &&
				   0 == strncmp(ts, "months", slen)) {
				num *= 60 * 60 * 24 * 30;
			} else if (slen == 4 &&
				   0 == strncmp(ts, "days", slen)) {
				num *= 60 * 60 * 24;
			} else if (slen == 5 &&
				   0 == strncmp(ts, "hours", slen)) {
				num *= 60 * 60;
			} else if (slen == 7 &&
				   0 == strncmp(ts, "minutes", slen)) {
				num *= 60;
			} else if (slen == 7 &&
				   0 == strncmp(ts, "seconds", slen)) {
				num *= 1;
			} else {
				log_error_write(srv, __FILE__, __LINE__, "ss", 
						"unknown type:", ts);
				return -1;
			}
			
			retts += num;
			
			ts = space + 1;
		} else {
			if (0 == strcmp(ts, "years")) {
				num *= 60 * 60 * 24 * 30 * 12;
			} else if (0 == strcmp(ts, "months")) {
				num *= 60 * 60 * 24 * 30;
			} else if (0 == strcmp(ts, "days")) {
				num *= 60 * 60 * 24;
			} else if (0 == strcmp(ts, "hours")) {
				num *= 60 * 60;
			} else if (0 == strcmp(ts, "minutes")) {
				num *= 60;
			} else if (0 == strcmp(ts, "seconds")) {
				num *= 1;
			} else {
				log_error_write(srv, __FILE__, __LINE__, "ss", 
						"unknown type:", ts);
				return -1;
			}
			
			retts += num;
			
			break;
		}
	}
	
	if (offset != NULL) *offset = retts;
	
	return type;
}


/* handle plugin config and check values */

SETDEFAULTS_FUNC(mod_expire_set_defaults) {
	plugin_data *p = p_d;
	size_t i = 0, k;
	
	config_values_t cv[] = { 
		{ "expire.url",                 NULL, T_CONFIG_ARRAY, T_CONFIG_SCOPE_CONNECTION },       /* 0 */
		{ NULL,                         NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
	};
	
	if (!p) return HANDLER_ERROR;
	
	/* 0 */
	cv[i++].destination = p->conf_expire_url;
	
	if (0 != config_insert_values(srv, cv)) {
		return HANDLER_ERROR;
	}
	
	for (k = 0; k < p->conf_expire_url->used; k++) {
		data_string *ds = (data_string *)p->conf_expire_url->data[k];
			
		/* parse lines */
		if (-1 == mod_expire_get_offset(srv, p, ds->value, NULL)) {
			log_error_write(srv, __FILE__, __LINE__, "sb", 
					"parsing expire.url failed:", ds->value);
			return HANDLER_ERROR;
		}
	}
	
	
	return HANDLER_GO_ON;
}

URIHANDLER_FUNC(mod_expire_uri_handler) {
	plugin_data *p = p_d;
	int s_len;
	size_t k;
	
	if (con->uri.path->used == 0) return HANDLER_GO_ON;
	
	s_len = con->uri.path->used - 1;
	
	for (k = 0; k < p->conf_expire_url->used; k++) {
		data_string *ds = (data_string *)p->conf_expire_url->data[k];
		int ct_len = ds->key->used - 1;
		
		if (ct_len > s_len) continue;
		if (ds->key->used == 0) continue;
		
		if (0 == strncmp(con->uri.path->ptr, ds->key->ptr, ct_len)) {
			int ts;
			time_t t;
			size_t len;
			
			mod_expire_get_offset(srv, p, ds->value, &ts);
			t = (ts += srv->cur_ts);
			
			if (0 == (len = strftime(p->expire_tstmp->ptr, p->expire_tstmp->size - 1, 
					   "%a, %d %b %Y %H:%M:%S GMT", gmtime(&(t))))) {
				/* could not set expire header, out of mem */
				
				return HANDLER_GO_ON;
				
			}
			    
			p->expire_tstmp->used = len + 1;
			
			response_header_overwrite(srv, con, CONST_STR_LEN("Expires"), CONST_BUF_LEN(p->expire_tstmp));
			
			return HANDLER_GO_ON;
		}
	}
	
	/* not found */
	return HANDLER_GO_ON;
}

/* this function is called at dlopen() time and inits the callbacks */

int mod_expire_plugin_init(plugin *p) {
	p->name        = buffer_init_string("expire");
	
	p->init        = mod_expire_init;
	p->handle_uri_clean  = mod_expire_uri_handler;
	p->set_defaults  = mod_expire_set_defaults;
	p->cleanup     = mod_expire_free;
	
	p->data        = NULL;
	
	return 0;
}
