#include <sys/types.h>
#include <sys/stat.h>

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>

#include "plugin.h"
#include "http_auth.h"
#include "log.h"
#include "response.h"


/**
 * the basic and digest auth framework
 * 
 * - config handling
 * - protocol handling
 * 
 * http_auth.c 
 * http_auth_digest.c 
 * 
 * do the real work
 */

INIT_FUNC(mod_auth_init) {
	mod_auth_plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	p->tmp_buf = buffer_init();
	p->conf_auth_plain_groupfile = buffer_init();
	p->conf_auth_plain_userfile = buffer_init();
	p->conf_auth_backend = buffer_init();
	
	p->conf_auth_ldap_hostname = buffer_init();
	p->conf_auth_ldap_basedn = buffer_init();
	p->conf_auth_ldap_filter = buffer_init();

#ifdef USE_LDAP
	p->ldap_filter = buffer_init();
	p->ldap_filter_pre = buffer_init();
	p->ldap_filter_post = buffer_init();
#endif
	
	p->conf_auth_require = array_init();
	
	p->auth_user = buffer_init();
	
	return p;
}

FREE_FUNC(mod_auth_free) {
	mod_auth_plugin_data *p = p_d;
	
	UNUSED(srv);

	if (!p) return HANDLER_GO_ON;
	
	array_free(p->conf_auth_require);
	
	buffer_free(p->tmp_buf);
	buffer_free(p->conf_auth_plain_groupfile);
	buffer_free(p->conf_auth_plain_userfile);
	buffer_free(p->conf_auth_backend);
	
	buffer_free(p->conf_auth_ldap_hostname);
	buffer_free(p->conf_auth_ldap_basedn);
	buffer_free(p->conf_auth_ldap_filter);
	
	buffer_free(p->auth_user);
	
#ifdef USE_LDAP
	buffer_free(p->ldap_filter);
	buffer_free(p->ldap_filter_pre);
	buffer_free(p->ldap_filter_post);
	
	if (p->ldap) ldap_unbind_s(p->ldap);
#endif
	
	free(p);
	
	return HANDLER_GO_ON;
}

static handler_t mod_auth_uri_handler(server *srv, connection *con, void *p_d) {
	size_t k;
	int auth_required = 0, auth_satisfied = 0;
	char *http_authorization = NULL;
	data_string *ds;
	mod_auth_plugin_data *p = p_d;
	array *req;
	
	if (p->conf_auth_require == NULL) return HANDLER_GO_ON;
	
	/*
	 * AUTH
	 *  
	 */
	
	/* do we have to ask for auth ? */
	
	auth_required = 0;
	auth_satisfied = 0;
	
	/* search auth-directives for path */
	for (k = 0; k < p->conf_auth_require->used; k++) {
		if (p->conf_auth_require->data[k]->key->used == 0) continue;
		
		if (0 == strncmp(con->uri.path->ptr, p->conf_auth_require->data[k]->key->ptr, p->conf_auth_require->data[k]->key->used - 1)) {
			auth_required = 1;
			break;
		}
	}
	
	/* nothing to do for us */
	if (auth_required == 0) return HANDLER_GO_ON;
	
	req = ((data_array *)(p->conf_auth_require->data[k]))->value;
	
	/* try to get Authorization-header */
		
	if (NULL != (ds = (data_string *)array_get_element(con->request.headers, "Authorization"))) {
		http_authorization = ds->value->ptr;
	}
	
	if (ds && ds->value && ds->value->used) {
		char *auth_realm;
		data_string *method;
		
		method = (data_string *)array_get_element(req, "method");
		
		/* parse auth-header */
		if (NULL != (auth_realm = strchr(http_authorization, ' '))) {
			int auth_type_len = auth_realm - http_authorization;
			
			if ((auth_type_len == 5) &&
			    (0 == strncmp(http_authorization, "Basic", auth_type_len))) {
				
				if (0 == strcmp(method->value->ptr, "basic")) {
					auth_satisfied = http_auth_basic_check(srv, con, p, req, con->uri.path, auth_realm+1);
				}
			} else if ((auth_type_len == 6) &&
				   (0 == strncmp(http_authorization, "Digest", auth_type_len))) {
				if (0 == strcmp(method->value->ptr, "digest")) {
					if (-1 == (auth_satisfied = http_auth_digest_check(srv, con, p, req, con->uri.path, auth_realm+1))) {
						con->http_status = 400;
						
						/* a field was missing */
						
						return HANDLER_FINISHED;
					}
				}
			} else {
				log_error_write(srv, __FILE__, __LINE__, "ss", 
						"unknown authentification type:",
						http_authorization);
			}
		}
	}
	
	if (!auth_satisfied) {
		data_string *method, *realm;
		method = (data_string *)array_get_element(req, "method");
		realm = (data_string *)array_get_element(req, "realm");
		
		con->http_status = 401;
			
		if (0 == strcmp(method->value->ptr, "basic")) {
			buffer_copy_string(p->tmp_buf, "Basic realm=\"");
			buffer_append_string_buffer(p->tmp_buf, realm->value);
			buffer_append_string(p->tmp_buf, "\"");
			
			response_header_insert(srv, con, CONST_STR_LEN("WWW-Authenticate"), CONST_BUF_LEN(p->tmp_buf));
		} else if (0 == strcmp(method->value->ptr, "digest")) {
			char hh[33];
			http_auth_digest_generate_nonce(srv, p, srv->tmp_buf, hh);
			
			buffer_copy_string(p->tmp_buf, "Digest realm=\"");
			buffer_append_string_buffer(p->tmp_buf, realm->value);
			buffer_append_string(p->tmp_buf, "\", nonce=\"");
			buffer_append_string(p->tmp_buf, hh);
			buffer_append_string(p->tmp_buf, "\", qop=\"auth\"");
			
			response_header_insert(srv, con, CONST_STR_LEN("WWW-Authenticate"), CONST_BUF_LEN(p->tmp_buf));
		} else {
			/* evil */
		}
		return HANDLER_FINISHED;
	} else {
		/* the REMOTE_USER header */
		
		buffer_copy_string_buffer(con->authed_user, p->auth_user);
	}
	
	return HANDLER_GO_ON;
}

SETDEFAULTS_FUNC(mod_auth_set_defaults) {
	mod_auth_plugin_data *p = p_d;
	data_unset *du;
	
	config_values_t cv[] = { 
		{ "auth.backend",                   NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ "auth.backend.plain.groupfile",   NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ "auth.backend.plain.userfile",    NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ "auth.require",                   NULL, T_CONFIG_LOCAL, T_CONFIG_SCOPE_CONNECTION },
		{ "auth.backend.ldap.hostname",     NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ "auth.backend.ldap.base-dn",      NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ "auth.backend.ldap.filter",       NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ NULL,                             NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
	};
	
	cv[0].destination = p->conf_auth_backend;
	cv[1].destination = p->conf_auth_plain_groupfile;
	cv[2].destination = p->conf_auth_plain_userfile;
	cv[3].destination = p->conf_auth_require;
	cv[4].destination = p->conf_auth_ldap_hostname;
	cv[5].destination = p->conf_auth_ldap_basedn;
	cv[6].destination = p->conf_auth_ldap_filter;
	
	if (0 != config_insert_values(srv, cv)) {
		return HANDLER_ERROR;
	}
	
	if (NULL != (du = array_get_element(srv->config, "auth.require"))) {
		if (du->type == TYPE_ARRAY) {
			size_t n;
			data_array *da = (data_array *)du;
			
			for (n = 0; n < da->value->used; n++) {
				if (da->value->data[n]->type == TYPE_ARRAY) {
					size_t m;
					data_array *da_file = (data_array *)da->value->data[n];
					const char *method, *realm, *require;
			
					method = realm = require = NULL;
					
					for (m = 0; m < da_file->value->used; m++) {
						if (da_file->value->data[m]->type == TYPE_STRING) {
							if (0 == strcmp(da_file->value->data[m]->key->ptr, "method")) {
								method = ((data_string *)(da_file->value->data[m]))->value->ptr;
							} else if (0 == strcmp(da_file->value->data[m]->key->ptr, "realm")) {
								realm = ((data_string *)(da_file->value->data[m]))->value->ptr;
							} else if (0 == strcmp(da_file->value->data[m]->key->ptr, "require")) {
								require = ((data_string *)(da_file->value->data[m]))->value->ptr;
							} else {
								log_error_write(srv, __FILE__, __LINE__, "sssbs", "unexpected type for key: ", "auth.require", "[", da_file->value->data[m]->key, "](string)");
								return HANDLER_ERROR;
							}
						} else {
							log_error_write(srv, __FILE__, __LINE__, "sssbs", "unexpected type for key: ", "auth.require", "[", da_file->value->data[m]->key, "](string)");
							
							return HANDLER_ERROR;
						}
					}
					
					if (method == NULL) {
						log_error_write(srv, __FILE__, __LINE__, "sssbs", "missing entry for key: ", "auth.require", "[", "method", "](string)");
						return HANDLER_ERROR;
					}
					
					if (realm == NULL) {
						log_error_write(srv, __FILE__, __LINE__, "sssbs", "missing entry for key: ", "auth.require", "[", "realm", "](string)");
						return HANDLER_ERROR;
					}
					
					if (require == NULL) {
						log_error_write(srv, __FILE__, __LINE__, "sssbs", "missing entry for key: ", "auth.require", "[", "require", "](string)");
						return HANDLER_ERROR;
					}
					
					if (method && realm && require) {
						data_string *ds;
						data_array *a;
						
						a = data_array_init();
						buffer_copy_string_buffer(a->key, da_file->key);
						
						ds = data_string_init();
						
						buffer_copy_string(ds->key, "method");
						buffer_copy_string(ds->value, method);
						
						array_insert_unique(a->value, (data_unset *)ds);
						
						ds = data_string_init();
						
						buffer_copy_string(ds->key, "realm");
						buffer_copy_string(ds->value, realm);
						
						array_insert_unique(a->value, (data_unset *)ds);
						
						ds = data_string_init();
						
						buffer_copy_string(ds->key, "require");
						buffer_copy_string(ds->value, require);
						
						array_insert_unique(a->value, (data_unset *)ds);
						
						array_insert_unique(p->conf_auth_require, (data_unset *)a);
					}
					
				} else {
					log_error_write(srv, __FILE__, __LINE__, "sssbs", 
							"unexpected type for key: ", "auth.require", "[", da->value->data[n]->key, "](string)");
					
					return HANDLER_ERROR;
				}
			}
		} else {
			log_error_write(srv, __FILE__, __LINE__, "sssd", 
					"unexpected type for key: ", "auth.require", du->type);
			
			return HANDLER_ERROR;
		}
	}
	
	if (p->conf_auth_backend->used) {
		int fd;
		
		if (0 == p->conf_auth_plain_userfile->used) {
			log_error_write(srv, __FILE__, __LINE__, "s", 
					"auth.backend.plain.userfile is not set");
			return HANDLER_ERROR;
		}
		
		/* try to read */
		if (-1 == (fd = open(p->conf_auth_plain_userfile->ptr, O_RDONLY))) {
			log_error_write(srv, __FILE__, __LINE__, "sbss", 
					"opening auth.backend.plain.userfile:", p->conf_auth_plain_userfile,
					"failed:", strerror(errno));
			return HANDLER_ERROR;
		}
		close(fd);
		
		
#ifdef USE_LDAP
		if (0 == strcmp(p->conf_auth_backend->ptr, "ldap")) {
			int ret;
			char *dollar;
			
			if (p->conf_auth_ldap_hostname->used == 0) {
				log_error_write(srv, __FILE__, __LINE__, "s", "ldap: auth.backend.ldap.hostname has to be set");
				
				return HANDLER_ERROR;
			}
			
			if (p->conf_auth_ldap_basedn->used == 0) {
				log_error_write(srv, __FILE__, __LINE__, "s", "ldap: auth.backend.ldap.base-dn has to be set");
				
				return HANDLER_ERROR;
			}
			
			if (p->conf_auth_ldap_filter->used == 0) {
				log_error_write(srv, __FILE__, __LINE__, "s", "ldap: auth.backend.ldap.filter has to be set");
				
				return HANDLER_ERROR;
			}
			
			/* parse filter */
			
			if (NULL == (dollar = strchr(p->conf_auth_ldap_filter->ptr, '$'))) {
				log_error_write(srv, __FILE__, __LINE__, "s", "ldap: auth.backend.ldap.filter is missing a replace-operator '$'");
				
				return HANDLER_ERROR;
			}
			
			buffer_copy_string_len(p->ldap_filter_pre, p->conf_auth_ldap_filter->ptr, dollar - p->conf_auth_ldap_filter->ptr);
			buffer_copy_string(p->ldap_filter_post, dollar+1);
			
			if (NULL == (p->ldap = ldap_init(p->conf_auth_ldap_hostname->ptr, LDAP_PORT))) {
				log_error_write(srv, __FILE__, __LINE__, "ss", "ldap ...", strerror(errno));
				
				return HANDLER_ERROR;
			}
			
			ret = LDAP_VERSION3;
			if (LDAP_OPT_SUCCESS != (ret = ldap_set_option(p->ldap, LDAP_OPT_PROTOCOL_VERSION, &ret))) {
				log_error_write(srv, __FILE__, __LINE__, "ss", "ldap:", ldap_err2string(ret));
				
				return HANDLER_ERROR;
			}
		
		
			/* 1. */
			if (LDAP_SUCCESS != (ret = ldap_simple_bind_s(p->ldap, NULL, NULL))) {
				log_error_write(srv, __FILE__, __LINE__, "ss", "ldap:", ldap_err2string(ret));
				
				return HANDLER_ERROR;
			}
		}
#else
		if (0 == strcmp(p->conf_auth_backend->ptr, "ldap")) {
			log_error_write(srv, __FILE__, __LINE__, "s", "no ldap support available");
			
			return HANDLER_ERROR;
		}
#endif
		
	}
	
	return HANDLER_GO_ON;
}

int mod_auth_plugin_init(plugin *p) {
	p->name        = buffer_init_string("auth");
	p->init        = mod_auth_init;
	p->set_defaults = mod_auth_set_defaults;
	p->handle_uri_clean = mod_auth_uri_handler;
	p->cleanup     = mod_auth_free;
	
	p->data        = NULL;
	
	return 0;
}
