#include "d-gcc-includes.h"
#include "d-lang.h"
#include "d-codegen.h"
#include <math.h>
#include <limits.h>
#include "total.h"
#include "init.h"
#include "symbol.h"

// struct rtx was modified for c++; this macro from rtl.h needs to
// be modified accordingly.
#undef XEXP
#define XEXP(RTX, N)	(RTL_CHECK2 (RTX, N, 'e', 'u').rtx_)

IRState gen; // Top-level IRState
ModuleInfo IRState::moduleInfo;
Array IRState::stmtExprList;
bool IRState::emitTemplates;

void
IRState::doLineNote(const Loc & loc)
{
    if (loc.filename) {
#if D_GCC_VER >= 34
	// gcc 3.4
	location_t gcc_loc = { loc.filename, loc.linnum };
	emit_line_note (gcc_loc);
#else
	emit_line_note ((const char *)loc.filename, loc.linnum);
#endif
    } else {
	// do nothing
    }
}

void
IRState::setDeclLoc(tree t, Dsymbol * decl)
{
    while (decl) {
	if (decl->loc.filename) {
	    setDeclLoc(t, decl->loc);
	    return;
	}
	decl = decl->parent;
    }
    Loc l;
    l.filename = "";
    l.linnum = 0;
    setDeclLoc(t, l); // fallback; backend sometimes crashes if not set
}

void
IRState::setupSymbolStorage(Dsymbol * dsym, tree decl_tree, bool force_static_public)
{
    Declaration * real_decl = dsym->isDeclaration();
    FuncDeclaration * func_decl = real_decl ? real_decl->isFuncDeclaration() : 0;
    
    if (force_static_public ||
	( TREE_CODE( decl_tree ) == VAR_DECL && (real_decl && real_decl->isDataseg()) ) ||
	( TREE_CODE( decl_tree ) == FUNCTION_DECL ) ) {

	bool is_template = false;
	Dsymbol * sym = dsym->parent;

	while (sym) {
	    if (sym->isTemplateInstance()) {
		is_template = true;
		break;
	    }
	    sym = sym->parent;
	}

	if (is_template || dsym->getModule() == getCurrentModule()) {
	    DECL_EXTERNAL( decl_tree ) = 0;
	    TREE_STATIC( decl_tree ) = 1; // %% don't set until there is a body?
	} else {
	    DECL_EXTERNAL( decl_tree ) = 1;
	    TREE_STATIC( decl_tree ) = 0;
	}

	if ( ( real_decl ) ) {
	    switch (real_decl->prot()) {
	    case PROTpublic:
	    case PROTprotected:
	    case PROTexport:
		// %% set for specials like init,vtbl ? -- otherwise,special case
		// for reverse the default
		TREE_PUBLIC( decl_tree ) = 1;
		break;
	    default:
		if ( (func_decl && (func_decl->isMain() || func_decl->isWinMain() ||
			  func_decl->isDllMain())) ||
		    (real_decl->isMember() && ! real_decl->isThis())) {
		    // DMD seems to ignore private in this case...
		    TREE_PUBLIC( decl_tree ) = 1;
		} else {
		    TREE_PUBLIC( decl_tree ) = 0;
		}
	    }
	}

	if ( func_decl ) {
	    if (func_decl->isNested()) {
		TREE_PUBLIC( decl_tree ) = 0;
	    }
	}
	
    } else {
	TREE_STATIC( decl_tree ) = 0;
	DECL_EXTERNAL( decl_tree ) = 0;
	TREE_PUBLIC( decl_tree ) = 0;
    }
}
void
IRState::setupStaticStorage(Dsymbol * dsym, tree decl_tree)
{
    setupSymbolStorage(dsym, decl_tree, true);
    TREE_PUBLIC( decl_tree ) = 1;
}
void
IRState::prepareSymbolOutput(Dsymbol * dsym, tree decl_tree)
{
    //	make_decl_one_only checks variables for DECL_INITIAL.
    //	If it doesn't have one, it makes in a common variable.
    //	This is no good because we don't have DECL_INITIAL
    //	until the variable is emitted.  So....
    //  Annoying to have to duplicate the test from setupSymbolStorage
    //  %%TODO:
    //  1. Store decision in lang_specific (still have to call this)
    //  2. Like DMD, make all D linkage symbols ONE_ONLY

    // at this point, staic variables should have TREE_STATIC set on them
    if ( TREE_STATIC(decl_tree) ||
	(TREE_CODE( decl_tree ) == FUNCTION_DECL ) ) {

	bool is_template = false;

	Dsymbol * sym = dsym->parent;
	while (sym) {
	    if (sym->isTemplateInstance()) {
		is_template = true;
		break;
	    }
	    sym = sym->parent;
	}
	
	if (is_template) {
	    if (SUPPORTS_ONE_ONLY) { // have to check, otherwise backend will abort
		// this sets TREE_PUBLIC
		make_decl_one_only( decl_tree );
#ifdef MAKE_DECL_COALESCED
		// for Apple gcc...
	    } else {
		// warn if coalescing disabled?
		MAKE_DECL_COALESCED( decl_tree );
#else
	    } else {
		static bool warned = false;
		if (! warned) {
		    ::warning("system does not support one-only linkage");
		    warned = true;
		}
#endif
	    }
	}
    }

    if (TREE_CODE(decl_tree) == FUNCTION_DECL &&
	DECL_STATIC_CONSTRUCTOR( decl_tree )) {
	    
	if (targetm.have_ctors_dtors) {
	    (* targetm.asm_out.constructor) (XEXP (DECL_RTL (decl_tree), 0),
		DEFAULT_INIT_PRIORITY);
	}
	// %%TODO: fallback if ! targetm.have_ctors_dtors
    }
}

bool
IRState::shouldEmit(Dsymbol * d_sym)
{
    if (emitTemplates)
	return true;

    Dsymbol * sym = d_sym->parent;
    while (sym) {
	if (sym->isTemplateInstance()) {
	    return false;
	    break;
	}
	sym = sym->parent;
    }
    return true;
}


void
IRState::declareType(tree t, Dsymbol * d_sym)
{
    //if (d_sym->getModule() != getCurrentModule())
    // return;
    
    bool top_level = true;

    tree ident = d_sym->ident ? get_identifier(d_sym->ident->string) : get_identifier("fix");
    tree decl = build_decl(TYPE_DECL, ident, t);

    TYPE_NAME( t ) = decl;

    while (d_sym->parent) {
	top_level = false;
	
	d_sym = d_sym->parent;
	if (d_sym->isFuncDeclaration() ||
	    d_sym->isAggregateDeclaration() ||
	    ( top_level = d_sym->isModule() )) {

	    DECL_CONTEXT( decl ) = d_sym->toSymbol()->ScontextDecl;
	    break;
	}
    }

    tree a_t = t;
    if (POINTER_TYPE_P( a_t )) {
	a_t = TREE_TYPE(a_t);
	if (AGGREGATE_TYPE_P(a_t)) {
	    /* Using the same ident as the reference type
	       doesn't seem to cause a problem.  A non-null identifier is required
	       for the debugging info to work on dbx. */
	    //TYPE_NAME(a_t) = ident; 
	    TYPE_STUB_DECL(a_t) = build_decl(TYPE_DECL, NULL_TREE, a_t);
	    rest_of_type_compilation(a_t, top_level); // this requires setting TYPE_STUB_DECL
	    DECL_CONTEXT( TYPE_STUB_DECL(a_t) ) = DECL_CONTEXT( decl );
	    rest_of_decl_compilation(TYPE_STUB_DECL(a_t), NULL, top_level, 0);
	}
    }

    
    rest_of_decl_compilation(decl, NULL, top_level, 0);
}

tree
IRState::convertTo(Expression * exp, Type * target_type)
{
    return convertTo(exp->toElem( this ), exp->type, target_type);
}

tree
IRState::convertTo(tree exp, Type * exp_type, Type * target_type)
{
    tree result = 0;
    target_type = target_type->toBasetype();

    if (exp_type->toBasetype()->equals(target_type))
	return exp;
    
    // %%TODO: fix this annoying exception
    // %% still needed for scalar types?
    switch (exp_type ? exp_type->toBasetype()->ty : -1) {
    case Tdelegate:
	// %%TODO:NOP/VIEW_CONVERT
	if (target_type->ty == Tdelegate) {
	    return delegateVal( delegateMethodRef(exp), delegateObjectRef(exp),
		target_type);
	} else {
	    ::error("can't convert a delegate expression to %s", target_type->toChars());
	    return error_mark_node;
	}
	break;
    case Tclass:
	if (target_type->ty == Tclass) {
	    ClassDeclaration * target_class_decl = ((TypeClass *) target_type)->sym;
	    ClassDeclaration * obj_class_decl = ((TypeClass *) exp_type)->sym;

	    int offset;
	    if (target_class_decl == obj_class_decl ||
		target_class_decl->isBaseOf(obj_class_decl, & offset)) {
		// Casting up the inheritance tree: Don't do anything special.
		// Cast class to an interface it implements: Handle at compile time.
		if (offset) {
		    result = build(PLUS_EXPR, target_class_decl->type->toCtype(),
			exp, size_int(offset));
		} else {
		    // d_convert will make a NOP cast
		}
	    } else if ( ! obj_class_decl->isInterfaceDeclaration() ) {
		// Otherwise, do dynamic cast
		tree args[2] = { exp, addressOf( target_class_decl->toSymbol()->Stree ) };
		return libCall(LIBCALL_DYNAMIC_CAST, 2, args);
	    } else {
		// %% warn that this will always fail?
		// %% Not handling the INTERFACE_OFFSET option
		result = convert(target_type->toCtype(), d_null_pointer);
		if (TREE_SIDE_EFFECTS( exp )) { // make sure the expression is still used if user expects it
		    result = build(COMPOUND_EXPR, TREE_TYPE(result), exp, result);
		}
		return result;
	    }
	} else {
	    // nothing; default
	}
	break;
    strings_are_arrays:
    case Tsarray:
	{
	    // %% TODO: stuffing types ok?
	    tree pointer_value = build1( ADDR_EXPR,
		build_pointer_type( TREE_TYPE( exp )), // could use TYPE_POINTER_TO, but probably not for strings
		exp );
	    if (target_type->ty == Tpointer) {
		result = pointer_value;
		TREE_TYPE(result) = target_type->toCtype();
	    } else if (target_type->ty == Tarray) {
		unsigned array_len;

		if ( TREE_CODE( exp ) == STRING_CST ) {
		    // %% may not be correct with null padding..
		    array_len = TREE_STRING_LENGTH( exp ) / target_type->next->size();
		} else {
		    TypeSArray * a_type = (TypeSArray*) exp_type;
		    
		    array_len = a_type->dim->toInteger();
		    unsigned sz_a = a_type->next->size();
		    unsigned sz_b = target_type->next->size();
		    
		    // conversions to different sizes
		    // %% assume tvoid->size() == 1
		    // %% TODO: handle misalign like _d_arraycast_xxx ?
		    if (a_type->next->isbit() != target_type->next->isbit()) {
			if (a_type->next->isbit()) {
			    array_len /= 8 * sz_b;
			} else {
			    array_len *= 8 * sz_a;
			}
		    } else if (sz_a != sz_b) {
			array_len = array_len * sz_a / sz_b;
		    }
		}
		TREE_TYPE(pointer_value) = target_type->next->pointerTo()->toCtype();

		// Assumes casting to dynamic array of same type or void
		return darrayVal(target_type, array_len, pointer_value);
	    }
	    // %% else error?
	}
	break;
    case Tarray:
	if (target_type->ty == Tpointer) {
	    return convert(target_type->toCtype(), darrayPtrRef( exp ));
	} else if (target_type->ty == Tarray) {
	    // assume tvoid->size() == 1

	    Type * src_elem_type = exp_type->next->toBasetype();
	    Type * dst_elem_type = target_type->next->toBasetype();
	    unsigned sz_a = src_elem_type->size();
	    unsigned sz_b = dst_elem_type->size();
	    
	    if ((sz_a  != sz_b) ||
		 src_elem_type->isbit() != dst_elem_type->isbit()) {
		
		//TY exp_elem_ty = exp_type->next->toBasetype()->ty;
		if (! src_elem_type->isbit()) {
		    tree args[3] = {
			// assumes Type::tbit->size() == 1
			integerConstant(sz_b, Type::tuns32),
			integerConstant(sz_a * (dst_elem_type->isbit() ? 8 : 1),
			    Type::tuns32),
			exp
		    };
		    return libCall(LIBCALL_ARRAYCAST, 3, args, target_type->toCtype());
		} else {
		    tree args[2] = {
			integerConstant(sz_b, Type::tuns32),
			exp
		    };
		    return libCall(LIBCALL_ARRAYCAST_FROMBIT, 2, args, target_type->toCtype());
		}
	    } else {
		// %% VIEW_CONVERT_EXPR or NOP_EXPR ?
		// Convert to/from void[] or elements are the same size -- don't change length
		return build1(NOP_EXPR, target_type->toCtype(), exp);
	    }
	} else {
	    abort();
	}
	break;
    default:
	if (exp_type->iscomplex()) {
	    // creal.re, .im implemented by cast to real or ireal
	    // Assumes target type is the same size as the original's components size
	    if (target_type->isreal()) {
		return realPart(exp);
	    } else if (target_type->isimaginary()) {
		return imagPart(exp);
	    }
	} else {
	    //if (exp->op == TOKstring)
	    //    goto strings_are_arrays;
	    switch( TREE_CODE( exp )) {
	    case STRING_CST:
		goto strings_are_arrays;
		break;
	    default:
		// %% convert(value, target_type->toCtype()) ?
		;
	    }
	}
    }

    if (! result)
	result = d_convert_basic(target_type->toCtype(), exp);
    return result;
}

tree
IRState::convertForInitialization(Expression * exp, Declaration * target_decl)
{
    return doConvertForInit(exp, NULL_TREE, exp->type, target_decl);
}

tree
IRState::convertForInitialization(tree exp_tree, Type * exp_type, Declaration * target_decl)
{
    return doConvertForInit(0, exp_tree, exp_type, target_decl);
}

// Link convertForInitialization, but uses the information from Argument instead
// of a Declaration.
tree
IRState::convertForArgument(Expression * exp, Argument * arg)
{
    tree exp_tree = exp->toElem(this);
    if ( isArgumentReferenceType(arg) ) {
	// front-end already sometimes automatically takes the address
	// TODO: find out why...
	// assert(exp->op == TOKaddress);
	if (exp->op != TOKaddress) 
	    return addressOf( exp_tree );
	else
	    return exp_tree;
    } else {
	return convertForAssignment( exp_tree, exp->type, arg->type );
    }
}

tree
IRState::doConvertForInit(Expression * exp, tree in_exp_tree, Type * in_exp_type, Declaration * target_decl)
{
    if ( isDeclarationReferenceType( target_decl ) ) {
	// %% not converting to exact reference type...
	return addressOf( exp ? exp->toElem(this) : in_exp_tree );
    } else {
	tree result;
	
	if ( exp ) {
	    result = convertForAssignment(exp, target_decl->type);
	} else {
	    result = convertForAssignment(in_exp_tree, in_exp_type, target_decl->type);
	}

	// We can't init static storage with a VAR_DECL (probably can't
	// do it with an auto variable, either -- but can just use
	// MODIFY_EXPR in that case).  Also applies to fields of
	// record in static storage
	
	// %%TODO: Are these all the cases?
	
	if (! target_decl->type->isscalar() &&
	    ( target_decl->isDataseg() ||
	      (target_decl->storage_class & STCfield)) ) {
	    result = stripVarDecl( result );
	}
	return result;
    }
}


// Apply semantics of assignment to a values of type <target_type> to <exp>
// (e.g., pointer = array -> pointer = & array[0])

// Expects base type to be passed
static Type *
final_sa_elem_type(Type * type)
{
    while (type->ty == Tsarray) {
	type = type->next->toBasetype();
    }
    return type;
}

tree
IRState::convertForAssignment(Expression * exp, Type * target_type)
{
    Type * exp_base_type = exp->type->toBasetype();
    Type * target_base_type = target_type->toBasetype();

    // Assuming this only has to handle converting a non Tsarray type to
    // arbitrarily dimensioned Tsarrays.
    if (target_base_type->ty == Tsarray &&
	final_sa_elem_type(target_base_type)->equals(exp_base_type)) { // %% what about implicit converions...?

	TypeSArray * sa_type = (TypeSArray *) target_base_type;
	xdmd_integer_t count = sa_type->dim->toInteger();
	
	tree ctor = make_node( CONSTRUCTOR );
	TREE_TYPE( ctor ) = target_type->toCtype();
	if (count) {
	    CONSTRUCTOR_ELTS( ctor ) = tree_cons( build(RANGE_EXPR, Type::tindex->toCtype(),
						      integer_zero_node, integerConstant( count - 1 )),
		stripVarDecl(convertForAssignment(exp, sa_type->next)), NULL_TREE );
	}
	TREE_READONLY( ctor ) = 1;
	TREE_CONSTANT( ctor ) = 1;
	return ctor;
    } else if (! target_type->isscalar() && exp_base_type->isintegral()) {
	// D Front end uses IntegerExp(0) to mean zero-init a structure
	// This could go in convert for assignment, but we only see this for
	// internal init code -- this also includes default init for _d_newarrayi...
	
	if (exp->toInteger() == 0) {
	    tree empty = make_node( CONSTRUCTOR );
	    TREE_TYPE( empty ) = target_type->toCtype();
	    CONSTRUCTOR_ELTS( empty ) = NULL_TREE; // %% will this zero-init?
	    TREE_CONSTANT( empty ) = 1;
	    // static?
	    return empty;
	    // %%TODO: Use a code (lang_specific in decl or flag) to memset instead?
	} else {
	    abort();
	}
    }
    
    tree exp_tree = exp->toElem(this);
    return convertForAssignment(exp_tree, exp->type, target_type);
}

tree
IRState::convertForAssignment(tree expr, Type * expr_type, Type * target_type)
{
    return convertTo(expr, expr_type, target_type);
}

// could be like C c-typeck.c:default_conversion
// todo:

/* Perform default promotions for C data used in expressions.
   Arrays and functions are converted to pointers;
   enumeral types or short or char, to int.
   In addition, manifest constants symbols are replaced by their values.  */

// what about float->double?

tree
IRState::convertForCondition(Expression * exp) {
    tree result = exp->toElem(this);
    switch (exp->type->toBasetype()->ty) {
    case Taarray:
    case Tarray:
	result = darrayLenRef(result);
	break;
    case Tdelegate:
	// DMD checks (function || object), but what good
	// is if if there is a null function pointer?
	{
	    tree a, b;
	    if ( D_IS_METHOD_CALL_EXPR(result) ) {
		extractMethodCallExpr(result, a, b);
	    } else {
		tree tmp = maybeMakeTemp(result);
		a = delegateObjectRef(tmp);
		b = delegateMethodRef(tmp);
	    }
	    // not worth using  or TRUTH_ORIF...
	    // %%TODO: Is this okay for all targets?
	    result = build(BIT_IOR_EXPR, TREE_TYPE(a), a, b);
	}
	break;
    default:
	break;
    }
    // see expr.c:do_jump for the types of trees that can be passed to expand_start_cond
    //TREE_USED( <result> ) = 1; // %% maybe not.. expr optimized away because constant expression?
    return d_truthvalue_conversion( result );
}



tree
IRState::convertForVarArg(Expression * exp)
{
    // The conversions are already done by the front end
    // This just handles stuff we build.
    
    if (exp->type) {
	exp = exp->integralPromotions();
	if (exp->type->ty == Tsarray)
	    exp = exp->addressOf();
    } else if (exp->op == TOKstring) {
	tree z = exp->toElem( this );
	return build1(ADDR_EXPR, build_pointer_type( TREE_TYPE( z )), z);
	    
	// %%? needed for StringExps which don't have a type... 
	// the tree type will be set by toElem
    } 
    return exp->toElem( this );
}

/* Convert to void[] without changing the length */
tree
IRState::rawArray(Expression * exp)
{
    TY ty = exp->type->toBasetype()->ty;
    tree val;
    if (ty == Tsarray) {
	val = convertTo(exp, exp->type->toBasetype()->next->arrayOf());
    } else if (ty == Taarray || ty == Tarray) {
	val = exp->toElem(this);
    } else {
	abort();
	return 0;
    }
    return build1(NOP_EXPR, Type::tvoid->arrayOf()->toCtype(), val);
}


tree
IRState::stripVarDecl(tree value) {
    if ( TREE_CODE( value ) != VAR_DECL ) {
	return value;
    } else if ( DECL_INITIAL(value) ) {
	return DECL_INITIAL(value);
    } else {
	Type * d_type = getDType( TREE_TYPE( value ));
	if (d_type) {
	    d_type = d_type->toBasetype();
	    switch (d_type->ty) {
	    case Tstruct:
		{
		    tree t = gen.aggregateInitializer(((TypeStruct *) d_type)->sym, NULL, NULL, NULL_TREE);
		    TREE_CONSTANT(t) = 1;
		    return t;
		}
		break;
	    default:
		// error below
		break;
	    }
	}
	abort();
    }
}

tree
IRState::pointerIntSum(tree ptr_node, tree idx_exp)
{
    tree result_type_node = TREE_TYPE( ptr_node );
    tree intop = idx_exp;
    tree size_exp;

    // %% TODO: real-not-long-double issues...

    // %% test for void case ...
    size_exp = size_in_bytes( TREE_TYPE( result_type_node ) ); // array element size
    if (integer_zerop(size_exp))
	size_exp = integer_one_node;

    if (TYPE_PRECISION (TREE_TYPE (intop)) != TYPE_PRECISION (sizetype)
	|| TREE_UNSIGNED (TREE_TYPE (intop)) != TREE_UNSIGNED (sizetype))
	intop = convert (d_type_for_size (TYPE_PRECISION (sizetype), 
			     TREE_UNSIGNED (sizetype)), intop);
    
    intop = convert (result_type_node,
	build/*_binary_op*/ (MULT_EXPR, TREE_TYPE(size_exp), intop,  // the type here may be wrong %%
	    convert (TREE_TYPE (intop), size_exp)));
    intop = fold(intop);

    if ( integer_zerop(intop) )
	return ptr_node;
    else
	return build(PLUS_EXPR, result_type_node, ptr_node, intop);
}

// Doesn't do some of the omptimizations in ::makeArrayElemRef
tree
IRState::checkedIndex(Loc loc, tree index, tree upper_bound, bool inclusive)
{
    if (global.params.useArrayBounds) {
	return build(COND_EXPR, TREE_TYPE(index),
	    boundsCond(index, upper_bound, inclusive),
	    index,
	    assertCall(loc, LIBCALL_ARRAY_BOUNDS));
    } else {
	return index;
    }
}


// index must be wrapped in a SAVE_EXPR to prevent multiple evaluation...
tree
IRState::boundsCond(tree index, tree upper_bound, bool inclusive)
{
    tree bound_check;

    // definately need to convert the index to unsigned
    bound_check = build(inclusive ? LE_EXPR : LT_EXPR, boolean_type_node, index, upper_bound);
    
    if (! TREE_UNSIGNED( TREE_TYPE( index ))) {
	bound_check = build(TRUTH_ANDIF_EXPR, boolean_type_node, bound_check,
	    // %% conversions
	    build(GE_EXPR, boolean_type_node, index, integer_zero_node));
    }

    return bound_check;
}

tree
IRState::assertCall(Loc loc, LibCall libcall)
{
    tree args[2] = { darrayString(loc.filename ? loc.filename : ""),
		     integerConstant(loc.linnum, Type::tuns32) };
    return libCall(libcall, 2, args);
}

tree
IRState::floatConstant( real_t value, TypeBasic * target_type )
{
    REAL_VALUE_TYPE converted_val;
    
    tree type_node = target_type->toCtype();
    real_convert(& converted_val, TYPE_MODE(type_node), & value.rv());
    
    return build_real(type_node, converted_val);
}

tree
IRState::functionPointer(FuncDeclaration * func_decl)
{
    // call function toSymbol before using the type to fix method decls..
    tree func_node = func_decl->toSymbol()->Stree;
    
    return build1(ADDR_EXPR, func_decl->type->pointerTo()->toCtype(), func_node);
}

void
IRState::grokSlice(Expression * slice_exp, tree & array_out, tree & lwr_out, tree & upr_out, int flags)
{
    Expression * array_exp, * lwr_exp, * upr_exp;
    extractSliceExp(slice_exp, array_exp, lwr_exp, upr_exp);

    array_out = array_exp->toElem(this);

    if (lwr_exp) {
	lwr_out = lwr_exp->toElem(this);
	if (flags & GS_BoundsCheck && global.params.useArrayBounds) {
	    // and not ((array-is-static && lwr-is-constant) or ( lwr-is-const-zero ))
	    abort(); // todo
	}
    } else if (flags & GS_CreateBounds)
	lwr_out = integerConstant(0, Type::tindex);
    else
	lwr_out = NULL_TREE;
    if (upr_exp) {
	upr_out = upr_exp->toElem(this);
	if (flags & GS_BoundsCheck && global.params.useArrayBounds) {
	    abort(); // todo
	}
    } else if (flags & GS_CreateBounds) {
	array_out = maybeMakeTemp(array_out);
	upr_out = arrayLength(array_out, array_exp->type);
    }

    Type * array_base_type = array_exp->type->toBasetype();

    switch (flags & GS_ProcessMask) {
    case GS_ExtractPointer:
	switch (array_base_type->ty) {
	case Tsarray:
	case Tarray:
	    array_out = convertTo(array_out, array_base_type, array_base_type->next->pointerTo());
	    break;
	case Tpointer:
	    // nothing;
	    break;
	default:
	    slice_exp->error("Unexpected type for slice: %s", array_base_type->toChars());
	    lwr_out = upr_out = array_out = error_mark_node;
	    return;
	}
	break;
    case 0:
	break;
    default:
	abort();
    }
}

tree
IRState::call(FuncDeclaration * func_decl, Array * args)
{
    Loc loc;
    Expression * exp = new VarExp(loc, func_decl);
    exp->type = func_decl->type;
    return call(exp, /*(TypeFunction *) func_decl->type, */ args);
}

tree
IRState::libCall(LibCall lib_call, unsigned n_args, tree *args, tree force_result_type)
{
    tree result;
    tree callee = functionPointer( getLibCallDecl( lib_call ));
    // for force_result_type, assumes caller knows what it is doing %%
    tree result_type = force_result_type != NULL_TREE ?
	force_result_type : TREE_TYPE(TREE_TYPE(TREE_OPERAND(callee, 0)));
    tree arg_list = NULL_TREE;
    for (int i = n_args - 1; i >= 0; i--) {
	arg_list = tree_cons(NULL_TREE, args[i], arg_list);
    }

    result = buildCall(result_type, callee, arg_list);
    return result;
}

tree
IRState::call(tree callable, tree object, Array * arguments)
{
    tree func_type_node = TREE_TYPE( callable );
    TypeFunction * func_type = 0;
    
    if ( POINTER_TYPE_P( func_type_node ) ) {
	func_type_node = TREE_TYPE( func_type_node );
    }
    assert( TREE_CODE(func_type_node) == FUNCTION_TYPE );
    assert( TYPE_LANG_SPECIFIC( func_type_node ) != 0 );
    func_type = (TypeFunction *) getDType(func_type_node)->toBasetype();
    assert( func_type->ty == Tfunction );

    tree actual_arg_list = NULL_TREE;
    tree actual_callee;
    tree result;

    if ( TREE_CODE(TREE_TYPE( callable )) == FUNCTION_TYPE ) {
	// || TREE_CODE(callable) == FUNCTION_DECL // redundant
	
	// The backend does not accept a FUNCTION_DECL (but it does accept a VAR_DECL
	// that is a function pointer.)  We also need to handle
	// (* pointer_to_function)(arg,arg,...)
	actual_callee = addressOf( callable );
    } else {
	actual_callee = callable;
    }
    if ( object != NULL_TREE ) {
	func_type_node = build_method_type( TREE_TYPE( object ), func_type_node );
    }

    Array * formal_args = func_type->arguments; // can be NULL for genCfunc decls
    unsigned n_formal_args = formal_args ? formal_args->dim : 0;
    int n_actual_args = arguments ? arguments->dim : 0;

    // assumes arguments->dim <= formal_args->dim if (! this->varargs)
    // Create args in reverse array order to get the correct list order
    for (int i = (int) n_actual_args - 1; i >= -1; i--) {
	tree actual_arg_tree;

	if (i == -1) {
	    // If this is a nested function being called as a delegate,
	    // object should not be NULL_TREE and will be the dummy argument.
	    // If calling a nested function directly, pass zero for the
	    // dummy argument.
	    if (object != NULL_TREE) {
		actual_arg_tree = object;
	    } else if ( TYPE_LANG_SPECIFIC( func_type_node )->is_nested_function ) {
		actual_arg_tree = integer_zero_node;
	    } else {
		continue;
	    }
	} else {
	    Expression * actual_arg_exp = (Expression *) arguments->data[i];
	    if ((unsigned)i < n_formal_args) {
		Argument * formal_arg = (Argument *) formal_args->data[i];

		actual_arg_tree = convertForArgument(actual_arg_exp, formal_arg);
	    } else {
		actual_arg_tree = convertForVarArg(actual_arg_exp);
	    }
	}

	// default args..
	
	// how to pass arguments ..
	// depends on in, out, etc...


	//TREE_USED( actual_arg_tree ) = 1; // needed ?
	
	//printf("%s: %d: %08x\n\n", expr->toChars(), i, actual_arg_tree);
	actual_arg_list = tree_cons(NULL_TREE, actual_arg_tree, actual_arg_list);
    }

    result = buildCall(TREE_TYPE(func_type_node), actual_callee, actual_arg_list);
    return maybeExpandSpecialCall(result);
}

tree
IRState::call(Expression * expr, /*TypeFunction * func_type, */ Array * arguments)
{
    tree callee = expr->toElem(this);
    tree object = NULL_TREE;

    if ( D_IS_METHOD_CALL_EXPR( callee ) ) {
	// This could be a delegate expression (TY == Tdelegate), but not
	// actually a delegate variable.
	
	extractMethodCallExpr(callee, callee, object);
    } else if (expr->type->toBasetype()->ty == Tdelegate) {
	object = delegateObjectRef(callee);
	callee = delegateMethodRef(callee);
    } else if (expr->op == TOKvar) {
	FuncDeclaration * func = ((VarExp *) expr)->var->isFuncDeclaration();
	if (func && func->isNested()) {
	    // Pass fake argument for nested functions
	    object = convert(ptr_type_node, integer_zero_node);
	}
    }
    return call(callee, object, arguments);
}

tree
IRState::libCall(LibCall lib_call, unsigned n_args, Expression ** args, tree result_type)
{
    Array a;
    a.setDim(n_args);
    memmove(a.data, args, n_args * sizeof(void *));
    return libCall(lib_call, & a, result_type);
}

static const char * libcall_ids[LIBCALL_count] =
    { "_d_assert", "_d_array_bounds", "_d_switch_error", "_d_invariant",
      "_d_newclass", "_d_new","_d_newarrayi", "_d_newbitarray",
      "_d_delclass", "_d_delarray", "_d_callfinalizer",
      "_d_arraysetlength", "_d_arraysetlengthb",
      "_d_dynamic_cast",
      "_adEq", "_adEqBit", "_adCmp", "_adCmpChar", "_adCmpBit",
      "_aaIn", "_aaGet", "_aaDel",
      "_d_arraycast", "_d_arraycast_frombit",
      "_d_arraycopy", "_d_arraycopybit",
      "_d_arraycat", "_d_arraycatn",
      "_d_arrayappend", "_d_arrayappendc",
      "_d_arraysetbit",
      "_d_monitorenter", "_d_monitorexit",
      "_d_criticalenter", "_d_criticalexit",
      "_d_throw",
      "_d_switch_string", "_d_switch_ustring",
      "_d_gnu_bitarrayslice", "_d_gnu_bitarrayslicep", "_d_gnu_copytobitarrayslice"
    };

static FuncDeclaration * libcall_decls[LIBCALL_count];

void
IRState::replaceLibCallDecl(FuncDeclaration * d_decl)
{
    if ( ! d_decl->ident )
	return;
    for (unsigned i = 0; i < LIBCALL_count; i++) {
	if ( strcmp(d_decl->ident->string, libcall_ids[i]) == 0 ) {
	    // %% warn if libcall already set?
	    // Only do this for the libcalls where it's a problem, otherwise
	    // it causes other problems...
	    switch ((LibCall) i) {
	    case LIBCALL_GNU_BITARRAYSLICEP:
	    case LIBCALL_ARRAYCOPY: // this could be solved by turning copy of char into memcpy
	    case LIBCALL_ARRAYCAST:
		// replace the function declaration
		break;
#if D_GCC_VER <= 33
		// may be safe for GCC 3.4, haven't tried it
	    case LIBCALL_ADCMPCHAR:
#endif
	    default:
		// don't replace
		return;
	    }
	    libcall_decls[i] = d_decl;
	    break;
	}
    }
}


FuncDeclaration *
IRState::getLibCallDecl(LibCall lib_call)
{
    FuncDeclaration * decl = libcall_decls[lib_call];
    Array arg_types;
    bool varargs = false; // cheap way to not declare arguments
    if (! decl) {
	Type * return_type = Type::tvoid;
	
	switch (lib_call) {
	case LIBCALL_ASSERT:
	case LIBCALL_ARRAY_BOUNDS:
	case LIBCALL_SWITCH_ERROR:
	    // need to spec chararray/string because internal code passes string constants
	    arg_types.reserve(2);
	    arg_types.push( Type::tchar->arrayOf() );
	    arg_types.push( Type::tuns32 );
	    break;
	case LIBCALL_NEWCLASS:
	    arg_types.push( ClassDeclaration::classinfo->type );
	    return_type = getObjectType();
	    break;
	case LIBCALL_NEW:
	    arg_types.push( Type::tuns32 );
	    arg_types.push( Type::tuns32 );
	    return_type = Type::tvoid->arrayOf();
	    break;
	case LIBCALL_NEWARRAYI:
	    arg_types.push( Type::tuns32 );
	    arg_types.push( Type::tuns32 );
	    varargs = true;
	    return_type = Type::tvoid->arrayOf();
	    break;
	case LIBCALL_NEWBITARRAY:
	    arg_types.push( Type::tuns32 );
	    arg_types.push( Type::tbit );
	    return_type = Type::tbit->arrayOf(); 
	    break;
	case LIBCALL_DELCLASS:
	    arg_types.push(getObjectType()->pointerTo());
	    break;
	case LIBCALL_DELARRAY:
	    arg_types.push(Type::tvoid->arrayOf()->pointerTo());
	    break;
	case LIBCALL_CALLFINALIZER:
	    arg_types.push(Type::tvoid->pointerTo()); // declared void*; could be Object....
	    break;
	case LIBCALL_ARRAYSETLENGTH:
	case LIBCALL_ARRAYSETLENGTH_B:
	    arg_types.push( Type::tuns32 );
	    if (lib_call == LIBCALL_ARRAYSETLENGTH)
		arg_types.push( Type::tuns32 );
	    arg_types.push( Type::tvoid->arrayOf()->pointerTo() );
	    return_type = Type::tvoid->arrayOf(); 
	    break;
	case LIBCALL_DYNAMIC_CAST:
	    arg_types.push( getObjectType() );
	    arg_types.push( ClassDeclaration::classinfo->type );
	    return_type = getObjectType();
	    break;
	case LIBCALL_ADEQ:
	case LIBCALL_ADCMP:
	    arg_types.reserve(3);
	    arg_types.push(Type::tvoid->arrayOf());
	    arg_types.push(Type::tvoid->arrayOf());
	    arg_types.push(Type::typeinfo->type);
	    return_type = Type::tint32;
	    break;
	case LIBCALL_ADCMPCHAR:
	    arg_types.reserve(2);
	    arg_types.push(Type::tchar->arrayOf());
	    arg_types.push(Type::tchar->arrayOf());
	    return_type = Type::tint32;
	    break;
	case LIBCALL_ADEQBIT:
	case LIBCALL_ADCMPBIT:
	    arg_types.reserve(2);
	    arg_types.push(Type::tbit->arrayOf());
	    arg_types.push(Type::tbit->arrayOf());
	    return_type = Type::tint32;
	    break;
	case LIBCALL_AAIN:
	case LIBCALL_AAGET:
	case LIBCALL_AADEL:
	    {
		Type * aa_type = Type::tvoid->pointerTo()->arrayOf(); // associated arrays are dynamic arrays of pointers
		if (lib_call == LIBCALL_AAGET)
		    aa_type->pointerTo();
		
		arg_types.reserve(3);
		arg_types.push(aa_type);
		arg_types.push(Type::typeinfo->type); // typeinfo reference
		if ( lib_call == LIBCALL_AAGET )
		    arg_types.push(Type::tint32); // %%
		
		varargs = true;
		
		switch (lib_call) {
		case LIBCALL_AAIN:  return_type = Type::tint32; break;
		case LIBCALL_AAGET: return_type = Type::tvoid->pointerTo(); break;
		case LIBCALL_AADEL: return_type = Type::tvoid; break;
		default:
		    abort();
		}
	    }
	    break;
	case LIBCALL_ARRAYCAST:
	    {
		Type * t = Type::tvoid->arrayOf();
		arg_types.push(Type::tuns32);
		arg_types.push(Type::tuns32);
		arg_types.push(t);
		return_type = t;
	    }
	    break;
	case LIBCALL_ARRAYCAST_FROMBIT:
	    {
		Type * t = Type::tvoid->arrayOf();
		arg_types.push(Type::tuns32);
		arg_types.push(t);
		return_type = t;
	    }
	    break;
	case LIBCALL_ARRAYCOPY:
	    {
		Type * t = Type::tvoid->arrayOf();
		arg_types.push(Type::tuns32);
		arg_types.push(t);
		arg_types.push(t);
		return_type = t;
	    }
	    break;
	case LIBCALL_ARRAYCOPYBIT:
	    {
		Type * t = Type::tbit->arrayOf();
		arg_types.push(t);
		arg_types.push(t);
		return_type = t;
	    }
	    break;
	case LIBCALL_ARRAYCAT:
	    {
		Type * t = Type::tvoid->arrayOf();
		arg_types.push(t);
		arg_types.push(t);
		arg_types.push(Type::tuns32);
		return_type = t;
	    }
	    break;
	case LIBCALL_ARRAYCATN:
	    {
		arg_types.push(Type::tuns32);
		arg_types.push(Type::tuns32);
		varargs = true;
		return_type = Type::tvoid->arrayOf();
	    }
	    break;
	case LIBCALL_ARRAYAPPEND:
	    {
		Type * t = Type::tuns8->arrayOf();
		arg_types.push(t->pointerTo());
		arg_types.push(t);
		arg_types.push(Type::tuns32);
		return_type = Type::tvoid->arrayOf();
	    }
	    break;
	case LIBCALL_ARRAYAPPENDC:
	    {
		Type * t = Type::tuns8->arrayOf()->pointerTo();
		arg_types.push(t->pointerTo());
		arg_types.push(Type::tuns32);
		varargs = true;
		return_type = Type::tvoid->arrayOf();
	    }
	    break;
	case LIBCALL_ARRAYSETBIT:
	    {
		Type * t = Type::tbit->arrayOf();
		arg_types.push(t);
		arg_types.push(Type::tuns32);
		arg_types.push(Type::tuns32);
		arg_types.push(Type::tbit);
		return_type = t;
	    }
	    break;
	case LIBCALL_MONITORENTER:
	case LIBCALL_MONITOREXIT:
	case LIBCALL_THROW:
	case LIBCALL_INVARIANT:
	    arg_types.push(getObjectType());
	    break;
	case LIBCALL_CRITICALENTER:
	case LIBCALL_CRITICALEXIT:
	    arg_types.push(Type::tvoid->pointerTo());
	    break;
	case LIBCALL_SWITCH_STRING:
	case LIBCALL_SWITCH_USTRING:
	    {
		Type * t = lib_call == LIBCALL_SWITCH_STRING ?
		    Type::tchar->arrayOf() : Type::twchar->arrayOf();
		arg_types.push(t->arrayOf());
		arg_types.push(t);
		return_type = Type::tint32;
	    }
	    break;
	case LIBCALL_GNU_BITARRAYSLICE:
	    {
		Type * t = Type::tbit->arrayOf();
		arg_types.push(t);
		arg_types.push(Type::tuns32);
		arg_types.push(Type::tuns32);
		return_type = t;
	    }
	    break;
	case LIBCALL_GNU_COPYTOBITARRAYSLICE:
	    {
		arg_types.push(Type::tbit->pointerTo());
		arg_types.push(Type::tuns32);
		arg_types.push(Type::tuns32);
		arg_types.push(Type::tbit->arrayOf());
		return_type = Type::tbit->arrayOf();
	    }
	    break;
	case LIBCALL_GNU_BITARRAYSLICEP:
	    arg_types.push(Type::tbit->pointerTo());
	    arg_types.push(Type::tuns32);
	    arg_types.push(Type::tuns32);
	    return_type = Type::tbit->arrayOf();
	    break;
	default:
	    abort();
	}
	decl = FuncDeclaration::genCfunc(return_type, (char *) libcall_ids[lib_call]);
	// %% have to do this because we don't declare arguments
	{
	    TypeFunction * tf = (TypeFunction *) decl->type;
	    tf->varargs = varargs;
	    Array * args = new Array;
	    args->setDim( arg_types.dim );
	    for (unsigned i = 0; i < arg_types.dim; i++)
		args->data[i] = new Argument( (Type *) arg_types.data[i],
		    NULL, In);
	    tf->arguments = args;
	}
	libcall_decls[lib_call] = decl;
    }
    return decl;
}

tree
IRState::libCall(LibCall lib_call, Array * args, tree result_type)
{
    FuncDeclaration * func_decl = getLibCallDecl(lib_call);

    tree result =  call(func_decl, args);
    if (result_type != NULL_TREE) {
	// %% assumes caller knows what it is doing
	TREE_TYPE( result ) = result_type;
    }
    return result;
}

tree IRState::maybeExpandSpecialCall(tree call_exp)
{
    // More code duplication from C

    tree callee = TREE_OPERAND(call_exp, 0);
    tree t, op;
    if (POINTER_TYPE_P(TREE_TYPE( callee ))) {
	callee = TREE_OPERAND(callee, 0);
    }
    if (TREE_CODE(callee) == FUNCTION_DECL) {
	if (DECL_BUILT_IN_CLASS(callee) == NOT_BUILT_IN) {
	    // the most common case
	    return call_exp;
	} else if (DECL_BUILT_IN_CLASS(callee) == BUILT_IN_NORMAL) {
	    switch (DECL_FUNCTION_CODE(callee)) {
	    case BUILT_IN_ABS:
	    case BUILT_IN_LABS:
	    case BUILT_IN_LLABS:
	    case BUILT_IN_IMAXABS:
		/* The above are required for both gcc3.3 and 3.4.  The below
		   are only needed for gcc3.3, but we might as well use it
		   for gcc3.4 as well. */
	    case BUILT_IN_FABS:
	    case BUILT_IN_FABSL:
	    case BUILT_IN_FABSF:
		op = TREE_VALUE( TREE_OPERAND( call_exp, 1 ));
		t = build1(ABS_EXPR, TREE_TYPE(op), op);
		return d_convert_basic(TREE_TYPE(call_exp), t);
		// probably need a few more cases:
	    default:
		return call_exp;
	    }
	} else if (0 /** WIP **/ && DECL_BUILT_IN_CLASS(callee) == BUILT_IN_FRONTEND) {
	    // %%TODO: need to handle BITS_BIG_ENDIAN
	    // %%TODO: need to make expressions unsigned
	    
	    Intrinsic intrinsic = (Intrinsic) DECL_FUNCTION_CODE(callee);
	    // Might as well do intrinsics here...
	    switch (intrinsic) {
	    case INTRINSIC_BSF: // This will use bsf on x86, but BSR becomes 31-(bsf)!!
		// drop through
	    case INTRINSIC_BSR:
		// %% types should be correct, but should still check..
		// %% 64-bit..
#if D_GCC_VER >= 34
		return build(CALL_EXPR, TREE_TYPE(call_exp),
		    built_in_decls[intrinsic == INTRINSIC_BSF ? BUILT_IN_CTZ : BUILT_IN_CLZ],
		    TREE_OPERAND(call_exp, 1));
#else
		
#endif
	    case INTRINSIC_BT:
	    case INTRINSIC_BTC:
	    case INTRINSIC_BTR:
	    case INTRINSIC_BTS:
		break;
	    case INTRINSIC_BSWAP:
		{
		}
		break;
	    case INTRINSIC_INP:
	    case INTRINSIC_INPW:
	    case INTRINSIC_INPL:
	    case INTRINSIC_OUTP:
	    case INTRINSIC_OUTPW:
	    case INTRINSIC_OUTPL:
#ifdef TARGET_386
#else
		::error("Port I/O intrinsic '%s' is only available on ix86 target",
		    IDENTIFIER_POINTER(DECL_NAME(callee)));
#endif
		break;
	    default:
		abort();
	    }
	}
    }

    return call_exp;
}



tree
IRState::doThunk(unsigned offset, FuncDeclaration * target)
{
    bool is_extern = target->getModule() != getCurrentModule();
    // %% for now, target->toSymbol()->Stree would be EXTERN or STATIC...
    Symbol * sym = target->toSymbol();
    if (! sym->Sspecial.thunks )
	sym->Sspecial.thunks = new Array;
    Array & thunks = * sym->Sspecial.thunks;
    Thunk * thunk;
    bool is_new_thunk = true;
    for (unsigned i = 0; i < thunks.dim; i++) {
	thunk = (Thunk *) thunks.data[i];
	if (thunk->offset == offset) {
	    is_new_thunk = false;
	    break;
	}
    }
    if (is_new_thunk) {
	thunk = new Thunk;
	thunk->offset = offset;
	thunks.push(thunk);
    }
    tree * p_decl = is_extern ? & thunk->externDecl : & thunk->staticDecl;
    if (! *p_decl) {
	if (is_extern) {
	    *p_decl = target->toThunkSymbol(offset)->Stree;
	    DECL_EXTERNAL( *p_decl ) = 1;
	} else {
	    *p_decl = doThunk1(offset, target);
	}
    }
    return addressOf(*p_decl);
}

// Copied from cp/method.c:use_thunk
tree
IRState::doThunk1(unsigned offset, FuncDeclaration * target)
{
    int delta = -offset;
    tree target_func_decl = target->toSymbol()->Stree;
    tree alias = target_func_decl;
    tree thunk_decl = target->toThunkSymbol(offset)->Stree;
    // can't use the thunk symbol yet because there may be more than one thunk
    // per implemented method -- not sure how DMD does this..

    TREE_ADDRESSABLE(target_func_decl) = 1;
    TREE_USED(target_func_decl) = 1;
#if D_GCC_VER >= 34
    DECL_VISIBILITY (thunk_decl) = DECL_VISIBILITY (target_func_decl);
    
#ifdef ASM_OUTPUT_DEF
  if (targetm.have_named_sections)
    {
      resolve_unique_section (target_func_decl, 0, flag_function_sections);

      if (DECL_SECTION_NAME (target_func_decl) != NULL && DECL_ONE_ONLY (target_func_decl))
	{
	  resolve_unique_section (thunk_decl, 0, flag_function_sections);

	  /* Output the thunk into the same section as function.  */
	  DECL_SECTION_NAME (thunk_decl) = DECL_SECTION_NAME (target_func_decl);
	}
    }
#endif
#endif

  // cp/method.c:
  /* The back-end expects DECL_INITIAL to contain a BLOCK, so we
     create one.  */
  // ... actually doesn't seem to be the case for output_mi_thunk
  DECL_INITIAL (thunk_decl) = make_node (BLOCK);
  BLOCK_VARS (DECL_INITIAL (thunk_decl)) = DECL_ARGUMENTS (thunk_decl);

    if (targetm.asm_out.can_output_mi_thunk(thunk_decl, delta, 0, alias)) {
	const char *fnname;
	current_function_decl = thunk_decl;
	DECL_RESULT( thunk_decl ) = build_decl(RESULT_DECL, 0, integer_type_node);
	
	fnname = XSTR(XEXP(DECL_RTL(thunk_decl), 0), 0);
	initFunctionStart(thunk_decl, 0);
	assemble_start_function (thunk_decl, fnname);
	targetm.asm_out.output_mi_thunk (asm_out_file, thunk_decl,
	    delta, 0, alias);
	assemble_end_function(thunk_decl, fnname);
	current_function_decl = 0;
	cfun = 0;
	/* Because init_function_start increments this, we must
	   decrement it.  */
	immediate_size_expand--;
	TREE_ASM_WRITTEN (thunk_decl) = 1;

	return thunk_decl;
    } else {
	::error("Sorry, backend for this machine doesn't support thunks");
	return error_mark_node;
    }

}

tree
IRState::doSimpleFunction(Module * mod, const char * name, tree expr, bool static_ctor)
{
    TypeFunction * func_type = new TypeFunction(0, Type::tvoid, 0, LINKc);
    FuncDeclaration * func = new FuncDeclaration(mod->loc, mod->loc, // %% locs may be wrong
	Lexer::idPool(name), STCstatic, func_type); // name is only to prevent crashes
    func->loc = Loc(mod, 1); // to prevent debug info crash // maybe not needed if DECL_ARTIFICIAL?
    func->linkage = func_type->linkage;
    func->parent = mod;
    func->protection = PROTprivate;

    tree func_decl = func->toSymbol()->Stree;
    if (static_ctor) {
	DECL_STATIC_CONSTRUCTOR( func_decl ) = 1; // apparently, the back end doesn't do anything with this
	TREE_USED( func_decl ) = 1; // only needed for GCC 3.3?
    }
    // %% maybe remove the identifier

    func->fbody = new ExpStatement(mod->loc,
	new WrappedExp(mod->loc, TOKcomma, expr, Type::tvoid));

    func->toObjFile();

    return addressOf(func_decl);
}

tree
IRState::doFunctionToCallFunctions(Module * mod, const char * name, Array * functions)
{
    // If there is only one function, just return that
    if (functions->dim == 1) {
	return addressOf( (Declaration *) functions->data[0] );
    } else if (functions->dim > 1) {
	// %% shouldn't front end build these?
	tree exp = NULL_TREE;
	tree call_exp = NULL_TREE;
	for (unsigned i = 0; i < functions->dim; i++) {
	    FuncDeclaration * fn_decl = (FuncDeclaration *) functions->data[i];
	    call_exp = gen.buildCall(void_type_node, gen.addressOf(fn_decl), NULL_TREE);
	    if (! exp)
		exp = call_exp;
	    else {
		exp = build(COMPOUND_EXPR, void_type_node, exp, call_exp);
	    }
	}
	Symbol * s = mod->toSymbolX(name, 0, 0);
	return gen.doSimpleFunction(mod, s->Sident, exp, false);
    } else {
	return NULL_TREE;
    }
}


/* Type of return value may be different from aer_exp->exp.
   Case 1: For array of bit, type is bitConfig.elemType
*/

tree
IRState::arrayElemRef(IndexExp * aer_exp, 
    tree * and_mask_out, tree * or_mask_out, tree * shift_count_out)
{
    if (and_mask_out)
	*and_mask_out = NULL_TREE;
    if (or_mask_out)
	*or_mask_out = NULL_TREE;
    if (shift_count_out)
	*shift_count_out = NULL_TREE;
    
    Expression * e1 = aer_exp->e1;
    Expression * e2 = aer_exp->e2;
	
    Type * base_type = e1->type->toBasetype();
    TY base_type_ty = base_type->ty;

    bool is_bit = base_type->next->isbit();

    tree index_expr; // logical index
    tree subscript_expr; // expr that indexes the array data; different from index_expr for bit for bounds check
    tree ptr_exp;  // base pointer to the elements
    tree elem_ref; // reference the the element.  built in two steps for bit

    index_expr = e2->toElem( this );
    subscript_expr = index_expr;

    switch (base_type_ty) {
    case Tarray:
    case Tsarray:
	{
	    tree e1_tree = e1->toElem(this);

	    if ( global.params.useArrayBounds &&
		// If it's a static array and the index is
		// constant, the front end has already
		// checked the bounds.
		! (base_type_ty == Tsarray && e2->isConst()) ) {

		
		tree array_len_expr, throw_expr, oob_cond;
		// implement bounds check as a conditional expression:
		// a[ inbounds(index) ? index : { throw ArrayBoundsError } ]
		//
		// First, set up the index expression to only be evaluated
		// once.
		// %% save_expr does this check: if (! TREE_CONSTANT( index_expr ))
		//   %% so we don't do a <0 check for a[2]...
		index_expr = maybeMakeTemp( index_expr );
		
		if (base_type_ty == Tarray) {
		    e1_tree = maybeMakeTemp(e1_tree);
		    array_len_expr = darrayLenRef(e1_tree);
		} else {
		    array_len_expr = ((TypeSArray *) base_type)->dim->toElem(this);
		}

		oob_cond = boundsCond(index_expr, array_len_expr, false);
		throw_expr = assertCall(aer_exp->loc, LIBCALL_ARRAY_BOUNDS);

		subscript_expr = build( COND_EXPR, TREE_TYPE( index_expr ),
		    oob_cond, index_expr, throw_expr );
	    }

	    if (is_bit) {
		// %% dup'd below for Tpointer case
		// %% do we want the orig index_expr or the subscript_expr saved?
		
		index_expr = maybeMakeTemp( index_expr );
		
		tree shift_node = bitConfig.shift_tree;
		// Type::tindex is signed, subscript_expr must be unsigned to generatate unsigned
		// right shift %% 
		//if ( ! TYPE_UNSIGNED( TREE_TYPE( subscript_expr )) )..
		// %% YOW! calling 'convert' breaks stuff
		subscript_expr = d_convert_basic( TREE_TYPE( shift_node ), subscript_expr ); // folding?
		subscript_expr = build( RSHIFT_EXPR, TREE_TYPE(shift_node),
		    subscript_expr, shift_node );

		subscript_expr = fold( subscript_expr ); // %% won't work for bound check..
		// would need COMPOUND_EXPR( bounds_check_expr, folded_index_expr )
		
	    }

	    // %% TODO: make this an ARRAY_REF?
	    if (base_type_ty == Tarray)
		ptr_exp = darrayPtrRef( e1_tree ); // %% do convert in darrayPtrRef?
	    else
		ptr_exp = addressOf( e1_tree );
	    // This conversion is required for static arrays and is just-to-be-safe
	    // for dynamic arrays
	    ptr_exp = d_convert_basic(base_type->next->pointerTo()->toCtype(), ptr_exp);
	}
	break;
    case Tpointer:
	if (is_bit) {
	    // %% dup'd above
	    if ( TREE_CODE( index_expr ) != SAVE_EXPR)
		index_expr = save_expr( index_expr );
	    subscript_expr = index_expr;
	    subscript_expr = build(RSHIFT_EXPR, TREE_TYPE(subscript_expr),
		subscript_expr,
		bitConfig.shift_tree/*integerConstant(bitarray_elem_shift, unsigned_type_node)*/);
	    subscript_expr = fold( subscript_expr );
	}
	ptr_exp = e1->toElem( this );
	break;
    default:
	abort();
    }

    if (! is_bit)
	ptr_exp = pvoidOkay( ptr_exp );
    else
	ptr_exp = d_convert_basic(bitConfig.elemType->pointerTo()->toCtype(), ptr_exp);
    elem_ref = build1(INDIRECT_REF, TREE_TYPE(TREE_TYPE(ptr_exp)),
	pointerIntSum( ptr_exp, subscript_expr));

    if (is_bit) {
	tree t;

	// (index & <mask>)
	tree bit_index_expr = build(BIT_AND_EXPR, TREE_TYPE(index_expr),
	    index_expr, bitConfig.mask_tree/*integerConstant(bitarray_elem_mask, bitArrayElemType)*/);
	bit_index_expr = fold( bit_index_expr );

	if (or_mask_out || shift_count_out)
	    bit_index_expr = save_expr( bit_index_expr );
	// 1 << (index & <mask>)
	t = build(LSHIFT_EXPR, bitConfig.elemType->toCtype()/*bitArrayElemType->toCtype()*/,
	    integer_one_node, bit_index_expr);
	if (and_mask_out)
	    * and_mask_out = fold(t);

	if (shift_count_out)
	    * shift_count_out = bit_index_expr;
    }

    return elem_ref;
}

tree
IRState::arrayElemRef(IndexExp * aer_exp)
{
    tree elem_ref, and_mask;
    elem_ref = arrayElemRef(aer_exp, & and_mask, (tree*) 0, (tree*) 0);
    if (and_mask) { // i.e., is the element type bit.
	/*
	// %% d_truthvalue_conversion will convert something that is already truthval cvt'd? .. would
	// be nice to optimize this out...
	// %% also assumptions about boolean_type_node == Type::tboolean
	if (TREE_CODE(and_mask) == BIT_NOT_EXPR)
	    and_mask = TREE_OPERAND( and_mask, 0 );
	else
	    and_mask = fold( build1(BIT_NOT_EXPR, TREE_TYPE(and_mask), and_mask) );
	*/
	return d_truthvalue_conversion( build(BIT_AND_EXPR,
				TREE_TYPE(elem_ref), elem_ref, and_mask ));
    } else
	return elem_ref;
}

tree
IRState::darrayPtrRef(tree exp)
{
    // Get the backend type for the array and pick out the array data
    // pointer field (assumed to be the second field.)
    tree ptr_field = TREE_CHAIN( TYPE_FIELDS( TREE_TYPE( exp )));
    return build(COMPONENT_REF, TREE_TYPE( ptr_field ), exp, ptr_field);
}

tree
IRState::darrayLenRef(tree exp)
{
    // Get the backend type for the array and pick out the array length
    // field (assumed to be the first field.)
    tree len_field = TYPE_FIELDS( TREE_TYPE( exp ));
    return build(COMPONENT_REF, TREE_TYPE( len_field ), exp, len_field);
}


tree
IRState::darrayVal(tree type, tree len, tree data)
{
    // %% assert type is a darray
    tree ctor = make_node( CONSTRUCTOR );
    tree len_field, ptr_field;
    tree elts;
    
    TREE_TYPE( ctor ) = type;
    TREE_READONLY( ctor ) = 1;
    TREE_STATIC( ctor ) = 0;   // can be set by caller if needed
    TREE_CONSTANT( ctor ) = 0; // "
    len_field = TYPE_FIELDS( TREE_TYPE( ctor ));
    ptr_field = TREE_CHAIN( len_field );

    elts = tree_cons( ptr_field, data, NULL_TREE ); // shouldn't need to convert the pointer...
    elts = tree_cons( len_field, len, elts );
    CONSTRUCTOR_ELTS( ctor ) = elts;

    return ctor;
}

tree
IRState::darrayVal(tree type, unsigned len, tree data)
{
    // %% assert type is a darray
    tree ctor = make_node( CONSTRUCTOR );
    tree len_value, ptr_value, len_field, ptr_field;
    tree elts;
    
    TREE_TYPE( ctor ) = type;
    TREE_READONLY( ctor ) = 1;
    TREE_STATIC( ctor ) = 0;   // can be set by caller if needed
    TREE_CONSTANT( ctor ) = 0; // "
    len_field = TYPE_FIELDS( TREE_TYPE( ctor ));
    ptr_field = TREE_CHAIN( len_field );

    if (data ) {
	assert( POINTER_TYPE_P( TREE_TYPE( data )));
	ptr_value = data;
    } else {
	ptr_value = convert( TREE_TYPE( ptr_field ), integer_zero_node);
    }

    elts = tree_cons( ptr_field, ptr_value, NULL_TREE ); // shouldn't need to convert the pointer...
    
    len_value = integerConstant(len, TREE_TYPE(len_field));
    elts = tree_cons( len_field, len_value, elts );
    CONSTRUCTOR_ELTS( ctor ) = elts;

    return ctor;
}

tree
IRState::darrayString(const char * str)
{
    unsigned len = strlen(str);
    // %% assumes str is null-terminated
    tree str_tree = build_string(len + 1, str);
    
    TREE_TYPE( str_tree ) = arrayType(Type::tchar, len);
    return darrayVal(Type::tchar->arrayOf()->toCtype(), len, addressOf(str_tree));
}


tree
IRState::arrayLength(tree exp, Type * exp_type)
{
    Type * base_type = exp_type->toBasetype();
    switch (base_type->ty) {
    case Tsarray:
	return integerConstant(((TypeSArray *) base_type)->dim->toInteger(), size_type_node);
    case Tarray:
	return darrayLenRef(exp);
    default:
	::error("can't determine the length of a %s", exp_type->toChars());
	return error_mark_node;
    }
}
    
tree
IRState::floatMod(tree a, tree b, Type * d_type)
{
#if D_GCC_VER >= 34
    enum built_in_function fn;
    switch (d_type->toBasetype()->ty) {
    case Tfloat32:
    case Timaginary32:
	fn = BUILT_IN_FMODF;
	break;
    case Tfloat64:
    case Timaginary64:
    no_long_double:
	fn = BUILT_IN_FMOD;
	break;
    case Tfloat80:
    case Timaginary80:
	if (! haveLongDouble())
	    goto no_long_double;
	fn = BUILT_IN_FMODL;
	break;
    default:
	::error("tried to perform floating-point modulo division on %s",
	    d_type->toChars());
	return error_mark_node;
    }
    tree decl = built_in_decls[fn];
    // %% assuming no arg conversion needed
    // %% bypassing buildCall since this shouldn't have
    // side effects
    return build(CALL_EXPR, TREE_TYPE(TREE_TYPE(decl)),
	addressOf(decl),
	tree_cons(NULL_TREE, a,
	    tree_cons(NULL_TREE, b, NULL_TREE)));
#else
    // could move this to d-builtins.c
    static tree float_mod_decls[3];
    static const char * decl_names[3] = { "fmodf", "fmod", "fmodl" };
    tree the_float_type;
    int decl_idx;
    switch (d_type->toBasetype()->ty) {
    case Tfloat32:
    case Timaginary32:
	decl_idx = 0;
	the_float_type = float_type_node;
	break;
    case Tfloat64:
    case Timaginary64:
    no_long_double:
	decl_idx = 1;
	the_float_type = double_type_node;
	break;
    case Tfloat80:
    case Timaginary80:
	if (! haveLongDouble())
	    goto no_long_double;
	decl_idx = 2;
	the_float_type = long_double_type_node;
	break;
    default:
	::error("tried to perform floating-point modulo division on %s",
	    d_type->toChars());
	return error_mark_node;
    }
    
    tree decl = float_mod_decls[decl_idx];
    if (! decl) {
	tree func_type = build_function_type(the_float_type,
	    tree_cons(NULL_TREE, the_float_type,
		tree_cons(NULL_TREE, the_float_type,
		    tree_cons(NULL_TREE, void_type_node, NULL_TREE))));
	decl = build_decl(FUNCTION_DECL, get_identifier(decl_names[decl_idx]),
	    func_type);
	DECL_EXTERNAL(decl) = 1;
	float_mod_decls[decl_idx] = decl;
    }
    // %% assuming no arg conversion needed
    // %% bypassing buildCall since this shouldn't have
    // side effects
    return build(CALL_EXPR, TREE_TYPE(TREE_TYPE(decl)),
	addressOf(decl),
	tree_cons(NULL_TREE, a,
	    tree_cons(NULL_TREE, b, NULL_TREE)));
#endif
}

Expression *
IRState::typeinfoReferenceExp(Type * t, const Loc loc)
{
    Expression * e = t->getProperty(loc, Id::typeinfo);
    assert( e->type->toBasetype()->ty == Tclass );
    return e;
}

tree
IRState::typeinfoReference(Type * t, const Loc loc)
{
    tree ti_ref = t->getProperty(loc, Id::typeinfo)->toElem(this);
    assert( POINTER_TYPE_P( TREE_TYPE( ti_ref )) );
    return ti_ref;
}


// delegate is
// struct delegate {
//   void * frame_or_object;
//   void * function;
// }

tree
IRState::delegateObjectRef(tree exp)
{
    // Get the backend type for the array and pick out the array data
    // pointer field (assumed to be the first field.)
    tree obj_field = TYPE_FIELDS( TREE_TYPE( exp ));
    return build(COMPONENT_REF, TREE_TYPE( obj_field ), exp, obj_field);
}

tree
IRState::delegateMethodRef(tree exp)
{
    // Get the backend type for the array and pick out the array length
    // field (assumed to be the second field.)
    tree method_field = TREE_CHAIN( TYPE_FIELDS( TREE_TYPE( exp )));
    return build(COMPONENT_REF, TREE_TYPE( method_field ), exp, method_field);
}

// Converts pointer types of method_exp and object_exp to match d_type
tree
IRState::delegateVal(tree method_exp, tree object_exp, Type * d_type)
{
    Type * base_type = d_type->toBasetype();
    if ( base_type->ty == Tfunction ) {
	// Called from DotVarExp.  These are just used to
	// make function calls and not to make Tdelegate variables.
	// Clearing the type makes sure of this.
	base_type = 0;
    } else {
	assert(base_type->ty == Tdelegate);
    }
    
    tree type = base_type ? base_type->toCtype() : NULL_TREE;
    tree ctor = make_node( CONSTRUCTOR );
    tree obj_field = NULL_TREE;
    tree func_field = NULL_TREE;
    
    TREE_READONLY( ctor ) = 1;
    if (type) {
	TREE_TYPE( ctor ) = type;
	obj_field = TYPE_FIELDS( type );
	func_field = TREE_CHAIN( obj_field );
    }
    CONSTRUCTOR_ELTS( ctor ) =
	tree_cons( obj_field,  object_exp,
	    tree_cons( func_field, method_exp, NULL));

    return ctor;
}

void
IRState::extractMethodCallExpr(tree mcr, tree & callee_out, tree & object_out) {
    assert( D_IS_METHOD_CALL_EXPR( mcr ));
    tree elts = CONSTRUCTOR_ELTS( mcr );
    object_out = TREE_VALUE( elts );
    callee_out = TREE_VALUE( TREE_CHAIN( elts ));
}

tree
IRState::objectInstanceMethod(Expression * obj_exp, FuncDeclaration * func, Type * d_type)
{
    Type * obj_type = obj_exp->type->toBasetype();
    if (func->isThis()) {
	tree this_expr = obj_exp->toElem( this );
	
	// Calls to super are static (func is the super's method)
	// Structs don't have vtables.
	// Final and private methods can be called directly. (Private
	//  methods don't show up in the vtable (?check 0.81))
	if (obj_exp->op == TOKsuper ||
	    obj_type->ty == Tstruct || obj_type->ty == Tpointer ||
	    func->isFinal() || func->protection == PROTprivate  ||
	    func->vtblIndex == -1) { // don't know if this is needed as a catchall
	    
	    if (obj_type->ty == Tstruct)
		this_expr = addressOf(this_expr);
	    return methodCallExpr(functionPointer(func), this_expr, d_type);
	} else {
	    // Interface methods are also in the class's vtable, so we don't
	    // need to convert from a class pointer to an interface pointer.
	    this_expr = maybeMakeTemp( this_expr );

	    tree vtbl_ref = build1( INDIRECT_REF, TREE_TYPE(TREE_TYPE( this_expr )),
		this_expr ); // dereference the pointer to the object
	    tree field = TYPE_FIELDS( TREE_TYPE( vtbl_ref )); // the vtbl is the first field
	    vtbl_ref = build( COMPONENT_REF, TREE_TYPE( field ), vtbl_ref, field ); // vtbl field (a pointer)
	    // %% better to do with array ref?
	    vtbl_ref = build( PLUS_EXPR, TREE_TYPE(vtbl_ref), vtbl_ref,
		size_int( getPointerSize() * func->vtblIndex ));
	    vtbl_ref = build1( INDIRECT_REF, TREE_TYPE( functionPointer(func) ),
		vtbl_ref );

	    return methodCallExpr(vtbl_ref, this_expr, d_type);
	}
    } else {
	// Static method; ignore the object instance
	return addressOf(func);
    }
}


tree
IRState::realPart(tree c) {
    return build1(REALPART_EXPR, TREE_TYPE(TREE_TYPE(c)), c);
}
tree
IRState::imagPart(tree c) {
    return build1(IMAGPART_EXPR, TREE_TYPE(TREE_TYPE(c)), c);
}

// Create a record type from two field types
tree
IRState::twoFieldType(Type * ft1, Type * ft2, Type * d_type)
{
    tree rec_type = make_node( RECORD_TYPE );
    tree f0 = build_decl(FIELD_DECL, get_identifier("_a"), ft1->toCtype());
    tree f1 = build_decl(FIELD_DECL, get_identifier("_b"), ft2->toCtype());
    DECL_CONTEXT(f0) = rec_type;
    DECL_CONTEXT(f1) = rec_type;
    TYPE_FIELDS(rec_type) = chainon(f0, f1);
    layout_type(rec_type);
    TYPE_STUB_DECL(rec_type) = build_decl(TYPE_DECL, NULL_TREE, rec_type);
    rest_of_type_compilation(rec_type, 1);
    if (d_type)
	TYPE_LANG_SPECIFIC(rec_type) = build_d_type_lang_specific(d_type);
    return rec_type;
}

tree
IRState::aggDeclStaticInit(AggregateDeclaration * agg_decl)
{
    SymbolDeclaration * symdec = new SymbolDeclaration(0, agg_decl->toInitializer());
    tree t = symdec->toSymbol()->Stree;
    delete symdec;
    return t;
}

Module * IRState::builtinsModule = 0;
Module * IRState::intrinsicModule = 0;

static inline const char * strip_builtin(const char * x) {
    if (! strncmp(x, "__builtin_", strlen("__builtin_")))
	return x + strlen("__builtin_");
    else
	return x;
}

bool
IRState::maybeSetUpBuiltin(Declaration * decl)
{
    Module * mod = decl->getModule();
    if (! mod || ! decl->ident)
	return false;
    
    if (mod == builtinsModule) {
	const char * decl_ident = strip_builtin( decl->ident->string );

	for (int i =  0; i < (int) END_BUILTINS; i++) {
	    tree bi_decl = built_in_decls[i];
	    if (! bi_decl)
		continue;
	    const char * bi_name = IDENTIFIER_POINTER(DECL_ASSEMBLER_NAME(bi_decl));
	    if ( ! strcmp( decl_ident, strip_builtin( bi_name ) )) {
		// %% assuming everything is a function...
		decl->csym  = new Symbol();
		decl->csym->Stree = bi_decl;

		// %% hack
		TYPE_LANG_SPECIFIC( TREE_TYPE( bi_decl )) =
		    build_d_type_lang_specific( decl->type );

		return true;
	    }
	}
    }
#if 0
    else if (mod == intrinsicModule) {
	// Matches order of Intrinsic enum
	static const char * intrinsic_names[] = {
	    "bsf", "bsr",
	    "bt", "btc", "btr", "bts",
	    "bswap",
	    "inp", "inpw", "inpl",
	    "outp", "outw", "outl"
	};
	for (int i = 0; i < (int) INTRINSIC_count; i++) {
	    if ( ! strcmp( decl->ident->string, intrinsic_names[i] ) ) {
		tree type = decl->type->toCtype();
		tree fn_decl = build_decl(FUNCTION_DECL,
		    get_identifier(intrinsic_names[i]), type);
		DECL_BUILT_IN_CLASS( fn_decl ) = BUILT_IN_FRONTEND;
		DECL_FUNCTION_CODE( fn_decl ) = i;
		
		decl->csym  = new Symbol();
		decl->csym->Stree = fn_decl;

		// %% hack (do I need this?)
		TYPE_LANG_SPECIFIC( TREE_TYPE( fn_decl )) =
		    build_d_type_lang_specific( decl->type );

		return true;
	    }
	}
    }
#endif
    return false;
}


bool
IRState::isDeclarationReferenceType(Declaration * decl)
{
    Type * base_type = decl->type->toBasetype();

    // D doesn't do this now..
    if ( base_type->ty == Treference ) {
	return true;
    }

    if (  decl->isOut() ||
	( decl->isParameter() && base_type->ty == Tsarray ) ) {
	return true;
    }

    return false;
}

tree
IRState::trueDeclarationType(Declaration * decl)
{
    // If D supported references, we would have to check twice for
    //   (out T &) -- disallow, maybe or make isDeclarationReferenceType return
    //   the number of levels to reference
    tree decl_type = decl->type->toCtype();
    if ( isDeclarationReferenceType( decl )) {
	return build_reference_type( decl_type );
    } else {
	return decl_type;
    }
}

// These should match the Declaration versions above
bool
IRState::isArgumentReferenceType(Argument * arg)
{
    Type * base_type = arg->type->toBasetype();

    if ( base_type->ty == Treference ) {
	return true;
    }

    if ( (arg->inout == Out) || (arg->inout == InOut) || base_type->ty == Tsarray ) {
	return true;
    }

    return false;
}

tree
IRState::trueArgumentType(Argument * arg)
{
    tree arg_type = arg->type->toCtype();
    if ( isArgumentReferenceType( arg )) {
	return build_reference_type( arg_type );
    } else {
	return arg_type;
    }
}

void
IRState::aggregateToObjFileCommon(AggregateDeclaration * agg_decl, tree type_node)
{
    tree struct_type = agg_decl->type->toCtype();
    if ( POINTER_TYPE_P(struct_type) ) {
	// Can't use a REFERENCE_TYPE for DECL_CONTEXT; decl_type_context() will abort 
	struct_type = TREE_TYPE( struct_type );
    }
    
    (*lang_hooks.decls.pushlevel) (0);
    this->pushDeclContext( struct_type );

    if (agg_decl->members) {
	for (unsigned i = 0; i < agg_decl->members->dim; i++) {
	    Dsymbol * sym = (Dsymbol *) agg_decl->members->data[i];
	    sym->toObjFile();
	}
    }

    this->popDeclContext();
    (*lang_hooks.decls.poplevel) (0, 0, 0); // do anything with the list?

    // Emit the static initializer, unless a anonymous, union, or interface
    // %% Changed: do unions now
    StructDeclaration * s = agg_decl->isStructDeclaration();
    if ( ! (s && (! s->ident/* || s->isUnion()*/)) && ! agg_decl->isInterfaceDeclaration() ) {
	tree init_decl = agg_decl->toInitializer()->Stree;
	
	prepareSymbolOutput(agg_decl, init_decl);
	TREE_PUBLIC( init_decl ) = 1;
	TREE_USED( init_decl ) = 1;
	d_mark_addressable( init_decl );
	
	layout_decl( init_decl, 0 );
	rest_of_decl_compilation( init_decl, NULL/*sym->Sident*/, 1, 0);
    }
}

tree
IRState::aggregateInitializer(AggregateDeclaration * agg_decl,
    Array * ini_fields, Array * ini_values, tree ini_nodes)
{
    tree struct_type = agg_decl->type->toCtype();	
    if (TREE_CODE( struct_type ) == REFERENCE_TYPE )
	struct_type = TREE_TYPE( struct_type );
    
    tree struct_fields = TYPE_FIELDS( struct_type );
    tree struct_field_i = struct_fields;
    unsigned i;

    // %% again, is ->fields only vardecls
    Array * dmd_field_decls = agg_decl->isClassDeclaration() ?
	agg_decl->toSymbol()->Sspecial.classDecl.flattenedFields :
	& agg_decl->fields;
    const unsigned field_count = dmd_field_decls->dim;
    Array   gcc_field_decls;
    Array   out_vals;

    gcc_field_decls.setDim(field_count);
    out_vals.setDim(field_count);

    i = 0;
    while (struct_field_i) {
	gcc_field_decls.data[i] = struct_field_i;
	out_vals.data[i] = NULL_TREE;
	i++;
	struct_field_i = TREE_CHAIN(struct_field_i);
    }

    if (ini_fields) {
	for (i = 0; i < ini_fields->dim; i++) {
	    bool found = false;
	    VarDeclaration * ini_decl = (VarDeclaration *) ini_fields->data[i];
	    for (unsigned j = 0; j < field_count; j++) {
		VarDeclaration * field_decl = (VarDeclaration *) dmd_field_decls->data[j];
		if (ini_decl == field_decl) {
		    // if expression, convert for ini
		    Initializer * init_val = (Initializer *) ini_values->data[i];
		    ExpInitializer * exp_init = init_val->isExpInitializer();
		    tree value;
		    if (exp_init) {
			value = convertForInitialization(exp_init->exp, field_decl);
		    } else {
			value = init_val->toDt();
		    }
		    out_vals.data[j] = value;
		    found = true;
		//} else if (ini_decl->offset == field_decl->offset) {
		} else if (field_decl->offset >= ini_decl->offset &&
			   field_decl->offset <  ini_decl->offset + ini_decl->type->size()) {
		    // %%TODO: need to handle overlapping definition
		    // new_value = convertFo|init_val|error_mark
		    // if (out_vals.data[j] && != new_value...
		    // %% could still have stuff like this (maybe ok)
		    // struct X { union { struct { int x, y, z; }; struct { int a, b, c; }}};
		    // X x = { x:1, y:2, c:3; }
		    
		    // don't allow this default initializer to be used
		    out_vals.data[j] = error_mark_node;
		} else if (ini_decl->offset < field_decl->offset) {
		    // we have found all the fields in the same anonymous (%%) union
		    if (! found)
			abort();
		    break;
		}
	    }
	}
    }

    i = 0;
    // expect ini_nodes to be in order
    while (ini_nodes) {
	tree ini_decl = TREE_PURPOSE( ini_nodes );
	// %% array bounds
	do {
	    if (ini_decl == (tree) gcc_field_decls.data[i]) {
		// Note: no conversions are done
		out_vals.data[i] = TREE_VALUE( ini_nodes );
		break;
	    }
	    i++;
	} while ( i < field_count );
	// %% checks...
	ini_nodes = TREE_CHAIN( ini_nodes );
    }

    // use default initializers
    // better to be able to check for union fields which probably shouldn't
    // be default initialized
    i = 0;
    while ( i < field_count ) {
	VarDeclaration * field_decl;

	if (! out_vals.data[i] && // init'd field above
	    (tree) out_vals.data[i] != error_mark_node && // init'd union sibling field above
	    dmd_field_decls->data[i] ) { // special field w/o default init

	    field_decl = (VarDeclaration *) dmd_field_decls->data[i];
	    tree value = NULL_TREE;
	    if (field_decl->init) {
		// dup'd from above..
		ExpInitializer * exp_init = field_decl->init->isExpInitializer();
		if (exp_init) {
		    value = convertForInitialization(exp_init->exp, field_decl);
		} else {
		    value = field_decl->init->toDt();
		}
	    } else {
		Expression * e = field_decl->type->defaultInit();
		if (e)
		    value = convertForInitialization(e, field_decl);
		else {
		    //printf("huh?\n");
		}
	    }
	    out_vals.data[i] = value;
	    
	    unsigned last_offset = field_decl->offset;
	    
	    while (++i < field_count) {
		field_decl = (VarDeclaration *) dmd_field_decls->data[i];
		if ( ! field_decl || field_decl->offset > last_offset ) {
		    break;
		}
	    }
	} else {
	    i++;
	}
    }

    tree elts = NULL_TREE;
    // go in reverse so list will be correct from the start
    for (int n = field_count - 1; n >= 0; n--) {
	tree value = (tree) out_vals.data[n];
	if (value && value != error_mark_node) {
	    /*
	    fprintf(stderr, "*** %4u %u %s\n",
		TREE_INT_CST_LOW( DECL_FIELD_BIT_OFFSET( ((tree)gcc_field_decls.data[n]) ) ),
		((tree)gcc_field_decls.data[n])->decl.name->identifier.id.str);
	    debug_tree( (tree)gcc_field_decls.data[n] );
	    //debug_tree(value);
	    //fprintf(stderr, "...***");
	    */
	    elts = tree_cons((tree) gcc_field_decls.data[n], value, elts);
	} else { // reallerror? see varasm.c:output_constructor -- probably need all fields defined
	    // 	    fprintf(stderr, "*** NOTHING FOR %s: \n",
	    // 		((tree)gcc_field_decls.data[n])->decl.name->identifier.id.str);
	    // %%
	    continue;
	    tree field = ((tree) gcc_field_decls.data[n]);
	    tree ident = field->decl.name;
	    // %% FIX: disabled to support unions
	    //::error("ICE: missing field #%d \"%s\"\n", n, ident ? (char*)ident->identifier.id.str : "<unnamed field>");
	    if (! ident)
		debug_tree(field);
	    abort();
	}
    }

    tree ctor = make_node(CONSTRUCTOR);
    TREE_TYPE( ctor ) = struct_type; // %% fix
    CONSTRUCTOR_ELTS( ctor ) = elts;
    TREE_READONLY( ctor ) = 1;
    
    return ctor;
}      

tree
IRState::vtable(Array * vtbl, ClassDeclaration * ci_override)
{
    tree ctor = make_node( CONSTRUCTOR );
    tree elem_list = NULL_TREE;

    TREE_TYPE( ctor ) = arrayType(Type::tvoid->pointerTo(), vtbl->dim);

    for (int i = vtbl->dim - 1; i >= 0 ; i--) { // backwards so elem_list is in order
	Dsymbol * vtbl_ent = (Declaration *) vtbl->data[i];
	Declaration * decl;
	tree value;
	
	if ((decl = vtbl_ent->isDeclaration()) && decl->isAbstract()) {
	    value = d_null_pointer;
	} else {
	    if (i == 0 && ci_override)
		vtbl_ent = ci_override;
	    value = addressOf( vtbl_ent );
	}
	
	elem_list = tree_cons(NULL_TREE, value, elem_list);
    }

    TREE_CONSTANT( ctor ) = 1;
    TREE_READONLY( ctor ) = 1;
    TREE_STATIC( ctor ) = 1;
    TREE_ADDRESSABLE( ctor ) = 1; // %% or just the decl?
    CONSTRUCTOR_ELTS( ctor ) = elem_list;
    return ctor;
}

tree
IRState::exceptionObject()
{
    tree obj_type = getObjectType()->toCtype();
    // Like gjc, the actual D exception object is one
    // pointer behind the exception header
    tree t = build (EXC_PTR_EXPR, ptr_type_node);
    t = build1(NOP_EXPR, build_pointer_type(obj_type), t); // treat exception header as ( Object* )
    t = build(MINUS_EXPR, TREE_TYPE(t), t, TYPE_SIZE_UNIT(TREE_TYPE(t)));
    t = build1(INDIRECT_REF, obj_type, t);
    return t;
}

tree
IRState::arrayType(tree type_node, xdmd_integer_t size)
{
    tree index_type_node;
    // -1 okay?
    if (size > 0)
	index_type_node = integerConstant(size - 1, size_type_node);
    else
	index_type_node = integer_minus_one_node;
    index_type_node = build_index_type(index_type_node);
    return build_array_type(type_node, index_type_node);
}




tree
IRState::makeStmtExpr(Statement * statement, IRState * irs)
{
    tree t = build1((enum tree_code) D_STMT_EXPR, void_type_node, NULL_TREE);
    TREE_SIDE_EFFECTS(t) = 1; // %% 
    stmtExprList.push(t);
    stmtExprList.push(statement);
    stmtExprList.push(irs);
    return t;
}

void
IRState::retrieveStmtExpr(tree t, Statement ** s_out, IRState ** i_out)
{
    for (int i = stmtExprList.dim - 3; i >= 0 ; i -= 3) {
	if ( (tree) stmtExprList.data[i] == t ) {
	    *s_out = (Statement *) stmtExprList.data[i + 1];
	    *i_out = (IRState *)   stmtExprList.data[i + 2];
	    // The expression could be evaluated multiples times, so we must
	    // keep the values forever --- %% go back to per-function list
	    /*
	    stmtExprList.remove(i);
	    stmtExprList.remove(i);
	    stmtExprList.remove(i);
	    */
	    return;
	}
    }
    abort();
    return;
}


BitConfig bitConfig;

void
BitConfig::setType(Type * type)
{
    elemType = type;
    shift = exact_log2( elemType->size() * 8 );
    mask = elemType->size() * 8 - 1;
    shift_tree = IRState::integerConstant(shift, unsigned_type_node); // %% 32/64?
    mask_tree = IRState::integerConstant(mask, elemType->toCtype());
    dkeep(shift_tree);
    dkeep(mask_tree);
}

WrappedExp::WrappedExp(Loc loc, enum TOK op, tree exp_node, Type * type)
    : Expression(loc, op, sizeof(WrappedExp))
{
    this->exp_node = exp_node;
    this->type = type;
}

void
WrappedExp::toCBuffer(OutBuffer *buf)
{
    buf->printf("< wrapped exprission >");
}

elem *
WrappedExp::toElem(IRState *)
{
    return exp_node;
}


void AggLayout::addFields(Array * fields)
{
    tree new_field_chain = NULL_TREE;
    for (unsigned i = 0; i < fields->dim; i++) {
	// %% D anonymous unions just put the fields into the outer struct...
	// does this cause problems?

	VarDeclaration * var_decl = (VarDeclaration *) fields->data[i];
	
	assert( var_decl->storage_class & STCfield );

	tree ident = var_decl->ident ? get_identifier(var_decl->ident->string) : NULL_TREE;
	tree field_decl = build_decl(FIELD_DECL, ident,
	    gen.trueDeclarationType( var_decl ));
	// %% may want to set var_decl->csym and lang_decl
	// %% there are issues with this (inherited members)
	gen.setDeclLoc( field_decl, var_decl->loc );
	var_decl->csym = new Symbol;
	var_decl->csym->Stree = field_decl;
	
	DECL_CONTEXT( field_decl ) = aggType;
	DECL_FCONTEXT( field_decl ) = aggType; // %% should point to base class
	DECL_FIELD_OFFSET (field_decl) = size_int( var_decl->offset );
	DECL_FIELD_BIT_OFFSET (field_decl) = bitsize_int( 0 );
	//SET_DECL_OFFSET_ALIGN (field_decl, BIGGEST_ALIGNMENT); // ?
	layout_decl( field_decl, 0 );
	new_field_chain = chainon(field_decl, new_field_chain);
    }
    TYPE_FIELDS( aggType ) = chainon(TYPE_FIELDS( aggType ), nreverse( new_field_chain ));
}

void AggLayout::addField(tree field_decl, unsigned offset)
{
    DECL_CONTEXT( field_decl ) = aggType;
    DECL_FCONTEXT( field_decl ) = aggType; // %% should point to base class
    DECL_FIELD_OFFSET (field_decl) = size_int( offset );
    DECL_FIELD_BIT_OFFSET (field_decl) = bitsize_int( 0 );
    Loc l(aggDecl->getModule(), 1); // Must set this or we crash with DWARF debugging
    // gen.setDeclLoc( field_decl, aggDecl->loc ); // aggDecl->loc isn't set
    gen.setDeclLoc(field_decl, l);
    
    layout_decl( field_decl, 0 );
    TYPE_FIELDS( aggType ) = chainon(TYPE_FIELDS( aggType ), field_decl);
}

void AggLayout::finish()
{
    unsigned size_to_use = aggDecl->structsize;
    unsigned align_to_use = aggDecl->alignsize;

    TYPE_SIZE( aggType ) = bitsize_int( size_to_use * BITS_PER_UNIT );
    TYPE_SIZE_UNIT( aggType ) = size_int( size_to_use );
    TYPE_ALIGN( aggType ) = align_to_use * BITS_PER_UNIT; // %%doc int, not a tree
    // TYPE_ALIGN_UNIT is not an lvalue
    TYPE_PACKED (aggType ) = TYPE_PACKED (aggType ); // %% todo
    compute_record_mode ( aggType );
    // %%  stor-layout.c:finalize_type_size ... it's private to that file

    
    // c-decl.c -- set up variants ? %%
    for (tree x = TYPE_MAIN_VARIANT( aggType ); x; x = TYPE_NEXT_VARIANT( x )) {
	TYPE_FIELDS (x) = TYPE_FIELDS (aggType);
	TYPE_LANG_SPECIFIC (x) = TYPE_LANG_SPECIFIC (aggType);
	TYPE_ALIGN (x) = TYPE_ALIGN (aggType);
	TYPE_USER_ALIGN (x) = TYPE_USER_ALIGN (aggType);
    }

}

CtorMaker::CtorMaker(AggregateDeclaration * agg_decl, int storage_class, IRState * irs)
{
    aggDecl = agg_decl;
    storageClass = storage_class;
    this->irs = irs;
    tree type = aggDecl->type->toCtype();

    if (TREE_CODE(type) == REFERENCE_TYPE)
	type = TREE_TYPE( type );
    if (TREE_CODE(type) == RECORD_TYPE ||
	TREE_CODE(type) == UNION_TYPE )
	curField = TYPE_FIELDS(type);

    elemList = NULL_TREE;
}

CtorMaker & CtorMaker::add(tree value)
{
    elemList = tree_cons(curField, value, elemList);
    curField = TREE_CHAIN(curField);
    return *this;
}

CtorMaker & CtorMaker::add(tree field, tree value)
{
    elemList = tree_cons(field, value, elemList);
    curField = TREE_CHAIN(field);
    return *this;
}

CtorMaker & CtorMaker::addArray(unsigned count, tree data)
{
    assert(data || ! count);
    //if ( ! POINTER_TYPE_P( value ) ) {
    if (count) {
	if ( TREE_CODE( data ) != ADDR_EXPR &&
	     TREE_CODE( data ) != REFERENCE_EXPR ) {
	    data = irs->addressOf(data);
	}
    } else {
	data = convert(ptr_type_node, integer_zero_node);
    }
    return add(irs->darrayVal(TREE_TYPE(curField), count, data));
}

tree CtorMaker::finish()
{
    tree ctor = irs->aggregateInitializer(aggDecl, NULL, NULL,
	nreverse( elemList ));

    if (storageClass & STCstatic)
	TREE_STATIC( ctor ) = 1;
    if (storageClass & STCconst)
	TREE_CONSTANT( ctor ) = 1;
    TREE_READONLY( ctor ) = 1;
    return ctor;
}

ArrayMaker::ArrayMaker(Type * elem_type, int storage_class)
{
    elemType = elem_type;
    storageClass = storage_class;
    
    elemList = NULL_TREE;
    count = 0;
}

ArrayMaker &
ArrayMaker::add(tree value)
{
    elemList = tree_cons(NULL_TREE, value, elemList);
    count++;
    return *this;
}

tree
ArrayMaker::finish()
{
    tree ctor = make_node(CONSTRUCTOR);
    tree type = gen.arrayType(elemType->toCtype(), count);
    TREE_TYPE( ctor ) = type;
    CONSTRUCTOR_ELTS( ctor ) = nreverse( elemList );
    if (storageClass & STCstatic)
	TREE_STATIC( ctor ) = 1;
    if (storageClass & STCconst)
	TREE_CONSTANT( ctor ) = 1;
    TREE_READONLY( ctor ) = 1;
    return ctor;
    
}

