/*
 * main.cpp
 *
 * Copyright (C) 2003 Ed Cogburn <ecogburn@xtn.net>
 *
 * This file is part of the program "d: The Directory Lister".
 *
 * "d: The Directory Lister" is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.
 *
 * "d: The Directory Lister" is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with "d: The Directory Lister" in a file named "COPYING".
 * If not, visit their website at http://www.gnu.org for a copy, or request a copy by writing to:
 *     Free Software Foundation, Inc.
 *     59 Temple Place, Suite 330
 *     Boston, MA 02111-1307 USA
*/



#include "globals.hpp"
#include <sys/types.h>
#include <dirent.h>
#include <ctime>
#include <cmath>
#include <fnmatch.h>



// Our option declarations
poc::opt_v_csv_str oColor('C', "color");
poc::opt_bool oColorize('c', "colorize", true);
poc::opt_bool oDirsFirst('d', "dirs-first", true);
poc::opt_bool oFollowLinks('f', "follow-links");
poc::opt_bool oHelp('h', "help");
poc::opt_bool oHiddenFiles('H', "hidden-files");
poc::opt_bool oKilobyteReal('k', "kilobyte-real", true);
poc::opt_int  oPadding('p', "padding", 2);
poc::opt_bool oReverseSort('r', "reverse-sort");
poc::opt_bool oRecurse('R', "recurse");
poc::opt_bool oSummary('S', "summary", true);
poc::opt_str  oSort('s', "sort", "name");
poc::opt_bool oTreeAscii('T', "tree-ascii");
poc::opt_bool oTree('t', "tree");
poc::opt_bool oVersion('v', "version");

poc::parse_opt_cfg poptcfg;

// These control how much space must be reserved for their section.  They're calculated on a first pass before printing.
unsigned uid_name_section_width;
unsigned gid_name_section_width;
unsigned file_size_section_width;

// This stuff is for colorization output.
struct color_setting
{
	ct::Color fg;
	ct::Color bg;
	ct::Attr at;
};

// Default colorizations used by d.  This array is indexed by colorize_index below.
color_setting colorize[] = {
	{  ct::Yellow,   ct::Red,   ct::Bold },
	{    ct::Cyan, ct::Black,   ct::Bold },
	{   ct::Green, ct::Black,   ct::Bold },
	{ ct::Magenta, ct::Black,   ct::Bold },
	{   ct::White, ct::Black, ct::Normal },
	{  ct::Yellow,  ct::Blue,   ct::Bold },
	{   ct::White,  ct::Blue,   ct::Bold },
	{     ct::Red,  ct::Blue,   ct::Bold }
};

enum colorize_index { ciBadLink, ciDir, ciExec, ciLink, ciNormal, ciChrDev, ciBlkDev, ciSpecial };

// This struct is for user provided colorizations.
struct custom_color_setting
{
	string name;
	ct::Color fg;
	ct::Color bg;
	ct::Attr at;
};

// This will hold the user provided colorizations, after being parsed.
vector<custom_color_setting> filecolorlist;

// Used when recursing a non-tree output
unsigned grand_total_files;
unsigned long grand_total_size;

// Characters to use for tree output - see dump_tree()
const char* dt_ld1 = "  \x1B(0\x78\x1B(B ";
const char* dt_ld2 = "  \x1B(0\x6D\x71\x1B(B";
const char* dt_ld3 = "  \x1B(0\x74\x71\x1B(B";
const char* dt_as1 = "  | ";
const char* dt_as2 = "  `-";
const char* dt_as3 = "  |-";
const char* dt_seq1 = NULL;
const char* dt_seq2 = NULL;
const char* dt_seq3 = NULL;

ct::colorized_text ctxt;

const char* vers = "1.2.0";

enum dl_subdir { NotASubdir, IsASubdir };



//-----------------------------------------------------------------------------
// Helper function for print_help_msg()
string prtstrcolor(const char* output, ct::Color cf, ct::Color cb, ct::Attr a)
{
	string s;
	
	if (oColorize.Set)
		s += ctxt.set(cf, cb, a);

	s += output;

	if (oColorize.Set)
		s += ctxt.reset();

	return s;
}



//-----------------------------------------------------------------------------
void print_help_msg(stringstream& ss)
{
	if (oColorize.Set)
		ss << ctxt.set(ct::Yellow, ct::Black, ct::Bold);
	ss << "d:  The Directory Lister.  Version " << vers << ".  Copyright 2003 by Ed Cogburn.";
	if (oColorize.Set)
		ss << ctxt.reset();
	ss << endl;

	ss << prtstrcolor("Released under GNU General Public License.  See the file COPYING for details.", 
					  ct::White, ct::Black, ct::Bold);
	ss << endl << endl;

	ss << "Usage:  d [options] [path]" << endl << endl;

	ss << "Options:" << endl;

	ss << "  ";
	ss << prtstrcolor("--help", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-h", ct::Green, ct::Black, ct::Bold);
	ss << "\t\tThis help screen." << endl;

	ss << "  ";
	ss << prtstrcolor("--version", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-v", ct::Green, ct::Black, ct::Bold);
	ss << "\t\tVersion number of 'd'." << endl;

	ss << "  ";
	ss << prtstrcolor("--color", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-C", ct::Green, ct::Black, ct::Bold);
	ss << "\t\tCustomize colorizations. See info docs." << endl;

	ss << "  ";
	ss << prtstrcolor("--colorize", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-c", ct::Green, ct::Black, ct::Bold);
	ss << "\t\tColorize output. Default: ON." << endl;

	ss << "  ";
	ss << prtstrcolor("--dirs-first", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-d", ct::Green, ct::Black, ct::Bold);
	ss << "\tAlways print directories first. Default: ON." << endl;

	ss << "  ";
	ss << prtstrcolor("--follow-links", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-f", ct::Green, ct::Black, ct::Bold);
	ss << "\tFollow links when listing recursively. Default: OFF." << endl;

	ss << "  ";
	ss << prtstrcolor("--hidden-files", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-H", ct::Green, ct::Black, ct::Bold);
	ss << "\tShow hidden files. Default: OFF." << endl;

	ss << "  ";
	ss << prtstrcolor("--kilobyte-real", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-k", ct::Green, ct::Black, ct::Bold);
	ss << "\tUse a real kilobyte, 1,024, instead of 1,000" << endl << "\t\t\tfor calculating file size. Default: ON." << endl;

	ss << "  ";
	ss << prtstrcolor("--recurse", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-R", ct::Green, ct::Black, ct::Bold);
	ss << "\t\tPrint directories recursively. Default: OFF." << endl;

	ss << "  ";
	ss << prtstrcolor("--reverse-sort", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-r", ct::Green, ct::Black, ct::Bold);
	ss << "\tPrint output in reverse order. Default: OFF." << endl;

	ss << "  ";
	ss << prtstrcolor("--summary", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-S", ct::Green, ct::Black, ct::Bold);
	ss << "\t\tPrint a summary line at the end. Default: ON." << endl;

	ss << "  ";
	ss << prtstrcolor("--tree-ascii", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-T", ct::Green, ct::Black, ct::Bold);
	ss << "\tPrint tree using ASCII. Implies -R. Default: OFF." << endl;

	ss << "  ";
	ss << prtstrcolor("--tree", ct::Green, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-t", ct::Green, ct::Black, ct::Bold);
	ss << "\t\tPrint using tree-like format. Implies -R. Default: OFF." << endl;

	ss << "  ";
	ss << prtstrcolor("--padding=", ct::Green, ct::Black, ct::Bold);
	ss << prtstrcolor("arg", ct::Magenta, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-p", ct::Green, ct::Black, ct::Bold);
	ss << prtstrcolor("Arg", ct::Magenta, ct::Black, ct::Bold);
	ss << "\tSet spacing between sections on a line"  << endl << "\t\t\t(Arg = 1 - 16). Default: 2." << endl;

	ss << "  ";
	ss << prtstrcolor("--sort=", ct::Green, ct::Black, ct::Bold);
	ss << prtstrcolor("Arg", ct::Magenta, ct::Black, ct::Bold);
	ss << ",";
	ss << prtstrcolor("-s", ct::Green, ct::Black, ct::Bold);
	ss << prtstrcolor("Arg", ct::Magenta, ct::Black, ct::Bold);
	ss << "\tSet sorting method.  Arg can be '(";
	ss << prtstrcolor("d", ct::Magenta, ct::Black, ct::Bold);
	ss << ")";
	ss << prtstrcolor("ate", ct::Magenta, ct::Black, ct::Bold);
	ss << "',"  << endl << "\t\t\t'(";
	ss << prtstrcolor("n", ct::Magenta, ct::Black, ct::Bold);
	ss << ")";
	ss << prtstrcolor("ame", ct::Magenta, ct::Black, ct::Bold);
	ss << "' or '(";
	ss << prtstrcolor("s", ct::Magenta, ct::Black, ct::Bold);
	ss << ")";
	ss << prtstrcolor("ize", ct::Magenta, ct::Black, ct::Bold);
	ss << "'. Default: '";
	ss << prtstrcolor("name", ct::Magenta, ct::Black, ct::Bold);
	ss << "'." << endl;
		
	if (oColorize.Set)
		ss << ctxt.set(ct::White, ct::Black, ct::Bold);
	ss << "Use '--option=false' or '-o-' to turn an option off, or use the /etc/default/d" << endl;
	ss << "or /etc/d.conf or $HOME/.d.conf configuration file.  See info doc for details." << endl;
	if (oColorize.Set)
		ss << ctxt.reset();
}



//-----------------------------------------------------------------------------
// Used in more than one place, by print_file_size, and to calculate total_size of all files for subdir and all listed if recursing
static string print_human_readable_file_size(unsigned long size)
{
	stringstream ss;
	long double val = size;
	long double divisor;
	char k, m, g, t, e;

	if (oKilobyteReal.Set) {
		divisor = 1024;
		k = 'K';
		m = 'M';
		g = 'G';
		t = 'T';
		e = 'E';
	} else {
		divisor = 1000;
		k = 'k';
		m = 'm';
		g = 'g';
		t = 't';
		e = 'e';
	}

	// For comments below I use 'k', 'm', 'g', 't' and 'e' but they could also be uppercase if oKilobyteReal == true.

	// Width = NNNN | ( (N.N | NN | 0NNN) + (k | m) )
	ss << fixed << setprecision(0);

	// Less than 1000/1024?
	if (val < divisor) {
		ss << val;
		return ss.str();
	}

	// First division, now = N + k
	val /= divisor;

	// Less than 10k?
	if (val < 10) {
		// N.N + k
		// Round up or down if within 0.1
		if (val - floor(val) > 0.9)
			val = floor(val) + 1.0;
		if (val - floor(val) >= 0.1)
			ss << setprecision(1);
		ss << val << k;
		return ss.str();
	}

	// Less than 100k?
	if (val < 100) {
		// NN + k
		ss << val << k;
		return ss.str();
	}

	// Less than 1000k?
	if (val < 1000) {
		// NNN + k
		ss << val << k;
		return ss.str();
	}

	val /= divisor;

	if (val < 10) {
		// N.N + m
		// Round up or down if within 0.1
		if (val - floor(val) > 0.9)
			val = floor(val) + 1.0;
		if (val - floor(val) >= 0.1)
			ss << setprecision(1);
		ss << val << m;
		return ss.str();
	}

	if (val < 100) {
		// NN + m
		ss << val << m;
		return ss.str();
	}

	if (val < 1000) {
		// NNN + m
		ss << val << m;
		return ss.str();
	}

	val /= divisor;

	if (val < 10) {
		// N.N + g
		// Round up or down if within 0.1
		if (val - floor(val) > 0.9)
			val = floor(val) + 1.0;
		if (val - floor(val) >= 0.1)
			ss << setprecision(1);
		ss << val << g;
		return ss.str();
	}

	if (val < 100) {
		// NN + g
		ss << val << g;
		return ss.str();
	}

	if (val < 1000) {
		// NNN + g
		ss << val << g;
		return ss.str();
	}

	val /= divisor;

	if (val < 10) {
		// N.N + t
		// Round up or down if within 0.1
		if (val - floor(val) > 0.9)
			val = floor(val) + 1.0;
		if (val - floor(val) >= 0.1)
			ss << setprecision(1);
		ss << val << t;
		return ss.str();
	}

	if (val < 100) {
		// NN + t
		ss << val << t;
		return ss.str();
	}

	// if (val < 1000) {
	// NNN + t
	ss << val << t;
	return ss.str();

	// Does anyone have files bigger than 999 terabytes?  :-)
}



//-----------------------------------------------------------------------------
// build the "rwxrwxrwx" string
static string print_file_permissions(file_info& f)
{
	char t;
	string perms;

	char d = '-';
	char r = 'r';
	char w = 'w';
	char x = 'x';

	if (f.perm.owner_r)
		t = r;
	else
		t = d;
	perms += t;

	if (f.perm.owner_w)
		t = w;
	else
		t = d;
	perms += t;

	if (f.perm.suid_x)
		t = 's';
	else
		if (f.perm.owner_x)
			t = x;
		else
			t = d;
	perms += t;

	if (f.perm.group_r)
		t = r;
	else
		t = d;
	perms += t;

	if (f.perm.group_w)
		t = w;
	else
		t = d;
	perms += t;

	if (f.perm.sgid_x)
		t = 's';
	else
		if (f.perm.group_x)
			t = x;
		else
			t = d;
	perms += t;

	if (f.perm.other_r)
		t = r;
	else
		t = d;
	perms += t;

	if (f.perm.other_w)
		t = w;
	else
		t = d;
	perms += t;

	if (f.perm.sticky)
		t = 't';
	else
		if (f.perm.other_x)
			t = x;
		else
			t = d;
	perms += t;

	return perms;
}



//-----------------------------------------------------------------------------
// Print date much like ls does.
static string print_file_date(file_info& f)
{
	string s;
	time_t T;
	struct tm dt;
	double tdiff;
	char buf[64];

	// first break down file time
	if (localtime_r(&f.mod_time, &dt) != &dt) {
		stringstream err;
		err << "print_file_date(): Can't convert time-stamp for file \"" << f.name.name_ext() << "\"!";
		throw runtime_error(err.str());
	}

	T = time(NULL);
	tdiff = difftime(T, f.mod_time);
	if (tdiff < 14515200.0) {   // minimum number of seconds in 6 months (28 days per month)
		// print month, day, time
		strftime(buf, 64, "%b %d %H:%M", &dt);
	} else {
		// print month, day, year
		strftime(buf, 64, "%b %d  %Y", &dt);
	}
	s = buf;

	return s;
}



//-----------------------------------------------------------------------------
// Printing the name is made complicated by colorization.
static string print_file_name(file_info& f, bool show_full_name)
{
	static int symlink_recurse_level = 0;
	string s;

	if (oColorize.Set && f.exists)
		// If all execute bits set then check for directory or symlink first.
		// Look for specials
		if (f.is_pipe || f.is_socket)
			s = ctxt.set(colorize[ciSpecial].fg, colorize[ciSpecial].bg, colorize[ciSpecial].at);
		else
			if (f.perm.owner_x || f.perm.group_x || f.perm.other_x)
				if (f.is_dir)
					s = ctxt.set(colorize[ciDir].fg, colorize[ciDir].bg, colorize[ciDir].at);
				else
					if (f.is_symlink)
						if (f.is_dangling_symlink)
							s = ctxt.set(colorize[ciBadLink].fg, colorize[ciBadLink].bg, colorize[ciBadLink].at);
						else
							s = ctxt.set(colorize[ciLink].fg, colorize[ciLink].bg, colorize[ciLink].at);
					else
						s = ctxt.set(colorize[ciExec].fg, colorize[ciExec].bg, colorize[ciExec].at);
			else
				if (f.is_char_device)
					s = ctxt.set(colorize[ciChrDev].fg, colorize[ciChrDev].bg, colorize[ciChrDev].at);
				else
					if (f.is_block_device)
						s = ctxt.set(colorize[ciBlkDev].fg, colorize[ciBlkDev].bg, colorize[ciBlkDev].at);
					else
						// Check against user's list of colorizations
						for (unsigned i = 0; i < filecolorlist.size(); i++)
							if (fnmatch(filecolorlist[i].name.c_str(), f.name.name_ext().c_str(), 0) == 0) {
								s = ctxt.set(filecolorlist[i].fg, filecolorlist[i].bg, filecolorlist[i].at);
								break;
							}

	// If not selected above, must be normal file.
	if (oColorize.Set && s.size() == 0)
		s = ctxt.set(colorize[ciNormal].fg, colorize[ciNormal].bg, colorize[ciNormal].at);

	if (f.is_socket)
		s += '=';

	if (f.is_pipe)
		s += '-';

	if (show_full_name)
		s += f.name.full();
	else
		s += f.name.name_ext();

	if (f.is_dir)
		s += "/";

	if (oColorize.Set)
		s += ctxt.reset();

	if (f.is_symlink && symlink_recurse_level < 3) {
		path fp;
		if (f.symlink_name[0] == '/')
			fp = f.symlink_name;
		else
			fp = f.name.dir() + '/' + f.symlink_name;
		file_info ff(fp.full().c_str());

		s += "->";

		// If symlinked file is in a different directory, print its full path
		symlink_recurse_level++;
		if (f.name.dir() != ff.name.dir())
			s += print_file_name(ff, true);
		else
			s += print_file_name(ff, false);
		symlink_recurse_level--;
	}

	return s;
}



//-----------------------------------------------------------------------------
static string print_file_line(file_info& f)
{
	stringstream ss;
	string pad;

	// Set padding for output lines
	for (int i = 0; i < oPadding.Value; i++)
		pad += " ";

	ss << print_file_permissions(f);

	ss << pad;

	// Owner/Group names
	ss << setw(uid_name_section_width) << f.uid_name << ':';
	ss << setw(gid_name_section_width) << left << f.gid_name << right << setw(0);

	ss << pad;

	// Either file size or major/minor if a device
	ss << setw(file_size_section_width);
	if (f.is_char_device || f.is_block_device) {
		ss << ((f.rdev_nbr >> 8) & 0xFF) << '-' << (f.rdev_nbr & 0xFF);
	} else
		ss << print_human_readable_file_size(f.size);
	ss << setw(0);

	ss << pad;

	ss << print_file_date(f);

	ss << pad;

	ss << print_file_name(f, false);

	ss << '\n';

	return ss.str();
}



//-----------------------------------------------------------------------------
// This is one of the "Comp" functions used for sorting the list.
// If you're wondering, no, I wouldn't have known how to do this without a reference book.  :-)
// If you want to know what this is, try "C++ The Complete Reference", 4th Ed,
// by Herbert Schildt, Chapter 24, "Using Function Objects".
class sort_list_compare_files_by_date : binary_function<file_info&, file_info&, bool>
{
public:
	result_type operator() (first_argument_type arg1, second_argument_type arg2)
	{ 
		if (arg1.mod_time == arg2.mod_time)
			return static_cast<result_type> (arg1.name.full() < arg2.name.full());
		else
			return static_cast<result_type> (arg1.mod_time < arg2.mod_time);
	}
};


		
//-----------------------------------------------------------------------------
// This is one of the "Comp" functions used for sorting the list.
class sort_list_compare_files_by_name : binary_function<file_info&, file_info&, bool>
{
public:
	result_type operator() (first_argument_type arg1, second_argument_type arg2)
	{ 
		return static_cast<result_type> (arg1.name.full() < arg2.name.full());
	}
};


		
//-----------------------------------------------------------------------------
// This is one of the "Comp" functions used for sorting the list.
class sort_list_compare_files_by_size : binary_function<file_info&, file_info&, bool>
{
public:
	result_type operator() (first_argument_type arg1, second_argument_type arg2)
	{ 
		if (arg1.size == arg2.size)
			return static_cast<result_type> (arg1.name.full() < arg2.name.full());
		else
			return static_cast<result_type> (arg1.size < arg2.size);
	}
};


		
//-----------------------------------------------------------------------------
static void sort_directory(list<file_info>& files)
{
	if (files.size() > 1) {
		// main sort via Comp functions above
		if (oSort.Value == "date" || oSort.Value == "d")
			files.sort(sort_list_compare_files_by_date());
		else
			if (oSort.Value == "name" || oSort.Value == "n")
				files.sort(sort_list_compare_files_by_name());
			else
				if (oSort.Value == "size" || oSort.Value == "s")
					files.sort(sort_list_compare_files_by_size());
				else
					throw logic_error("sort_list(): Illegal oSort value!");

		// reverse the sort?
		if (oReverseSort.Set)
			reverse(files.begin(), files.end());

		// dirs-first sort?  Use bubble sort to bring subdirs to the top
		if (oDirsFirst.Set) {
			bool Cont;
			do {
				Cont = false;
				list<file_info>::iterator it1 = files.begin();
				it1++;
				while (it1 != files.end()) {
					list<file_info>::iterator it2 = it1;
					it2--;
					if ( (*it1).is_dir && !(*it2).is_dir ) {
						swap(*it1, *it2);
						Cont = true;
					}
					it1++;
				}
			} while (Cont == true);
		}
	}
}



//-----------------------------------------------------------------------------
// Use libc's opendir/readdir to suck in a directory's contents.
static void read_directory(file_info& df, list<file_info>& filelist)
{
	struct dirent *de;
	DIR *d = opendir(df.name.full().c_str());
	string s;

	while (d != NULL) {
		de = readdir(d);
		if (de == NULL)
			break;

		if (de->d_name[0] == '.') {
			if (de->d_name[1] == '\0')
				continue;
			if (de->d_name[1] == '.' && de->d_name[2] == '\0')
				continue;
			if (!oHiddenFiles.Set)
				continue;
		}

		s = df.name.full();
		s += '/';
		s += de->d_name;

		file_info f = s;
		filelist.push_back(f);
	}
}




//-----------------------------------------------------------------------------
static void dump_list(list<file_info>& files, dl_subdir subdir)
{
	unsigned long total_size;
	unsigned total_files;

	// initialize vars
	file_size_section_width = 1;
	uid_name_section_width = 0;
	gid_name_section_width = 0;
	total_size = 0;
	total_files = 0;

	// This is our first pass to calculate certain vars for optimizing output on the second pass.
	// Check size of user and group names.
	// Check whether devices are being displayed to decide amount of space needed for the file-size/device-major-minor section.
	// Add up number of normal files and their total size.
	for (list<file_info>::iterator it1 = files.begin(); it1 != files.end(); it1++) {
		unsigned tmp;

		// Track the maximum these vars need to be
		tmp = (*it1).uid_name.size();
		if (tmp > uid_name_section_width)
			uid_name_section_width = tmp;
		tmp = (*it1).gid_name.size();
		if (tmp > gid_name_section_width)
			gid_name_section_width = tmp;

		// If displaying char/block devices we need more space for that output.
		if ((*it1).is_char_device || (*it1).is_block_device)
			file_size_section_width = 7;
		else {
			tmp = print_human_readable_file_size((*it1).size).size();
			if (file_size_section_width < tmp)
				file_size_section_width = tmp;
		}

		// For the one line summary at the end of the printout.
		if ((*it1).is_regular_file) {
			total_files++;
			total_size += (*it1).size;
		}
	}

	grand_total_files += total_files;
	grand_total_size += total_size;

	// Go thru the list and print
	for (list<file_info>::iterator it1 = files.begin(); it1 != files.end(); it1++) {
		cout << print_file_line(*it1);
	}

	// One line summary
	if (oSummary.Set && total_files) {
		if (oColorize.Set)
			cout << ctxt.set(ct::White, ct::Black, ct::Bold);
		cout << total_files << " regular files";
		if (subdir == IsASubdir)
			cout << " in directory";
		cout << ", with a total size of ";
		cout <<  print_human_readable_file_size(total_size) << '.';
		if (oColorize.Set)
			cout << ctxt.reset();
		cout << endl;
	}
}



//-----------------------------------------------------------------------------
static void dump_tree(file_info df, bool show_full_name)
{
	static vector<bool> show_vline;
	static unsigned recurse_level = 0;

	list<file_info> files;
	list<file_info>::iterator it1;
	list<file_info>::iterator it2;


	read_directory(df, files);

	sort_directory(files);

	cout << print_file_name(df, show_full_name);

	cout << ':' << endl;

	if (files.size() > 0) {

		if (files.size() > 1)
			show_vline.push_back(true);
		else
			show_vline.push_back(false);

		for (it1 = files.begin(); it1 != files.end(); it1++) {
			for (unsigned i = 0; i < recurse_level; i++)
				if (show_vline[i])
					cout << dt_seq1;
				else
					cout << "    ";

			it2 = it1;
			it2++;
			if (it2 == files.end()) {
				show_vline.back() = false;
				cout << dt_seq2;
			} else
				cout << dt_seq3;

			if ((*it1).is_dir) {
				recurse_level++;
				dump_tree((*it1), false);
				recurse_level--;
				continue;
			}

			if ((*it1).is_symlink && oFollowLinks.Set) {
				path fp;
				if ((*it1).symlink_name[0] == '/')
					fp = (*it1).symlink_name;
				else
					fp = (*it1).name.dir() + '/' + (*it1).symlink_name;

				file_info ff(fp.full().c_str());

				if (ff.is_dir) {
					cout << ctxt.set(colorize[ciLink].fg, colorize[ciLink].bg, colorize[ciLink].at);
					cout << (*it1).name.name_ext() << ctxt.reset() << "->";
					recurse_level++;
					dump_tree(ff, false);
					recurse_level--;
					continue;
				}
			}

			cout << print_file_name((*it1), false) << endl;
		}

		show_vline.pop_back();
	}
}



//-----------------------------------------------------------------------------
// Use libc's opendir/readdir to suck in a directory's contents.
static void dump_directory(file_info df, const char* lnkname)
{
	list<file_info> files;

	read_directory(df, files);

	sort_directory(files);

	list<file_info>::iterator it1 = files.begin();
	while (it1 != files.end()) {
		if (oRecurse.Set && (*it1).is_dir) {
			dump_directory(*it1, lnkname);
			it1 = files.erase(it1);
			continue;
		}
		if (oRecurse.Set && oFollowLinks.Set && (*it1).is_symlink) {
			path fp;
			if ((*it1).symlink_name[0] == '/')
				fp = (*it1).symlink_name;
			else
				fp = (*it1).name.dir() + '/' + (*it1).symlink_name;
			file_info ff(fp.full().c_str());
			if (ff.is_dir) {
				dump_directory(ff, (*it1).name.name_ext().c_str());
				it1 = files.erase(it1);
				continue;
			}
		}
		it1++;
	}

	cout << endl;

	if (lnkname != "")
		cout << ctxt.set(colorize[ciLink].fg, colorize[ciLink].bg, colorize[ciLink].at) << lnkname << ctxt.reset() << "->";

	cout << print_file_name(df, true) << ":";

	cout << endl;

	if (files.size() > 0)
		dump_list(files, IsASubdir);
}



//-----------------------------------------------------------------------------
// Part of parsing a "color" option in a configuration file.
static ct::Color read_config_validate_color(string& c, const char* filename, int line)
{
	if (c == "black")
		return ct::Black;
	if (c == "red")
		return ct::Red;
	if (c == "green")
		return ct::Green;
	if (c == "yellow")
		return ct::Yellow;
	if (c == "blue")
		return ct::Blue;
	if (c == "magenta")
		return ct::Magenta;
	if (c == "cyan")
		return ct::Cyan;
	if (c == "white")
		return ct::White;

	stringstream err;
	err << "validate_color(): Invalid color \"" << c << '\"';
	if (filename == "" || line == -1)
		err << "given on the command line" << '!';
	else
		err << "given in \"" << filename << "\" at line " << line << '!';
	err << "  Must be 'black', 'red', 'green', 'yellow', 'blue', 'magenta', 'cyan', or 'white'.";
	throw runtime_error(err.str());
}



//-----------------------------------------------------------------------------
// Part of parsing a "color" option in a configuration file.
static ct::Attr read_config_validate_attr(string& a, const char* filename, int line)
{
	if (a == "normal")
		return ct::Normal;
	if (a == "bold")
		return ct::Bold;
	if (a == "underline")
		return ct::Underline;
	if (a == "blink")
		return ct::Blink;
	if (a == "reverse")
		return ct::Reverse;

	stringstream err;
	err << "validate_attr(): Invalid attribute \"" << a << '\"';
	if (filename == "" || line == -1)
		err << "given on the command line" << '!';
	else
		err << "given in \"" << filename << "\" at line " << line << '!';
	err << "  Must be 'normal', 'bold', 'underline', 'blink', or 'reverse'.";
	throw runtime_error(err.str());
}



//-----------------------------------------------------------------------------
// Part of parsing a "color" option in a configuration file.
static void read_config_color(string& target, string& fcolor, string& bcolor, string& attr, const char* filename, int line)
{
	if (target == "::badlink::") {
		colorize[ciBadLink].fg = read_config_validate_color(fcolor, filename, line);
		colorize[ciBadLink].bg = read_config_validate_color(bcolor, filename, line);
		colorize[ciBadLink].at = read_config_validate_attr(attr, filename, line);
		return;
	}
	if (target == "::dir::") {
		colorize[ciDir].fg = read_config_validate_color(fcolor, filename, line);
		colorize[ciDir].bg = read_config_validate_color(bcolor, filename, line);
		colorize[ciDir].at = read_config_validate_attr(attr, filename, line);
		return;
	}
	if (target == "::exec::") {
		colorize[ciExec].fg = read_config_validate_color(fcolor, filename, line);
		colorize[ciExec].bg = read_config_validate_color(bcolor, filename, line);
		colorize[ciExec].at = read_config_validate_attr(attr, filename, line);
		return;
	}
	if (target == "::link::") {
		colorize[ciLink].fg = read_config_validate_color(fcolor, filename, line);
		colorize[ciLink].bg = read_config_validate_color(bcolor, filename, line);
		colorize[ciLink].at = read_config_validate_attr(attr, filename, line);
		return;
	}
	if (target == "::normal::") {
		colorize[ciNormal].fg = read_config_validate_color(fcolor, filename, line);
		colorize[ciNormal].bg = read_config_validate_color(bcolor, filename, line);
		colorize[ciNormal].at = read_config_validate_attr(attr, filename, line);
		return;
	}
	if (target == "::chrdev::") {
		colorize[ciChrDev].fg = read_config_validate_color(fcolor, filename, line);
		colorize[ciChrDev].bg = read_config_validate_color(bcolor, filename, line);
		colorize[ciChrDev].at = read_config_validate_attr(attr, filename, line);
		return;
	}
	if (target == "::blkdev::") {
		colorize[ciBlkDev].fg = read_config_validate_color(fcolor, filename, line);
		colorize[ciBlkDev].bg = read_config_validate_color(bcolor, filename, line);
		colorize[ciBlkDev].at = read_config_validate_attr(attr, filename, line);
		return;
	}
	if (target == "::special::") {
		colorize[ciSpecial].fg = read_config_validate_color(fcolor, filename, line);
		colorize[ciSpecial].bg = read_config_validate_color(bcolor, filename, line);
		colorize[ciSpecial].at = read_config_validate_attr(attr, filename, line);
		return;
	}

	custom_color_setting ccs;

	ccs.name = target;
	ccs.fg = read_config_validate_color(fcolor, filename, line);
	ccs.bg = read_config_validate_color(bcolor, filename, line);
	ccs.at = read_config_validate_attr(attr, filename, line);

	filecolorlist.push_back(ccs);
}


//-----------------------------------------------------------------------------
// Parse a configuration file.  Parsing is simple for everything but the "color" option.
static void read_config(file_info& f)
{
	string s = f.name.full();
	poptcfg.parse_configfile(s);

	if (oSort.Value == "d")
		oSort.Value = "date";
	if (oSort.Value == "n")
		oSort.Value = "name";
	if (oSort.Value == "s")
		oSort.Value = "size";
	if (oSort.Value != "name" && oSort.Value != "date" && oSort.Value != "size") {
		stringstream err;
		err << "read_config(): Invalid value for sort option in \"";
		err << f.name.name_ext() << "\" at line " << oSort.LineNbr << '!';
		throw runtime_error(err.str());
	}

	if (oColorize.Set && oColor.Value.size() > 0) {
		for (unsigned i = 0; i < oColor.Value.size(); i++) {
			if (oColor.Value[i].size() != 4) {
				stringstream err;
				err << "read_config(): Invalid syntax for color option in \"";
				err << f.name.name_ext() << "\" at line " << oColor.LineNbr[i] << '!';
				err << "  Syntax must be \"color=target,fg_color,bg_color,attr\"!";
				throw runtime_error(err.str());
			}

			read_config_color(oColor.Value[i][0], oColor.Value[i][1],
							  oColor.Value[i][2], oColor.Value[i][3],
							  f.name.name_ext().c_str(), oColor.LineNbr[i]);
		}
	}

}



//-----------------------------------------------------------------------------
static bool read_command_line(int argc, char *argv[], vector<string>& noa)
{
	poptcfg.parse_cloptions(argc, argv, noa);

	if (oHelp.Set) {
		stringstream ss;
		print_help_msg(ss);
		cout << ss.str();
		return true;
	}

	if (oVersion.Set) {
		cout << "d v" << vers << endl;
		return true;
	}

	if (oSort.Value == "d")
		oSort.Value = "date";
	if (oSort.Value == "n")
		oSort.Value = "name";
	if (oSort.Value == "s")
		oSort.Value = "size";
	if (oSort.Value != "name" && oSort.Value != "date" && oSort.Value != "size") {
		stringstream err;
		err << "main(): \"" << oSort.Value << "\" is not a valid sort option!";
		throw runtime_error(err.str());
	}

	if (oColorize.Set && oColor.Value.size() > 0) {
		for (unsigned i = 0; i < oColor.Value.size(); i++) {
			if (oColor.Value[i].size() != 4) {
				stringstream err;
				err << "main(): Invalid syntax for color option!";
				err << "  Syntax must be \"color=target,fg_color,bg_color,attr\"!";
				throw runtime_error(err.str());
			}

			read_config_color(oColor.Value[i][0], oColor.Value[i][1], oColor.Value[i][2], oColor.Value[i][3], "", -1);
		}
	}

	if (oTree.Set || oTreeAscii.Set) {
		if (oTreeAscii.Set) {
			dt_seq1 = dt_as1;
			dt_seq2 = dt_as2;
			dt_seq3 = dt_as3;
		} else {
			dt_seq1 = dt_ld1;
			dt_seq2 = dt_ld2;
			dt_seq3 = dt_ld3;
		}
	}

	return false;
}



//-----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
	try {
		list<file_info> filelist;
		vector<string> non_opt_args;

		// Get configs (system)
		file_info cf;
		cf = "/etc/default/d";
		if (cf.exists)
			read_config(cf);
		else {
			cf = "/etc/d.conf";
			if (cf.exists)
				read_config(cf);
		}

		// Get configs (user)
		cf = "~/.d.conf";
		if (cf.exists)
			read_config(cf);

		// Get command line options
		if (read_command_line(argc, argv, non_opt_args) == true)
			return 0;

		// Get current path
		path init_path = ".";

		// Only used when recursing thru multiple directories
		grand_total_files = 0;
		grand_total_size = 0;

		// Decide what we're listing
		if (non_opt_args.size() == 0) {
			// Nothing on the command line, assume current path
			string tmp = init_path.full();
			file_info f = tmp;
			filelist.push_back(f);
		} else {
			// One item, if its a directory, dump its contents
			if (non_opt_args.size() == 1) {
				file_info f = non_opt_args[0];
				if (f.exists)
					filelist.push_back(f);
				else {
					cout << f.name.full() << " doesn't exist!" << endl;
					return 1;
				}
			} else {
				// Otherwise list the things that were passed to us.
				for (unsigned i = 0; i < non_opt_args.size(); i++) {
					file_info f = non_opt_args[i];
					filelist.push_back(f);
				}
			}
		}

		if (filelist.size() == 0) {
			cout << "Nothing to list!" << endl;
			return 1;
		}

		if (filelist.size() == 1) {
			if (filelist.front().is_dir)
				if (oTree.Set || oTreeAscii.Set)
					dump_tree(filelist.front(), true);
				else
					dump_directory(filelist.front(), "");
			else
				if (filelist.front().is_symlink) {
					path fp;
					if (filelist.front().symlink_name[0] == '/')
						fp = filelist.front().symlink_name;
					else
						fp = filelist.front().name.dir() + '/' + filelist.front().symlink_name;
					file_info ff(fp.full().c_str());
					if (ff.is_dir)
						if (oTree.Set || oTreeAscii.Set)
							dump_tree(ff, true);
						else
							dump_directory(ff, "");
					else
						dump_list(filelist, NotASubdir);
				} else
					dump_list(filelist, NotASubdir);
		} else
			dump_list(filelist, NotASubdir);

		if (oSummary.Set && oRecurse.Set) {
			cout << endl;
			if (oColorize.Set)
				cout << ctxt.set(ct::White, ct::Black, ct::Bold);
			cout << grand_total_files << " regular files listed, with a grand total size of ";
			cout << setw(0) << print_human_readable_file_size(grand_total_size) << '.';
			if (oColorize.Set)
				cout << ctxt.reset();
			cout << endl;
		}

		return 0;
	}
	catch (runtime_error re) {
		if (oColorize.Set)
			cerr << ctxt.set(ct::White, ct::Red, ct::Bold);
		cerr << "d: runtime error in " << re.what();
		if (oColorize.Set)
			cerr << ctxt.reset();
		cerr << endl;
		return 1;
	}
	catch (logic_error le) {
		if (oColorize.Set)
			cerr << ctxt.set(ct::Yellow, ct::Red, ct::Bold);
		cerr << "d: logic error in " << le.what();
		cerr << "  Logic errors should never happen.  Please report this.";
		if (oColorize.Set)
			cerr << ctxt.reset();
		cerr << endl;
		return 1;
	}
	catch (exception ex) {
		if (oColorize.Set)
			cerr << ctxt.set(ct::Yellow, ct::Red, ct::Bold);
		cerr << "d: unknown exception: " << ex.what();
		cerr << "  Unkown exceptions should never happen.  Please report this.";
		if (oColorize.Set)
			cerr << ctxt.reset();
		cerr << endl;
		return 1;
	}
}
