/* 
 * agetty.c - simple, portable getty
 *
 * Original author:	Wietse Venema, Eindhoven University of Technology
 *                  	wietse@wzv.win.tue.nl
 *
 * Maintained by:	John DiMarco,  University of Toronto, CDF
 *        		jdd@cdf.toronto.edu
 *
 * Extended baud rate, explicit line param support added by:
 *			Dennis Cronin, formerly of Central Data Corporation
 *          (now Digi International)
 *
 * TABS: 8
 */
#ifndef	lint
static char rcsid[] = "$Id: agetty.c,v 1.15 1995/04/19 16:13:23 jdd Exp $";
#endif

#include "patchlevel.h"
#include <termio.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <utmp.h>
#include <string.h>

 /* If USE_SYSLOG is undefined all diagnostics go directly to /dev/console. */
#ifdef	USE_SYSLOG
#include <syslog.h>
extern void closelog();
#endif

#ifndef LOGFILE
#define	LOGFILE	"/dev/console"
#endif /* LOGFILE */

 /* Extended baud rate support for Digi EtherLite and SCSI Terminal Server
  * modules */
#if defined(DIGI_EXT_BAUDS)
# include "cd_ext.h"			/* Digi-specific header file */
#endif

 /*
  * Some heuristics to find out what environment we are in: if it is not
  * System V, assume it is SunOS 4.
  */

#ifdef	LOGIN_PROCESS			/* defined in System V utmp.h */
#define	SYSV_STYLE			/* select System V style getty */
#endif

#ifdef __CYGWIN__
#undef SYSV_STYLE
#endif

 /*
  * What follows is an attempt to unify varargs.h and stdarg.h. I prefer this
  * over #ifdefs within the actual code.
  */

#if defined(__STDC__)
#define __P(x) x
#include <stdarg.h>
#define VARARGS(func,type,arg) func(type arg, ...)
#define VASTART(ap,type,name)  va_start(ap,name)
#define VAEND(ap)              va_end(ap)
#else
#define __P(x) ()
#ifndef va_dcl
#include <varargs.h>
#endif /* va_start */
#define VARARGS(func,type,arg) func(va_alist) va_dcl
#define VASTART(ap,type,name)  {type name; va_start(ap); name = va_arg(ap, type)
#define VAEND(ap)              va_end(ap);}
#endif

extern void exit();
extern long time();
#ifdef SYSV_STYLE
extern struct utmp *getutent();
#endif
extern long atol();

 /*
  * Things you may want to modify.
  * 
  * If ISSUE is not defined, agetty will never display the contents of the
  * /etc/issue file. You will not want to spit out large "issue" files at the
  * wrong baud rate. Relevant for System V only.
  * 
  * You may disagree with the default line-editing etc. characters defined
  * below. Note, however, that DEL cannot be used for interrupt generation
  * and for line editing at the same time.
  */

#ifdef	SYSV_STYLE
#define	ISSUE "/etc/issue"		/* displayed before the login prompt */
#endif

#define LOGIN "login: "			/* login prompt */

/* Some shorthands for control characters. */

#define CTL(x)		(x ^ 0100)	/* Assumes ASCII dialect */
#define	CR		CTL('M')	/* carriage return */
#define	NL		CTL('J')	/* line feed */
#define	BS		CTL('H')	/* back space */
#define	DEL		CTL('?')	/* delete */

/* Defaults for line-editing etc. characters; you may want to change this. */

#ifndef DEF_ERASE
#define DEF_ERASE	DEL		/* default erase character */
#endif /* DEF_ERASE */
#ifndef DEF_INTR
#define DEF_INTR	CTL('C')	/* default interrupt character */
#endif /* DEF_INTR */
#ifndef DEF_QUIT
#define DEF_QUIT	CTL('\\')	/* default quit char */
#endif /* DEF_QUIT */
#ifndef DEF_KILL
#define DEF_KILL	CTL('U')	/* default kill char */
#endif /* DEF_KILL */
#ifndef DEF_EOF
#define DEF_EOF		CTL('D')	/* default EOF char */
#endif /* DEF_EOF */
#ifndef DEF_EOL
#define DEF_EOL		0
#endif /* DEF_EOL */
#ifndef DEF_SWITCH
#define DEF_SWITCH	0		/* default switch char */
#endif /* DEF_SWITCH */

 /*
  * SunOS 4.1.x termio is broken. We must use the termios stuff instead,
  * because the termio -> termios translation does not clear the termios
  * CIBAUD bits. Therefore, the tty driver would sometimes report that input
  * baud rate != output baud rate. I did not notice that problem with SunOS
  * 4.1. We will use termios where available, and termio otherwise.
  */

#if !defined(SYSV_STYLE) || defined(SOLARIS)
#ifdef	TCGETS
#undef	TCGETA
#undef	TCSETA
#undef	TCSETAW
#define	termio	termios
#define	TCGETA	TCGETS
#define	TCSETA	TCSETS
#define	TCSETAW	TCSETSW
#endif
#endif					/* SYSV_STYLE */

 /*
  * This program tries to not use the standard-i/o library.  This keeps the
  * executable small on systems that do not have shared libraries (System V
  * Release <3).
  */

#define	BUFSIZ		1024

 /*
  * When multiple baud rates are specified on the command line, the first one
  * we will try is the first one specified.
  */

#define	FIRST_SPEED	0

/* Storage for command-line options. */

#define	MAX_SPEED	10		/* max. nr. of baud rates */

struct options {
    int     flags;			/* toggle switches, see below */
    int     timeout;			/* time-out period */
    char   *login;			/* login program */
    char   *banner;			/* login banner */
    char   *connectmsg;			/* Connection Message */
    char   *name;			/* (host)name passed to login */
    char   *command;			/* command to run before line open */
    int	    parity;			/* none, odd, even (0,1,2) */
    int	    csize;			/* 7 or 8 bit chars */
    int	    stop_bits;			/* 1 or 2 stop bits */
    int     numspeed;			/* number of baud rates to try */
    int     speeds[MAX_SPEED];		/* baud rates to be tried */
    char   *tty;			/* name of tty */
};

#define	F_PARSE		(1<<0)		/* process modem status messages */
#define	F_ISSUE		(1<<1)		/* display /etc/issue */
#define	F_RTSCTS	(1<<2)		/* enable RTS/CTS flow control */
#define F_AUTOBAUD	(1<<3)		/* enable autobauding on newline */
#define	F_LOCAL		(1<<4)		/* use CLOCAL mode */
#define	F_CLPARAMS	(1<<5)		/* line params from cmd line */

/* Storage for things detected while the login name was read. */

struct chardata {
    int     erase;			/* erase character */
    int     kill;			/* kill character */
    int     eol;			/* end-of-line character */
    int     parity;			/* what parity did we see */
    int     capslock;			/* upper case without lower case */
};

/* Initial values for the above. */

struct chardata init_chardata = {
    DEF_ERASE,				/* default erase character */
    DEF_KILL,				/* default kill character */
    0,					/* always filled in at runtime */
    0,					/* space parity */
    0,					/* always filled in at runtime */
};

/* Function protos */

void    parse_args __P((int argc, char  **argv, struct options *op));
void    parse_speeds __P((struct options *op, char   *arg));
#ifdef SYSV_STYLE
void    update_utmp __P((char *line));
#endif
void	preopen __P((char *tty, char *command));
void    open_tty __P((struct options *op, struct termio *tp));
void    termio_init __P((struct options *op, struct termio *tp));
void    auto_baud __P((struct termio *tp));
void    do_prompt __P((struct options *op, struct termio *tp));
void    next_speed __P((struct options *op, struct termio *tp));
int	do_autobaud __P((struct options *op));
char   *get_logname __P((struct options *op, struct termio *tp,
		struct chardata *cp));
void    termio_final __P((struct options *op, struct termio *tp, 
		struct chardata *cp));
int     caps_lock __P((char *s));
int     bcode __P((char *s));
void    usage __P((void));
void    error __P((char *fmt,...));
void	set_baud_extension __P((char *tty, int baudcode));
#ifdef __CYGWIN__
void    doit __P((struct options *, char *));
void	protocol __P((int, int));
void	cleanup __P((int));
#endif


/* The following is used for understandable diagnostics. */

char   *progname;

/* 
 *				main
 */
int     main(argc, argv)
int     argc;
char  **argv;
{
    char   *logname;			/* login name, given to /bin/login */
    struct chardata chardata;		/* set by get_logname() */
    struct termio termio;		/* terminal mode bits */
    static struct options options = {
	F_ISSUE,			/* show /etc/issue (SYSV_STYLE) */
	0,				/* no timeout */
	"/bin/login",			/* default login program */
	LOGIN,				/* default login banner */
	NULL,				/* default connection message */
	NULL,				/* default (host)name passed to login */
	NULL,				/* command to run before line open */
	0,				/* no parity */
	8,				/* 8 bit chars default */
	1,				/* 1 stop bit default */
	0,				/* no baud rates known yet */
    };

    /* The BSD-style init command passes us a useless process name. */

#ifdef	SYSV_STYLE
    progname = argv[0];
#else
    progname = "agetty";
#endif

    /* Parse command-line arguments. */

    parse_args(argc, argv, &options);

    /* Update the utmp file. */

#ifdef	SYSV_STYLE

    update_utmp(options.tty);
#endif

    /* Run the pre-open command */

    preopen(options.tty, options.command);

    /* Open the tty as standard { input, output, error }. */

    open_tty(&options, &termio);

    /* See if hardware supports extended baud rates by setting 0 extension */

    set_baud_extension(options.tty, 0);

    /* Initialize the termio settings (raw mode blocking i/o). */

    termio_init(&options, &termio);

    /* Optionally detect the baud rate from the modem status message. */

    if (options.flags & F_PARSE)
	auto_baud(&termio);

    /* Set the optional timer. */

    if (options.timeout) {
	(void) alarm((unsigned) options.timeout);
    }

    /* Do autobauding */

    if(F_AUTOBAUD & options.flags){
        while (!do_autobaud(&options)) {
	    next_speed(&options, &termio);
	}
    }

    /* Read the login name. */

    while ((logname = get_logname(&options, &termio, &chardata)) == 0)
	next_speed(&options, &termio);
	

    /* Disable timer. */

    if (options.timeout)
	(void) alarm(0);

    /* Finalize the termio settings. */

    termio_final(&options, &termio, &chardata);

    /* Now the newline character should be properly written. */

    (void) write(1, "\n\r", 2);

    /* Let the login program take care of password validation. */

#ifndef __CYGWIN__
    if(NULL==options.name){
	    (void) execl(options.login, options.login, logname, (char *) 0);
    } else {
	    (void) execl(options.login, options.login, "-h", options.name, 
		logname, (char *) 0);
    }
#else
    doit(&options, logname);
#endif
    error("%s: can't exec %s: %m", options.tty, options.login);
    /* NOTREACHED */
}

/*
 *				parse_args
 *
 *    Parse command-line arguments.
 *
 *    Returns void, aborts thru error() if an error occurs.
 */
void    parse_args(argc, argv, op)
int     argc;
char  **argv;
struct options *op;
{
    extern char *optarg;		/* getopt */
    extern int optind;			/* getopt */
    int     c;
    static char termenv[30] = "TERM=";
#ifdef __CYGWIN__
    static char ttystr[30] = "/dev/";
#endif

    while (isascii(c = getopt(argc, argv, "1278ELNOab:c:e:hil:mn:t:T:"))) {

	switch (c) {
	case '1':				/* one stop bit */
	    op->stop_bits = 1;
	    op->flags |= F_CLPARAMS;
	    break;
	case '2':				/* two stop bits */
	    op->stop_bits = 2;
	    op->flags |= F_CLPARAMS;
	    break;
	case '7':				/* 7 bit characters */
	    op->csize = 7;
	    op->flags |= F_CLPARAMS;
	    break;
	case '8':				/* 8 bit characters */
	    op->csize = 8;
	    op->flags |= F_CLPARAMS;
	    break;
	case 'E':				/* even parity */
	    op->parity = 2;
	    op->flags |= F_CLPARAMS;
	    break;
	case 'L':				/* local mode */
	    op->flags |= F_LOCAL;
	    break;
	case 'N':				/* no parity */
	    op->parity = 0;
	    op->flags |= F_CLPARAMS;
	    break;
	case 'O':				/* odd parity */
	    op->parity = 1;
	    op->flags |= F_CLPARAMS;
	    break;
	case 'T':				/* terminal type */
	    (void) strncpy(termenv + 5, optarg, 24);
	    if (putenv(termenv))
		error("putenv: %m");
	    break;
	case 'a':				/* autobaud */
	    op->flags |= F_AUTOBAUD;
	    break;
	case 'b':
            op->banner = optarg;		/* non-default login banner */
	    break;
	case 'c':
            op->connectmsg = optarg;		/* connection message */
	    break;
	case 'e':
	    op->command = optarg;		/* cmd before line open */
	    break;
	case 'h':				/* enable h/w flow control */
	    op->flags |= F_RTSCTS;
	    break;
	case 'i':				/* do not show /etc/issue */
	    op->flags &= ~F_ISSUE;
	    break;
	case 'l':
	    op->login = optarg;			/* non-default login program */
	    break;
	case 'm':				/* parse modem status message */
	    op->flags |= F_PARSE;
	    break;
	case 'n':				/* (host)name for login */
	    op->name = optarg;
	    break;
	case 't':				/* time out */
	    if ((op->timeout = atoi(optarg)) <= 0)
		error("bad timeout value: %s", optarg);
	    break;
	default:
	    {
		char err_char[2];
		err_char[0] = (char) c;
		err_char[1] = 0;
		error("%s: bad command line option '-%s'",op->tty, err_char);
	    }
	    break;
	}
    }
    if (argc != optind + 2)			/* check parameter count */
	error("line and speed params expected after options");

#ifdef	SYSV_STYLE
    op->tty = argv[optind++];			/* tty name */
    parse_speeds(op, argv[optind]);		/* baud rate(s) */
#else
    parse_speeds(op, argv[optind++]);		/* baud rate(s) */
#ifndef __CYGWIN__
    op->tty = argv[optind];			/* tty name */
#else
    strcat(ttystr, argv[optind]);
    op->tty = ttystr;
#endif

#endif
}

/*
 *				parse_speeds
 *
 *    Parse alternate baud rates.  Expects comma-separated list.
 *
 *    Returns void, aborts thru error() if an error occurs.
 */
void    parse_speeds(op, arg)
struct options *op;
char   *arg;
{
    char   *cp;

    for (cp = strtok(arg, ","); cp != 0; cp = strtok((char *) 0, ",")) {
	if ((op->speeds[op->numspeed++] = bcode(cp)) <= 0)
	    error("bad speed: %s", cp);
	if (op->numspeed > MAX_SPEED)
	    error("too many alternate speeds");
    }
}

#ifdef	SYSV_STYLE

/*
 *				update_utmp
 *    Update our utmp entry.
 *
 *    Returns void, aborts thru error() if an error occurs.
 */
void    update_utmp(line)
char   *line;
{
    struct utmp *utp;
    int     mypid = getpid();

    while (utp = getutent()) {
	if (utp->ut_type == INIT_PROCESS && utp->ut_pid == mypid) {
	    utp->ut_type = LOGIN_PROCESS;
	    utp->ut_time = time((long *) 0);
	    (void) strncpy(utp->ut_name, "LOGIN", sizeof(utp->ut_name));
	    (void) strncpy(utp->ut_line, line, sizeof(utp->ut_line));
	    pututline(utp);
	    endutent();
	    return;
	}
    }
    error("%s: no utmp entry", line);
}

#endif

/*
 *				preopen
 *    
 *    Execute a command prior to opening the tty device.  Forks
 *    another process to execute the spec'd command and waits
 *    for completion of child.  We ignore status return from child.
 *
 *    Returns void, aborts thru error() if we can't fork or exec.
 */
void preopen(tty, command)
char	*tty;
char	*command;
{
    if(command){

	switch(fork()){
	case -1:
            /* error */
	    error("%s: fork: %m", tty);
            /* NOTREACHED */
	    break;
	case 0: 
	    /* child */
	    (void) execl(command, command, tty, (char *) 0);
            error("%s: can't exec %s: %m", tty, command);
            /* NOTREACHED */
	    break;
	default:
	    /* parent */
            (void) wait((int *)0);
	    break;
	}
    }
}

/*
 *  				open_tty
 *
 *    Set up tty as standard { input, output, error }.
 *
 *    Returns void, aborts thru error() if an error occurs.
 */
void    open_tty(op, tp)
struct options *op;
struct termio *tp;
{
    /* Get rid of the present standard { output, error} if any. */

    (void) close(1);
    (void) close(2);

    /* Set up new standard input, unless we are given an already opened port. */

    if (strcmp(op->tty, "-")) {
	struct stat st;

	/* Sanity checks... */

#ifndef __CYGWIN__
	if (chdir("/dev"))
	    error("/dev: chdir() failed: %m");
#endif
	if (stat(op->tty, &st) < 0)
	    error("/dev/%s: %m", op->tty);
	if ((st.st_mode & S_IFMT) != S_IFCHR)
	    error("/dev/%s: not a character device", op->tty);

	/* Open the tty as standard input. */

	(void) close(0);

	/* If they've req'd local mode, open non-blocking style */
	if(op->flags & F_LOCAL) {

	    /* allow 10 secs to succeed */
	    alarm(10);

	    if (open(op->tty, O_RDWR | O_NDELAY) != 0)
	    	error("/dev/%s: cannot open as standard input: %m", op->tty);

	    alarm(0);

	    /* clear non-blocking now so I/O isn't affected */
	    fcntl(0, F_SETFL, 0);
	}
	/* Otherwise, do normal blocking open */
	else {
	    if (open(op->tty, O_RDWR) != 0)
	   	 error("/dev/%s: cannot open as standard input: %m", op->tty);
	}
    } else {

	/*
	 * Standard input should already be connected to an open port. Make
	 * sure it is open for read/write.
	 */

	if ((fcntl(0, F_GETFL, 0) & O_RDWR) != O_RDWR)
	    error("%s: not open for read/write", op->tty);
    }

    /* Set up standard output and standard error file descriptors. */

    if (dup(0) != 1 || dup(0) != 2)		/* set up stdout and stderr */
	error("%s: dup problem: %m", op->tty);	/* we have a problem */

    /*
     * The following ioctl will fail if stdin is not a tty, but also when
     * there is noise on the modem control lines. In the latter case, the
     * common course of action is (1) fix your cables (2) give the modem more
     * time to properly reset after hanging up. SunOS users can achieve (2)
     * by patching the SunOS kernel variable "zsadtrlow" to a larger value; 5
     * seconds seems to be a good value.
     */

    if (ioctl(0, TCGETA, tp) < 0)
	error("%s: ioctl: %m", op->tty);

    /*
     * It seems to be a terminal. Set proper protections and ownership. Mode
     * 0622 is suitable for SYSV <4 because /bin/login does not change
     * protections. SunOS 4 login will change the protections to 0620 (write
     * access for group tty) after the login has succeeded. Ignore errors
     * because we may be called from an unprivileged call-back program.
     */

    (void) chown(op->tty, 0, 0);		/* root, sys */
    (void) chmod(op->tty, 0622);		/* crw--w--w- */
}

/*
 *				termio_init
 *
 *    Initialize termio settings to defaults.  If command line params
 *    were spec'd, allow them to override defaults.
 *
 *    Returns void, aborts thru error() if an error occurs.
 */
void    termio_init(op, tp)
struct options *op;
struct termio *tp;
{
    /*
     * Initial termio settings: 8-bit characters, raw-mode, blocking i/o.
     * Special characters are set after we have read the login name; all
     * reads will be done in raw mode anyway. Errors will be dealt with
     * lateron. Turn on h/w flow control for huge /etc/issue files.
     */

    tp->c_cflag = CS8 | HUPCL | CREAD | (op->speeds[FIRST_SPEED] & CBAUD);

    /* see command line params for line settings were spec'd */
    if(op->flags & F_CLPARAMS) {

	/* One or more of the defaults has been overridden from cmd line */

	tp->c_cflag &= ~CSIZE;	
	if(op->csize == 8)			/* set character size */
		tp->c_cflag |= CS8;
	else
 		tp->c_cflag |= CS7;

    	switch(op->parity) {			/* set parity mode */
	case 0:					/* none */
		break;
	case 1:					/* odd */
		tp->c_cflag |= (PARODD | PARENB);
		break;
	case 2:					/* even */
		tp->c_cflag |= PARENB;
	}

	if(op->stop_bits == 2)			/* set stop bits */
		tp->c_cflag |= CSTOPB;
    }

    /* see if local mode req'd from command line */
    if(op->flags & F_LOCAL) {
	tp->c_cflag |= CLOCAL;
    }

#ifdef	CRTSCTS
    if (op->flags & F_RTSCTS)
	tp->c_cflag |= CRTSCTS;
#endif
    tp->c_iflag = tp->c_lflag = tp->c_oflag = 0;
#ifndef SOLARIS
    tp->c_line = 0;
#endif
    tp->c_cc[VMIN] = 1;
    tp->c_cc[VTIME] = 0;
    set_baud_extension(op->tty, op->speeds[FIRST_SPEED]);
#ifdef __CYGWIN__
    tp->c_ispeed = tp->c_ospeed = tp->c_cflag & CBAUD;
#endif
    if(ioctl(0, TCSETA, tp) < 0)
	error("%s: ioctl: TCSETA: %m", op->tty);
}

/*
 *				auto_baud
 *
 *    Extract baud rate from modem status message.
 *
 *    Returns void, ignores errors.
 */
void    auto_baud(tp)
struct termio *tp;
{
    int     speed;
    int     vmin;
    unsigned long iflag;
    char    buf[BUFSIZ];
    char   *bp;
    int     nread;

    /*
     * This works only if the modem produces its status code AFTER raising
     * the DCD line, and if the computer is fast enough to set the proper
     * baud rate before the message has gone by. We expect a message of the
     * following format:
     * 
     * <junk><number><junk>
     * 
     * The number is interpreted as the baud rate of the incoming call. If the
     * modem does not tell us the baud rate within one second, we will keep
     * using the current baud rate. It is advisable to enable BREAK
     * processing (comma-separated list of baud rates) if the processing of
     * modem status messages is enabled.
     */

    /*
     * Use 7-bit characters, don't block if input queue is empty. Errors will
     * be dealt with later on.
     */

    iflag = tp->c_iflag;
    tp->c_iflag |= ISTRIP;			/* enable 8th-bit stripping */
    vmin = tp->c_cc[VMIN];
    tp->c_cc[VMIN] = 0;				/* don't block if queue empty */
    (void) ioctl(0, TCSETA, tp);

    /*
     * Wait for a while, then read everything the modem has said so far and
     * try to extract the speed of the dial-in call.
     */

    (void) sleep(1);
    if ((nread = read(0, buf, sizeof(buf) - 1)) > 0) {
	buf[nread] = '\0';
	for (bp = buf; bp < buf + nread; bp++) {
	    if (isascii(*bp) && isdigit(*bp)) {
		if (speed = bcode(bp)) {
		    tp->c_cflag &= ~CBAUD;
		    tp->c_cflag |= (speed & CBAUD);
		}
		break;
	    }
	}
    }
    /* Restore terminal settings. Errors will be dealt with lateron. */

    tp->c_iflag = iflag;
    tp->c_cc[VMIN] = vmin;
#ifdef __CYGWIN__
    tp->c_ispeed = tp->c_ospeed = tp->c_cflag & CBAUD;
#endif
    (void) ioctl(0, TCSETA, tp);
}

/*
 *				do_prompt
 *
 *    Show login prompt.  Three possible parts:  (optional)
 *    /etc/issue contents, (optional) connect message, and final
 *    login prompt.
 *
 *    Returns void, ignores errors.
 */
void    do_prompt(op, tp)
struct options *op;
struct termio *tp;
{
#ifdef	ISSUE
    int     fd;
    int     oflag;
    int     n;
    char    buf[BUFSIZ];
#endif

    (void) write(1, "\r\n", 2);			/* start a new line */
#ifdef	ISSUE					/* optional: show /etc/issue */
    if ((op->flags & F_ISSUE) && (fd = open(ISSUE, 0)) >= 0) {
	oflag = tp->c_oflag;			/* save current setting */
	tp->c_oflag |= (ONLCR | OPOST);		/* map NL in output to CR-NL */
	(void) ioctl(0, TCSETAW, tp);
	while ((n = read(fd, buf, sizeof(buf))) > 0)
	    (void) write(1, buf, n);
	tp->c_oflag = oflag;			/* restore settings */
	(void) ioctl(0, TCSETAW, tp);		/* wait till output is gone */
	(void) close(fd);
    }
#endif
    if(op->connectmsg != (char*) NULL)
    {
    	(void) write(1, op->connectmsg, 	/* show connection msg */
	    strlen(op->connectmsg)*sizeof(char));
    	(void) write(1, "\r\n", 2);		/* start a new line */
	op->connectmsg=(char *)NULL;
    }
    (void) write(1, op->banner, 
	strlen(op->banner)*sizeof(char));	/* show prompt */
}

/*
 *				next_speed
 *				
 *    Select next baud rate using list provided on the command line.
 *    If only one speed was spec'd, stays at the same speed.
 *
 *    Returns void, aborts thru error() if an error occurs.
 */
void    next_speed(op, tp)
struct options *op;
struct termio *tp;
{
    static int baud_index = FIRST_SPEED;/* current speed index */

    /* make sure there are multiple possible speeds */
    if(op->numspeed == 1)
	return;
    
    /* step to next one on list */
    baud_index = (baud_index + 1) % op->numspeed;
    tp->c_cflag &= ~CBAUD;
    tp->c_cflag |= (op->speeds[baud_index] & CBAUD);
    set_baud_extension(op->tty, op->speeds[baud_index]);
#ifdef __CYGWIN__
    tp->c_ispeed = tp->c_ospeed = tp->c_cflag & CBAUD;
#endif
    if(ioctl(0, TCSETA, tp) < 0)
	error("%s: ioctl: TCSETA: %m", op->tty);
}

/*
 *				do_autobaud
 *    
 *    Select baud rate by watching for CR or NL.  If we don't get
 *    a match on the char against either of those two chars,
 *    we try the next baud rate.
 *
 *    Returns true if we got a CR or NL, else false.  Aborts thru
 *    error() if read is interrupted.
 */
int	do_autobaud(op)
struct options *op;
{
    char    c = '\0';

    (void) sleep(1);
    (void) ioctl(0, TCFLSH, (struct termio *) 0);
    if (read(0, &c, 1) < 1) {
        if (errno == EINTR || errno == EIO)
            exit(0);
    }
    c &= (char)127; /* strip parity bit */
    if('\n'==c || '\r'==c ) return (1);
    return(0);	
}

/*
 *				get_logname
 *				
 *    Get user name, establish parity, speed, erase, kill, eol.
 *
 *    Returns pointer to logname.  Ignores most errors.
 */
char   *get_logname(op, tp, cp)
struct options *op;
struct termio *tp;
struct chardata *cp;
{
    static char    logname[BUFSIZ];
    char   *bp;
    char    c;				/* input character, full eight bits */
    char    ascval;			/* low 7 bits of input character */
    int     bits;			/* # of "1" bits per character */
    int     mask;			/* mask with 1 bit up */
    static char *erase[] = {		/* backspace-space-backspace */
	"\010\040\010",			/* space parity */
	"\010\040\010",			/* odd parity */
	"\210\240\210",			/* even parity */
	"\210\240\210",			/* no parity */
    };

    /* Initialize kill, erase, parity etc. (also after switching speeds). */

    *cp = init_chardata;

    /* Flush pending input (esp. after parsing or switching the baud rate). */

    (void) sleep(1);
    (void) ioctl(0, TCFLSH, (struct termio *) 0);

    /* Prompt for and read a login name. */

    for (*logname = 0; *logname == 0; /* void */ ) {

	/* Write issue file and prompt, with "parity" bit == 0. */

	do_prompt(op, tp);

	/* Read name, watch for break, parity, erase, kill, end-of-line. */

	for (bp = logname, cp->eol = 0; cp->eol == 0; /* void */ ) {

	    /* Do not report trivial EINTR/EIO errors. */

	    if (read(0, &c, 1) < 1) {
		if (errno == EINTR || errno == EIO) {
		    exit(0);
		}
		error("%s: read: %m", op->tty);
	    }
	    /* Do BREAK handling elsewhere. */

	    if ((c == 0) && op->numspeed > 1)
		return (0);

	    /* Do parity bit handling. */

	    if (c != (ascval = (c & 0177))) {	/* "parity" bit on ? */
		for (bits = 1, mask = 1; mask & 0177; mask <<= 1)
		    if (mask & ascval)
			bits++;			/* count "1" bits */
		cp->parity |= ((bits & 1) ? 1 : 2);
	    }
	    /* Do erase, kill and end-of-line processing. */

	    switch (ascval) {
	    case CR:
	    case NL:
		*bp = 0;			/* terminate logname */
		cp->eol = ascval;		/* set end-of-line char */
		break;
	    case BS:
	    case DEL:
	    case '#':
		cp->erase = ascval;		/* set erase character */
		if (bp > logname) {
		    (void) write(1, erase[cp->parity], 3);
		    bp--;
		}
		break;
	    case CTL('U'):
	    case '@':
		cp->kill = ascval;		/* set kill character */
		while (bp > logname) {
		    (void) write(1, erase[cp->parity], 3);
		    bp--;
		}
		break;
	    case CTL('D'):
		exit(0);
	    default:
		if (!isascii(ascval) || !isprint(ascval)) {
		     /* ignore garbage characters */ ;
		} else if (bp - logname >= sizeof(logname) - 1) {
		    error("%s: input overrun", op->tty);
		} else {
		    (void) write(1, &c, 1);	/* echo the character */
		    *bp++ = ascval;		/* and store it */
		}
		break;
	    }
	}
	if (*logname == '-') {
	    static char text[] = "\r\nlogin names may not begin with '-'.\r\n";
	    write(2, text, sizeof(text) - 1);
	    *logname = 0;
	}
    }
    /* Handle names with upper case and no lower case. */

    if (cp->capslock = caps_lock(logname)) {
	for (bp = logname; *bp; bp++)
	    if (isupper(*bp))
		*bp = tolower(*bp);		/* map name to lower case */
    }
    return (logname);
}

/*
 *				termio_final
 *    
 *    Set the final tty mode bits.  Install special characters as
 *    guessed by login procedure.  Optionally installs parity mode
 *    if not spec'd explicitly on the command line.
 *
 *    Returns void, aborts thru error() if an error occurs.
 */
void    termio_final(op, tp, cp)
struct options *op;
struct termio *tp;
struct chardata *cp;
{
    /* General terminal-independent stuff. */

    tp->c_iflag |= IXON | IXOFF;		/* 2-way flow control */
    tp->c_lflag |= ICANON | ISIG | ECHO | ECHOE | ECHOK;
    tp->c_oflag |= OPOST;
    tp->c_cc[VINTR] = DEF_INTR;			/* default interrupt */
    tp->c_cc[VQUIT] = DEF_QUIT;			/* default quit */
    tp->c_cc[VEOF] = DEF_EOF;			/* default EOF character */
    tp->c_cc[VEOL] = DEF_EOL;
#ifndef __CYGWIN__
    tp->c_cc[VSWTCH] = DEF_SWITCH;		/* default switch character */
#endif

    /* Account for special characters seen in input. */

    if (cp->eol == CR) {
	tp->c_iflag |= ICRNL;			/* map CR in input to NL */
	tp->c_oflag |= ONLCR;			/* map NL in output to CR-NL */
    }
    tp->c_cc[VERASE] = cp->erase;		/* set erase character */
    tp->c_cc[VKILL] = cp->kill;			/* set kill character */

    /*
     *	If no command line params for line settings, see what we've
     *	guessed about parity.
     */
    if((op->flags & F_CLPARAMS) == 0) {

	/* Account for the presence or absence of parity bits in input. */

	switch (cp->parity) {
	case 0:					/* space (always 0) parity */
	    break;
	case 1:					/* odd parity */
	    tp->c_cflag |= PARODD;
	    /* FALLTHROUGH */
	case 2:					/* even parity */
	    tp->c_cflag |= PARENB;
	    tp->c_iflag |= INPCK | ISTRIP;
	    /* FALLTHROUGH */
	case (1 | 2):				/* no parity bit */
	    tp->c_cflag &= ~CSIZE;
	    tp->c_cflag |= CS7;
	    break;
	}
    }

    /* Account for upper case without lower case. */

    if (cp->capslock) {
	tp->c_iflag |= IUCLC;
#ifndef __CYGWIN__
	tp->c_lflag |= XCASE;
#endif
	tp->c_oflag |= OLCUC;
    }

    /* Optionally enable hardware flow control */

#ifdef	CRTSCTS
    if (op->flags & F_RTSCTS)
	tp->c_cflag |= CRTSCTS;
#endif

    /* Finally, make the new settings effective */

    if (ioctl(0, TCSETA, tp) < 0)
	error("%s: ioctl: TCSETA: %m", op->tty);
}

/*
 *				caps_lock
 *				
 *    Returns true is string contains upper case without lower case,
 *    else returns false.
 */
int     caps_lock(s)
char   *s;
{
    int     capslock;

    for (capslock = 0; *s; s++) {
	if (islower(*s))
	    return (0);
	if (capslock == 0)
	    capslock = isupper(*s);
    }
    return (capslock);
}

/*
 *				bcode
 *    
 *    Convert speed string to speed code.
 *
 *    Returns baud code, or 0 if baud rate not found.
 */
int     bcode(s)
char   *s;
{
/* defs for OS's that don't know about 19200 or 38400 */
#if	!defined(B19200)
#	define	B19200	EXTA
#	define	B38400	EXTB
#endif
    struct Speedtab {
	long    speed;
	int     code;
    };
    static struct Speedtab speedtab[] = {
	50,     B50,
	75,     B75,
	110,    B110,
	134,    B134,
	150,    B150,
	200,    B200,
	300,    B300,
	600,    B600,
	1200,   B1200,
	1800,   B1800,
	2400,   B2400,
	4800,   B4800,
	9600,   B9600,
	19200,  B19200,
	38400,  B38400,
#if	defined(DIGI_EXT_BAUDS)
	/* note: baud extensions all in 2nd nibble */
	57600,  B38400 | (1 << XC_BAUDSHIFT),
	76800,  B38400 | (2 << XC_BAUDSHIFT),
	115200, B38400 | (3 << XC_BAUDSHIFT),
#endif
	0, 0,
    };
    struct Speedtab *sp;
    long    speed = atol(s);

    for (sp = speedtab; sp->speed; sp++)
	if (sp->speed == speed)
	    return (sp->code);
    return (0);
}

/*
 *				usage
 *    
 *    Explain (briefly) the options.  Always exits thru error().
 */
void    usage()
{
#ifdef	SYSV_STYLE
    static char msg[] =
"[-aih1278ELOP] [-e command] [-n name] [-b loginbanner]\n\r\
           [-l login] [-m] [-t timeout] [-T term] line speed,...";
#else
    static char msg[] =
"[-ah1278ELOP] [-e command] [-n name] [-b loginbanner]\n\r\
           [-l login] [-m] [-t timeout] speed,... line";
#endif

    error("usage: %s %s", progname, msg);
}

/*
 *				error
 *    
 *    Report errors to console or syslog; only understands %s and %m.
 *
 *    When finished, we delay for 10 seconds (to avoid causing the
 *    "respawning too rapidly" message) and then exit with status 1.
 */

#define	str2cpy(b,s1,s2)	strcat(strcpy(b,s1),s2)

/* VARARGS */

void    VARARGS(error, char *, fmt)
{
    va_list ap;
#ifndef	USE_SYSLOG
    int     fd;
#endif
    char    buf[BUFSIZ];
    char   *bp;
#ifndef __CYGWIN__
    extern char *sys_errlist[];
#endif

    /*
     * If the diagnostic is reported via syslog(3), the process name is
     * automatically prepended to the message. If we write directly to
     * /dev/console, we must prepend the process name ourselves.
     */

#ifdef USE_SYSLOG
    buf[0] = '\0';
    bp = buf;
#else
    (void) str2cpy(buf, progname, ": ");
    bp = buf + strlen(buf);
#endif

    /*
     * %s expansion is done by hand. On a System V Release 2 system without
     * shared libraries and without syslog(3), linking with the stdio library
     * used to make the program three times as big...
     * 
     * %m expansion is done here as well. Too bad syslog(3) does not have a
     * vsprintf() like interface.
     */

    VASTART(ap, char *, fmt);
    while (*fmt) {
	if (strncmp(fmt, "%s", 2) == 0) {
	    (void) strcpy(bp, va_arg(ap, char *));
	    bp += strlen(bp);
	    fmt += 2;
	} else if (strncmp(fmt, "%m", 2) == 0) {
	    (void) strcpy(bp, sys_errlist[errno]);
	    bp += strlen(bp);
	    fmt += 2;
	} else if (strncmp(fmt, "%%", 2) == 0) {
	    *bp++ = *fmt++;
	    fmt++;
	} else {
	    *bp++ = *fmt++;
	}
    }
    *bp = 0;
    VAEND(ap);

    /*
     * Write the diagnostic directly to /dev/console if we do not use the
     * syslog(3) facility.
     */

#ifdef	USE_SYSLOG
    (void) openlog(progname, LOG_PID, LOG_AUTH);
    (void) syslog(LOG_ERR, "%s", buf);
    closelog();
#else
    /* Terminate with CR-LF since the console mode is unknown. */
    (void) strcat(bp, "\r\n");
    if ((fd = open(LOGFILE, O_RDWR)) >= 0) {
	(void) write(fd, buf, strlen(buf));
	(void) close(fd);
    }
#endif
    (void) sleep((unsigned) 10);		/* be kind to init(8) */
    exit(1);
}


/*
 *				set_baud_extension
 *    
 *    Access extended baud rates is supported.
 *
 *    If the version of agetty doesn't support extended baudrates,
 *    just a NOP.
 *
 *    If agetty supports extended baudrates, but hardware doesn't,
 *    it determines that on first call and further calls force
 *    an abort thru error().
 *
 *    Otherwise, it sets the extension factor (hi bits above the
 *    CBAUD nibble).  That should never fail, altho a subsequent
 *    SETA call may fail if the resultant baud rate is not
 *    available at the particular unit.
 */
void	set_baud_extension(tty, baudcode)
char   *tty;
int	baudcode;
{

#if	!defined(DIGI_EXT_BAUDS)

	return;				/* not supported this version */

#else

    static not_supported;
    stsextctl xc;

    /* see if we've already determined they're not supported */
    if(not_supported) {
	/* see if they are in fact asking for extended rate */
	if(baudcode & ~CBAUD)
	    error("%s: extended baud rates not supported.");
	    
	/* no extension needed, just return */
	return;
    }

    /* issue proprietary ioctl to fetch extended params */
    if(ioctl(0, STSGETEXTCTL, &xc) < 0) {
	not_supported = 1;		/* save future probing */
	return;				/* not supported */
    }	

    /* clear out any previous value (also make sure loopback off) */
    xc.x_flags &= ~(XC_BAUDEXT | XC_LOOP);

    /* bring in extension portion of baudcode */
    xc.x_flags |= (baudcode & XC_BAUDEXT);

    /* set the new baud extension */
    if(ioctl(0, STSSETEXTCTL, &xc) < 0)
	error("%s: set baud extension failed", tty);

    /* made it to here, extension successfully set */
#endif
}

#ifdef __CYGWIN__
#include <limits.h>
#include <unistd.h>

int openpty(amaster, aslave, name, termp, winp)
	int *amaster, *aslave;
	char *name;
	struct termios *termp;
	struct winsize *winp;
{
        int master, slave;
        char line[PATH_MAX];
        char *ptsname();

        master = open("/dev/ptmx", O_RDWR);
        if (master > 0) {
                grantpt(master);
                unlockpt(master);
                strcpy(line, ptsname(master));
                if ((slave = open(line, O_RDWR | O_NOCTTY)) != -1) {
                        if (amaster)
                                *amaster = master;
                        if (aslave)
                                *aslave = slave;
                        if (name)
                                strcpy(name, line);
                        if (termp)
                                (void) tcsetattr(slave, TCSAFLUSH, termp);
                        if (winp)
                                (void) ioctl(slave, TIOCSWINSZ, (char *) winp);
                        return (0);
                }
                (void) close(master);
        }
	errno = ENOENT;	/* out of ptys */
	return (-1);
}

int forkpty(amaster, name, termp, winp)
	int *amaster;
	char *name;
	struct termios *termp;
	struct winsize *winp;
{
	int master, slave, pid;

	if (openpty(&master, &slave, name, termp, winp) == -1)
		return (-1);
	switch (pid = fork()) {
	case -1:
		return (-1);
	case 0:
		/* 
		 * child
		 */
		close(master);
		login_tty(slave);
		return (0);
	}
	/*
	 * parent
	 */
	*amaster = master;
	close(slave);
	return (pid);
}

int
login_tty(fd)
	int fd;
{
	(void) setsid();
#ifdef TIOCSCTTY
	if (ioctl(fd, TIOCSCTTY, (char *)NULL) == -1)
		return (-1);
#else
	{
	  /* This might work.  */
	  char *fdname = ttyname (fd);
	  int newfd;
	  if (fdname)
	    {
	      if (fd != 0)
		(void) close (0);
	      if (fd != 1)
		(void) close (1);
	      if (fd != 2)
		(void) close (2);
	      newfd = open (fdname, O_RDWR);
	      (void) close (newfd);
	    }
	}
#endif
	(void) dup2(fd, 0);
	(void) dup2(fd, 1);
	(void) dup2(fd, 2);
	if (fd > 2)
		(void) close(fd);
	return (0);
}

/* `defaults' for tty settings.  */
#ifndef TTYDEF_IFLAG
#define	TTYDEF_IFLAG	(BRKINT | ISTRIP | ICRNL | IMAXBEL | IXON | IXANY)
#endif
#ifndef TTYDEF_OFLAG
#ifndef OXTABS
#define OXTABS 0
#endif
#define TTYDEF_OFLAG	(OPOST | ONLCR | OXTABS)
#endif
#ifndef TTYDEF_LFLAG
#define TTYDEF_LFLAG	(ECHO | ICANON | ISIG | IEXTEN | ECHOE|ECHOKE|ECHOCTL)
#endif

void
setup_term(fd)
	int fd;
{
	struct termios tt;

	tcgetattr(fd, &tt);

	tt.c_iflag = TTYDEF_IFLAG;
	tt.c_oflag = TTYDEF_OFLAG;
	tt.c_lflag = TTYDEF_LFLAG;
	tcsetattr(fd, TCSAFLUSH, &tt);
}

int	child;
int	netf;
char	line[1024];		/* XXX */
int	confirmed;

struct winsize win = { 24, 80, 0, 0 };


void
doit(struct options *options, char *logname)
{
	int master, pid, on = 1;
	int f = 0;

	netf = 0;

	pid = forkpty(&master, line, NULL, &win);
	if (pid < 0) {
		if (errno == ENOENT)
			error("Out of ptys");
		else
			error("Forkpty");
	}
	if (pid == 0) {
		setup_term(0);
		if(NULL==options->name){
	    		(void) execl(options->login, options->login, logname, (char *) 0);
    		} else {
	    		(void) execl(options->login, options->login, "-h", options->name, 
				logname, (char *) 0);
    		}
		return;
		/*NOTREACHED*/
	}
	ioctl(f, FIONBIO, &on);
	ioctl(master, FIONBIO, &on);
	signal(SIGCHLD, cleanup);
	protocol(f, master);
	signal(SIGCHLD, SIG_IGN);
	cleanup(0);
}

/*
 * rlogin "protocol" machine.
 */
void
protocol(f, p)
	register int f, p;
{
	char pibuf[1024+1], fibuf[1024+1];
	int fcc, pcc;
	int cc, nfd, n;

	/*
	 * Must ignore SIGTTOU, otherwise we'll stop
	 * when we try and set slave pty's window shape
	 * (our controlling tty is the master pty).
	 */
	(void) signal(SIGTTOU, SIG_IGN);

	if (f > p)
		nfd = f + 1;
	else
		nfd = p + 1;
	if (nfd > FD_SETSIZE) {
		syslog(LOG_ERR, "select mask too small, increase FD_SETSIZE");
		error("internal error (select mask too small)");
	}
	for (;;) {
		fd_set ibits, obits, ebits;

		FD_ZERO(&ebits);
		FD_ZERO(&ibits);
		FD_ZERO(&obits);

		FD_SET(f, &ibits);
		FD_SET(p, &ibits);
		FD_SET(p, &ebits);

		if ((n = select(nfd, &ibits, NULL, &ebits, 0)) < 0) {
			if (errno == EINTR)
				continue;
			error("select");
		}
		if (n == 0) {
			/* shouldn't happen... */
			sleep(5);
			continue;
		}
		if (FD_ISSET(f, &ibits)) {
			fcc = read(f, fibuf, sizeof(fibuf));
			if (fcc < 0 && errno == EWOULDBLOCK)
				fcc = 0;
			else {
				if (fcc <= 0)
					break;
				cc = write(p, fibuf, fcc);
			}
		}

		if (FD_ISSET(p, &ibits)) {
			pcc = read(p, pibuf, sizeof (pibuf));
			if (pcc < 0 && errno == EWOULDBLOCK)
				pcc = 0;
			else if (pcc <= 0)
				break;
			cc = write(f, pibuf, pcc);
		}
	}
}

void
cleanup(signo)
	int signo;
{
	exit(1);
}


#endif
