
/*
 * bltTree.h --
 *
 * Copyright 1998-1999 Lucent Technologies, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the names
 * of Lucent Technologies or any of their entities not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and
 * fitness.  In no event shall Lucent Technologies be liable for any
 * special, indirect or consequential damages or any damages
 * whatsoever resulting from loss of use, data or profits, whether in
 * an action of contract, negligence or other tortuous action, arising
 * out of or in connection with the use or performance of this
 * software.
 *
 *	The "tree" data object was created by George A. Howlett.
 */

#ifndef BLT_TREE_H
#define BLT_TREE_H
/*
t2 configure -mtime
t2 configure t1 -mtime 25505
t2 cget t1 -mtime
t2 cget 0 -mtime
t3 cget 2 -size
t3 find -exact -glob -count -mtime 10101 -size 102002 -
?firstIndex lastIndex? -exec { %n }
t3 open 0 -opencommand
t4 close -closecommand

t0 configure -opencommand {
	foreach i [glob %P] {
		%W insert end $i -mtime [file mtime $i] -
}

id = TreeCreateEntry();
add the info to the tree.

entryPtr
-button yes
-bindtags
-closecommand
-data
-button
-icons
-images
-label
-labelcolor
-labelfont
-labelshadow
-opencommand
-text
-textcolor
-textfont
-textshadow

t0 configure -closecommand {
}

append
apply
children
create		Blt_TreeCreateNode
cursor
delete		Blt_TreeDeleteNode
deselect
destroy
dup
find		Blt_TreeFindNode
insert		Blt_TreeInsertNode
isFirst
isLast
move
next		Blt_TreeNextNode
last		Blt_TreeLastNode
first		Blt_TreeFirstNode
parent		Blt_TreeParentNode
previous	Blt_TreePrevNode
prune (pack)
relink
rename
select
slink
sort
isancestor

*/
#define APPLY_BEFORE		(1<<3)
#define APPLY_AFTER		(1<<1)
#define APPLY_OPEN_ONLY		(1<<2)

#define NOTIFY_CREATE		(1<<0)
#define NOTIFY_DELETE		(1<<1)
#define NOTIFY_SET		(1<<2)
#define NOTIFY_UNSET		(1<<3)
#define NOTIFY_ALL		\
	(NOTIFY_CREATE | NOTIFY_DELETE | NOTIFY_SET | NOTIFY_UNSET)

typedef char *Blt_TreeAtom;
typedef struct Blt_TreeNode_ *Blt_TreeNode;
typedef struct Blt_TreeObject_ *Blt_TreeObject;
typedef struct Blt_TreeToken_ *Blt_Tree;

/*
 * Blt_TreeObject --
 *
 *	Structure providing the internal representation of the tree
 *	object.	A tree is uniquely identified by a combination of 
 *	its name and originating namespace.  Two trees in the same 
 *	interpreter can have the same names but reside in different 
 *	namespaces.
 *
 *	The tree object represents a general-ordered tree of node.
 *	Each node may contain a heterogeneous collection of data.  
 *	Nodes do not need to contain the same data fields.  Each 
 *	piece of data is identified by a field name.  Data field
 *	names are saved as reference counted strings and can be 
 *	shared among nodes.
 *
 *	The tree is threaded.  A node contains both a pointer to 
 *	back its parents and another to its siblings.  Therefore
 *	the tree maybe traversed non-recursively.
 * 
 *	A tree object can be shared by several clients.  When a
 *	client wants to use a tree object, it is given a token
 *	that represents the tree.  The tree object uses the tokens
 *	to keep track of its clients.  When all clients have 
 *	released their tokens the tree is automatically destroyed.
 */
struct Blt_TreeObject_ {

    Tcl_Interp *interp;		/* Interpreter associated with this tree. */

    Tcl_Namespace *nsPtr;	/* Trees are uniquely identified by the
				 * namespace they were created in and their
				 * name. */
    Tk_Uid nameUid;		/* Name of tree */
    char *fullName;

    Tcl_HashEntry *hashPtr;	/* Pointer to this tree object in tree
				 * object hash table. */

    Blt_TreeNode root;		/* Root of the entire tree. */

    int nNodes;			/* Always counts root node. */

    int depth;			/* Maximum depth of the tree. */

    unsigned int flags;		/* Notification flags. See definitions
				 * below. */
    unsigned int notifyFlags;	/* Notification flags. See definitions
				 * below. */

    Tcl_HashTable nodeTable;	/* Table of node identifiers. Used to
				 * search for a node pointer given an inode.*/
    unsigned int nextNode;

    Tcl_HashTable atomTable;	/* Hash table containing the names of 
				 * the data fields within the tree. */

    char *sortCmd;		/* Tcl command to invoke to sort entries */

    Blt_Chain *chainPtr;	/* List of clients using this tree */

};

/*
 * Blt_TreeNode_ --
 *
 *	Structure representing a node in a general ordered tree.
 *	Nodes are identified by their index, or inode.  Nodes also
 *	have names, but nodes names are not unique and can be 
 *	changed.  Inodes are valid even if the node is moved.
 *
 *	Each node can contain a list of data fields.  Fields are
 *	name-value pairs.  The values are represented by Tcl_Objs.
 *	
 */
struct Blt_TreeNode_ {
    Blt_TreeAtom atom;		/* Name of node (doesn't have to be 
				 * unique */

    unsigned int inode;		/* Unique identifier for node. */

    Blt_TreeObject treePtr;

    Blt_TreeNode parent;	/* Pointer to the parent node. If
				 * NULL, this is the root node. */

    short int depth;		/* The depth of this node in the tree. */

    unsigned short flags;
    
    Blt_Chain *chainPtr;	/* Pointer to a chain of subnodes. The
				 * chain isn't allocated until the
				 * node has children. */

    Blt_ChainLink *linkPtr;	/* Pointer to this node in the
				 * parent's chain of children nodes.
				 * Used to remove the node from its
				 * parent when destroying the node. */

    Blt_Chain *dataPtr;		/* A chain of Tcl_Obj's. Hold the
				 * various fields of the node.  Can be
				 * NULL, if no data is current held.
				 */
};

typedef int (Blt_TreeNotifyCreateProc) _ANSI_ARGS_((Blt_Tree tree, 
	Blt_TreeNode node, ClientData clientData));
typedef int (Blt_TreeNotifyDeleteProc) _ANSI_ARGS_((Blt_Tree tree,
	Blt_TreeNode node, ClientData clientData));
typedef int (Blt_TreeNotifySetProc) _ANSI_ARGS_((Blt_Tree tree, 
	Blt_TreeNode node, Blt_TreeAtom fieldAtom, ClientData clientData));
typedef int (Blt_TreeNotifyUnsetProc) _ANSI_ARGS_((Blt_Tree tree, 
	Blt_TreeNode node, Blt_TreeAtom fieldAtom, ClientData clientData));

typedef int (Blt_TreeNotifyUpdateProc) _ANSI_ARGS_((Blt_Tree tree, 
	ClientData clientData, unsigned int flags));

typedef struct Blt_TreeNotifyProcs_ {
    Blt_TreeNotifyUpdateProc *notifyUpdateProc;

    /* Node-specific callbacks */
    Blt_TreeNotifyCreateProc *notifyCreateProc;
    Blt_TreeNotifyDeleteProc *notifyDeleteProc;
    Blt_TreeNotifySetProc *notifySetProc; 
    Blt_TreeNotifyUnsetProc *notifyUnsetProc; 
} Blt_TreeNotifyProcs;

/*
 * Blt_TokenToken --
 *
 *	A tree can be shared by several clients.  Each client allocates
 *	this structure which acts as a key for using the tree.  Clients
 *	can designate notifier routines that are automatically invoked
 *	by the tree object whenever the tree is changed is specific
 *	ways by other clients.
 */
struct Blt_TreeToken_ {
    unsigned int magic;		/* Magic value designating whether this
				 * really is a tree token or not */
    unsigned int notifyFlags;	/* Notification flags. See definitions
				 * in bltTree.h. */
    Blt_TreeNotifyProcs *procsPtr;
    ClientData clientData;
    Blt_ChainLink *linkPtr;
    Blt_TreeObject tree;	/* Pointer to the structure containing
				 * the master information about the
				 * tree used by the client.  If NULL,
				 * this indicates that the tree has
				 * been destroyed (but as of yet, this
				 * client hasn't recognized it). */
};

EXTERN Blt_TreeNode Blt_TreeCreateNode _ANSI_ARGS_((Blt_Tree tree, 
	Blt_TreeNode parent, char *name, int position)); 

EXTERN int Blt_TreeDeleteNode _ANSI_ARGS_((Blt_Tree tree, Blt_TreeNode node));

EXTERN int Blt_TreeMoveNode _ANSI_ARGS_((Blt_Tree tree, Blt_TreeNode node, 
	Blt_TreeNode parent, Blt_TreeNode before));

EXTERN Blt_TreeNode Blt_TreeGetNode _ANSI_ARGS_((Blt_Tree tree, 
	unsigned int inode));

EXTERN Blt_TreeNode Blt_TreeFindChild _ANSI_ARGS_((Blt_TreeNode parent, 
	char *name));

EXTERN Blt_TreeAtom Blt_TreeGetAtom _ANSI_ARGS_((Blt_Tree tree, char *string));

EXTERN void Blt_TreeFreeAtom _ANSI_ARGS_((Blt_Tree tree, Blt_TreeAtom atom));

EXTERN Blt_Tree Blt_TreeAllocate _ANSI_ARGS_((Tcl_Interp *interp, char *name));

EXTERN void Blt_TreeRelease _ANSI_ARGS_((Blt_Tree tree));

EXTERN Blt_TreeNode Blt_TreeFirstChild _ANSI_ARGS_((Blt_TreeNode parent));

EXTERN Blt_TreeNode Blt_TreeNextSibling _ANSI_ARGS_((Blt_TreeNode node));

EXTERN Blt_TreeNode Blt_TreeLastChild _ANSI_ARGS_((Blt_TreeNode parent));

EXTERN Blt_TreeNode Blt_TreePrevSibling _ANSI_ARGS_((Blt_TreeNode node));

EXTERN Blt_TreeNode Blt_TreeNextNode _ANSI_ARGS_((Blt_TreeNode root, 
	Blt_TreeNode node, unsigned int nodeFlags));

EXTERN Blt_TreeNode Blt_TreePrevNode _ANSI_ARGS_((Blt_TreeNode root,
	Blt_TreeNode node, unsigned int nodeFlags));

EXTERN Blt_TreeNode Blt_TreeRootNode _ANSI_ARGS_((Blt_Tree tree));

EXTERN Blt_TreeNode Blt_TreeEndNode _ANSI_ARGS_((Blt_TreeNode node,
	unsigned int nodeFlags));

EXTERN int Blt_TreeIsBefore _ANSI_ARGS_((Blt_TreeNode node1, 
	Blt_TreeNode node2));

EXTERN int Blt_TreeIsAncestor _ANSI_ARGS_((Blt_TreeNode root, 
	Blt_TreeNode node));

EXTERN void Blt_TreeExposeAncestors _ANSI_ARGS_((Blt_TreeNode node));

EXTERN int Blt_TreeMapAncestors _ANSI_ARGS_((Blt_TreeNode node));

EXTERN int Blt_TreeGetValue _ANSI_ARGS_((Blt_TreeNode node, 
	Blt_TreeAtom fieldAtom, Tcl_Obj **valuePtr));

EXTERN int Blt_TreeSetValue _ANSI_ARGS_((Blt_Tree tree, Blt_TreeNode node, 
	Blt_TreeAtom fieldAtom, Tcl_Obj *valuePtr));

EXTERN int Blt_TreeUnsetValue _ANSI_ARGS_((Blt_Tree tree, Blt_TreeNode node, 
	Blt_TreeAtom fieldAtom));

typedef int (Blt_TreeEnumProc) _ANSI_ARGS_((Blt_TreeNode node, char *name,
	Tcl_Obj *valuePtr));

EXTERN int Blt_TreeEnumFields _ANSI_ARGS_((Blt_TreeNode node, 
	Blt_TreeEnumProc *proc));

typedef int (Blt_TreeCompareNodesProc) _ANSI_ARGS_((Blt_TreeNode node1, 
	Blt_TreeNode node2));

typedef int (Blt_TreeApplyProc) _ANSI_ARGS_((Blt_TreeNode node, 
	ClientData clientData));

EXTERN int Blt_TreeApply _ANSI_ARGS_((Blt_TreeNode root, 
	Blt_TreeApplyProc *proc, ClientData clientData));

EXTERN int Blt_TreeSortNode _ANSI_ARGS_((Blt_TreeNode node, 
	Blt_TreeCompareNodesProc *proc));

EXTERN int Blt_TreeCreate _ANSI_ARGS_((Tcl_Interp *interp, char *pathName));

EXTERN Blt_Tree Blt_TreeGetToken _ANSI_ARGS_((Tcl_Interp *interp, 
	char *name, Blt_TreeNotifyProcs *procsPtr, ClientData clientData));
EXTERN void Blt_TreeReleaseToken _ANSI_ARGS_((Blt_Tree tree));

EXTERN unsigned int Blt_TreeGetNotifyFlags _ANSI_ARGS_((Blt_Tree tree));
EXTERN void Blt_TreeSetNotifyFlags _ANSI_ARGS_((Blt_Tree tree, 
	unsigned int mask));

#define Blt_TreeName(token)	((token)->tree->nameUid)
#define Blt_TreeFullName(token)  ((token)->tree->fullName)
#define Blt_TreeRootNode(token)	((token)->tree->root)

#define Blt_TreeNodeDepth(node)	((node)->depth)
#define Blt_TreeNodeName(node)	((node)->atom)
#define Blt_TreeNodeIndex(node)	((node)->inode)
#define Blt_TreeNodeParent(node) ((node)->parent)
#define Blt_TreeNodeChildren(node) \
	(((node) == NULL) ? 0 : Blt_ChainGetLength((node)->chainPtr))
#define Blt_TreeNodeFlags(node) ((node)->flags)

#endif /* BLT_TREE_H */
