/*
 * bltHtCmd.c --
 *
 *	This module implements an hierarchy widget for the BLT toolkit.
 *
 * Copyright 1998-1999 Lucent Technologies, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the names
 * of Lucent Technologies or any of their entities not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and
 * fitness.  In no event shall Lucent Technologies be liable for any
 * special, indirect or consequential damages or any damages
 * whatsoever resulting from loss of use, data or profits, whether in
 * an action of contract, negligence or other tortuous action, arising
 * out of or in connection with the use or performance of this
 * software.
 *
 *	The "hiertable" widget was created by George A. Howlett.
 */

/*
 * TODO:
 *
 * BUGS:
 *   1.  "open" operation should change scroll offset so that as many
 *	 new entries (up to half a screen) can be seen.
 *   2.  "open" needs to adjust the scrolloffset so that the same entry
 *	 is seen at the same place.
 */
#include "bltInt.h"

#ifndef NO_HIERTABLE

#include "bltHiertable.h"
#include "bltList.h"
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#define TOGGLE(x, mask)		\
	(((x) & (mask)) ? ((x) & ~(mask)) : ((x) | (mask)))
#define CLAMP(val,low,hi)	\
	(((val) < (low)) ? (low) : ((val) > (hi)) ? (hi) : (val))

#define DEF_COLUMN_BACKGROUND		RGB_WHITE
#define DEF_COLUMN_BORDER_WIDTH		STD_BORDERWIDTH
#define DEF_COLUMN_COLOR		RGB_BLACK
#define DEF_COLUMN_FONT			STD_FONT
#define DEF_COLUMN_FORMAT_COMMAND	(char *)NULL
#define DEF_COLUMN_HIDE			"no"
#define DEF_COLUMN_JUSTIFY		"center"
#define DEF_COLUMN_MAX			"0"
#define DEF_COLUMN_MIN			"0"
#define DEF_COLUMN_PAD			"2"
#define DEF_COLUMN_RELIEF		"flat"
#define DEF_COLUMN_STATE		"normal"
#define DEF_COLUMN_TEXT			(char *)NULL
#define DEF_COLUMN_TITLE		(char *)NULL
#define DEF_COLUMN_TITLE_BACKGROUND	STD_COLOR_NORMAL_BG
#define DEF_COLUMN_TITLE_FONT		STD_FONT
#define DEF_COLUMN_TITLE_FOREGROUND	STD_COLOR_NORMAL_FG
#define DEF_COLUMN_TITLE_LABEL		(char *)NULL
#define DEF_COLUMN_TITLE_SHADOW		(char *)NULL
#define DEF_COLUMN_WEIGHT		"1.0"
#define DEF_COLUMN_WIDTH		"0"

#ifdef __STDC__
static CompareProc 
	ExactCompare, 
	GlobCompare, 
	RegexpCompare;
static Blt_TreeApplyProc 
	SizeApplyProc, 
	SortApplyProc, 
	OpenApplyProc, 
	CloseApplyProc, 
	UnmapApplyProc, 
	MapAncestorsApplyProc, 
	FixUnmappedSelectionsApplyProc;
static Blt_TreeCompareNodesProc CompareNodesByTclCmd, CompareNodesByName;
static Tk_LostSelProc LostSelection;
static ApplyProc SelectEntry;
static IterProc PrevNode;
#endif /* __STDC__ */

static int StringToData _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, Tk_Window tkwin, char *string, char *widgRec,
	int offset));
static char *DataToString _ANSI_ARGS_((ClientData clientData,
	Tk_Window tkwin, char *widgRec, int offset,
	Tcl_FreeProc **freeProcPtrPtr));

Tk_CustomOption bltHiertableDataOption =
{
    StringToData, DataToString, (ClientData)0,
};

extern Tk_CustomOption bltDistanceOption;
extern Tk_CustomOption bltPadOption;
extern Tk_CustomOption bltShadowOption;
extern Tk_CustomOption bltStateOption;

static Tk_ConfigSpec columnSpecs[] =
{
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	DEF_COLUMN_BACKGROUND, Tk_Offset(Column, border), 0},
    {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *)NULL, (char *)NULL, 0, 0},
    {TK_CONFIG_SYNONYM, "-bg", "background", (char *)NULL, (char *)NULL, 0, 0},
    {TK_CONFIG_CUSTOM, "-borderwidth", "borderWidth", "BorderWidth",
	DEF_COLUMN_BORDER_WIDTH, Tk_Offset(Column, borderWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &bltDistanceOption},
    {TK_CONFIG_SYNONYM, "-fg", "foreground", (char *)NULL, (char *)NULL, 0, 0},
    {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_COLUMN_COLOR, Tk_Offset(Column, fgColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_FONT, "-font", "font", "Font",
	DEF_COLUMN_FONT, Tk_Offset(Column, font), 0},
    {TK_CONFIG_BOOLEAN, "-hide", "hide", "Hide",
	DEF_COLUMN_HIDE, Tk_Offset(Column, hidden),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_JUSTIFY, "-justify", "justify", "Justify",
	DEF_COLUMN_JUSTIFY, Tk_Offset(Column, justify), 
        TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-max", "max", "Max",
	DEF_COLUMN_MAX, Tk_Offset(Column, reqMax),
        TK_CONFIG_DONT_SET_DEFAULT, &bltDistanceOption},
    {TK_CONFIG_CUSTOM, "-min", "min", "Min",
	DEF_COLUMN_MIN, Tk_Offset(Column, reqMin),
        TK_CONFIG_DONT_SET_DEFAULT, &bltDistanceOption},
    {TK_CONFIG_CUSTOM, "-pad", "pad", "Pad",
	DEF_COLUMN_PAD, Tk_Offset(Column, pad), 
        TK_CONFIG_DONT_SET_DEFAULT, &bltPadOption},
    {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
	DEF_COLUMN_RELIEF, Tk_Offset(Column, relief), 
        TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-state", "state", "State",
	DEF_COLUMN_STATE, Tk_Offset(Column, state), 
	TK_CONFIG_DONT_SET_DEFAULT, &bltStateOption},
    {TK_CONFIG_STRING, "-text", "text", "Text",
	DEF_COLUMN_TITLE_LABEL, Tk_Offset(Column, text), 0},
    {TK_CONFIG_BORDER, "-titlebackground", "titleBackground", "TitleBackground",
	DEF_COLUMN_TITLE_BACKGROUND, Tk_Offset(Column, titleBorder), 0},
    {TK_CONFIG_FONT, "-titlefont", "titleFont", "Font",
	DEF_COLUMN_TITLE_FONT, Tk_Offset(Column, titleFont), 0},
    {TK_CONFIG_COLOR, "-titleforeground", "titleForeground", "TitleForeground",
	DEF_COLUMN_TITLE_FOREGROUND, Tk_Offset(Column, titleFgColor), 0},
    {TK_CONFIG_CUSTOM, "-titleshadow", "titleShadow", "TitleShadow",
	DEF_COLUMN_TITLE_SHADOW, Tk_Offset(Column, titleShadow), 0,
        &bltShadowOption},
    {TK_CONFIG_DOUBLE, "-weight", (char *)NULL, (char *)NULL,
	DEF_COLUMN_WEIGHT, Tk_Offset(Column, weight), 
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-width", "width", "Width",
	DEF_COLUMN_WIDTH, Tk_Offset(Column, reqWidth),
        TK_CONFIG_DONT_SET_DEFAULT, &bltDistanceOption},
    {TK_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL,
	(char *)NULL, 0, 0}
};

/*
 *----------------------------------------------------------------------
 *
 * SkipSeparators --
 *
 *	Moves the character pointer past one of more separators.
 *
 * Results:
 *	Returns the updates character pointer.
 *
 *----------------------------------------------------------------------
 */
static char *
SkipSeparators(path, pathSep, length)
    char *path, *pathSep;
    int length;
{
    while ((*path == pathSep[0]) && (strncmp(path, pathSep, length) == 0)) {
	path += length;
    }
    return path;
}

/*
 *----------------------------------------------------------------------
 *
 * SplitPath --
 *
 *	Returns the trailing component of the given path.  Trailing
 *	separators are ignored.
 *
 * Results:
 *	Returns the string of the tail component.
 *
 *----------------------------------------------------------------------
 */
static int
SplitPath(htabPtr, path, depthPtr, compPtrPtr)
    Hiertable *htabPtr;
    char *path;
    int *depthPtr;
    char ***compPtrPtr;
{
    int skipLen, pathLen;
    char *sep;
    int depth, listSize;
    char **components;
    register char *p;

    if ((htabPtr->pathSep == TCL_LIST) || (htabPtr->pathSep == NO_SEPARATOR)) {
	if (Tcl_SplitList(htabPtr->interp, path, depthPtr, compPtrPtr) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
	return TCL_OK;
    }
    skipLen = strlen(htabPtr->pathSep);
    path = SkipSeparators(path, htabPtr->pathSep, skipLen);
    pathLen = strlen(path);

    depth = pathLen / skipLen;
    listSize = (depth + 1) * sizeof(char *);
    components = (char **)malloc(listSize + (pathLen + 1));
    assert(components);
    p = (char *)components + listSize;
    strcpy(p, path);

    sep = strstr(p, htabPtr->pathSep);
    depth = 0;
    while ((*p != '\0') && (sep != NULL)) {
	*sep = '\0';
	components[depth++] = p;
	p = SkipSeparators(sep + skipLen, htabPtr->pathSep, skipLen);
	sep = strstr(p, htabPtr->pathSep);
    }
    if (*p != '\0') {
	components[depth++] = p;
    }
    components[depth] = NULL;
    *depthPtr = depth;
    *compPtrPtr = components;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * PrevNode --
 *
 *	Returns the "previous" node in the tree.  This node (in 
 *	depth-first order) is its parent, if the node has no siblings
 *	that are previous to it.  Otherwise it is the last descendant 
 *	of the last sibling.  In this case, descend the sibling's
 *	hierarchy, using the last child at any ancestor, with we
 *	we find a leaf.
 *
 *----------------------------------------------------------------------
 */
static Blt_TreeNode
PrevNode(htabPtr, node, mask)
    Hiertable *htabPtr;
    Blt_TreeNode node;
    unsigned int mask;
{
    Blt_TreeNode prev;
    Entry *entryPtr;

    if (node == htabPtr->rootNode) {
	return NULL;		/* The root is the first node. */
    }
    prev = Blt_TreePrevSibling(node);
    if (prev == NULL) {
	/* There are no siblings previous to this one, so pick the parent. */
	return Blt_TreeNodeParent(node);
    }
    /*
     * Traverse down the right-most thread, in order to select the
     * next entry.  Stop if we find a "closed" entry or reach a leaf.
     */
    node = prev;
    entryPtr = GetEntry(htabPtr, node);
    while ((entryPtr->flags & mask) == mask) {
	prev = Blt_TreeLastChild(node);
	if (prev == NULL) {
	    break;		/* Found a leaf. */
	}
	node = prev;
	entryPtr = GetEntry(htabPtr, node);
    }
    return node;
}

static Blt_TreeNode
EndNode(htabPtr, node, mask)
    Hiertable *htabPtr;
    Blt_TreeNode node;
    unsigned int mask;
{
    Entry *entryPtr;
    Blt_TreeNode next;
    
    next = Blt_TreeLastChild(node);
    while (next != NULL) {
	entryPtr = GetEntry(htabPtr, next);
	if ((entryPtr->flags & mask) != mask) {
	    break;
	}
	node = next;
	next = Blt_TreeLastChild(next);
    }
    return node;
}

/*
 *----------------------------------------------------------------------
 *
 * OpenApplyProc --
 *
 *	Sets the selection flag for a node.  The selection flag is
 *	set/cleared/toggled based upon the flag set in the hierarchy
 *	widget.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 *----------------------------------------------------------------------
 */
static int
OpenApplyProc(node, clientData)
    Blt_TreeNode node;
    ClientData clientData;
{
    Hiertable *htabPtr = (Hiertable *)clientData;

    return Blt_HtOpenEntry(htabPtr, GetEntry(htabPtr, node));
}

/*
 *----------------------------------------------------------------------
 *
 * CloseApplyProc --
 *
 *	Sets the selection flag for a node.  The selection flag is
 *	set/cleared/toggled based upon the flag set in the hierarchy
 *	widget.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 *----------------------------------------------------------------------
 */
static int
CloseApplyProc(node, clientData)
    Blt_TreeNode node;
    ClientData clientData;
{
    Hiertable *htabPtr = (Hiertable *)clientData;

    return Blt_HtCloseEntry(htabPtr, GetEntry(htabPtr, node));
}

/*
 *----------------------------------------------------------------------
 *
 * SortApplyProc --
 *
 *	Sorts the subnodes at a given node.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 *----------------------------------------------------------------------
 */
static int
SortApplyProc(node, clientData)
    Blt_TreeNode node;
    ClientData clientData;
{
    if (Blt_TreeNodeChildren(node) > 1) {
	Hiertable *htabPtr = (Hiertable *)clientData;

	if (htabPtr->sortCmd != NULL) {
	    bltHiertableLastInstance = htabPtr;
	    Blt_TreeSortNode(node, CompareNodesByTclCmd);
	} else {
	    Blt_TreeSortNode(node, CompareNodesByName);
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * MapApplyProc --
 *
 * Results:
 *	Always returns TCL_OK.
 *
 *----------------------------------------------------------------------
 */
static int
MapApplyProc(node, clientData)
    Blt_TreeNode node;
    ClientData clientData;
{
    Hiertable *htabPtr = (Hiertable *)clientData;
    Entry *entryPtr;

    entryPtr = GetEntry(htabPtr, node);
    entryPtr->flags |= ENTRY_MAPPED;
    return TCL_OK;
}

#ifdef notdef
int
EntryIsMapped(nodePtr)
    Node *nodePtr;
{
    if (nodePtr != NULL) {
	unsigned int nodeFlags;

	if (!(nodePtr->flags & NODE_MAPPED)) {
	    return FALSE;
	}
	nodePtr = nodePtr->parentPtr;
	nodeFlags = (NODE_OPEN | NODE_MAPPED);
	while (nodePtr != NULL) {
	    if ((nodePtr->flags & nodeFlags) != nodeFlags) {
		return FALSE;
	    }
	    nodePtr = nodePtr->parentPtr;
	}
    }
    return TRUE;
}
#endif

/*
 *----------------------------------------------------------------------
 *
 * UnmapApplyProc --
 *
 * Results:
 *	Always returns TCL_OK.
 *
 *----------------------------------------------------------------------
 */
static int
UnmapApplyProc(node, clientData)
    Blt_TreeNode node;
    ClientData clientData;
{
    Hiertable *htabPtr = (Hiertable *)clientData;
    Entry *entryPtr;

    entryPtr = GetEntry(htabPtr, node);
    entryPtr->flags &= ~ENTRY_MAPPED;
    return TCL_OK;
}

static void
MapAncestors(htabPtr, node)
    Hiertable *htabPtr;
    Blt_TreeNode node;
{
    Entry *entryPtr;

    while (node != NULL) {
	entryPtr = GetEntry(htabPtr, node);
	entryPtr->flags |= (ENTRY_MAPPED | ENTRY_OPEN);
	node = Blt_TreeNodeParent(node);
    }
}


/*
 *----------------------------------------------------------------------
 *
 * MapAncestorsApplyProc --
 *
 *	If a node in mapped, then all its ancestors must be mapped also.
 *	This routine traverses upwards and maps each unmapped ancestor.
 *	It's assumed that for any mapped ancestor, all it's ancestors
 *	will already be mapped too.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 *----------------------------------------------------------------------
 */
static int
MapAncestorsApplyProc(node, clientData)
    Blt_TreeNode node;
    ClientData clientData;
{
    Hiertable *htabPtr = (Hiertable *)clientData;
    Entry *entryPtr;
    /*
     * Make sure that all the ancestors of this entry are mapped too.
     */
    node = Blt_TreeNodeParent(node);
    while (node != NULL) {
	entryPtr = GetEntry(htabPtr, node);
	if (entryPtr->flags & ENTRY_MAPPED) {
	    break;		/* Assume ancestors are also mapped. */
	}
	entryPtr->flags |= (ENTRY_MAPPED | ENTRY_OPEN);
	node = Blt_TreeNodeParent(node);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FindPath --
 *
 *	Finds the node designated by the given path.  Each path
 *	component is searched for as the tree is traversed.
 *
 *	A leading character string is trimmed off the path if it
 *	matches the one designated (see the -trimleft option).
 *
 *	If no separator is designated (see the -separator
 *	configuration option), the path is considered a Tcl list.
 *	Otherwise the each component of the path is separated by a
 *	character string.  Leading and trailing separators are
 *	ignored.  Multiple separators are treated as one.
 *
 * Results:
 *	Returns the pointer to the designated node.  If any component
 *	can't be found, NULL is returned.
 *
 *----------------------------------------------------------------------
 */
static Blt_TreeNode
FindPath(htabPtr, node, path)
    Hiertable *htabPtr;
    Blt_TreeNode node;
    char *path;
{
    Blt_TreeNode child;
    char **compArr;
    int nComp;
    register char **p;
    
    /* Trim off characters that we don't want */
    if (htabPtr->trimLeft != NULL) {
	register char *s1, *s2;
	
	/* Trim off leading character string if one exists. */
	for (s1 = path, s2 = htabPtr->trimLeft; *s2 != '\0'; s2++, s1++) {
	    if (*s1 != *s2) {
		break;
	    }
	}
	if (*s2 == '\0') {
	    path = s1;
	}
    }
    if (*path == '\0') {
	return node;
    }
    if (SplitPath(htabPtr, path, &nComp, &compArr) != TCL_OK) {
	return NULL;
    }
    for (p = compArr; *p != NULL; p++) {
	child = Blt_TreeFindChild(node, *p);
	if (child == NULL) {
	    Tcl_DString dString;

	    Tcl_DStringInit(&dString);
	    Blt_HtGetFullPath(htabPtr, node, &dString);
	    Tcl_AppendResult(htabPtr->interp, "can't find node \"", *p,
		     "\" in parent node \"", Tcl_DStringValue(&dString), "\"", 
		     (char *)NULL);
	    Tcl_DStringFree(&dString);
	    free((char *)compArr);
	    return NULL;
	}
	node = child;
    }
    free((char *)compArr);
    return node;
}

/*
 *----------------------------------------------------------------------
 *
 * GetNode --
 *
 *	Converts a string into node pointer.  The string may be in one
 *	of the following forms:
 *
 *	    @x,y		- Closest node to the specified X-Y position.
 *	    NNN			- inode.
 *	    "active"		- Currently active node.
 *	    "anchor"		- anchor of selected region.
 *	    "current"		- Currently picked node in bindtable.
 *	    "focus"		- The node currently with focus.
 *	    "root"		- Root node.
 *	    "end"		- Last open node in the entire hierarchy.
 *	    "next"		- Next open node from the currently active
 *				  node. Wraps around back to top.
 *	    "last"		- Previous open node from the currently active
 *				  node. Wraps around back to bottom.
 *	    "up"		- Next open node from the currently active
 *				  node. Does not wrap around.
 *	    "down"		- Previous open node from the currently active
 *				  node. Does not wrap around.
 *	    "nextsibling"	- Next sibling of the current node.
 *	    "prevsibling"	- Previous sibling of the current node.
 *	    "parent"		- Parent of the current node.
 *	    "view.top"		- Top of viewport.
 *	    "view.bottom"	- Bottom of viewport.
 *	    @path		- Absolute path to a node.
 *
 * Results:
 *	If the string is successfully converted, TCL_OK is returned.
 *	The pointer to the node is returned via nodePtr.
 *	Otherwise, TCL_ERROR is returned and an error message is left
 *	in interpreter's result field.
 *
 *----------------------------------------------------------------------
 */
static int
GetNode2(htabPtr, string, nodePtr)
    Hiertable *htabPtr;
    char *string;
    Blt_TreeNode *nodePtr;
{
    Blt_TreeNode node, from;
    char c;
    int mask;

    c = string[0];
    from = *nodePtr;
    *nodePtr = NULL;

    mask = ENTRY_OPEN | ENTRY_MAPPED;
    node = NULL;
    if (isdigit(UCHAR(string[0]))) {
	int inode;

	if (Tcl_GetInt(htabPtr->interp, string, &inode) != TCL_OK) {
	    return TCL_ERROR;
	}
	node = Blt_TreeGetNode(htabPtr->tree, inode);
    } else if ((c == 'e') && (strcmp(string, "end") == 0)) {
	node = EndNode(htabPtr, htabPtr->rootNode, mask);
    } else if ((c == 'a') && (strcmp(string, "anchor") == 0)) {
	node = htabPtr->selAnchorNode;
    } else if ((c == 'f') && (strcmp(string, "focus") == 0)) {
	node = htabPtr->focusNode;
	if ((node == htabPtr->rootNode) && (htabPtr->hideRoot)) {
	    node = Blt_HtNextNode(htabPtr, htabPtr->rootNode, mask);
	}
    } else if ((c == 'r') && (strcmp(string, "root") == 0)) {
	node = htabPtr->rootNode;
    } else if ((c == 'p') && (strcmp(string, "parent") == 0)) {
	node = from;
	if (Blt_TreeNodeParent(node) != NULL) {
	    node = Blt_TreeNodeParent(node);
	}
    } else if ((c == 'c') && (strcmp(string, "current") == 0)) {
	/* Can't trust picked item, if entries have been 
	 * added or deleted. */
	if (!(htabPtr->flags & HIERTABLE_DIRTY)) {
	    node = (Blt_TreeNode)Blt_GetCurrentItem(htabPtr->bindTable);
	    if (node == NULL) {
		node = (Blt_TreeNode)
		    Blt_GetCurrentItem(htabPtr->buttonBindTable);
	    }
	}
    } else if ((c == 'u') && (strcmp(string, "up") == 0)) {
	node = PrevNode(htabPtr, from, mask);
	if (node == NULL) {
	    node = from;
	}
	if ((node == htabPtr->rootNode) && (htabPtr->hideRoot)) {
	    node = Blt_HtNextNode(htabPtr, node, mask);
	}
    } else if ((c == 'd') && (strcmp(string, "down") == 0)) {
	node = Blt_HtNextNode(htabPtr, from, mask);
	if (node == NULL) {
	    node = from;
	}
	if ((node == htabPtr->rootNode) && (htabPtr->hideRoot)) {
	    node = Blt_HtNextNode(htabPtr, node, mask);
	}
    } else if (((c == 'l') && (strcmp(string, "last") == 0)) ||
	((c == 'p') && (strcmp(string, "prev") == 0))) {
	node = PrevNode(htabPtr, from, mask);
	if (node == NULL) {
	    node = EndNode(htabPtr, htabPtr->rootNode, mask);
	}
	if ((node == htabPtr->rootNode) && (htabPtr->hideRoot)) {
	    node = Blt_HtNextNode(htabPtr, node, mask);
	}
    } else if ((c == 'n') && (strcmp(string, "next") == 0)) {
	node = Blt_HtNextNode(htabPtr, from, mask);
	if (node == NULL) {
	    if (htabPtr->hideRoot) {
		node = Blt_HtNextNode(htabPtr, htabPtr->rootNode, mask);
	    } else {
		node = htabPtr->rootNode;
	    }
	}
    } else if ((c == 'n') && (strcmp(string, "nextsibling") == 0)) {
	node = Blt_TreeNextSibling(from);
	if (node == NULL) {
	    node = from;
	}
    } else if ((c == 'p') && (strcmp(string, "prevsibling") == 0)) {
	node = Blt_TreePrevSibling(from);
	if (node == NULL) {
	    node = from;
	}
    } else if ((c == 'v') && (strcmp(string, "view.top") == 0)) {
	if (htabPtr->nVisible > 0) {
	    node = htabPtr->visibleArr[0]->node;
	}
    } else if ((c == 'v') && (strcmp(string, "view.bottom") == 0)) {
	if (htabPtr->nVisible > 0) {
	    node = htabPtr->visibleArr[htabPtr->nVisible - 1]->node;
	}
    } else if (c == '@') {
	int x, y;

	if (Blt_GetXY(htabPtr->interp, htabPtr->tkwin, string, &x, &y)
	    == TCL_OK) {
	    node = Blt_HtNearestNode(htabPtr, x, y, TRUE);
	} else {
	    node = FindPath(htabPtr, htabPtr->rootNode, string + 1);
	}
	if (node == NULL) {
	    Tcl_ResetResult(htabPtr->interp);
	    Tcl_AppendResult(htabPtr->interp, "can't find entry \"", string,
		"\" in \"", Tk_PathName(htabPtr->tkwin), "\"", (char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	node = FindPath(htabPtr, htabPtr->rootNode, string);
	if (node == NULL) {
	    Tcl_ResetResult(htabPtr->interp);
	    Tcl_AppendResult(htabPtr->interp, "can't find entry \"", string, 
		"\" in \"", Tk_PathName(htabPtr->tkwin), "\"", (char *)NULL);
	    return TCL_ERROR;
	}
    }
    *nodePtr = node;
    return TCL_OK;
}

static int
GetNode(htabPtr, string, nodePtr)
     Hiertable *htabPtr;
     char *string;
     Blt_TreeNode *nodePtr; 
{
    *nodePtr = htabPtr->focusNode; /* Default to focus node */
    return GetNode2(htabPtr, string, nodePtr);
}

/*
 *----------------------------------------------------------------------
 *
 * StringToNode --
 *
 *	Like GetNode but also finds nodes by serial number.
 *	If the string starts with a digit, it's converted into a
 *	number and then looked-up in a hash table.  This means that
 *	serial identifiers take precedence over node names with
 *	the contain only numbers.
 *
 * Results:
 *	If the string is successfully converted, TCL_OK is returned.
 *	The pointer to the node is returned via nodePtr.
 *	Otherwise, TCL_ERROR is returned and an error message is left
 *	in interpreter's result field.
 *
 *----------------------------------------------------------------------
 */

static int
StringToNode(htabPtr, string, nodePtr)
    Hiertable *htabPtr;
    char *string;
    Blt_TreeNode *nodePtr;
{
    if (GetNode(htabPtr, string, nodePtr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (*nodePtr == NULL) {
	Tcl_ResetResult(htabPtr->interp);
	Tcl_AppendResult(htabPtr->interp, "can't find entry \"", string,
	    "\" in \"", Tk_PathName(htabPtr->tkwin), "\"", (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static Blt_TreeNode 
GetNthNode(parent, position)
    Blt_TreeNode parent;
    int position;
{
    Blt_TreeNode node;
    int count;

    count = 0;
    for(node = Blt_TreeFirstChild(parent); node != NULL; 
	node = Blt_TreeNextSibling(node)) {
	if (count == position) {
	    return node;
	}
    }
    return Blt_TreeLastChild(parent);
}

/*
 *--------------------------------------------------------------
 *
 * GetPosition --
 *
 *	Convert a string representing a position in the list of children
 *	to its numeric value.  A position index can be in one of
 *	the following forms.
 *
 * 	  number	- number of the item in the hierarchy, indexed
 *			  from zero.
 *	  "end"		- last position in the hierarchy.
 *
 * Results:
 *	A standard Tcl result.  If "string" is a valid index, then
 *	*indexPtr is filled with the corresponding numeric index.
 *	Otherwise an error message is left in interp->result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static int
GetPosition(interp, string, indexPtr)
    Tcl_Interp *interp;		/* Interpreter to report results back
				 * to. */
    char *string;		/* String representation of the index.
				 * Can be an integer or "end" to refer
				 * to the last index. */
    int *indexPtr;		/* Holds the converted index. */
{
    if ((string[0] == 'e') && (strcmp(string, "end") == 0)) {
	*indexPtr = END;	/* Indicates last position in hierarchy. */
    } else {
	int position;

	if (Tcl_GetInt(interp, string, &position) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (position < 0) {
	    Tcl_AppendResult(interp, "bad position \"", string, "\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
	*indexPtr = position;
    }
    return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * SelectEntry --
 *
 *	Sets the selection flag for a node.  The selection flag is
 *	set/cleared/toggled based upon the flag set in the hierarchy
 *	widget.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 *----------------------------------------------------------------------
 */
static int
SelectEntry(htabPtr, entryPtr)
    Hiertable *htabPtr;
    Entry *entryPtr;
{
    switch (htabPtr->flags & SELECTION_MASK) {
    case SELECTION_CLEAR:
	entryPtr->flags &= ~ENTRY_SELECTED;
	break;

    case SELECTION_SET:
	entryPtr->flags |= ENTRY_SELECTED;
	break;

    case SELECTION_TOGGLE:
	entryPtr->flags = TOGGLE(entryPtr->flags, ENTRY_SELECTED);
	break;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * EventuallyInvokeSelectCmd --
 *
 *      Queues a request to execute the -selectcommand code associated
 *      with the widget at the next idle point.  Invoked whenever the
 *      selection changes.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Tcl code gets executed for some application-specific task.
 *
 *----------------------------------------------------------------------
 */
static void
EventuallyInvokeSelectCmd(htabPtr)
    Hiertable *htabPtr;
{
    if (!(htabPtr->flags & SELECTION_PENDING)) {
	htabPtr->flags |= SELECTION_PENDING;
	Tk_DoWhenIdle(Blt_HtSelectCmdProc, (ClientData)htabPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ClearSelections --
 *
 *	Recursively clears the selection flags of the given node and
 *	and each subnode.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	If a selection command is specified (see the "-selectcommand"
 *	option), the Tcl proc is invoked at the next idle point.
 *
 *----------------------------------------------------------------------
 */
static void
ClearSelections(htabPtr, node)
    Hiertable *htabPtr;
    Blt_TreeNode node;
{
    unsigned int flags;
    Entry *entryPtr;

    flags = (htabPtr->flags & SELECTION_MASK);
    htabPtr->flags &= ~SELECTION_MASK;
    htabPtr->flags |= SELECTION_CLEAR;
    for (node = Blt_TreeFirstChild(node); node != NULL;
	 node = Blt_TreeNextSibling(node)) {
	entryPtr = GetEntry(htabPtr, node);
	Blt_HtTreeApply(htabPtr, entryPtr, SelectEntry);
    }
    htabPtr->flags &= ~SELECTION_MASK;
    htabPtr->flags |= flags;
    if (htabPtr->selectCmd != NULL) {
	EventuallyInvokeSelectCmd(htabPtr);
    }
}

static int
GetColumn(htabPtr, fieldName, columnPtrPtr)
    Hiertable *htabPtr;
    char *fieldName;
    Column **columnPtrPtr;
{
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_FindHashEntry(&(htabPtr->columnTable), fieldName);
    if (hPtr == NULL) {
	Tcl_AppendResult(htabPtr->interp, "can't find column \"", fieldName, 
		 "\" in \"", Blt_TreeFullName(htabPtr->tree), "\"",
		(char *)NULL);
	return TCL_ERROR;
    }
    *columnPtrPtr = (Column *)Tcl_GetHashValue(hPtr);
    return TCL_OK;
}


static void
ConfigureColumn(htabPtr, columnPtr)
    Hiertable *htabPtr;
    Column *columnPtr;
{
    unsigned long gcMask;
    XGCValues gcValues;
    GC newGC;
    TextLayout *layoutPtr;
    TextStyle style;

    gcMask = GCForeground | GCFont;
    gcValues.foreground = columnPtr->fgColor->pixel;
    gcValues.font = Tk_FontId(columnPtr->font);
    newGC = Tk_GetGC(htabPtr->tkwin, gcMask, &gcValues);
    if (columnPtr->gc != NULL) {
	Tk_FreeGC(htabPtr->display, columnPtr->gc);
    }
    columnPtr->gc = newGC;
    gcValues.foreground = columnPtr->titleFgColor->pixel;
    gcValues.font = Tk_FontId(columnPtr->titleFont);
    newGC = Tk_GetGC(htabPtr->tkwin, gcMask, &gcValues);
    if (columnPtr->titleGC != NULL) {
	Tk_FreeGC(htabPtr->display, columnPtr->titleGC);
    }
    columnPtr->titleGC = newGC;

    memset(&style, 0, sizeof(TextStyle));
    style.font = columnPtr->titleFont;
    style.justify = TK_JUSTIFY_LEFT;
    style.shadow.offset = columnPtr->titleShadow.offset;
    layoutPtr = Blt_GetTextLayout(columnPtr->text, &style);
    if (columnPtr->layoutPtr != NULL) {
	free((char *)columnPtr->layoutPtr);
    }
    columnPtr->layoutPtr = layoutPtr;
}

static void
DestroyColumn(htabPtr, columnPtr)
    Hiertable *htabPtr;
    Column *columnPtr;
{
    Tcl_HashEntry *hPtr;

    Tk_FreeOptions(columnSpecs, (char *)columnPtr, htabPtr->display, 0);

    if (columnPtr->gc != NULL) {
	Tk_FreeGC(htabPtr->display, columnPtr->gc);
    }

    if (columnPtr->titleGC != NULL) {
	Tk_FreeGC(htabPtr->display, columnPtr->titleGC);
    }

    hPtr = Tcl_FindHashEntry(&(htabPtr->columnTable), columnPtr->atom);
    if (hPtr != NULL) {
	Tcl_DeleteHashEntry(hPtr);
    }

    if (columnPtr->linkPtr != NULL) {
	Blt_ChainDeleteLink(htabPtr->chainPtr, columnPtr->linkPtr);
    }
    if (columnPtr->text != NULL) {
	free(columnPtr->text);
    }
    if (columnPtr->layoutPtr != NULL) {
	free((char *)columnPtr->layoutPtr);
    }
    Blt_TreeFreeAtom(htabPtr->tree, columnPtr->atom);
    free ((char *)columnPtr);
}

void
Blt_HtDestroyColumns(htabPtr)
    Hiertable *htabPtr;
{
    Column *columnPtr;
    Blt_ChainLink *linkPtr;

    for (linkPtr = Blt_ChainFirstLink(htabPtr->chainPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	columnPtr = (Column *)Blt_ChainGetValue(linkPtr);
	columnPtr->linkPtr = NULL;
	DestroyColumn(htabPtr, columnPtr);
    }
    if (htabPtr->chainPtr != NULL) {
	Blt_ChainDestroy(htabPtr->chainPtr);
    }
    htabPtr->chainPtr = NULL;
}

Column *
Blt_HtCreateColumn(htabPtr, name, defTitle, nOptions, options)
    Hiertable *htabPtr;
    char *name, *defTitle;
    int nOptions;
    char **options;
{
    Column *columnPtr;
    Tcl_HashEntry *hPtr;
    int isNew;

    columnPtr = (Column *)calloc(1, sizeof(Column));
    assert(columnPtr);
    if (htabPtr->tree != NULL) {
	columnPtr->atom = Blt_TreeGetAtom(htabPtr->tree, name);
    }
    columnPtr->text = strdup(defTitle);
    columnPtr->justify = TK_JUSTIFY_CENTER;
    columnPtr->relief = TK_RELIEF_FLAT;
    columnPtr->borderWidth = 2;
    columnPtr->pad.side1 = columnPtr->pad.side2 = 2;
    columnPtr->state = STATE_NORMAL;
    columnPtr->weight = 1.0;
    hPtr = Tcl_CreateHashEntry(&(htabPtr->columnTable), name, &isNew);
    Tcl_SetHashValue(hPtr, (ClientData)columnPtr);
    if (Blt_ConfigureWidgetComponent(htabPtr->interp, htabPtr->tkwin, 
	Tk_GetUid(name), Tk_GetUid("Column"), columnSpecs,
	nOptions, options, (char *)columnPtr, 0) != TCL_OK) {
	DestroyColumn(htabPtr, columnPtr);
	return NULL;
    }
    ConfigureColumn(htabPtr, columnPtr);
    return columnPtr;
}


/*
 * --------------------------------------------------------------
 *
 * Hiertable operations
 *
 * --------------------------------------------------------------
 */

/*ARGSUSED*/
static int
FocusOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc == 3) {
	Blt_TreeNode node;

	if (GetNode(htabPtr, argv[2], &node) != TCL_OK) {
	    return TCL_ERROR;
	}
	if ((node != NULL) && (node != htabPtr->focusNode)) {
	    Entry *entryPtr;

	    entryPtr = GetEntry(htabPtr, node);
	    if (!(entryPtr->flags & ENTRY_MAPPED)) {
		/* Doesn't make sense to set focus to a node you can't see. */
		/* FIXME: make open */
		MapAncestors(htabPtr, node);
	    }
	    /* Changing focus can only affect the visible entries.  The
	     * entry layout stays the same. */
	    htabPtr->flags |= HIERTABLE_SCROLL;
	    htabPtr->focusNode = node;
	}
	Blt_HtEventuallyRedraw(htabPtr);
    }
    Blt_SetFocusItem(htabPtr->bindTable, htabPtr->focusNode);
    if (htabPtr->focusNode != NULL) {
	Tcl_SetResult(interp, Blt_HtNodeToString(htabPtr->focusNode),
	    TCL_VOLATILE);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * BboxOp --
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
BboxOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node;
    register int i;
    Entry *entryPtr;
    char string[200];
    int width, height, yBot;
    int left, top, right, bottom;
    int screen;
    int lWidth;

    if (htabPtr->flags & HIERTABLE_LAYOUT) {
	/*
	 * The layout is dirty.  Recompute it now, before we use the
	 * world dimensions.  But remember, the "bbox" operation isn't
	 * valid for hidden entries (since they're not visible, they
	 * don't have world coordinates).
	 */
	Blt_HtComputeLayout(htabPtr);
    }
    left = htabPtr->worldWidth;
    top = htabPtr->worldHeight;
    right = bottom = 0;

    screen = FALSE;
    if ((argv[2][0] == '-') && (strcmp(argv[2], "-screen") == 0)) {
	screen = TRUE;
	argc--, argv++;
    }
    for (i = 2; i < argc; i++) {
	if ((argv[i][0] == 'a') && (strcmp(argv[i], "all") == 0)) {
	    left = top = 0;
	    right = htabPtr->worldWidth;
	    bottom = htabPtr->worldHeight;
	    break;
	}
	if (GetNode(htabPtr, argv[i], &node) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (node == NULL) {
	    continue;
	}
	entryPtr = GetEntry(htabPtr, node);
	if (!(entryPtr->flags & ENTRY_MAPPED)) {
	    continue;
	}
	yBot = entryPtr->worldY + entryPtr->height;
	height = VPORTHEIGHT(htabPtr);
	if ((yBot <= htabPtr->yOffset) &&
	    (entryPtr->worldY >= (htabPtr->yOffset + height))) {
	    continue;
	}
	if (bottom < yBot) {
	    bottom = yBot;
	}
	if (top > entryPtr->worldY) {
	    top = entryPtr->worldY;
	}
	lWidth = ICONWIDTH(DEPTH(htabPtr, node));
	if (right < (entryPtr->worldX + entryPtr->width + lWidth)) {
	    right = (entryPtr->worldX + entryPtr->width + lWidth);
	}
	if (left > entryPtr->worldX) {
	    left = entryPtr->worldX;
	}
    }

    if (screen) {
	width = VPORTWIDTH(htabPtr);
	height = VPORTHEIGHT(htabPtr);
	/*
	 * Do a min-max text for the intersection of the viewport and
	 * the computed bounding box.  If there is no intersection, return
	 * the empty string.
	 */
	if ((right < htabPtr->xOffset) || (bottom < htabPtr->yOffset) ||
	    (left >= (htabPtr->xOffset + width)) ||
	    (top >= (htabPtr->yOffset + height))) {
	    return TCL_OK;
	}
	/* Otherwise clip the coordinates at the view port boundaries. */
	if (left < htabPtr->xOffset) {
	    left = htabPtr->xOffset;
	} else if (right > (htabPtr->xOffset + width)) {
	    right = htabPtr->xOffset + width;
	}
	if (top < htabPtr->yOffset) {
	    top = htabPtr->yOffset;
	} else if (bottom > (htabPtr->yOffset + height)) {
	    bottom = htabPtr->yOffset + height;
	}
	left = SCREENX(htabPtr, left), top = SCREENY(htabPtr, top);
	right = SCREENX(htabPtr, right), bottom = SCREENY(htabPtr, bottom);
    }
    if ((left < right) && (top < bottom)) {
	sprintf(string, "%d %d %d %d", left, top, right - left, bottom - top);
	Tcl_SetResult(interp, string, TCL_VOLATILE);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ButtonActivateOp --
 *
 *	Selects the button to appear active.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ButtonActivateOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node, old;

    if (argv[3][0] == '\0') {
	node = NULL;
    } else if (GetNode(htabPtr, argv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    if (htabPtr->hierColumnPtr->hidden) {
	return TCL_OK;
    }
    old = htabPtr->activeButtonNode;
    htabPtr->activeButtonNode = node;
    if (node != old) {
#ifdef notdef
	/* FIXME: Entries changed, how do you draw old? */
	if (htabPtr->flags & HIERTABLE_DIRTY) {
	    Blt_HtEventuallyRedraw(htabPtr);
	} else {
	    Drawable drawable;

	    drawable = Tk_WindowId(htabPtr->tkwin);
	    if (old != NULL) {
		Hiertable_DrawButton(htabPtr, old, drawable);
	    }
	    if (node != NULL) {
		Hiertable_DrawButton(htabPtr, node, drawable);
	    }
	    Hiertable_DrawOuterBorders(htabPtr, drawable);
	}
#else
	/* FIXME: Entries changed, how do you draw old? */
	Drawable drawable;

	drawable = Tk_WindowId(htabPtr->tkwin);
	if ((old != NULL) && (old != htabPtr->rootNode)) {
	    Blt_HtDrawButton(htabPtr, GetEntry(htabPtr, old), drawable);
	}
	if ((node != NULL) && (node != htabPtr->rootNode)) {
	    Blt_HtDrawButton(htabPtr, GetEntry(htabPtr, node), drawable);
	}
	Blt_HtDrawOuterBorders(htabPtr, drawable);
#endif
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ButtonBindOp --
 *
 *	  .t bind tag sequence command
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ButtonBindOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    ClientData object;
    int inode;
    /*
     * Individual entries are selected by inode only.  All other
     * strings are interpreted as a binding tag.  For example, if one
     * binds to "focus", it is assumed that this refers to a bind tag,
     * not the entry with focus.
     */
    if (Tcl_GetInt(interp, argv[3], &inode) != TCL_OK) {
	Tcl_ResetResult(interp);
	object = (ClientData)Blt_HtGetUid(htabPtr, argv[3]);
    } else {
	Blt_TreeNode node;

	node = Blt_TreeGetNode(htabPtr->tree, inode);
	object = (ClientData)node;
    }
    return Blt_ConfigureBindings(interp, htabPtr->buttonBindTable, object,
	argc - 4, argv + 4);
}

/*
 *----------------------------------------------------------------------
 *
 * ButtonCgetOp --
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ButtonCgetOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    return Tk_ConfigureValue(interp, htabPtr->tkwin, htabPtr->buttonSpecs, 
	(char *)htabPtr, argv[3], 0);
}

/*
 *----------------------------------------------------------------------
 *
 * ButtonConfigureOp --
 *
 * 	This procedure is called to process a list of configuration
 *	options database, in order to reconfigure the one of more
 *	entries in the widget.
 *
 *	  .h button configure option value
 *
 * Results:
 *	A standard Tcl result.  If TCL_ERROR is returned, then
 *	interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for htabPtr; old resources get freed, if there
 *	were any.  The hypertext is redisplayed.
 *
 *----------------------------------------------------------------------
 */
static int
ButtonConfigureOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc == 3) {
	return Tk_ConfigureInfo(interp, htabPtr->tkwin, htabPtr->buttonSpecs, 
		(char *)htabPtr, (char *)NULL, 0);
    } else if (argc == 4) {
	return Tk_ConfigureInfo(interp, htabPtr->tkwin, htabPtr->buttonSpecs, 
		(char *)htabPtr, argv[3], 0);
    }
    if (Tk_ConfigureWidget(htabPtr->interp, htabPtr->tkwin, 
		htabPtr->buttonSpecs, argc - 3, argv + 3, (char *)htabPtr, 
		TK_CONFIG_ARGV_ONLY) != TCL_OK) {
	return TCL_ERROR;
    }
    Blt_HtConfigureButtons(htabPtr);
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ButtonOp --
 *
 *	This procedure handles button operations.
 *
 * Results:
 *	A standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
static Blt_OpSpec buttonOps[] =
{
    {"activate", 1, (Blt_OpProc)ButtonActivateOp, 4, 4, "node",},
    {"bind", 1, (Blt_OpProc)ButtonBindOp, 4, 6,
	"tagName ?sequence command?",},
    {"cget", 2, (Blt_OpProc)ButtonCgetOp, 4, 4, "option",},
    {"configure", 2, (Blt_OpProc)ButtonConfigureOp, 3, 0,
	"?option value?...",},
};

static int nButtonOps = sizeof(buttonOps) / sizeof(Blt_OpSpec);

static int
ButtonOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Blt_OpProc opProc;
    int result;

    opProc = Blt_GetOperation(interp, nButtonOps, buttonOps, BLT_OPER_ARG2, 
	argc, argv);
    if (opProc == NULL) {
	return TCL_ERROR;
    }
    result = (*opProc) (htabPtr, interp, argc, argv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * CgetOp --
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
CgetOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    return Tk_ConfigureValue(interp, htabPtr->tkwin, htabPtr->widgetSpecs,
	(char *)htabPtr, argv[2], 0);
}

/*ARGSUSED*/
static int
CloseOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;		/* Not used. */
    int argc;
    char **argv;
{
    Blt_TreeNode node;
    int recurse, result;
    int length;
    register int i;

    recurse = FALSE;
    length = strlen(argv[2]);
    if ((argv[2][0] == '-') && (length > 1) &&
	(strncmp(argv[2], "-recurse", length) == 0)) {
	argv++, argc--;
	recurse = TRUE;
    }
    for (i = 2; i < argc; i++) {
	if (GetNode(htabPtr, argv[i], &node) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (node == NULL) {
	    continue;
	}
	/*
	 * Clear any selected entries that may become hidden by
	 * closing the node.
	 */
	ClearSelections(htabPtr, node);

	/*
	 * -----------------------------------------------------------
	 *
	 *  Check if either the "focus" entry or selection anchor
	 *  is in this hierarchy.  Must move it or disable it before
	 *  we close the node.  Otherwise it may be deleted by a Tcl
	 *  "close" script, and we'll be left pointing to a bogus
	 *  memory location.
	 *
	 * -----------------------------------------------------------
	 */
	if (Blt_TreeIsAncestor(node, htabPtr->focusNode)) {
	    htabPtr->focusNode = node;
	    Blt_SetFocusItem(htabPtr->bindTable, htabPtr->focusNode);
	}
	if (Blt_TreeIsAncestor(node, htabPtr->selAnchorNode)) {
	    htabPtr->selMarkNode = htabPtr->selAnchorNode = NULL;
	}
	if (Blt_TreeIsAncestor(node, htabPtr->activeNode)) {
	    htabPtr->activeNode = node;
	}
	if (recurse) {
	    result = Blt_TreeApply(node, CloseApplyProc, (ClientData)htabPtr);
	} else {
	    result = Blt_HtCloseEntry(htabPtr, GetEntry(htabPtr, node));
	}
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}	
    }
    /* Closing a node may affect the visible entries but not the 
     * the world layout of the entries. */
    htabPtr->flags |= HIERTABLE_SCROLL;
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureOp --
 *
 * 	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or reconfigure)
 *	the widget.
 *
 * Results:
 *	A standard Tcl result.  If TCL_ERROR is returned, then
 *	interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for htabPtr; old resources get freed, if there
 *	were any.  The widget is redisplayed.
 *
 *----------------------------------------------------------------------
 */
static int
ConfigureOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc == 2) {
	return Tk_ConfigureInfo(interp, htabPtr->tkwin, htabPtr->widgetSpecs, 
		(char *)htabPtr, (char *)NULL, 0);
    } else if (argc == 3) {
	return Tk_ConfigureInfo(interp, htabPtr->tkwin, htabPtr->widgetSpecs, 
		(char *)htabPtr, argv[2], 0);
    }
    bltHiertableLastInstance = htabPtr;
    if (Tk_ConfigureWidget(interp, htabPtr->tkwin, htabPtr->widgetSpecs, 
	argc - 2, argv + 2, (char *)htabPtr, TK_CONFIG_ARGV_ONLY) != TCL_OK) {
	return TCL_ERROR;
    }
    Blt_HtConfigure(htabPtr);
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * ColumnCgetOp --
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ColumnCgetOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Column *columnPtr;

    if (GetColumn(htabPtr, argv[3], &columnPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return Tk_ConfigureValue(interp, htabPtr->tkwin, columnSpecs, 
	(char *)columnPtr, argv[4], 0);
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnConfigureOp --
 *
 * 	This procedure is called to process a list of configuration
 *	options database, in order to reconfigure the one of more
 *	entries in the widget.
 *
 *	  .h entryconfigure node node node node option value
 *
 * Results:
 *	A standard Tcl result.  If TCL_ERROR is returned, then
 *	interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for htabPtr; old resources get freed, if there
 *	were any.  The hypertext is redisplayed.
 *
 *----------------------------------------------------------------------
 */
static int
ColumnConfigureOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Column *columnPtr;
    int nOptions, start;
    register int i;

    /* Figure out where the option value pairs begin */
    for(i = 3; i < argc; i++) {
	if (argv[i][0] == '-') {
	    break;
	}
	if (GetColumn(htabPtr, argv[i], &columnPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    start = i;
    nOptions = argc - start;
    
    for (i = 3; i < start; i++) {
	if (GetColumn(htabPtr, argv[i], &columnPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (nOptions == 0) {
	    return Tk_ConfigureInfo(interp, htabPtr->tkwin, columnSpecs, 
		(char *)columnPtr, (char *)NULL, 0);
	} else if (nOptions == 1) {
	    return Tk_ConfigureInfo(interp, htabPtr->tkwin, columnSpecs, 
		(char *)columnPtr, argv[start], 0);
	}
	if (Tk_ConfigureWidget(htabPtr->interp, htabPtr->tkwin, columnSpecs, 
		nOptions, argv + start, (char *)columnPtr, 
		TK_CONFIG_ARGV_ONLY) != TCL_OK) {
	    return TCL_ERROR;
	}
	ConfigureColumn(htabPtr, columnPtr);
    }
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnDeleteOp --
 *
 *----------------------------------------------------------------------
 */
static int
ColumnDeleteOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Column *columnPtr;
    register int i;
    Blt_TreeNode node;
    Entry *entryPtr;

    for(i = 3; i < argc; i++) {
	if (GetColumn(htabPtr, argv[i], &columnPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	/* Traverse the tree deleting column entries as necessary.  */
	for(node = Blt_TreeRootNode(htabPtr->tree); node != NULL;
	    node = Blt_HtNextNode(htabPtr, node, 0)) {
	    entryPtr = GetEntry(htabPtr, node);
	    if (entryPtr != NULL) {
		Blt_ChainLink *linkPtr;
		Field *fieldPtr;
		
		for (linkPtr = Blt_ChainFirstLink(entryPtr->chainPtr); 
		     linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
		    fieldPtr = (Field *)Blt_ChainGetValue(linkPtr);
		    if (fieldPtr->columnPtr == columnPtr) {
			Blt_HtDestroyField(fieldPtr);
			Blt_ChainDeleteLink(entryPtr->chainPtr, linkPtr);
		    }
		}
	    }
	}
	DestroyColumn(htabPtr, columnPtr);
    }
    /* Deleting a column may affect the height of an entry. */
    htabPtr->flags |= (HIERTABLE_LAYOUT | HIERTABLE_DIRTY);
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnInsertOp --
 *
 *	Add new columns to the tree.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ColumnInsertOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int insertPos;
    register int i;
    int nOptions;
    char **options;
    int start;
    Column *columnPtr;
    Blt_ChainLink *beforePtr;
    Blt_TreeNode node;
    Entry *entryPtr;

    if (GetPosition(htabPtr->interp, argv[3], &insertPos) != TCL_OK) {
	return TCL_ERROR;
    }
    if ((insertPos == -1) || 
	(insertPos >= Blt_ChainGetLength(htabPtr->chainPtr))) {
	beforePtr = NULL;
    } else {
	beforePtr =  Blt_ChainGetNthLink(htabPtr->chainPtr, insertPos);
    }
    /*
     * Count the pathnames that follow.  Count the arguments until we
     * spot one that looks like a configuration option (i.e. starts
     * with a minus ("-")).
     */
    for (i = 4; i < argc; i++) {
	if (argv[i][0] == '-') {
	    break;
	}
    }
    start = i;
    nOptions = argc - i;
    options = argv + start;

    for (i = 4; i < start; i++) {
	if (GetColumn(htabPtr, argv[i], &columnPtr) == TCL_OK) {
	    Tcl_AppendResult(interp, "column \"", argv[i], "\" already exists",
		     (char *)NULL);
	    return TCL_ERROR;
	}
	Tcl_ResetResult(interp);
	columnPtr = Blt_HtCreateColumn(htabPtr, argv[i], argv[i], 
		nOptions, options);
	if (columnPtr == NULL) {
	    return TCL_ERROR;
	}
	if (beforePtr == NULL) {
	    columnPtr->linkPtr = 
		Blt_ChainAppend(htabPtr->chainPtr, (ClientData)columnPtr);
	} else {
	    columnPtr->linkPtr = Blt_ChainNewLink();
	    Blt_ChainSetValue(columnPtr->linkPtr, (ClientData)columnPtr);
	    Blt_ChainLinkBefore(htabPtr->chainPtr, columnPtr->linkPtr, 
		beforePtr);
	}
	/* 
	 * Traverse the tree adding column entries where needed.
	 */
	for(node = Blt_TreeRootNode(htabPtr->tree); node != NULL;
	    node = Blt_HtNextNode(htabPtr, node, 0)) {
	    entryPtr = GetEntry(htabPtr, node);
	    Blt_HtAddField(entryPtr, columnPtr);
	}
    }
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnMoveOp --
 *
 *	Move a column.
 *
 * .h column move field1 position
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 *
 * ColumnNamesOp --
 *
 *----------------------------------------------------------------------
 */
static int
ColumnNamesOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Blt_ChainLink *linkPtr;
    Column *columnPtr;

    for(linkPtr = Blt_ChainFirstLink(htabPtr->chainPtr); linkPtr != NULL;
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	columnPtr = (Column *)Blt_ChainGetValue(linkPtr);
	Tcl_AppendElement(interp, columnPtr->atom);
    }
    return TCL_OK;
}


static Blt_OpSpec columnOps[] =
{
    {"cget", 2, (Blt_OpProc)ColumnCgetOp, 4, 4, "field option",},
    {"configure", 2, (Blt_OpProc)ColumnConfigureOp, 3, 0, 
	"field ?option value?...",},
    {"delete", 1, (Blt_OpProc)ColumnDeleteOp, 3, 0, "field ?field...?",},
    {"insert", 1, (Blt_OpProc)ColumnInsertOp, 5, 0, 
	"position field ?field...? ?option value?...",},
    {"names", 1, (Blt_OpProc)ColumnNamesOp, 3, 3, "",},
};
static int nColumnOps = sizeof(columnOps) / sizeof(Blt_OpSpec);

/*
 *----------------------------------------------------------------------
 *
 * ColumnOp --
 *
 *----------------------------------------------------------------------
 */
static int
ColumnOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Blt_OpProc opProc;
    int result;

    opProc = Blt_GetOperation(interp, nColumnOps, columnOps, BLT_OPER_ARG2, 
	argc, argv);
    if (opProc == NULL) {
	return TCL_ERROR;
    }
    result = (*opProc) (htabPtr, interp, argc, argv);
    return result;
}

/*ARGSUSED*/
static int
CurselectionOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;		/* Not used. */
    int argc;			/* Not used. */
    char **argv;		/* Not used. */
{
    Blt_TreeNode node;
    Entry *entryPtr;

    for (node = htabPtr->rootNode; node != NULL; 
	 node = Blt_HtNextNode(htabPtr, node, ENTRY_OPEN | ENTRY_MAPPED)) {
	entryPtr = GetEntry(htabPtr, node);
	if (entryPtr->flags & ENTRY_SELECTED) {
	    Tcl_AppendElement(htabPtr->interp, Blt_HtNodeToString(node));
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * BindOp --
 *
 *	  .t bind index sequence command
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
BindOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    ClientData object;
    int inode;

    /*
     * Individual entries are selected by inode only.  All other strings
     * are interpreted as a binding tag.
     */
    if (Tcl_GetInt(interp, argv[2], &inode) != TCL_OK) {
	Tcl_ResetResult(interp);
	object = (ClientData)Blt_HtGetUid(htabPtr, argv[2]);
    } else {
	Blt_TreeNode node;

	node = Blt_TreeGetNode(htabPtr->tree, inode);
	object = (ClientData)node;
    }
    return Blt_ConfigureBindings(interp, htabPtr->bindTable, object, argc - 3, 
	argv + 3);
}

/*
 *----------------------------------------------------------------------
 *
 * EntryActivateOp --
 *
 *	Selects the entry to appear active.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
EntryActivateOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node, old;
    Entry *entryPtr;

    if (argv[3][0] == '\0') {
	node = NULL;
    } else if (GetNode(htabPtr, argv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    if (htabPtr->hierColumnPtr->hidden) {
	return TCL_OK;
    }
    old = htabPtr->activeNode;
    htabPtr->activeNode = node;
    if (node != old) {
	if (htabPtr->flags & HIERTABLE_DIRTY) {
	    Blt_HtEventuallyRedraw(htabPtr);
	} else {
	    Drawable drawable;
	    int x, y;

	    drawable = Tk_WindowId(htabPtr->tkwin);
	    if (old != NULL) {
		entryPtr = GetEntry(htabPtr, old);
		x = SCREENX(htabPtr, entryPtr->worldX) +
		    ICONWIDTH(DEPTH(htabPtr, old));
		y = SCREENY(htabPtr, entryPtr->worldY);
		entryPtr->flags |= ENTRY_ICON;
		Blt_HtDrawIcon(htabPtr, entryPtr, x, y, drawable);
	    }
	    if (node != NULL) {
		entryPtr = GetEntry(htabPtr, node);
		x = SCREENX(htabPtr, entryPtr->worldX) + 
		    ICONWIDTH(DEPTH(htabPtr, node));
		y = SCREENY(htabPtr, entryPtr->worldY);
		entryPtr->flags |= ENTRY_ICON;
		Blt_HtDrawIcon(htabPtr, entryPtr, x, y, drawable);
	    }
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * EntryCgetOp --
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
EntryCgetOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node;
    Entry *entryPtr;
    int result;

    if (StringToNode(htabPtr, argv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    entryPtr = GetEntry(htabPtr, node);
    result = Tk_ConfigureValue(interp, htabPtr->tkwin, 
	htabPtr->entrySpecs, (char *)entryPtr, argv[4], 0);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * EntryConfigureOp --
 *
 * 	This procedure is called to process a list of configuration
 *	options database, in order to reconfigure the one of more
 *	entries in the widget.
 *
 *	  .h entryconfigure node node node node option value
 *
 * Results:
 *	A standard Tcl result.  If TCL_ERROR is returned, then
 *	interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for htabPtr; old resources get freed, if there
 *	were any.  The hypertext is redisplayed.
 *
 *----------------------------------------------------------------------
 */
static int
EntryConfigureOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int nIds, nOpts;
    char **options;
    register int i;
    Blt_TreeNode node;
    Entry *entryPtr;
    int result;

    /* Figure out where the option value pairs begin */
    argc -= 3, argv += 3;
    for (i = 0; i < argc; i++) {
	if (argv[i][0] == '-') {
	    break;
	}
	if (StringToNode(htabPtr, argv[i], &node) != TCL_OK) {
	    return TCL_ERROR;	/* Can't find node. */
	}
    }
    nIds = i;			/* Number of element names specified */
    nOpts = argc - i;		/* Number of options specified */
    options = argv + i;		/* Start of options in argv  */

    for (i = 0; i < nIds; i++) {
	StringToNode(htabPtr, argv[i], &node);
	entryPtr = GetEntry(htabPtr, node);
	if (nOpts == 0) {
	    return Tk_ConfigureInfo(interp, htabPtr->tkwin, 
		htabPtr->entrySpecs, (char *)entryPtr, (char *)NULL, 0);
	} else if (nOpts == 1) {
	    return Tk_ConfigureInfo(interp, htabPtr->tkwin, 
		htabPtr->entrySpecs, (char *)entryPtr, options[0], 0);
	}
	bltHiertableLastInstance = htabPtr;
	result = Tk_ConfigureWidget(htabPtr->interp, htabPtr->tkwin, 
		htabPtr->entrySpecs, nOpts, options, (char *)entryPtr, 
		TK_CONFIG_ARGV_ONLY);
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}
	Blt_HtConfigureEntry(htabPtr, entryPtr);
    }
    htabPtr->flags |= (HIERTABLE_DIRTY | HIERTABLE_LAYOUT | HIERTABLE_SCROLL);
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * EntryIsHiddenOp --
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
EntryIsHiddenOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node;
    Entry *entryPtr;

    if (StringToNode(htabPtr, argv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    entryPtr = GetEntry(htabPtr, node);
    Tcl_SetResult(interp, (entryPtr->flags & ENTRY_MAPPED) ? "0" : "1", 
	TCL_STATIC);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * EntryIsOpenOp --
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
EntryIsOpenOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node;
    Entry *entryPtr;

    if (StringToNode(htabPtr, argv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    entryPtr = GetEntry(htabPtr, node);
    Tcl_SetResult(interp, (entryPtr->flags & ENTRY_OPEN) ? "1" : "0", 
	TCL_STATIC);
    return TCL_OK;
}

/*ARGSUSED*/
static int
EntryChildrenOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Blt_TreeNode parent, node;

    if (StringToNode(htabPtr, argv[3], &parent) != TCL_OK) {
	return TCL_ERROR;
    }
    if (argc == 4) {
	for (node = Blt_TreeFirstChild(parent); node != NULL;
	    node = Blt_TreeNextSibling(node)) {
	    Tcl_AppendElement(interp, Blt_HtNodeToString(node));
	}
    } else if (argc == 6) {
	Blt_TreeNode first, last;
	int firstPos, lastPos;
	int nNodes;

	if ((GetPosition(interp, argv[4], &firstPos) != TCL_OK) ||
	    (GetPosition(interp, argv[5], &lastPos) != TCL_OK)) {
	    return TCL_ERROR;
	}
	nNodes = Blt_TreeNodeChildren(parent);
	if (nNodes == 0) {
	    return TCL_OK;
	}
	if (lastPos == END) {
	    last = Blt_TreeLastChild(parent);
	} else {
	    if (lastPos >= nNodes) {
		lastPos = nNodes - 1;
	    }
	    last = GetNthNode(parent, lastPos);
	}
	if (firstPos == END) {
	    first = Blt_TreeLastChild(parent);
	} else {
	    if (firstPos >= nNodes) {
		firstPos = nNodes - 1;
	    }
	    first = GetNthNode(parent, firstPos);
	}
	if (firstPos > lastPos) {
	    for (/*empty*/; last != NULL; last = PrevNode(htabPtr, last, 0)) {
		Tcl_AppendElement(interp, Blt_HtNodeToString(last));
		if (last == first) {
		    break;
		}
	    }
	} else {
	    for (/*empty*/; first != NULL; 
		  first = Blt_HtNextNode(htabPtr, first, 0)) {
		Tcl_AppendElement(interp, Blt_HtNodeToString(first));
		if (first == last) {
		    break;
		}
	    }
	}
    } else {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " ",
	    argv[1], " ", argv[2], " index ?first last?", (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * EntryDeleteOp --
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
EntryDeleteOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node, first, last, next;
    int firstPos, lastPos;
    int nEntries;

    if (StringToNode(htabPtr, argv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    if (argc == 5) {
	/*
	 * Delete a single node from hierarchy specified by its
	 * numeric position.
	 */
	if (GetPosition(interp, argv[3], &firstPos) != TCL_OK) {
	    return TCL_ERROR;
	}
	if ((firstPos != END) && (firstPos >= Blt_TreeNodeChildren(node))) {
	    return TCL_OK;	/* Bad first index */
	}
	if (firstPos == END) {
	    node = Blt_TreeLastChild(node);
	} else {
	    node = GetNthNode(node, firstPos);
	}
	first = last = node;
	for (node = first; node != NULL; node = next) {
	    next = Blt_TreeNextSibling(node);
	    Blt_TreeDeleteNode(htabPtr->tree, node);
	    if (node == last) {
		break;
	    }
	}
    } else {
	/*
	 * Delete range of nodes in hierarchy specified by first/last
	 * positions.
	 */
	if ((GetPosition(interp, argv[4], &firstPos) != TCL_OK) ||
	    (GetPosition(interp, argv[5], &lastPos) != TCL_OK)) {
	    return TCL_ERROR;
	}
	nEntries = Blt_TreeNodeChildren(node);
	if (nEntries == 0) {
	    return TCL_OK;
	}
	if ((firstPos == END) || (firstPos >= nEntries)) {
	    firstPos = nEntries - 1;
	}
	if ((lastPos == END) || (lastPos >= nEntries)) {
	    lastPos = nEntries - 1;
	}
	if (firstPos > lastPos) {
	    Tcl_AppendResult(interp, "bad range: \"", argv[4], " > ", argv[5],
			     "\"", (char *)NULL);
	    return TCL_ERROR;
	}
	first = GetNthNode(node, firstPos);
	last = GetNthNode(node, lastPos);
	for (node = first; node != NULL; node = next) {
	    next = Blt_TreeNextSibling(node);
	    Blt_TreeDeleteNode(htabPtr->tree, node);
	    if (node == last) {
		break;
	    }
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SizeApplyProc --
 *
 *	Returns the number of children at the given node. The sum
 *	is passed via the clientData field of the hierarchy widget.
 *
 * Results:
 *	Always TCL_OK.
 *
 *----------------------------------------------------------------------
 */
static int
SizeApplyProc(node, clientData)
    Blt_TreeNode node;
    ClientData clientData;
{
    int *sumPtr = (int *)clientData;

    *sumPtr = *sumPtr + 1;
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * SizeOpOp --
 *
 *	Counts the number of entries at this node.
 *
 * Results:
 *	A standard Tcl result.  If an error occurred TCL_ERROR is
 *	returned and interp->result will contain an error message.
 *	Otherwise, TCL_OK is returned and interp->result contains
 *	the number of entries.
 *
 *----------------------------------------------------------------------
 */
static int
EntrySizeOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int length;
    Blt_TreeNode node;
    int sum;
    int recurse;

    recurse = FALSE;
    length = strlen(argv[3]);
    if ((argv[3][0] == '-') && (length > 1) &&
	(strncmp(argv[3], "-recurse", length) == 0)) {
	argv++, argc--;
	recurse = TRUE;
    }
    if (argc == 3) {
	Tcl_AppendResult(interp, "missing node argument: should be \"",
	    argv[0], " entry open node\"", (char *)NULL);
	return TCL_ERROR;
    }
    if (StringToNode(htabPtr, argv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    if (recurse) {
	sum = 0;
	Blt_TreeApply(node, SizeApplyProc, (ClientData)&sum);
    } else {
	sum = Blt_TreeNodeChildren(node);
    }
    Tcl_SetResult(interp, Blt_Itoa(sum), TCL_VOLATILE);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * EntryOp --
 *
 *	This procedure handles entry operations.
 *
 * Results:
 *	A standard Tcl result.
 *
 *----------------------------------------------------------------------
 */

static Blt_OpSpec entryOps[] =
{
    {"activate", 1, (Blt_OpProc)EntryActivateOp, 4, 4, "index",},
    /*bbox*/
    /*bind*/
    {"cget", 2, (Blt_OpProc)EntryCgetOp, 5, 5, "index option",},
    {"children", 2, (Blt_OpProc)EntryChildrenOp, 4, 6, 
	"index firstPos lastPos",},
    /*close*/
    {"configure", 2, (Blt_OpProc)EntryConfigureOp, 4, 0,
	"index ?index...? ?option value?...",},
    {"delete", 2, (Blt_OpProc)EntryDeleteOp, 5, 6, "index firstPos ?lastPos?",},
    /*focus*/
    /*hide*/
    /*index*/
    {"ishidden", 3, (Blt_OpProc)EntryIsHiddenOp, 4, 4, "index",},
    {"isopen", 3, (Blt_OpProc)EntryIsOpenOp, 4, 4, "index",},
    /*move*/
    /*nearest*/
    /*open*/
    /*see*/
    /*show*/
    {"size", 1, (Blt_OpProc)EntrySizeOp, 4, 5, "?-recurse? index",},
    /*toggle*/
};
static int nEntryOps = sizeof(entryOps) / sizeof(Blt_OpSpec);

static int
EntryOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Blt_OpProc opProc;
    int result;

    opProc = Blt_GetOperation(interp, nEntryOps, entryOps, BLT_OPER_ARG2, 
	argc, argv);
    if (opProc == NULL) {
	return TCL_ERROR;
    }
    result = (*opProc) (htabPtr, interp, argc, argv);
    return result;
}

/*ARGSUSED*/
static int
ExactCompare(interp, name, pattern)
    Tcl_Interp *interp;		/* Not used. */
    char *name;
    char *pattern;
{
    return (strcmp(name, pattern) == 0);
}

/*ARGSUSED*/
static int
GlobCompare(interp, name, pattern)
    Tcl_Interp *interp;		/* Not used. */
    char *name;
    char *pattern;
{
    return Tcl_StringMatch(name, pattern);
}

static int
RegexpCompare(interp, name, pattern)
    Tcl_Interp *interp;
    char *name;
    char *pattern;
{
    return Tcl_RegExpMatch(interp, name, pattern);
}

/*
 *----------------------------------------------------------------------
 *
 * FindOp --
 *
 *	Find one or more nodes based upon the pattern provided.
 *
 * Results:
 *	A standard Tcl result.  The interpreter result will contain a
 *	list of the node serial identifiers.
 *
 *----------------------------------------------------------------------
 */
static int
FindOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    register Blt_TreeNode node;
    Blt_TreeNode first, last;
    int nMatches, maxMatches;
    char c;
    int length;
    CompareProc *compareProc;
    IterProc *nextProc;
    int invertMatch;		/* normal search mode (matching entries) */
    char *namePattern, *fullPattern;
    char *execCmd;
    register int i;
    int result;
    char *pattern, *option, *value;
    Tcl_DString dString, pathString;
    Blt_List configOptList;
    register Blt_ListEntry lEntry;
    Entry *entryPtr;

    invertMatch = FALSE;
    maxMatches = 0;
    execCmd = namePattern = fullPattern = NULL;
    compareProc = ExactCompare;
    nextProc = Blt_HtNextNode;
    Blt_InitList(&configOptList, TCL_STRING_KEYS);

    entryPtr = GetEntry(htabPtr, htabPtr->rootNode);
    /*
     * Step 1:  Process flags for find operation.
     */
    for (i = 2; i < argc; i++) {
	if (argv[i][0] != '-') {
	    break;
	}
	option = argv[i] + 1;
	length = strlen(option);
	c = option[0];
	if ((c == 'e') && (length > 2) &&
	    (strncmp(option, "exact", length) == 0)) {
	    compareProc = ExactCompare;
	} else if ((c == 'g') && (strncmp(option, "glob", length) == 0)) {
	    compareProc = GlobCompare;
	} else if ((c == 'r') && (strncmp(option, "regexp", length) == 0)) {
	    compareProc = RegexpCompare;
	} else if ((c == 'n') && (length > 1) &&
	    (strncmp(option, "nonmatching", length) == 0)) {
	    invertMatch = TRUE;
	} else if ((c == 'n') && (length > 1) &&
	    (strncmp(option, "name", length) == 0)) {
	    if ((i + 1) == argc) {
		goto missingArg;
	    }
	    i++;
	    namePattern = argv[i];
	} else if ((c == 'f') && (strncmp(option, "full", length) == 0)) {
	    if ((i + 1) == argc) {
		goto missingArg;
	    }
	    i++;
	    fullPattern = argv[i];
	} else if ((c == 'e') && (length > 2) &&
	    (strncmp(option, "exec", length) == 0)) {
	    if ((i + 1) == argc) {
		goto missingArg;
	    }
	    i++;
	    execCmd = argv[i];
	} else if ((c == 'c') && (strncmp(option, "count", length) == 0)) {
	    if ((i + 1) == argc) {
		goto missingArg;
	    }
	    i++;
	    if (Tcl_GetInt(interp, argv[i], &maxMatches) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (maxMatches < 0) {
		Tcl_AppendResult(interp, "bad match count \"", argv[i],
		    "\": should be a positive number", (char *)NULL);
		Blt_ListReset(&configOptList);
		return TCL_ERROR;
	    }
	} else if ((option[0] == '-') && (option[1] == '\0')) {
	    break;
	} else {
	    /*
	     * Verify that the switch is actually an entry configuration
	     * option.
	     */
	    if (Tk_ConfigureValue(interp, htabPtr->tkwin, htabPtr->entrySpecs,
		 (char *)entryPtr, argv[i], 0) != TCL_OK) {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "bad find switch \"", argv[i], "\"",
		    (char *)NULL);
		Blt_ListReset(&configOptList);
		return TCL_ERROR;
	    }
	    if ((i + 1) == argc) {
		goto missingArg;
	    }
	    /* Save the option in the list of configuration options */
	    lEntry = Blt_ListFind(&configOptList, argv[i]);
	    if (lEntry == NULL) {
		lEntry = Blt_ListNewEntry(&configOptList, argv[i]);
		Blt_ListAppendEntry(&configOptList, lEntry);
	    }
	    Blt_ListSetValue(lEntry, (ClientData)argv[i + 1]);
	    i++;
	}
    }

    if ((argc - i) > 2) {
	Blt_ListReset(&configOptList);
	Tcl_AppendResult(interp, "too many args", (char *)NULL);
	return TCL_ERROR;
    }
    /*
     * Step 2:  Find the range of the search.  Check the order of two
     *		nodes and arrange the search accordingly.
     *
     *	Note:	Be careful to treat "end" as the end of all nodes, instead
     *		of the end of visible nodes.  That way, we can search the
     *		entire tree, even if the last folder is closed.
     */
    first = htabPtr->rootNode;	/* Default to root node */
    last = EndNode(htabPtr, first, 0);

    if (i < argc) {
	if ((argv[i][0] == 'e') && (strcmp(argv[i], "end") == 0)) {
	    first = EndNode(htabPtr, htabPtr->rootNode, 0);
	} else if (StringToNode(htabPtr, argv[i], &first) != TCL_OK) {
	    return TCL_ERROR;
	}
	i++;
    }
    if (i < argc) {
	if ((argv[i][0] == 'e') && (strcmp(argv[i], "end") == 0)) {
	    last = EndNode(htabPtr, htabPtr->rootNode, 0);
	} else if (StringToNode(htabPtr, argv[i], &last) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    if (Blt_TreeIsBefore(last, first)) {
	nextProc = PrevNode;
    }
    nMatches = 0;

    /*
     * Step 3:	Search through the tree and look for nodes that match the
     *		current pattern specifications.  Save the name of each of
     *		the matching nodes.
     */
    Tcl_DStringInit(&dString);
    for (node = first; node != NULL; 
	 node = (*nextProc) (htabPtr, node, 0)) {
	if (namePattern != NULL) {
	    result = (*compareProc) (interp, Blt_TreeNodeName(node), 
			     namePattern);
	    if (result == invertMatch) {
		goto nextNode;	/* Failed to match */
	    }
	}
	if (fullPattern != NULL) {
	    Tcl_DStringInit(&pathString);
	    Blt_HtGetFullPath(htabPtr, node, &pathString);
	    result = (*compareProc) (interp, Tcl_DStringValue(&pathString),
		fullPattern);
	    if (result == invertMatch) {
		goto nextNode;	/* Failed to match */
	    }
	}
	entryPtr = GetEntry(htabPtr, node);
	for (lEntry = Blt_ListFirstEntry(&configOptList); lEntry != NULL;
	    lEntry = Blt_ListNextEntry(lEntry)) {
	    option = Blt_ListGetKey(lEntry);
	    Tcl_ResetResult(interp);
	    if (Tk_ConfigureValue(interp, htabPtr->tkwin, htabPtr->entrySpecs,
		    (char *)entryPtr, option, 0) != TCL_OK) {
		goto error;	/* This shouldn't happen. */
	    }
	    pattern = (char *)Blt_ListGetValue(lEntry);
	    value = Tcl_GetStringResult(interp);
	    result = (*compareProc) (interp, value, pattern);
	    if (result == invertMatch) {
		goto nextNode;	/* Failed to match */
	    }
	}
	/* 
	 * Someone may actually delete the current node in the "exec"
	 * callback.  Preserve the entry.
	 */
	Tcl_Preserve(entryPtr);
	if (execCmd != NULL) {
	    Tcl_DString cmdString;

	    Blt_HtPercentSubst(htabPtr, node, execCmd, &cmdString);
	    result = Tcl_GlobalEval(interp, Tcl_DStringValue(&cmdString));
	    Tcl_DStringFree(&cmdString);
	    if (result != TCL_OK) {
		Tcl_Release(node);
		goto error;
	    }
	}
	if (entryPtr->node != NULL) {
	    /* Finally, save the matching node name. */
	    Tcl_DStringAppendElement(&dString, Blt_HtNodeToString(node));
	}
	Tcl_Release(entryPtr);
	nMatches++;
	if ((nMatches == maxMatches) && (maxMatches > 0)) {
	    break;
	}
      nextNode:
	if (node == last) {
	    break;
	}
    }
    Tcl_ResetResult(interp);
    Blt_ListReset(&configOptList);
    Tcl_DStringResult(interp, &dString);
    return TCL_OK;

  missingArg:
    Tcl_AppendResult(interp, "missing argument for find option \"", argv[i],
	"\"", (char *)NULL);
  error:
    Tcl_DStringFree(&dString);
    Blt_ListReset(&configOptList);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * GetOp --
 *
 *	Converts one or more node identifiers to its path component.
 *	The path may be either the single entry name or the full path
 *	of the entry.
 *
 * Results:
 *	A standard Tcl result.  The interpreter result will contain a
 *	list of the convert names.
 *
 *----------------------------------------------------------------------
 */
static int
GetOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int fullName;
    Blt_TreeNode node;
    Tcl_DString dString;
    Tcl_DString pathString;
    register int i;

    fullName = FALSE;
    if ((argc > 2) && (argv[2][0] == '-') && (strcmp(argv[2], "-full") == 0)) {
	fullName = TRUE;
	argv++, argc--;
    }
    Tcl_DStringInit(&dString);
    Tcl_DStringInit(&pathString);
    for (i = 2; i < argc; i++) {
	if (GetNode(htabPtr, argv[i], &node) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (node == NULL) {
	    Tcl_DStringAppendElement(&dString, "");
	    continue;
	}
	if (fullName) {
	    Blt_HtGetFullPath(htabPtr, node, &pathString);
	    Tcl_DStringAppendElement(&dString, Tcl_DStringValue(&pathString));
	} else {
	    Tcl_DStringAppendElement(&dString, Blt_TreeNodeName(node));
	}
    }
    Tcl_DStringFree(&pathString);
    Tcl_DStringResult(interp, &dString);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SearchAndApplyToTree --
 *
 *	Searches through the current tree and applies a procedure
 *	to matching nodes.  The search specification is taken from
 *	the following command-line arguments:
 *
 *      ?-exact? ?-glob? ?-regexp? ?-nonmatching?
 *      ?-data string?
 *      ?-name string?
 *      ?-full string?
 *      ?--?
 *      ?inode...?
 *
 * Results:
 *	A standard Tcl result.  If the result is valid, and if the
 *      nonmatchPtr is specified, it returns a boolean value
 *      indicating whether or not the search was inverted.  This
 *      is needed to fix things properly for the "hide nonmatching"
 *      case.
 *
 *----------------------------------------------------------------------
 */
static int
SearchAndApplyToTree(htabPtr, interp, argc, argv, proc, nonMatchPtr)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
    Blt_TreeApplyProc *proc;
    int *nonMatchPtr;		/* returns: inverted search indicator */
{
    CompareProc *compareProc;
    int invertMatch;		/* normal search mode (matching entries) */
    char *namePattern, *fullPattern;
    register int i;
    int length;
    int result;
    char *option, *pattern, *value;
    Blt_TreeNode node;
    char c;
    Blt_List configOpts;
    Entry *entryPtr;
    register Blt_ListEntry lEntry;

    Blt_InitList(&configOpts, TCL_STRING_KEYS);
    invertMatch = FALSE;
    namePattern = fullPattern = NULL;
    compareProc = ExactCompare;

    entryPtr = GetEntry(htabPtr, htabPtr->rootNode);
    for (i = 2; i < argc; i++) {
	if (argv[i][0] != '-') {
	    break;
	}
	option = argv[i] + 1;
	length = strlen(option);
	c = option[0];
	if ((c == 'e') && (strncmp(option, "exact", length) == 0)) {
	    compareProc = ExactCompare;
	} else if ((c == 'g') && (strncmp(option, "glob", length) == 0)) {
	    compareProc = GlobCompare;
	} else if ((c == 'r') && (strncmp(option, "regexp", length) == 0)) {
	    compareProc = RegexpCompare;
	} else if ((c == 'n') && (length > 1) &&
	    (strncmp(option, "nonmatching", length) == 0)) {
	    invertMatch = TRUE;
	} else if ((c == 'f') && (strncmp(option, "full", length) == 0)) {
	    if ((i + 1) == argc) {
		goto missingArg;
	    }
	    i++;
	    fullPattern = argv[i];
	} else if ((c == 'n') && (length > 1) &&
	    (strncmp(option, "name", length) == 0)) {
	    if ((i + 1) == argc) {
		goto missingArg;
	    }
	    i++;
	    namePattern = argv[i];
	} else if ((option[0] == '-') && (option[1] == '\0')) {
	    break;
	} else {
	    /*
	     * Verify that the switch is actually an entry configuration option.
	     */
	    if (Tk_ConfigureValue(interp, htabPtr->tkwin, htabPtr->entrySpecs,
		    (char *)entryPtr, argv[i], 0) != TCL_OK) {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "bad switch \"", argv[i],
	    "\": must be -exact, -glob, -regexp, -name, -full, or -nonmatching",
		    (char *)NULL);
		return TCL_ERROR;
	    }
	    if ((i + 1) == argc) {
		goto missingArg;
	    }
	    /* Save the option in the list of configuration options */
	    lEntry = Blt_ListFind(&configOpts, argv[i]);
	    if (lEntry == NULL) {
		lEntry = Blt_ListNewEntry(&configOpts, argv[i]);
		Blt_ListAppendEntry(&configOpts, lEntry);
	    }
	    Blt_ListSetValue(lEntry, (ClientData)argv[i + 1]);
	}
    }

    if ((namePattern != NULL) || (fullPattern != NULL) ||
	(Blt_ListGetLength(&configOpts) > 0)) {
	/*
	 * Search through the tree and look for nodes that match the
	 * current spec.  Apply the input procedure to each of the
	 * matching nodes.
	 */
	for (node = htabPtr->rootNode; node != NULL; 
	     node = Blt_HtNextNode(htabPtr, node, 0)) {
	    if (namePattern != NULL) {
		result = (*compareProc) (interp, Blt_TreeNodeName(node), 
			namePattern);
		if (result == invertMatch) {
		    continue;	/* Failed to match */
		}
	    }
	    if (fullPattern != NULL) {
		Tcl_DString dString;

		Tcl_DStringInit(&dString);
		Blt_HtGetFullPath(htabPtr, node, &dString);
		result = (*compareProc) (interp, Tcl_DStringValue(&dString),
		    fullPattern);
		Tcl_DStringFree(&dString);
		if (result == invertMatch) {
		    continue;	/* Failed to match */
		}
	    }
	    for (lEntry = Blt_ListFirstEntry(&configOpts); lEntry != NULL;
		lEntry = Blt_ListNextEntry(lEntry)) {
		option = Blt_ListGetKey(lEntry);
		Tcl_ResetResult(interp);
		entryPtr = GetEntry(htabPtr, node);
		if (Tk_ConfigureValue(interp, htabPtr->tkwin, 
			htabPtr->entrySpecs, (char *)entryPtr, 
			option, 0) != TCL_OK) {
		    return TCL_ERROR;	/* This shouldn't happen. */
		}
		pattern = (char *)Blt_ListGetValue(lEntry);
		value = Tcl_GetStringResult(interp);
		result = (*compareProc) (interp, value, pattern);
		if (result == invertMatch) {
		    continue;	/* Failed to match */
		}
	    }
	    /* Finally, apply the procedure to the node */
	    (*proc) (node, (ClientData)htabPtr);
	}
	Tcl_ResetResult(interp);
	Blt_ListReset(&configOpts);
    }
    /*
     * Apply the procedure to nodes that have been specified
     * individually.
     */
    for ( /*empty*/ ; i < argc; i++) {
	if ((argv[i][0] == 'a') && (strcmp(argv[i], "all") == 0)) {
	    return Blt_TreeApply(htabPtr->rootNode, proc, (ClientData)htabPtr);
	}
	if (StringToNode(htabPtr, argv[i], &node) != TCL_OK) {
	    return TCL_ERROR;
	}
	if ((*proc) (node, (ClientData)htabPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
    }

    if (nonMatchPtr != NULL) {
	*nonMatchPtr = invertMatch;	/* return "inverted search" status */
    }
    return TCL_OK;

  missingArg:
    Blt_ListReset(&configOpts);
    Tcl_AppendResult(interp, "missing pattern for search option \"", argv[i],
	"\"", (char *)NULL);
    return TCL_ERROR;

}


static int
FixUnmappedSelectionsApplyProc(node, clientData)
    Blt_TreeNode node;
    ClientData clientData;
{
    Hiertable *htabPtr = (Hiertable *)clientData;
    Entry *entryPtr;

    entryPtr = GetEntry(htabPtr, node);
    if (entryPtr->flags & ENTRY_MAPPED) {
	entryPtr->flags &= ~ENTRY_SELECTED;
	if (Blt_TreeIsAncestor(node, htabPtr->focusNode)) {
	    htabPtr->focusNode = Blt_TreeNodeParent(node);
	    if (htabPtr->focusNode == NULL) {
		htabPtr->focusNode = htabPtr->rootNode;
	    }
	    Blt_SetFocusItem(htabPtr->bindTable, htabPtr->focusNode);
	}
	if (Blt_TreeIsAncestor(node, htabPtr->selAnchorNode)) {
	    htabPtr->selMarkNode = htabPtr->selAnchorNode = NULL;
	}
	ClearSelections(htabPtr, node);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * HideOp --
 *
 *	Hides one or more nodes.  Nodes can be specified by their
 *      inode, or by matching a name or data value pattern.  By
 *      default, the patterns are matched exactly.  They can also
 *      be matched using glob-style and regular expression rules.
 *
 * Results:
 *	A standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
static int
HideOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int status, nonmatching;

    status = SearchAndApplyToTree(htabPtr, interp, argc, argv, UnmapApplyProc, 
	&nonmatching);

    if (status != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     * If this was an inverted search, scan back through the
     * tree and make sure that the parents for all visible
     * nodes are also visible.  After all, if a node is supposed
     * to be visible, its parent can't be hidden.
     */
    if (nonmatching) {
	Blt_TreeApply(htabPtr->rootNode, MapAncestorsApplyProc, 
		(ClientData)htabPtr);
    }
    /*
     * Make sure that selections arme cleared from any hidden
     * nodes.  This wasn't done earlier--we had to delay it until
     * we fixed the visibility status for the parents.
     */
    Blt_TreeApply(htabPtr->rootNode, FixUnmappedSelectionsApplyProc, 
	(ClientData)htabPtr);

    /* Hiding an entry only effects the visible nodes. */
    htabPtr->flags |= (HIERTABLE_LAYOUT | HIERTABLE_SCROLL);
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ShowOp --
 *
 *	Mark one or more nodes to be exposed.  Nodes can be specified
 *	by their inode, or by matching a name or data value pattern.  By
 *      default, the patterns are matched exactly.  They can also
 *      be matched using glob-style and regular expression rules.
 *
 * Results:
 *	A standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
static int
ShowOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (SearchAndApplyToTree(htabPtr, interp, argc, argv, MapApplyProc,
	    (int *)NULL) != TCL_OK) {
	return TCL_ERROR;
    }
    htabPtr->flags |= (HIERTABLE_LAYOUT | HIERTABLE_SCROLL);
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * IndexOp --
 *
 *	Converts one of more words representing indices of the entries
 *	in the hierarchy widget to their respective serial identifiers.
 *
 * Results:
 *	A standard Tcl result.  Interp->result will contain the
 *	identifier of each inode found. If an inode could not be found,
 *	then the serial identifier will be the empty string.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
IndexOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node;

    node = htabPtr->focusNode;
    if ((argv[2][0] == '-') && (strcmp(argv[2], "-at") == 0)) {
	if (StringToNode(htabPtr, argv[3], &node) != TCL_OK) {
	    return TCL_ERROR;
	}
	argv += 2, argc -= 2;
    }
    if (argc > 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	    " index ?-at index? index\"", (char *)NULL);
	return TCL_ERROR;
    }
    if ((GetNode2(htabPtr, argv[2], &node) == TCL_OK) && (node != NULL)) {
	Tcl_SetResult(interp, Blt_HtNodeToString(node), TCL_VOLATILE);
    } else {
	Tcl_SetResult(interp, "", TCL_STATIC);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * InsertOp --
 *
 *	Add new entries into a hierarchy.  If no node is specified,
 *	new entries will be added to the root of the hierarchy.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
InsertOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode root, node, parent;
    int insertPos;
    int depth, count;
    char *path, **options;
    Tcl_DString dString;
    char **compArr;
    register char **p;
    register int n;

    root = htabPtr->rootNode;
    if ((argv[2][0] == '-') && (strcmp(argv[2], "-at") == 0)) {
	if (StringToNode(htabPtr, argv[3], &root) != TCL_OK) {
	    return TCL_ERROR;
	}
	argv += 2, argc -= 2;
    }
    if (GetPosition(htabPtr->interp, argv[2], &insertPos) != TCL_OK) {
	return TCL_ERROR;
    }
    argc -= 3, argv += 3;
    Tcl_DStringInit(&dString);
    while (argc > 0) {
	path = argv[0];
	argv++, argc--;
	/*
	 * Count the option-value pairs that follow.  Count until we
	 * spot one that looks like an entry name (i.e. doesn't start
	 * with a minus "-").
	 */
	for (count = 0; count < argc; count += 2) {
	    if (argv[count][0] != '-') {
		break;
	    }
	}
	if (count > argc) {
	    count = argc;
	}
	options = argv;
	argc -= count, argv += count;

	if (htabPtr->trimLeft != NULL) {
	    register char *s1, *s2;

	    /* Trim off leading character string if one exists. */
	    for (s1 = path, s2 = htabPtr->trimLeft; *s2 != '\0'; s2++, s1++) {
		if (*s1 != *s2) {
		    break;
		}
	    }
	    if (*s2 == '\0') {
		path = s1;
	    }
	}
	/*
	 * Split the path and find the parent node of the path.
	 */
	compArr = &path;
	depth = 1;
	if (SplitPath(htabPtr, path, &depth, &compArr) != TCL_OK) {
	    goto error;
	}
	if (depth == 0) {
	    continue;		/* Root already exists. */
	}
	parent = root;
	depth--;		

	/* Verify the path preceding the tail component.  */
	for (n = 0, p = compArr; n < depth; n++, p++) {
	    node = Blt_TreeFindChild(parent, *p);
	    if (node == NULL) {
		if (!htabPtr->autoFill) {
		    Tcl_AppendResult(interp, "can't find path component \"",
		         *p, "\" in \"", path, "\"", (char *)NULL);
		    goto error;
		}
		node = Blt_TreeCreateNode(htabPtr->tree, parent, *p, END);
	    }
	    parent = node;
	}
	if ((!htabPtr->allowDups) && (Blt_TreeFindChild(parent, *p) != NULL)) {
	    Tcl_AppendResult(interp, "entry \"", *p, "\" already exists in \"",
		 path, "\"", (char *)NULL);
	    goto error;
	}
	node = Blt_TreeCreateNode(htabPtr->tree, parent, *p, insertPos);
	if (node == NULL) {
	    goto error;
	}
	if (Blt_HtCreateEntry(htabPtr, node, count, options) != TCL_OK) {
	    goto error;
	}
	if (compArr != &path) {
	    free((char *)compArr);
	}
	Tcl_DStringAppendElement(&dString, Blt_HtNodeToString(node));
    }
    htabPtr->flags |= (HIERTABLE_LAYOUT | HIERTABLE_SCROLL | HIERTABLE_DIRTY);
    Blt_HtEventuallyRedraw(htabPtr);
    Tcl_DStringResult(htabPtr->interp, &dString);
    return TCL_OK;
  error:
    if (compArr != &path) {
	free((char *)compArr);
    }
    Tcl_DStringFree(&dString);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DeleteOp --
 *
 *	Deletes nodes from the hierarchy. Deletes either a range of
 *	entries from a hierarchy or a single node (except root).
 *	In all cases, nodes are removed recursively.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
DeleteOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node;
    register int i;

    for (i = 2; i < argc; i++) {
	if (StringToNode(htabPtr, argv[i], &node) != TCL_OK) {
	    return TCL_ERROR;	/* Node or path doesn't already exist */
	}
	Blt_TreeDeleteNode(htabPtr->tree, node);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * MoveOp --
 *
 *	Move an entry into a new location in the hierarchy.
 *
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
MoveOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node, dest, parent;
    char c;
    int action;

#define MOVE_INTO	(1<<0)
#define MOVE_BEFORE	(1<<1)
#define MOVE_AFTER	(1<<2)
    if (StringToNode(htabPtr, argv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    c = argv[3][0];
    action = MOVE_INTO;
    if ((c == 'i') && (strcmp(argv[3], "into") == 0)) {
	action = MOVE_INTO;
    } else if ((c == 'b') && (strcmp(argv[3], "before") == 0)) {
	action = MOVE_BEFORE;
    } else if ((c == 'a') && (strcmp(argv[3], "after") == 0)) {
	action = MOVE_AFTER;
    } else {
	Tcl_AppendResult(interp, "bad position \"", argv[3],
	    "\": should be into, before, or after", (char *)NULL);
	return TCL_ERROR;
    }
    if (StringToNode(htabPtr, argv[4], &dest) != TCL_OK) {
	return TCL_ERROR;
    }
    /* Verify they aren't ancestors. */
    if (Blt_TreeIsAncestor(node, dest)) {
	Tcl_AppendResult(interp, "can't move node: \"", argv[2],
	    "\" is an ancestor of \"", argv[4], "\"", (char *)NULL);
	return TCL_ERROR;
    }
    parent = Blt_TreeNodeParent(dest);
    if (parent == NULL) {
	action = MOVE_INTO;
    }
    switch (action) {
    case MOVE_INTO:
	Blt_TreeMoveNode(htabPtr->tree, node, dest, (Blt_TreeNode)NULL);
	break;

    case MOVE_BEFORE:
	Blt_TreeMoveNode(htabPtr->tree, node, parent, dest);
	break;

    case MOVE_AFTER:
	dest = Blt_TreeNextSibling(dest);
	Blt_TreeMoveNode(htabPtr->tree, node, parent, dest);
	break;
    }
    htabPtr->flags |= (HIERTABLE_LAYOUT | HIERTABLE_DIRTY);
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*ARGSUSED*/
static int
NearestOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Button *buttonPtr = &(htabPtr->button);
    int x, y;			/* Screen coordinates of the test point. */
    register Entry *entryPtr;
    register Blt_TreeNode node;

    if ((Tk_GetPixels(interp, htabPtr->tkwin, argv[2], &x) != TCL_OK) ||
	(Tk_GetPixels(interp, htabPtr->tkwin, argv[3], &y) != TCL_OK)) {
	return TCL_ERROR;
    }
    if (htabPtr->nVisible == 0) {
	return TCL_OK;
    }
    node = Blt_HtNearestNode(htabPtr, x, y, TRUE);
    if (node == NULL) {
	return TCL_OK;
    }
    x = WORLDX(htabPtr, x);
    y = WORLDY(htabPtr, y);
    entryPtr = GetEntry(htabPtr, node);
    if (argc > 4) {
	char *where;
	int labelX, depth;

	where = "";
	if (entryPtr->flags & ENTRY_BUTTON) {
	    int buttonX, buttonY;

	    buttonX = entryPtr->worldX + entryPtr->buttonX;
	    buttonY = entryPtr->worldY + entryPtr->buttonY;
	    if ((x >= buttonX) && (x < (buttonX + buttonPtr->width)) &&
		(y >= buttonY) && (y < (buttonY + buttonPtr->height))) {
		where = "button";
	    }
	}
	depth = DEPTH(htabPtr, node);
	labelX = entryPtr->worldX + ICONWIDTH(depth);
	if ((x >= labelX) &&
	    (x < (labelX + ICONWIDTH(depth + 1) + entryPtr->width))) {
	    where = "select";
	}
	if (Tcl_SetVar(interp, argv[4], where, TCL_LEAVE_ERR_MSG) == NULL) {
	    return TCL_ERROR;
	}
    }
    Tcl_SetResult(interp, Blt_HtNodeToString(node), TCL_VOLATILE);
    return TCL_OK;
}


/*ARGSUSED*/
static int
OpenOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;		/* Not used. */
    int argc;
    char **argv;
{
    Blt_TreeNode node;
    int length;
    int recurse, result;
    register int i;

    recurse = FALSE;
    length = strlen(argv[2]);
    if ((argv[2][0] == '-') && (length > 1) &&
	(strncmp(argv[2], "-recurse", length) == 0)) {
	argv++, argc--;
	recurse = TRUE;
    }
    for (i = 2; i < argc; i++) {
	if (GetNode(htabPtr, argv[i], &node) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (node == NULL) {
	    continue;
	}
	if (recurse) {
	    result = Blt_TreeApply(node, OpenApplyProc, (ClientData)htabPtr);
	} else {
	    result = Blt_HtOpenEntry(htabPtr, GetEntry(htabPtr, node));
	}
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}
	/* Make sure ancestors of this node aren't hidden. */
	MapAncestors(htabPtr, node);
    }
    htabPtr->flags |= HIERTABLE_SCROLL;
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RangeOp --
 *
 *	Returns the node identifiers in a given range.
 *
 *----------------------------------------------------------------------
 */
static int
RangeOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Blt_TreeNode first, last;
    Blt_TreeNode node;
    Entry *entryPtr;
    unsigned int mask;
    int length;

    length = strlen(argv[2]);
    if ((argv[2][0] == '-') && (length > 1) &&
	(strncmp(argv[2], "-open", length) == 0)) {
	argv++, argc--;
	mask |= ENTRY_OPEN;
    }
    if (StringToNode(htabPtr, argv[2], &first) != TCL_OK) {
	return TCL_ERROR;
    }
    if (argc > 3) {
	if (StringToNode(htabPtr, argv[3], &last) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	last = EndNode(htabPtr, first, mask);
    }    
    if (mask & ENTRY_OPEN) {
	entryPtr = GetEntry(htabPtr, first);
	if (!(entryPtr->flags & ENTRY_MAPPED)) {
	    Tcl_AppendResult(interp, "first node \"", argv[2], "\" is hidden.",
		(char *)NULL);
	    return TCL_ERROR;
	}
	entryPtr = GetEntry(htabPtr, last);
	if (!(entryPtr->flags & ENTRY_MAPPED)) {
	    Tcl_AppendResult(interp, "last node \"", argv[3], "\" is hidden.",
		(char *)NULL);
	    return TCL_ERROR;
	}
    }
    /*
     * The relative order of the first/last markers determines the
     * direction.
     */
    if (Blt_TreeIsBefore(last, first)) {
	for (node = last; node != NULL; node = PrevNode(htabPtr, node, mask)) {
	    Tcl_AppendElement(interp, Blt_HtNodeToString(node));
	    if (node == first) {
		break;
	    }
	}
    } else {
	for (node = first; node != NULL; 
	     node = Blt_HtNextNode(htabPtr, node, mask)) {
	    Tcl_AppendElement(interp, Blt_HtNodeToString(node));
	    if (node == last) {
		break;
	    }
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ScanOp --
 *
 *	Implements the quick scan.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ScanOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    int x, y;
    char c;
    unsigned int length;
    int oper;

#define SCAN_MARK		1
#define SCAN_DRAGTO	2
    c = argv[2][0];
    length = strlen(argv[2]);
    if ((c == 'm') && (strncmp(argv[2], "mark", length) == 0)) {
	oper = SCAN_MARK;
    } else if ((c == 'd') && (strncmp(argv[2], "dragto", length) == 0)) {
	oper = SCAN_DRAGTO;
    } else {
	Tcl_AppendResult(interp, "bad scan operation \"", argv[2],
	    "\": should be either \"mark\" or \"dragto\"", (char *)NULL);
	return TCL_ERROR;
    }
    if ((Tk_GetPixels(interp, htabPtr->tkwin, argv[3], &x) != TCL_OK) ||
	(Tk_GetPixels(interp, htabPtr->tkwin, argv[4], &y) != TCL_OK)) {
	return TCL_ERROR;
    }
    if (oper == SCAN_MARK) {
	htabPtr->scanAnchorX = x;
	htabPtr->scanAnchorY = y;
	htabPtr->scanX = htabPtr->xOffset;
	htabPtr->scanY = htabPtr->yOffset;
    } else {
	int worldX, worldY;
	int dx, dy;

	dx = htabPtr->scanAnchorX - x;
	dy = htabPtr->scanAnchorY - y;
	worldX = htabPtr->scanX + (10 * dx);
	worldY = htabPtr->scanY + (10 * dy);

	if (worldX < 0) {
	    worldX = 0;
	} else if (worldX >= htabPtr->worldWidth) {
	    worldX = htabPtr->worldWidth - htabPtr->xScrollUnits;
	}
	if (worldY < 0) {
	    worldY = 0;
	} else if (worldY >= htabPtr->worldHeight) {
	    worldY = htabPtr->worldHeight - htabPtr->yScrollUnits;
	}
	htabPtr->xOffset = worldX;
	htabPtr->yOffset = worldY;
	htabPtr->flags |= HIERTABLE_SCROLL;
	Blt_HtEventuallyRedraw(htabPtr);
    }
    return TCL_OK;
}

/*ARGSUSED*/
static int
SeeOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;		/* Not used. */
    int argc;
    char **argv;
{
    Entry *entryPtr;
    int width, height;
    int x, y;
    Blt_TreeNode node;
    Tk_Anchor anchor;
    int left, right, top, bottom;

    anchor = TK_ANCHOR_W;	/* Default anchor is West */
    if ((argv[2][0] == '-') && (strcmp(argv[2], "-anchor") == 0)) {
	if (argc == 3) {
	    Tcl_AppendResult(interp, "missing \"-anchor\" argument",
		(char *)NULL);
	    return TCL_ERROR;
	}
	if (Tk_GetAnchor(interp, argv[3], &anchor) != TCL_OK) {
	    return TCL_ERROR;
	}
	argc -= 2, argv += 2;
    }
    if (argc == 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	    "see ?-anchor anchor? index\"", (char *)NULL);
	return TCL_ERROR;
    }
    if (GetNode(htabPtr, argv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    if (node == NULL) {
	return TCL_OK;
    }
    entryPtr = GetEntry(htabPtr, node);
    if (!(entryPtr->flags & ENTRY_MAPPED)) {
	/* FIXME: open  */
	MapAncestors(htabPtr, node);
	htabPtr->flags |= HIERTABLE_SCROLL;
	/*
	 * If the entry wasn't previously exposed, its world coordinates
	 * aren't likely to be valid.  So re-compute the layout before
	 * we try to see the viewport to the entry's location.
	 */
	Blt_HtComputeLayout(htabPtr);
    }
    width = VPORTWIDTH(htabPtr);
    height = VPORTHEIGHT(htabPtr);

    /*
     * XVIEW:	If the entry is left or right of the current view, adjust
     *		the offset.  If the entry is nearby, adjust the view just
     *		a bit.  Otherwise, center the entry.
     */
    left = htabPtr->xOffset;
    right = htabPtr->xOffset + width;

    switch (anchor) {
    case TK_ANCHOR_W:
    case TK_ANCHOR_NW:
    case TK_ANCHOR_SW:
	x = 0;
	break;
    case TK_ANCHOR_E:
    case TK_ANCHOR_NE:
    case TK_ANCHOR_SE:
	x = entryPtr->worldX + entryPtr->width + 
	    ICONWIDTH(DEPTH(htabPtr, node)) - width;
	break;
    default:
	if (entryPtr->worldX < left) {
	    x = entryPtr->worldX;
	} else if ((entryPtr->worldX + entryPtr->width) > right) {
	    x = entryPtr->worldX + entryPtr->width - width;
	} else {
	    x = htabPtr->xOffset;
	}
	break;
    }
    /*
     * YVIEW:	If the entry is above or below the current view, adjust
     *		the offset.  If the entry is nearby, adjust the view just
     *		a bit.  Otherwise, center the entry.
     */
    top = htabPtr->yOffset;
    bottom = htabPtr->yOffset + height;

    switch (anchor) {
    case TK_ANCHOR_N:
	y = htabPtr->yOffset;
	break;
    case TK_ANCHOR_NE:
    case TK_ANCHOR_NW:
	y = entryPtr->worldY - (height / 2);
	break;
    case TK_ANCHOR_S:
    case TK_ANCHOR_SE:
    case TK_ANCHOR_SW:
	y = entryPtr->worldY + entryPtr->height - height;
	break;
    default:
	if (entryPtr->worldY < top) {
	    y = entryPtr->worldY;
	} else if ((entryPtr->worldY + entryPtr->height) > bottom) {
	    y = entryPtr->worldY + entryPtr->height - height;
	} else {
	    y = htabPtr->yOffset;
	}
	break;
    }
    if ((y != htabPtr->yOffset) || (x != htabPtr->xOffset)) {
	/* htabPtr->xOffset = x; */
	htabPtr->yOffset = y;
	htabPtr->flags |= HIERTABLE_SCROLL;
    }
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * LostSelection --
 *
 *	This procedure is called back by Tk when the selection is grabbed
 *	away.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The existing selection is unhighlighted, and the window is
 *	marked as not containing a selection.
 *
 *----------------------------------------------------------------------
 */
static void
LostSelection(clientData)
    ClientData clientData;	/* Information about the widget. */
{
    Hiertable *htabPtr = (Hiertable *)clientData;
    Entry *entryPtr;
    Blt_TreeNode node;

    if ((htabPtr->selAnchorNode == NULL) || (!htabPtr->exportSelection)) {
	return;
    }
    htabPtr->selMarkNode = htabPtr->selAnchorNode = NULL;

    htabPtr->flags &= ~SELECTION_MASK;
    htabPtr->flags |= SELECTION_CLEAR;
    for (node = htabPtr->rootNode; node != NULL; 
	 node = Blt_HtNextNode(htabPtr, node, ENTRY_OPEN|ENTRY_MAPPED)) {
	entryPtr = GetEntry(htabPtr, node);
	SelectEntry(htabPtr, entryPtr);
    }
    htabPtr->flags &= ~SELECTION_MASK;
    Blt_HtEventuallyRedraw(htabPtr);
    if (htabPtr->selectCmd != NULL) {
	EventuallyInvokeSelectCmd(htabPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SelectRange --
 *
 *	Sets the selection flag for a range of nodes.  The range is
 *	determined by two pointers which designate the first/last
 *	nodes of the range.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 *----------------------------------------------------------------------
 */
static int
SelectRange(htabPtr, first, last)
    Hiertable *htabPtr;
    Blt_TreeNode first, last;
{
    register Blt_TreeNode node;

    /* Swap first and last pointers if the range is reversed. */
    if (Blt_TreeIsBefore(last, first)) {
	node = last;
	last = first;
	first = node;
    }
    for (node = first; node != NULL; 
	 node = Blt_HtNextNode(htabPtr, node, ENTRY_MAPPED | ENTRY_OPEN)) {
	SelectEntry(htabPtr, GetEntry(htabPtr, node));
	if (node == last) {
	    break;
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SelectionAnchorOp --
 *
 *	Sets the selection anchor to the element given by a index.
 *	The selection anchor is the end of the selection that is fixed
 *	while dragging out a selection with the mouse.  The index
 *	"anchor" may be used to refer to the anchor element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The selection changes.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
SelectionAnchorOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;		/* Not used. */
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node;

    if (GetNode(htabPtr, argv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    /* Set both the anchor and the mark. Indicates that a single node is selected. */
    htabPtr->selAnchorNode = htabPtr->selMarkNode = node;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SelectionCancelOp --
 *
 *	Cancels the temporary selection.  This is typically done when
 *	the mouse is released outside of the widget's window.
 *
 * Results:
 *	Always TCL_OK.
 *
 * Side effects:
 *	The appearance of the selection changes.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
SelectionCancelOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;		/* Not used. */
    int argc;			/* Not used. */
    char **argv;
{
    htabPtr->selMarkNode = htabPtr->selAnchorNode; /* Clear the mark. */
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SelectionIncludesOp
 *
 *	Returns 1 if the element indicated by index is currently
 *	selected, 0 if it isn't.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The selection changes.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
SelectionIncludesOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node;
    Entry *entryPtr;

    if (StringToNode(htabPtr, argv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    entryPtr = GetEntry(htabPtr, node);
    Tcl_SetResult(interp, (entryPtr->flags & ENTRY_SELECTED) 
		  ? "1" : "0", TCL_STATIC);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SelectionDragOp --
 *
 *	Sets the selection anchor to the element given by a index.
 *	The selection anchor is the end of the selection that is fixed
 *	while dragging out a selection with the mouse.  The index
 *	"anchor" may be used to refer to the anchor element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The selection changes.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
SelectionDragOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode node;
    char c;
    int length;

    if (GetNode(htabPtr, argv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    c = argv[4][0];
    length = strlen(argv[4]);
    htabPtr->flags &= ~SELECTION_MASK;
    if ((c == 'c') && (strncmp(argv[4], "clear", length) == 0)) {
	htabPtr->flags |= SELECTION_CLEAR;
    } else if ((c == 's') && (strncmp(argv[4], "set", length) == 0)) {
	htabPtr->flags |= SELECTION_SET;
    } else if ((c == 't') && (strncmp(argv[4], "toggle", length) == 0)) {
	htabPtr->flags |= SELECTION_TOGGLE;
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[4],
	    "\": should be \"clear\", \"set\", or \"toggle\"", (char *)NULL);
	return TCL_ERROR;
    }
    htabPtr->selMarkNode = node;
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SelectionSetOp
 *
 *	Selects, deselects, or toggles all of the elements in the
 *	range between first and last, inclusive, without affecting the
 *	selection state of elements outside that range.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The selection changes.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
SelectionSetOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;			/* Not used. */
    char **argv;
{
    Blt_TreeNode first, last;
    Entry *entryPtr;

    htabPtr->flags &= ~SELECTION_MASK;
    switch (argv[2][0]) {
    case 's':
	htabPtr->flags |= SELECTION_SET;
	break;
    case 'c':
	htabPtr->flags |= SELECTION_CLEAR;
	break;
    case 't':
	htabPtr->flags |= SELECTION_TOGGLE;
	break;
    }

    if (StringToNode(htabPtr, argv[3], &first) != TCL_OK) {
	return TCL_ERROR;
    }
    entryPtr = GetEntry(htabPtr, first);
    if ((!(entryPtr->flags & ENTRY_MAPPED)) && 
	(!(htabPtr->flags & SELECTION_CLEAR))) {
	Tcl_AppendResult(interp, "can't select hidden node \"", argv[3], "\"",
	    (char *)NULL);
	return TCL_ERROR;
    }
    last = first;
    if (argc > 4) {
	Entry *lastPtr;

	if (StringToNode(htabPtr, argv[4], &last) != TCL_OK) {
	    return TCL_ERROR;
	}
	lastPtr = GetEntry(htabPtr, last);
	if ((!(lastPtr->flags & ENTRY_MAPPED)) && 
		(!(htabPtr->flags & SELECTION_CLEAR))) {
	    Tcl_AppendResult(interp, "can't select hidden node \"", argv[4],
		"\"", (char *)NULL);
	    return TCL_ERROR;
	}
    }
    if ((htabPtr->flags & SELECTION_CLEAR) &&
	(htabPtr->selAnchorNode == htabPtr->selMarkNode)) {
	/* 
	 * Special case: If there's only one node selected, we don't need 
	 *		 to run through the entire tree to clear it.  
	 */
	if (htabPtr->selAnchorNode != NULL) {
	    entryPtr = GetEntry(htabPtr, htabPtr->selAnchorNode);
	    entryPtr->flags &= ~ENTRY_SELECTED;
	}
    } else {
	if (first == last) {
	    SelectEntry(htabPtr, entryPtr);
	} else {
	    SelectRange(htabPtr, first, last);
	}
	htabPtr->flags &= ~SELECTION_MASK;
    }
    if (htabPtr->flags & SELECTION_EXPORT) {
	Tk_OwnSelection(htabPtr->tkwin, XA_PRIMARY, LostSelection,
	    (ClientData)htabPtr);
    }
    Blt_HtEventuallyRedraw(htabPtr);
    if (htabPtr->selectCmd != NULL) {
	EventuallyInvokeSelectCmd(htabPtr);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SelectionOp --
 *
 *	This procedure handles the individual options for text
 *	selections.  The selected text is designated by start and end
 *	indices into the text pool.  The selected segment has both a
 *	anchored and unanchored ends.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The selection changes.
 *
 *----------------------------------------------------------------------
 */
static Blt_OpSpec selectionOps[] =
{
    {"anchor", 1, (Blt_OpProc)SelectionAnchorOp, 4, 4, "index",},
    {"cancel", 2, (Blt_OpProc)SelectionCancelOp, 3, 3, "",},
    {"clear", 2, (Blt_OpProc)SelectionSetOp, 4, 5, "firstIndex ?lastIndex?",},
    {"dragto", 1, (Blt_OpProc)SelectionDragOp, 5, 5, "index action",},
    {"includes", 2, (Blt_OpProc)SelectionIncludesOp, 4, 4, "index",},
    {"set", 1, (Blt_OpProc)SelectionSetOp, 4, 5, "firstIndex ?lastIndex?",},
    {"toggle", 1, (Blt_OpProc)SelectionSetOp, 4, 5, "firstIndex ?lastIndex?",},
};
static int nSelectionOps = sizeof(selectionOps) / sizeof(Blt_OpSpec);

static int
SelectionOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Blt_OpProc opProc;
    int result;

    opProc = Blt_GetOperation(interp, nSelectionOps, selectionOps,
	BLT_OPER_ARG2, argc, argv);
    if (opProc == NULL) {
	return TCL_ERROR;
    }
    result = (*opProc) (htabPtr, interp, argc, argv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * CompareNodesByName --
 *
 *	Comparison routine (used by qsort) to sort a chain of subnodes.
 *	A simple string comparison is performed on each node name.
 *
 * Results:
 *	1 is the first is greater, -1 is the second is greater, 0
 *	if equal.
 *
 *----------------------------------------------------------------------
 */
static int
CompareNodesByName(node1, node2)
    Blt_TreeNode node1, node2;
{
    return strcmp(Blt_TreeNodeName(node1), Blt_TreeNodeName(node2));
}

/*
 *----------------------------------------------------------------------
 *
 * CompareNodesByTclCmd --
 *
 *	Comparison routine (used by qsort) to sort a list of subnodes.
 *	A specified Tcl proc is invoked to compare the nodes.
 *
 * Results:
 *	1 is the first is greater, -1 is the second is greater, 0
 *	if equal.
 *
 *----------------------------------------------------------------------
 */
static int
CompareNodesByTclCmd(node1, node2)
    Blt_TreeNode node1, node2;
{
    int result;
    Hiertable *htabPtr = bltHiertableLastInstance;
    Tcl_Interp *interp = htabPtr->interp;
    Tcl_DString dString;

    result = 0;			/* Hopefully this will be Ok even if the
				 * Tcl command fails to return the correct
				 * result. */
    Tcl_DStringInit(&dString);
    Tcl_DStringAppendElement(&dString, htabPtr->sortCmd);
    Tcl_DStringAppendElement(&dString, Tk_PathName(htabPtr->tkwin));
    Tcl_DStringAppendElement(&dString, Blt_HtNodeToString(node1));
    Tcl_DStringAppendElement(&dString, Blt_HtNodeToString(node2));
    result = Tcl_GlobalEval(interp, Tcl_DStringValue(&dString));
    Tcl_DStringFree(&dString);

    if ((result != TCL_OK) ||
	(Tcl_GetInt(interp, Tcl_GetStringResult(interp), &result) != TCL_OK)) {
	Tk_BackgroundError(interp);
    }
    Tcl_ResetResult(interp);
    return result;
}


/*ARGSUSED*/
static int
SortOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;		/* Not used. */
    int argc;
    char **argv;
{
    int length;
    Blt_TreeNode node;
    register int i;
    int recurse, result;

    recurse = FALSE;
    htabPtr->sortCmd = NULL;
    for (i = 2; i < argc; i++) {
	if (argv[i][0] != '-') {
	    break;		/* Found start of indices */
	}
	length = strlen(argv[i]);
	if ((length > 1) && (strncmp(argv[i], "-recurse", length) == 0)) {
	    recurse = TRUE;
	} else if ((length > 1) && (strncmp(argv[i], "-command", length) == 0)) {
	    if ((i + 1) == argc) {
		Tcl_AppendResult(interp, "\"-command\" must be",
		    " followed by comparison command", (char *)NULL);
		return TCL_ERROR;
	    }
	    i++;
	    htabPtr->sortCmd = argv[i];
	} else if ((argv[i][1] == '-') && (argv[i][2] == '\0')) {
	    break;		/* Allow first index to start with a '-' */
	} else {
	    Tcl_AppendResult(interp, "bad switch \"", argv[i],
		"\": must be -command or -recurse", (char *)NULL);
	    return TCL_ERROR;
	}
    }
    for ( /*empty*/ ; i < argc; i++) {
	if (StringToNode(htabPtr, argv[i], &node) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (recurse) {
	    result = Blt_TreeApply(node, SortApplyProc, (ClientData)htabPtr);
	} else {
	    result = SortApplyProc(node, (ClientData)htabPtr);
	}
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    htabPtr->flags |= HIERTABLE_LAYOUT;
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*ARGSUSED*/
static int
ToggleOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;		/* Not used. */
    int argc;
    char **argv;
{
    Blt_TreeNode node;
    Entry *entryPtr;

    if (GetNode(htabPtr, argv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    if (node == NULL) {
	return TCL_OK;
    }
    entryPtr = GetEntry(htabPtr, node);
    if (entryPtr->flags & ENTRY_OPEN) {
	ClearSelections(htabPtr, node);
	if (Blt_TreeIsAncestor(node, htabPtr->focusNode)) {
	    htabPtr->focusNode = node;
	    Blt_SetFocusItem(htabPtr->bindTable, htabPtr->focusNode);
	}
	if (Blt_TreeIsAncestor(node, htabPtr->selAnchorNode)) {
	    htabPtr->selMarkNode = htabPtr->selAnchorNode = NULL;
	}
	Blt_HtCloseEntry(htabPtr, entryPtr);
    } else {
	Blt_HtOpenEntry(htabPtr, entryPtr);
    }
    htabPtr->flags |= HIERTABLE_SCROLL;
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

static int
XViewOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int width, worldWidth;

    width = VPORTWIDTH(htabPtr);
    worldWidth = htabPtr->worldWidth;
    if (argc == 2) {
	double fract;

	/*
	 * Note that we are bounding the fractions between 0.0 and 1.0
	 * to support the "canvas"-style of scrolling.
	 */
	fract = (double)htabPtr->xOffset / worldWidth;
	Tcl_AppendElement(interp, Blt_Dtoa(interp, CLAMP(fract, 0.0, 1.0)));
	fract = (double)(htabPtr->xOffset + width) / worldWidth;
	Tcl_AppendElement(interp, Blt_Dtoa(interp, CLAMP(fract, 0.0, 1.0)));
	return TCL_OK;
    }
    if (Blt_GetScrollInfo(interp, argc - 2, argv + 2, &(htabPtr->xOffset),
	    worldWidth, width, htabPtr->xScrollUnits, htabPtr->scrollMode) 
	    != TCL_OK) {
	return TCL_ERROR;
    }
    htabPtr->flags |= HIERTABLE_XSCROLL;
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

static int
YViewOp(htabPtr, interp, argc, argv)
    Hiertable *htabPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int height, worldHeight;

    height = VPORTHEIGHT(htabPtr);
    worldHeight = htabPtr->worldHeight;
    if (argc == 2) {
	double fract;

	/* Report first and last fractions */
	fract = (double)htabPtr->yOffset / worldHeight;
	Tcl_AppendElement(interp, Blt_Dtoa(interp, CLAMP(fract, 0.0, 1.0)));
	fract = (double)(htabPtr->yOffset + height) / worldHeight;
	Tcl_AppendElement(interp, Blt_Dtoa(interp, CLAMP(fract, 0.0, 1.0)));
	return TCL_OK;
    }
    if (Blt_GetScrollInfo(interp, argc - 2, argv + 2, &(htabPtr->yOffset),
	    worldHeight, height, htabPtr->yScrollUnits, htabPtr->scrollMode)
	!= TCL_OK) {
	return TCL_ERROR;
    }
    htabPtr->flags |= HIERTABLE_SCROLL;
    Blt_HtEventuallyRedraw(htabPtr);
    return TCL_OK;
}

/*
 * --------------------------------------------------------------
 *
 * Blt_HtWidgetInstCmd --
 *
 * 	This procedure is invoked to process commands on behalf of
 *	the hiertable widget.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 * --------------------------------------------------------------
 */
static Blt_OpSpec operSpecs[] =
{
    {"bbox", 2, (Blt_OpProc)BboxOp, 3, 0, "index...",},
    {"bind", 2, (Blt_OpProc)BindOp, 3, 5, "tagName ?sequence command?",},
    {"button", 2, (Blt_OpProc)ButtonOp, 2, 0, "args",},
    {"cget", 2, (Blt_OpProc)CgetOp, 3, 3, "option",},
    {"close", 2, (Blt_OpProc)CloseOp, 2, 0, "?-recurse? index...",},
    {"column", 3, (Blt_OpProc)ColumnOp, 2, 0, "oper args",},
    {"configure", 3, (Blt_OpProc)ConfigureOp, 2, 0, "?option value?...",},
    {"curselection", 2, (Blt_OpProc)CurselectionOp, 2, 2, "",},
    {"delete", 1, (Blt_OpProc)DeleteOp, 2, 0, "index ?index...?",},
    {"entry", 1, (Blt_OpProc)EntryOp, 2, 0, "oper args",},
    {"find", 2, (Blt_OpProc)FindOp, 2, 0, "?flags...? ?firstIndex lastIndex?",},
    {"focus", 2, (Blt_OpProc)FocusOp, 3, 3, "index",},
    {"get", 1, (Blt_OpProc)GetOp, 2, 0, "?-full? index ?index...?",},
    {"hide", 1, (Blt_OpProc)HideOp, 2, 0, "?-exact? ?-glob? ?-regexp? ?-nonmatching? ?-name string? ?-full string? ?-data string? ?--? ?index...?",},
    {"index", 3, (Blt_OpProc)IndexOp, 3, 5, "?-at index? string",},
    {"insert", 3, (Blt_OpProc)InsertOp, 3, 0, "?-at index? position label ?label...? ?option value?",},
    {"move", 1, (Blt_OpProc)MoveOp, 5, 5, "index into|before|after index",},
    {"nearest", 1, (Blt_OpProc)NearestOp, 4, 5, "x y ?varName?",},
    {"open", 1, (Blt_OpProc)OpenOp, 2, 0, "?-recurse? index...",},
    {"range", 1, (Blt_OpProc)RangeOp, 4, 5, "?-open? firstIndex lastIndex",},
    {"scan", 2, (Blt_OpProc)ScanOp, 5, 5, "dragto|mark x y",},
    {"see", 3, (Blt_OpProc)SeeOp, 3, 0, "?-anchor anchor? index",},
    {"selection", 3, (Blt_OpProc)SelectionOp, 2, 0, "oper args",},
    {"show", 2, (Blt_OpProc)ShowOp, 2, 0, "?-exact? ?-glob? ?-regexp? ?-nonmatching? ?-name string? ?-full string? ?-data string? ?--? ?index...?",},
    {"sort", 2, (Blt_OpProc)SortOp, 3, 0, "?-recurse? ?-command string? index...",},
    {"text", 2, (Blt_OpProc)Blt_HtTextOp, 2, 0, "args",},
    {"toggle", 2, (Blt_OpProc)ToggleOp, 3, 3, "index",},
    {"xview", 1, (Blt_OpProc)XViewOp, 2, 5, "?moveto fract? ?scroll number what?",},
    {"yview", 1, (Blt_OpProc)YViewOp, 2, 5, "?moveto fract? ?scroll number what?",},
};

static int nSpecs = sizeof(operSpecs) / sizeof(Blt_OpSpec);

int
Blt_HtWidgetInstCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Information about the widget. */
    Tcl_Interp *interp;		/* Interpreter to report errors back to. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Vector of argument strings. */
{
    Blt_OpProc opProc;
    Hiertable *htabPtr = (Hiertable *)clientData;
    int result;

    opProc = Blt_GetOperation(interp, nSpecs, operSpecs, BLT_OPER_ARG1, 
	argc, argv);
    if (opProc == NULL) {
	return TCL_ERROR;
    }
    Tcl_Preserve((ClientData)htabPtr);
    result = (*opProc) (htabPtr, interp, argc, argv);
    Tcl_Release((ClientData)htabPtr);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * StringToData --
 *
 *	Convert the string reprsenting a scroll mode, to its numeric
 *	form.
 *
 * Results:
 *	If the string is successfully converted, TCL_OK is returned.
 *	Otherwise, TCL_ERROR is returned and an error message is left
 *	in interpreter's result field.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
StringToData(clientData, interp, tkwin, string, widgRec, offset)
    ClientData clientData;	/* Node of entry. */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* Not used. */
    char *string;		/* New data string */
    char *widgRec;		/* Widget record */
    int offset;			/* atom of data field. */
{
    Entry *entryPtr = (Entry *)widgRec;
    char **elemArr;
    int nElem;
    Tcl_Obj *objPtr;
    register int i;
    Column *columnPtr;

    if ((string == NULL) || (*string == '\0')) {
	return TCL_OK;
    } 
    if (Tcl_SplitList(interp, string, &nElem, &elemArr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (nElem == 0) {
	return TCL_OK;
    }
    if (nElem & 0x1) {
	Tcl_AppendResult(interp, "data \"", string, 
		 "\" must in even name-value pairs", 
		(char *)NULL);
	return TCL_ERROR;
    }
    for (i = 0; i < nElem; i += 2) {
	if (GetColumn(entryPtr->htabPtr, elemArr[i], &columnPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	objPtr = Tcl_NewStringObj(elemArr[i + 1], strlen(elemArr[i + 1]));
	Blt_TreeSetValue(entryPtr->htabPtr->tree, entryPtr->node, 
		columnPtr->atom, objPtr);
	Blt_HtAddField(entryPtr, columnPtr);
    }
    free((char *)elemArr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DataToString --
 *
 * Results:
 *	The string representation of the button boolean is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
DataToString(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* Not used. */
    Tk_Window tkwin;		/* Not used. */
    char *widgRec;		/* Widget record */
    int offset;			/* atom of data field. */
    Tcl_FreeProc **freeProcPtr;	/* Memory deallocation scheme to use */
{
    Entry *entryPtr = (Entry *)widgRec;
    Tcl_DString dString;
    Blt_ChainLink *linkPtr;
    char *result;
    Field *dataPtr;
    Blt_TreeAtom atom;
    char *string;

    Tcl_DStringInit(&dString);
    for (linkPtr = Blt_ChainFirstLink(entryPtr->chainPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	dataPtr = (Field *)Blt_ChainGetValue(linkPtr);
	atom = dataPtr->columnPtr->atom;
	Tcl_DStringAppendElement(&dString, atom);
	string = Blt_HtGetData(entryPtr, atom);
	if (string == NULL) {
	    Tcl_DStringAppendElement(&dString, "");
	} else {
	    Tcl_DStringAppendElement(&dString, string);
	}
    }
    result = strdup(Tcl_DStringValue(&dString));
    Tcl_DStringFree(&dString);
    *freeProcPtr = (Tcl_FreeProc *)free;
    return result;
}

#endif /* NO_HIERTABLE */
