// Turmite.m

//  Turmites are FSM's living on a 2d tape. 
//  Behavior is determined by a transition table - which might change over time
//  multiple machines can be on the same tape square:
//        - last turmite to write on tape square is what other tmites
//           will see next time step.
//        - gives a chance to experiment with random ordering of
//           tmite list, or with a montecarlo update scheme, etc....

// note - the lookuptable should be a 2D array, but the collection library
//           doesn't support arrays yet..... :-( 
//           I fake it here with the ol' offset hack.....

#import "Turmite.h"
#include <random.h>

// Defining the methods for a Turmite.

@implementation Turmite

// Initialize crucial state for the turmite.

- setTape:  t
{
  tape = t;
  return self;
}

- setNumStates: (int)n
{
     numStates = n;
     return self;
}

- setNumTapeStates: (int)t
{
     numTapeStates = t;
     return self;
}

- setNumMoves: (int)m
{
     numMoves = m;
     return self;
}

// CreateEnd. This is a good place to put code that does various sorts
// of initialization that can only be done after some parameters of the
// object are set. It's also a good time to check for errors in creation.

- createEnd
{
  // make sure the user gave us a tape to live on.

  if (tape == nil)
    [InvalidCombination raiseEvent: "Turmite was created without a tape.\n"];
  
  // Cache the tapeSize for speed of later access. Note how we do
  // this in createEnd - it could also have been done when setTape:
  // was called, but this is a good place to do it, too. If an object
  // needed to allocate extra memory, this is the right place to do it.

  tapeXSize = [tape getSizeX];
  tapeYSize = [tape getSizeY];

  // Someday, I'd like the space library to be powerful enough that
  // the turmites never need to be aware how big their world is.

  // Cache the transition table offset for looking up newstate, move, and output

  offset = numStates * numTapeStates;
  offset2 = offset * 2;

  // compute the log_2 of the number of tapestates.
  // only works for powers of 2 - fix this to be more robust

  ii = 0;

  nn = numTapeStates;
  while (nn > 0)
    {
      nn >>= 1;
      ii++;
    }
  
  logt = ii - 1;


  // Finally, allocate the space for the transitionTable
  // The transition table is of type State (unsigned char), and we need three
  // rows of size offset, for nextstate, move, and output, respectively

  transitionTable = (State *) [[self getZone] alloc: (offset * 3)];

  return self;			  // CRUCIAL! 
}

// Methods for reading/writing a Turmite's state during runtime.


// Simple set methods for Turmite state. Some of these are probably not
// going to normally change in a Turmite's lifetime, but there's no reason
// they couldn't change.

- setState: (State)s
{
  state = s;
  return self;
}


// initialize a turmite with a random state, coordinates, and t-table.

- setRandom
{
  int i;
  
  // set random coordinates
  
  //  xpos = [uniformRandom rMax: tapeXSize];
  //  ypos = [uniformRandom rMax: tapeYSize];
  
  xpos = [uniformIntRand getIntegerWithMin: 0 withMax: tapeXSize-1];
  ypos = [uniformIntRand getIntegerWithMin: 0 withMax: tapeYSize-1];
  
  
  // set random state
  
  //  state = [uniformRandom rMax: numStates];
  state = [uniformIntRand getIntegerWithMin: 0 withMax: numStates-1];
  
  // set random transition table - bit of a hack using offset (defined at create).
  
  for (i = 0; i < offset; i++ )
    {
      *(transitionTable + i) =
        [uniformIntRand getIntegerWithMin: 0 withMax: numStates - 1];
      *(transitionTable + offset + i) =
        [uniformIntRand getIntegerWithMin: 0 withMax: numMoves - 1];
      *(transitionTable + offset2 + i) = 
        [uniformIntRand getIntegerWithMin: 0 withMax: numTapeStates - 1];
    }
  return self;
}


// All of the previous code is basic Swarm object programming. The
// real simulation code follows here.

// Turmite behaviour is actually implemented here. The notion of 
// a "step" method is a nice simplification for basic simulations.
// Some of this code could potentially be moved to a generic 2d
// FSM, and a lot of it could be cleaner if the space libary were
// smarter about coordinates, implemented boundary conditions and
// coordinate arithmetic and the like.


- step
{
  unsigned int index, move;

  
  // First get the state of the tape-cell we are sitting on.....
  
  input = [tape getValueAtX: xpos Y: ypos];
  
  // now use our state and the input to lookup what to do in the table
  // offsets were cached at create time....
  // we shift left the internal state by the log of numTapeStates
  // which was cached at create time - then we OR in the tape input to
  // get the lookuptable index. This means that only powers of
  // two should be used for numTapeStates and numStates.
  
  index = ((state << logt) | input );  	// yikes!! 
  
  state = transitionTable[index];			// multiply offset by 0
  move = transitionTable[offset + index];		// multiply offset by 1
  output = transitionTable[offset2 + index];            // multiply offset by 2
  
  // new state is already set
  
  // write output to tape - 
  // Output of last turmite to write is what mites on this cell see next
  // time step.
  
  [tape putValue: output atX: xpos Y: ypos];
  
  // now do the move.....
  
  switch (move)
    {
    case 0:  ypos++; break;			// move up
    case 1:  xpos++; break;			// move right
    case 2:  ypos--; break;			// move down
    case 3:  xpos--; break;			// move left
    default: xpos = xpos; ypos = ypos;		// if move illegal, stay put!
    }
  
  xpos = (xpos + tapeXSize) % tapeXSize;		// wrap if necessary
  ypos = (ypos + tapeYSize) % tapeYSize;
  
  // all done updating this tmite !  Return self.

  return self;
}

// Extra bit of display code: drawing on a window 
// Color = state

// This code works, but it'd be better if there were a generic object
// that knew how to draw agents on grids.

- drawSelfOn: (id <Raster>)r
{
  [r drawPointX: xpos Y: ypos Color: (state+02)];	// use state for color
  return self;
}

@end
