/* 
  FHPNeighborhood.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  FEB-1997

*/

#import "FHPneighborhood.h"
#import <random.h>

@implementation FHPneighborhood

+ create: aZone setGenerator: generator
{
  FHPneighborhood * neighborhood;
  
  neighborhood = [super create: aZone];
  neighborhood->uniformUnsigned =
    [UniformUnsignedDist create: aZone setGenerator: generator];
  
  return neighborhood;
}

- (void)drop
{
  if (uniformUnsigned != nil)
    [uniformUnsigned drop];
  [super drop];
}

- (NeighborCode)getInitialNeighborCode
{
  return TriNorth;
}

- (BOOL)isValidNeighborCode: (NeighborCode)inNeighborCode
{
  BOOL isValid;
  
  switch (inNeighborCode)
    {
    case TriNorth: 
    case TriNorthEast:  
    case TriSouthEast: 
    case TriSouth:
    case TriSouthWest:
    case TriNorthWest:  
      isValid = YES;
      break;
    default:
      isValid = NO;
    }
  
  return isValid;
}

/* 
   The -next, -previous and -randomise methods *could* be made
   marginally more efficient if they took advantage of the
   underlying ordering of the TriNeighborCode enumerated type;
   but that would then mean that any change in that underlying
   ordering would (silently) break the TriNeighborhood class.
   I prefer to accept the small performace penalty in making
   this class robust against changes in the underlying ordering
   of TriNeighborCode.
*/


- (NeighborCode)getNextNeighborCode: (NeighborCode)inNeighborCode
{
  NeighborCode nextNeighborCode;
  
  switch (inNeighborCode)
    {
    case TriNorth:      nextNeighborCode = TriNorthEast;   break;
    case TriNorthEast:  nextNeighborCode = TriSouthEast;   break;
    case TriSouthEast:  nextNeighborCode = TriSouth;       break;
    case TriSouth:      nextNeighborCode = TriSouthWest;   break;
    case TriSouthWest:  nextNeighborCode = TriNorthWest;   break;
    case TriNorthWest:  nextNeighborCode = TriNorth;       break;
    default:
      [InternalError raiseEvent:
                       "Invalid Tri neighbor code (%d) encountered.\n", 
                     inNeighborCode];
      nextNeighborCode = TriNorth; // Defensive...
    }
  
  return nextNeighborCode;
}

- (NeighborCode)getPreviousNeighborCode: (NeighborCode)inNeighborCode
{
  NeighborCode previousNeighborCode;
  
  switch (inNeighborCode)
    {
    case TriNorth:     previousNeighborCode = TriNorthWest;    break;
    case TriNorthWest: previousNeighborCode = TriSouthWest;    break;
    case TriSouthWest: previousNeighborCode = TriSouth;        break;
    case TriSouth:     previousNeighborCode = TriSouthEast;    break;
    case TriSouthEast: previousNeighborCode = TriNorthEast;    break;
    case TriNorthEast: previousNeighborCode = TriNorth;        break;
    default:
      [InternalError raiseEvent:
                       "Invalid Tri neighbor code (%d) encountered.\n", 
                     inNeighborCode];
      previousNeighborCode = TriNorth; // Defensive...
    }

  return previousNeighborCode;
}

- (NeighborCode)getOppositeNeighborCode: (NeighborCode)inNeighborCode
{
  NeighborCode oppositeNeighborCode;
  
  switch (inNeighborCode)
    {
    case TriNorth:      oppositeNeighborCode = TriSouth;         break;
    case TriNorthEast:  oppositeNeighborCode = TriSouthWest;     break;
    case TriSouthEast:  oppositeNeighborCode = TriNorthWest;     break;
    case TriSouth:      oppositeNeighborCode = TriNorth;         break;
    case TriSouthWest:  oppositeNeighborCode = TriNorthEast;     break;
    case TriNorthWest:  oppositeNeighborCode = TriSouthEast;     break;
    default:
      [InternalError raiseEvent:
                       "Invalid Tri neighbor code (%d) encountered.\n", 
                     inNeighborCode];
      oppositeNeighborCode = TriNorth; // Defensive...
    }
  
  return (oppositeNeighborCode);
}

- (NeighborCode)getRandomNeighborCode
{
  unsigned randomNumber;
  NeighborCode randomNeighborCode;
  
  randomNeighborCode = TriNorth; // Defensive...
  
  if (uniformUnsigned == nil)
    [InternalError
      raiseEvent:
        "Attempt to -getRandomNeighborCode, but uniformUnsigned is nil.\n"];
  else
    {
      randomNumber = [uniformUnsigned getUnsignedWithMin: 0 withMax: 5];
      switch (randomNumber)
        {
        case 0: randomNeighborCode = TriNorth;        break;
        case 1: randomNeighborCode = TriNorthEast;    break;
        case 2: randomNeighborCode = TriSouthEast;    break;
        case 3: randomNeighborCode = TriSouth;        break;
        case 4: randomNeighborCode = TriSouthWest;    break;
        case 5: randomNeighborCode = TriNorthWest;    break;
        default:
          [InternalError
            raiseEvent:
              "Out-of-range random number (%d) encountered.\n", randomNumber];
        }
    }
  
  return randomNeighborCode;
}

- (NeighborCode)getRandomNeighborCodeExcluding: (NeighborCode)inNeighborCode
{
  // This is a cheap `n cheerful, totally unoptimised version...
  
  unsigned randomNumber;
  NeighborCode randomNeighborCode;
  
  randomNeighborCode = [self getNextNeighborCode: inNeighborCode];
  
  if (uniformUnsigned == nil)
    [InternalError raiseEvent:
                     "Attempt to -getRandomNeighborCodeExcluding,\n"
                   "but uniformUnsigned is nil.\n"];
  else
    {
      randomNumber = [uniformUnsigned getUnsignedWithMin: 0 withMax: 4];
      while (randomNumber > 0)
        {
          randomNeighborCode = [self getNextNeighborCode: inNeighborCode];
          randomNumber--;
        }
    }
  
  return randomNeighborCode;
}

@end
