/* 
  LatticeAgent.h

  Barry McMullin <mcmullin@eeng.dcu.ie>
  FEB-1997


  This class provides an object abstraction for the
  idea of an "agent" embedded in a discrete
  lattice. LatticeAgent objects are things that can
  retain their individuality (more specifically, their
  *state*) as they move around in a Lattice (discrete
  space).

  This class is not of much use in itself; it is rather
  intended for subclassing to implement specific kinds
  of agents.

  An LatticeAgent object exists at a position in a
  Lattice.  It has methods to move itself to other
  positions in its "neighborhood" (defined in more
  detail below). LatticeAgent objects won't share
  positions in space - so if you tell one to move to
  somewhere that is already occupied (i.e. by another
  LatticeAgent) this will cause an InternalError
  exception to be raised, and terminate the program.  A
  LatticeAgent also has methods for accessing or
  referencing other LatticeAgent's in its neighborhood.

  Several of the LatticeAgent methods require an
  argument specifying a "neighboring" position.  This
  argument should be a value of the an enumerated type
  appropriate to the particular class of Lattice (e.g.,
  see SqrNeighborCode.h, TriLatticeCode.h).

*/

#import "Coord.h"
#import "Agent.h"
#import <simtools.h>

@interface LatticeAgent: Agent
{
  id coord;
  /* If this is nil the agent is not currently embedded
     in any Lattice. */
}

- copyCoord: aZone;

- (BOOL)attemptUnwarpToCoord: newCoord;
/* This embeds the agent at the specified absolute
  coordinate.  It will fail (raising InternalError) if
  the agent is *already* embedded, or if the specified
  coordinate is already occupied.  Its primary intended
  use is for setting initial agent coordinate
  immediately after creation; thereafter, motion is
  most robustly accomplished using the -swap methods.  */

- warp;
/* This "pulls" the agent out of world - i.e. sets the
  coordinate to nil. self->coord is dropped. Has no
  effect if agent is not already embedded.  */


- moveToCoord: newCoord;
/* Will fail (raising InternalError) if the destination
  coordinate is already occupied.  */

- moveToNeighbor: (NeighborCode)neighborCode;
/* Will fail (raising InternalError) if self->coord ==
  nil or if the destination coordinate is already
  occupied.  */


- (BOOL)attemptSwapWithAgent: (LatticeAgent *)agent;
/*
  Returns YES if the swap is completed, NO otherwise.

  Will fail (raising InternalError) if self->coord ==
  nil, or agent == nil, or agent->coord == nil. */

- (BOOL)attemptSwapToCoord: newCoord;
- (BOOL)attemptSwapToNeighbor: (NeighborCode)neighborCode;
/*
  Both return YES if the swap is completed, NO otherwise.

  Both will work regardless of whether the specified position
  contains another agent or is nil.

  Both will fail (raising InternalError) 
  if self->coord == nil.
*/
  

- saveTo: (id <OutFile>)file;   // Overridden...
- loadFrom: (id <InFile>)file toLattice: inLattice;  

@end
