/* 
  ModelSwarm.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  JAN-1997

*/

#import "Bernoulli.h"
#import "Globals.h"
#import "Lattice2d.h"
#import "Coord2d.h"
#import "Tipsybug.h"
#import "ModelSwarm.h"

#import <simtoolsgui.h>
#import <random.h>

@implementation ModelSwarm

+ createBegin: aZone
{
  ModelSwarm *obj;
  id <ProbeMap> probeMap;

  obj = [super createBegin: aZone];

  obj->numBugs = 100;
  obj->latticeXSize = 80;
  obj->latticeYSize = 80;
  obj->randomMoveProbability = 0.529123;
  obj->tilingCode = SqrTiling;

  // And build a customized probe map. Without a probe map, the default
  // is to show all variables and messages. Here we choose to
  // customize the appearance of the probe, give a nicer interface.

  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [self class]];
  probeMap = [probeMap createEnd];

  [probeMap addProbe: [probeLibrary getProbeForVariable: "numBugs"
				    inClass: [self class]]];
    [probeMap addProbe: [probeLibrary getProbeForVariable: "latticeXSize"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "latticeYSize"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "rasterScaleFactor"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "tilingCode"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "neighborhoodCode"
				    inClass: [self class]]];
  [probeMap addProbe: [[probeLibrary getProbeForVariable: "randomMoveProbability"
				    inClass: [self class]]
                         setFloatFormat: "%.3f"]];
  
  // Now install our custom probeMap into the probeLibrary.

  [probeLibrary setProbeMap: probeMap For: [self class]];
  
  return obj;
}

- (void)refreshNeighborhoodCode
{
  if (lattice != nil)
    {
      [lattice setNeighborhoodCode: neighborhoodCode];
      tilingCode = [lattice getTilingCode];
      [self refreshDisplay];
    }
}

- (void)setVNneighborhood
{
  neighborhoodCode = VN;
  [self refreshNeighborhoodCode];
}


- (void)setMooreNeighborhood
{
  neighborhoodCode = Moore;
  [self refreshNeighborhoodCode];
}

- (void)setFHPneighborhood
{
  neighborhoodCode = FHP;
  [self refreshNeighborhoodCode];
}

- (void)refreshTilingCode
{
  if (lattice != nil)
    {
      [lattice setTilingCode: tilingCode];
      neighborhoodCode = [lattice getNeighborhoodCode];
      [self refreshDisplay];
    }
}

- (void)setSqrTiling
{
  tilingCode = SqrTiling;
  [self refreshTilingCode];
}

- (void)setTriTiling
{
  tilingCode = TriTiling;
  [self refreshTilingCode];
}

- (void)refreshDisplay
{
  Color color;
  int tbugIndex, tbugCount;
  Tipsybug *tbug;
  id tbugCoord;
  int rasterX, rasterY;

  [latticeRaster erase];

  tbugCount = [tipsybugList getCount];
  for (tbugIndex = 0; tbugIndex < tbugCount; tbugIndex++)
    {
      tbug = [tipsybugList atOffset: tbugIndex];
      tbugCoord = [tbug copyCoord: [self getZone]];
      rasterX = [tbugCoord getRasterX];
      rasterY = [tbugCoord getRasterY];

      if ([tbug getMark] == 0) color = 0;
      else color = 1;
      [latticeRaster drawPointX: rasterX Y: rasterY Color: color];
    }
  
  [latticeRaster drawSelf];
  [probeDisplayManager update];
}


- (void)makeProbeForAgentAtX: (int)rasterX Y: (int)rasterY
{
  id agent;
  id coord;
  
  coord = [lattice createOriginCoord: [self getZone]];
  [coord setRasterX: rasterX];
  [coord setRasterY: rasterY];
  agent = [coord getObject];
  [coord drop];
  
  if (agent)
    [probeDisplayManager createProbeDisplayFor: agent];
  else
    GUI_BEEP ();
}

- (void)toggleMarkForAgentAtX: (int)rasterX Y: (int)rasterY
{
  id agent;
  id coord;
  
  coord = [lattice createOriginCoord: [self getZone]];
  [coord setRasterX: rasterX];
  [coord setRasterY: rasterY];
  agent = [coord getObject];
  [coord drop];
  
  if (agent)
    {
      if ([agent getMark] == 0)
        [agent setMark: 1];
      else
        [agent setMark: 0];
    }
  else
    GUI_BEEP ();
  
  [self refreshDisplay];
}

- createEnd
{
  ModelSwarm *finalSelf;
  
  finalSelf = [super createEnd];
  finalSelf->tilingCode = SqrTiling;
  finalSelf->neighborhoodCode = VN;
  finalSelf->rasterScaleFactor = 2;
  finalSelf->modelTime = 0;
  [probeDisplayManager createProbeDisplayFor: finalSelf];
  
  return (finalSelf);
}

- buildObjects
{
  int i;
  unsigned currentNeighborCode;
  BOOL done;
  Tipsybug *tbug;
  id coord, sortie;
  
  [super buildObjects];
  
  prng = [PMMLCG1 create: [self getZone] setStateFromSeed: 42]; 
  bernoulli = [Bernoulli create: [self getZone] setGenerator: prng];
  
  colormap = [Colormap create: [self getZone]];
  
  [colormap setColor: 0 ToName: "white"];
  [colormap setColor: 1 ToName: "red"];
  [colormap setColor: 2 ToName: "LimeGreen"];
  [colormap setColor: 3 ToName: "blue"];
  [colormap setColor: 4 ToName: "SkyBlue"];
  [colormap setColor: 5 ToName: "SteelBlue"];
  [colormap setColor: 6 ToName: "cyan"];
  [colormap setColor: 7 ToName: "black"];

  // Instruct the control panel to wait for a button
  // event: we halt here until someone hits a control
  // panel button so the user can get a chance to fill
  // in parameters before the simulation runs...
  [self setStateStopped];

  lattice = [Lattice2d createBegin: [self getZone]];
  [lattice setSizeX: latticeXSize Y: latticeYSize];
  [lattice setGenerator: prng];
  lattice = [lattice createEnd];

  [lattice setRasterScaleFactor: rasterScaleFactor];

  // Pass on the tiling and neighborhood settings to
  // the lattice.  Note that these two settings
  // potentially interact - so the requested settings
  // may *not* take effect...
  [self refreshTilingCode];
  [self refreshNeighborhoodCode];

  tipsybugList = [List create: [self getZone]];
  
  // Now a loop to create a bunch of tipsybugs.
  coord = [lattice createOriginCoord: [self getZone]];
  [coord setX: (latticeXSize / 2)];
  [coord setY: (latticeYSize / 2)];
  i = 0;
  currentNeighborCode = [lattice getInitialNeighborCode];
  done = NO;
  while (!done)
    {
      tbug = [Tipsybug create: [self getZone]];
      
      if (![tbug attemptUnwarpToCoord: coord])
        [InternalError
          raiseEvent: 
            "Failed unwarp of tbug %08x.\n", tbug];
      
      [tbug setRandomMoveProbability: randomMoveProbability];
      [tipsybugList addLast: tbug];
      
      i++;
      done = (i >= numBugs);
      
      if (!done)
        {
          // Figure position for next bug; I'm trying to
          // lay them out in a compact, spiral, pattern -
          // the exact pattern will depend on the
          // neighborhood set in the lattice.
          
          sortie = [coord copy: [self getZone]];
          [sortie translateByNeighborCode: currentNeighborCode];
          if ([[sortie getObject] 
                isMemberOf: [Tipsybug class]]) {
            // Oops - can't turn yet...
            currentNeighborCode = 
              [lattice getPreviousNeighborCode: currentNeighborCode];
            [sortie drop];
            sortie = [coord copy: [self getZone]];
            [sortie translateByNeighborCode: currentNeighborCode];
            if ([[sortie getObject] 
                  isMemberOf: [Tipsybug class]])
              [InternalError
                raiseEvent: 
                  "Unexpected Tipsybug object at next neighboring position.\n"];
          }
          [coord drop];
          coord = sortie;
          sortie = nil;
          currentNeighborCode = 
	[lattice getNextNeighborCode: currentNeighborCode];
        }
    }
  
  latticeRaster = [ZoomRaster create: [self getZone]];
  [latticeRaster setColormap: colormap];
  [latticeRaster setZoomFactor: 2];
  [latticeRaster setWidth: ([lattice getSizeX] * rasterScaleFactor) 
                 Height: ([lattice getSizeY] * rasterScaleFactor) ];
  [latticeRaster setWindowTitle: "Tipsy Lattice"];
  [latticeRaster pack];
  
  [latticeRaster setButton: ButtonRight Client: self
                 Message: M(makeProbeForAgentAtX:Y:)];
  [latticeRaster setButton: ButtonLeft Client: self
                 Message: M(toggleMarkForAgentAtX:Y:)];
  
  [self refreshDisplay];
  [self setStateStopped];
  
  return self;
}


- (void)step
{
  modelTime++;
  [tipsybugList forEach: M(step)];
  
  [actionCache doTkEvents];
  [self refreshDisplay];
}

- buildActions
{
  [super buildActions];
  
  modelActions = [ActionGroup create: [self getZone]];
  
  [modelActions createActionTo: self message: M(step)];

  modelSchedule = [Schedule createBegin: [self getZone]];
  [modelSchedule setRepeatInterval: 1];
  modelSchedule = [modelSchedule createEnd];
  [modelSchedule at: 0 createAction: modelActions];

  return self;
}

- activateIn: swarmContext
{
  [super activateIn: swarmContext];
  
  [modelSchedule activateIn: self];
  
  return [self getSwarmActivity];
}

- (void)setStateStopped
{
  /* 
    I don't simply use [controlPanel setStateStopped]
    because if we are already stopped that forces a wait
    for a click on some control panel button other than
    stop ... *not* what I want!
  */
  
  [controlPanel setState: ControlStateStopped];
  [actionCache waitForControlEvent];
}


@end
