/* 
  Lattice2d.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  FEB-1997

*/

#import "VNneighborhood.h"
#import "MooreNeighborhood.h"
#import "FHPneighborhood.h"
#import "Coord2d.h"
#import "Lattice2d.h"

@implementation Lattice2d

- setGenerator: generator
{
  if (neighborhood != nil)
    [InternalError raiseEvent:
      "Lattice2d -setGenerator can only be used at create phase.\n"];
  else
    prng = generator;

  return self;
}

- setSizeX: (int)x Y: (int)y
{
  if ((x % 2) != 0)
    [InternalError raiseEvent:
                     "Lattice2d xsize *must* be even.\n"];
  // This is a restriction of the specific way support
  // for the toroidal triangular tiling has been
  // implemented...
  
  return [super setSizeX: x Y: y];
}

- (Lattice2d *)createEnd
{
  Lattice2d * finalSelf;
  
  finalSelf = [super createEnd];
  finalSelf->tilingCode = SqrTiling;
  if (prng == nil)
    finalSelf->neighborhood = 
      [VNneighborhood create: [self getZone]];
  else
    finalSelf->neighborhood = 
      [VNneighborhood create: [self getZone] setGenerator: prng];
  finalSelf->neighborhoodCode = VN;
  finalSelf->rasterScaleFactor = 2; // Default - may not be sensible...
  
  return finalSelf;
}

+ create: aZone withSizeX: (int)sizeX Y: (int)sizeY
{
  Lattice2d *lattice2d;
  
  lattice2d = [self createBegin: aZone];
  [lattice2d setSizeX: sizeX Y: sizeY];
  lattice2d = [lattice2d createEnd];

  return lattice2d;
}

- (void)drop
{
  if (neighborhood != nil)
    [neighborhood drop];
  [super drop];
}

/* The following methods provide for forwarding
of messages in the <Neighborhood> protocol.  Most of this
could also be done with a -forward method - perhaps
more simply and/or efficiently.  But doing it this way
ensures that there is complete flexibility in the
method interfaces (any arguments, any return type);
and also, incidentally, saves the compiler from
getting upset that the protocol doesn't seem to be fully
implemented.  Also, there seems to be a (benign)
declaration conflict for the method -performv:, used
in forwarding, between Object.h and tclObjc.h...  */

+ create: aZone setGenerator: generator
{
  [InternalError
    raiseEvent:
      "+create:setGenerator: not supported for Lattice classes.\n"];
  
  return nil;
}

- (BOOL)isValidNeighborCode: (NeighborCode)inNeighbor
{
  return [neighborhood isValidNeighborCode: inNeighbor];
}

- (NeighborCode)getInitialNeighborCode
{
  return [neighborhood getInitialNeighborCode];
}

- (NeighborCode)getNextNeighborCode: (NeighborCode)inNeighbor
{
  return [neighborhood getNextNeighborCode: inNeighbor];
}

- (NeighborCode)getPreviousNeighborCode: (NeighborCode)inNeighbor
{
  return [neighborhood getPreviousNeighborCode: inNeighbor];
}

- (NeighborCode)getOppositeNeighborCode: (NeighborCode)inNeighbor
{
  return [neighborhood getOppositeNeighborCode: inNeighbor];
}

- (NeighborCode)getRandomNeighborCode
{
  return [neighborhood getRandomNeighborCode];
}

- (NeighborCode)getRandomNeighborCodeExcluding: (NeighborCode)inNeighbor
{
  return [neighborhood getRandomNeighborCodeExcluding: inNeighbor];
}

- (void)setRasterScaleFactor: (unsigned)newScaleFactor
{
  rasterScaleFactor = newScaleFactor;
}

- (unsigned)getRasterScaleFactor
{
  return rasterScaleFactor;
}

- (TilingCode)getTilingCode
{
  return tilingCode;
}

- (void)setTilingCode: (TilingCode)newTilingCode
{
  if (newTilingCode == SqrTiling)
    {
      tilingCode = newTilingCode;
      neighborhoodCode = VN;
      [neighborhood drop];
      if (prng == nil)
        neighborhood = 
          [VNneighborhood create: [self getZone]];
      else
        neighborhood = 
          [VNneighborhood create: [self getZone] setGenerator: prng];
    }
  else if (newTilingCode == TriTiling)
    {
      tilingCode = newTilingCode;
      neighborhoodCode = FHP;
      [neighborhood drop];
      if (prng == nil)
        neighborhood = 
          [FHPneighborhood create: [self getZone]];
      else
        neighborhood = 
          [FHPneighborhood create: [self getZone] setGenerator: prng];
    }
  else
    [InternalError raiseEvent:
                     "Invalid TilingCode %d.\n", newTilingCode];
}

- (NeighborhoodCode)getNeighborhoodCode
{
  return neighborhoodCode;
}

- (void)setNeighborhoodCode: (NeighborhoodCode)newNeighborhoodCode
{
  switch (newNeighborhoodCode)
    {
    case VN:    
      neighborhoodCode = newNeighborhoodCode;
      [neighborhood drop];
      if (prng == nil)
        neighborhood = 
          [VNneighborhood create: [self getZone]];
      else
        neighborhood = 
          [VNneighborhood create: [self getZone] setGenerator: prng];
      tilingCode = SqrTiling;
      break;
    case Moore: 
      neighborhoodCode = newNeighborhoodCode;
      [neighborhood drop];
      if (prng == nil)
      neighborhood = 
	[MooreNeighborhood create: [self getZone]];
      else
        neighborhood = 
          [MooreNeighborhood create: [self getZone] setGenerator: prng];
      tilingCode = SqrTiling;
      break;
    case FHP:   
      neighborhoodCode = newNeighborhoodCode;
      [neighborhood drop];
      if (prng == nil)
        neighborhood = 
          [FHPneighborhood create: [self getZone]];
      else
        neighborhood = 
          [FHPneighborhood create: [self getZone] setGenerator: prng];
      tilingCode = TriTiling;
      break;
    default:
      [InternalError raiseEvent:
                       "Invalid NeighborhoodCode %d.\n", newNeighborhoodCode];
    }
}


- createOriginCoord: aZone
{
  Coord2d *coord;

  coord = [Coord2d create: aZone withLattice: self];
  [coord setX: 0];  // Just to be explicit...
  [coord setY: 0];
  
  return (coord);
}

- createMaxCoord: aZone
{
  Coord2d *coord;
  
  coord = [Coord2d create: aZone withLattice: self];
  [coord setX: (xsize - 1)];
  [coord setY: (ysize - 1)];
  
  return (coord);
}

@end
