// Swarm library. Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

/* 
  VNneighbor.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  26-SEP-1996

*/

#import <simtools.h>
#import "VNneighbor.h"


@implementation VNneighbor


-(BOOL) isValidNeighborCode: (neighbor_t) inNeighbor {
  BOOL isValid;

  switch (inNeighbor) {
    case North: 
    case East:  
    case South: 
    case West:  
      isValid = YES;
      break;
    default:
      isValid = NO;
  }

  return(isValid);
}




/* 
   The -next, -previous and -randomise methods *could* be made
   marginally more efficient if they took advantage of the
   underlying ordering of the neighbor_t enumerated type;
   but that would then mean that any change in that underlying
   ordering would (silently) break the VNneighbor class.
   I prefer to accept the small performace penalty in making
   this class robust against changes in the underlying ordering
   of neighbor_t.
*/


-(neighbor_t) getNext: (neighbor_t) inNeighbor {
  neighbor_t nextNeighbor;

  switch (inNeighbor) {
  case North: nextNeighbor = East;    break;
  case East:  nextNeighbor = South;   break;
  case South: nextNeighbor = West;    break;
  case West:  nextNeighbor = North;   break;
  default:
    [InternalError raiseEvent:
     "Invalid von Neumann neighbor code (%d) encountered.\n", 
     inNeighbor];
    nextNeighbor = North; // Defensive...
  }

  return (nextNeighbor);
}

-(neighbor_t) getPrevious: (neighbor_t) inNeighbor {
  neighbor_t previousNeighbor;

  switch (inNeighbor) {
  case North: previousNeighbor = West;    break;
  case West:  previousNeighbor = South;   break;
  case South: previousNeighbor = East;    break;
  case East:  previousNeighbor = North;   break;
  default:
    [InternalError raiseEvent:
     "Invalid von Neumann neighbor code (%d) encountered.\n", 
      inNeighbor];
    previousNeighbor = North; // Defensive...
  }

  return (previousNeighbor);
}

-(neighbor_t) getOpposite: (neighbor_t) inNeighbor {
  neighbor_t oppositeNeighbor;

  switch (inNeighbor) {
  case North: oppositeNeighbor = South;    break;
  case West:  oppositeNeighbor = East;     break;
  case South: oppositeNeighbor = North;    break;
  case East:  oppositeNeighbor = West;   break;
  default:
    [InternalError raiseEvent:
     "Invalid von Neumann neighbor code (%d) encountered.\n", 
      inNeighbor];
    oppositeNeighbor = North; // Defensive...
  }

  return (oppositeNeighbor);

}

-(neighbor_t) getRandom {
  int randomNumber;
  neighbor_t randomNeighbor;

  randomNumber = [uniformRandom rMin: 0 Max: 4];
  switch (randomNumber) {
  case 0: randomNeighbor = North;    break;
  case 1: randomNeighbor = East;     break;
  case 2: randomNeighbor = South;    break;
  case 3: randomNeighbor = West;     break;
  default:
    [InternalError raiseEvent:
     "Out-of-range random number (%d) encountered.\n", randomNumber];
    randomNeighbor = North; // Defensive...
  }

  return (randomNeighbor);
}

-(neighbor_t) getRandomExcluding: (neighbor_t) inNeighbor {
  // This is a cheap `n cheerful, totally unoptimised version...

  int randomNumber;
  neighbor_t randomNeighbor;

  randomNumber = [uniformRandom rMin: 0 Max: 3];
  randomNeighbor = [self getNext: inNeighbor];
  while (randomNumber > 0) {
    randomNeighbor = [self getNext: inNeighbor];
    randomNumber--;
  }

  return (randomNeighbor);
}


@end
