// Swarm library. Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

/* 
  MooreNeighbor.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  08-OCT-1996

*/

#import <simtools.h>
#import "MooreNeighbor.h"


@implementation MooreNeighbor


-(BOOL) isValidNeighborCode: (neighbor_t) inNeighbor {
  BOOL isValid;

  switch (inNeighbor) {
    case North: 
    case NorthEast:
    case East:  
    case SouthEast:
    case South: 
    case SouthWest:
    case West:  
    case NorthWest:
      isValid = YES;
      break;
    default:
      isValid = NO;
  }

  return(isValid);
}





-(neighbor_t) getNext: (neighbor_t) inNeighbor {
  neighbor_t nextNeighbor;

  switch (inNeighbor) {
  case North:     nextNeighbor = NorthEast;    break;
  case NorthEast: nextNeighbor = East;         break;
  case East:      nextNeighbor = SouthEast;    break;
  case SouthEast: nextNeighbor = South;        break;
  case South:     nextNeighbor = SouthWest;    break;
  case SouthWest: nextNeighbor = West;         break;
  case West:      nextNeighbor = NorthWest;    break;
  case NorthWest: nextNeighbor = North;        break;
  default:
    [InternalError raiseEvent:
     "Invalid Moore neighbor code (%d) encountered.\n", 
     inNeighbor];
    nextNeighbor = North; // Defensive...
  }

  return (nextNeighbor);
}

-(neighbor_t) getPrevious: (neighbor_t) inNeighbor {
  neighbor_t previousNeighbor;

  switch (inNeighbor) {
  case North:     previousNeighbor = NorthWest;    break;
  case NorthWest: previousNeighbor = West;         break;
  case West:      previousNeighbor = SouthWest;    break;
  case SouthWest: previousNeighbor = South;        break;
  case South:     previousNeighbor = SouthEast;    break;
  case SouthEast: previousNeighbor = East;         break;
  case East:      previousNeighbor = NorthEast;    break;
  case NorthEast: previousNeighbor = North;        break;
  default:
    [InternalError raiseEvent:
     "Invalid Moore neighbor code (%d) encountered.\n", 
      inNeighbor];
    previousNeighbor = North; // Defensive...
  }

  return (previousNeighbor);
}

-(neighbor_t) getOpposite: (neighbor_t) inNeighbor {
  neighbor_t oppositeNeighbor;

  switch (inNeighbor) {
  case North:     oppositeNeighbor = South;        break;
  case NorthEast: oppositeNeighbor = SouthWest;    break;
  case West:      oppositeNeighbor = East;         break;
  case NorthWest: oppositeNeighbor = SouthEast;    break;
  case South:     oppositeNeighbor = North;        break;
  case SouthEast: oppositeNeighbor = NorthWest;    break;
  case East:      oppositeNeighbor = West;         break;
  case SouthWest: oppositeNeighbor = NorthEast;    break;
  default:
    [InternalError raiseEvent:
     "Invalid Moore neighbor code (%d) encountered.\n", 
      inNeighbor];
    oppositeNeighbor = North; // Defensive...
  }

  return (oppositeNeighbor);

}

-(neighbor_t) getRandom {
  int randomNumber;
  neighbor_t randomNeighbor;

  randomNumber = [uniformRandom rMin: 0 Max: 8];
  switch (randomNumber) {
  case 0: randomNeighbor = North;        break;
  case 1: randomNeighbor = NorthEast;    break;
  case 2: randomNeighbor = East;         break;
  case 3: randomNeighbor = SouthEast;    break;
  case 4: randomNeighbor = South;        break;
  case 5: randomNeighbor = SouthWest;    break;
  case 6: randomNeighbor = West;         break;
  case 7: randomNeighbor = NorthWest;    break;
  default:
    [InternalError raiseEvent:
     "Out-of-range random number (%d) encountered.\n", randomNumber];
    randomNeighbor = North; // Defensive...
  }

  return (randomNeighbor);
}

-(neighbor_t) getRandomExcluding: (neighbor_t) inNeighbor {
  // This is a cheap `n cheerful, totally unoptimised version...

  int randomNumber;
  neighbor_t randomNeighbor;

  randomNumber = [uniformRandom rMin: 0 Max: 7];
  randomNeighbor = [self getNext: inNeighbor];
  while (randomNumber > 0) {
    randomNeighbor = [self getNext: inNeighbor];
    randomNumber--;
  }

  return (randomNeighbor);
}


@end
