// Copyright (C) 1997 The Santa Fe Institute, JJ Merelo
// No warranty implied, see LICENSE for terms.


#import <collections.h>
#import <random.h>

#import "AgentSwarm.h"

#include <stdlib.h>		// for atof

#define STRSIZE 20

@implementation AgentSwarm

// Comparison function
int compFunc( id a, id b ) {

    float tmp1, tmp2;    
    sscanf( (char*) a, "%f", &tmp1 );
    sscanf( (char*) b, "%f", &tmp2 );
    if ( tmp1 > tmp2 ) 
	return -1;
    else
	return 1;
}

// ------------------------------ Swarm functions --------------

+ createBegin: aZone
{

  AgentSwarm *obj;
  id <ProbeMap> probeMap;

  obj = [super createBegin:aZone];

  // init defaults
  obj->initPopSize = 100;
  //  obj->generations=1000;
  obj->popSize=obj->initPopSize; // Without probe
  obj->mating = 0.1;		// Good for the demo proble
  obj->cloning = 0.1;		// Copying without doing anything
  obj->mut = 0.05;
  obj->dup = 0;
  obj->split = 0;
  obj->verbose= 0;		// Print genealogy or not
  obj->numIterations = 100;
  obj->minMemory = 1;
  obj->maxMemory = 5;

  // Create probes
  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [self class]];
  probeMap = [probeMap createEnd];

  // MODIFY: add your own probes for your parameters.
  [probeMap addProbe: [probeLibrary getProbeForVariable: "initPopSize"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "numIterations"
				     inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "minMemory"
				     inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "maxMemory"
				     inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "mating"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "cloning"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "mut"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "dup"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "split"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "verbose"
				    inClass: [self class]]];

  // Message probes
  [probeMap addProbe: [probeLibrary getProbeForMessage: "addPopulation:"
				    inClass: [self class]]];
  
  [probeLibrary setProbeMap: probeMap For: [self class]];
  
  return obj;
}

// __________________________
// Build simulation objects
-buildObjects {

  [super buildObjects];

  // Unevaluated guys
  popList = [List create:[self getZone]];
  // Create initial random population
  [self addPopulation: initPopSize];

  // Create intial map, to keep evaluated genes
  evalPopList = [Map createBegin: [self getZone]];
  [evalPopList setCompareFunction: compFunc];
  evalPopList = [evalPopList createEnd];

  return self;
}

-buildActions {
  [super buildActions];

  // MODIFY: schedule your own agents as appropriate.
  modelActions = [ActionGroup create: [self getZone]];
  [modelActions createActionTo: self message: M(evaluate)];
  [modelActions createActionTo: self message: M(newGeneration) ];

  modelSchedule = [Schedule createBegin: [self getZone]];
  [modelSchedule setRepeatInterval: 1];
  modelSchedule = [modelSchedule createEnd];
  [modelSchedule at: 0 createAction: modelActions];
  
  return self;
}

//_______________________________________________________________
-activateIn: (id) swarmContext  {
//_______________________________________________________________
  [super activateIn: swarmContext];
  [modelSchedule activateIn: self];
  return [self getSwarmActivity];
}

-createEnd {
  return [super createEnd];
}

// From protocol
//_______________________________________________________________________
-(void) addPopulation: (unsigned) _num {
				// Adds new chromosomes to the population
				// Set popSize variable
//_______________________________________________________________________

  unsigned i;
  unsigned max32=0x6fffffff;
  for ( i = 0; i < _num; i++ ) {
    EvolvableIPDagent* gen;
    unsigned mem, strategy;
    gen = [EvolvableIPDagent createBegin: [self getZone]];
    if ( minMemory != maxMemory ) 
      mem = [uniformUnsRand getUnsignedWithMin: minMemory withMax: maxMemory ];
    else 
      mem = minMemory;    
    strategy = [uniformUnsRand getUnsignedWithMin: 0L withMax: max32 ]; 
    [gen setStrategy: strategy withMemory: mem];
    gen = [gen createEnd];
    [popList addFirst: gen];
  }
  popSize+= _num;
    
}

//_______________________________________________________________
-evaluate
{			// Evaluates all member of the population
//_______________________________________________________________

  // Takes them from the new list and places then in the Population map,
  // keyed with fitness

  id member, index;
  IPDagent* agent[2];
  unsigned i, j, k, l, action[2];
  char saux[20];
  
  // Take the evaluated to unevaluated status
  // Some will be evaluated twice, but what the heck...
  index = [evalPopList begin: [self getZone]];
  while ((member = [index next])) {
    [member reset];
    [popList addFirst: member];
    [index remove];
  }
  [index drop];
  
  //  printf( "Evalpop count 1 is %d\n", [evalPopList count] );
  
  for (  i = 0; i < [popList count]; i++ ) {
    agent[0] = [popList atOffset: i ];
    for ( j = i+1; j < [popList count]; j++ ) {
      agent[1] = [popList atOffset: j ];
      for ( k = 0; k < numIterations; k++ ) {	// Fight
	for ( l=0; l < 2; l ++ ) {
	  action[l] = [ agent[l] action ]; // get agent action given history
	}
	[agent[0] updateAgent: ((action[0] << 1) | action[1])];
	[agent[1] updateAgent: ((action[1] << 1) | action[0])];
	
      }
    }
    sprintf( saux, "%.3f", [agent[0] getAverage] );
    if (! [evalPopList at: (id) saux insert: agent[0] ] ) {
      [SourceMessage raiseEvent : "Something is wrong\n" ];
    }
    if ( verbose )
      [agent[0] printState];
    
  }
   if ( verbose )
      printf("\n");
  //  printf( "Evalpop count 2 is %d\n", [evalPopList count] );
  return self;
}

// Two dummy population protocol definitions
-(unsigned) getGenerationCount {
  return generationCount;
}

-(unsigned) getGenerations {
  //  return generations;
  return 0;
}

-newGeneration {

    id index, stiff;
    unsigned killQuant, i;

    generationCount++;

    popSize = [evalPopList count];
    if ( ( mating + mut + dup + split + cloning ) > 1 ) {
      [InvalidCombination raiseEvent: 
			    "Genetic operator rates bigger than 1"];
    }
    killQuant=popSize*( mating + mut + dup + split + cloning ),
      
    // Eliminate the worst mutProb, and substitute them by the siblings of the
    // others
    i = 0; 
    //    printf( "Evalpop count 3 is %d\n", [evalPopList count] );
    index = [ evalPopList begin: [self getZone]];
    [index setLoc: End];

    while( ( i < killQuant) && (stiff = [index prev] ) ){
      //      if ( verbose ) {
      //	printf("[%s %f]\n", [stiff getID], [stiff getFitness] );
      //}
      [stiff drop];		// Eliminate gen
      [index remove];
      i++;
    }
    [index drop];
    // printf( "Evalpop count 4 is %d\n", [evalPopList count] );

    // Poner a 0 tambien popList
    index = [ popList begin: [self getZone]];
    while( [index next] ) {
      [index remove];
    }

    // Now make the new ones  siblings from the first split percent
    // and another randomly chosen

    // Crossover
    for ( i = 0; i < popSize*mating; i ++ ) {
      EvolvableIPDagent *gen, *parentA, *parentB;
      unsigned popCount = [evalPopList count] - 1;
      parentA = [evalPopList atOffset: 
			       [uniformUnsRand getUnsignedWithMin: 0L 
					       withMax: popCount ] ];
      parentB = [evalPopList atOffset: 
			       [uniformUnsRand getUnsignedWithMin: 0L 
					       withMax: popCount ] ];
	
      gen = [EvolvableIPDagent createBegin: [self getZone]];
      [gen copyAgent: parentA];
      [gen cross: parentB];	
      if ( ! (gen = [gen createEnd] ) ) {
	[SourceMessage raiseEvent: "Error creating chromosome\n"];
	return nil;
      }
      [popList addFirst: gen];
    }

    // Cloning
    for ( i = 0; i < popSize*cloning; i ++ ) {
      EvolvableIPDagent *gen, *parent;
      parent = [evalPopList atOffset:
			  [uniformUnsRand getUnsignedWithMin: 0L 
					  withMax:[evalPopList count] - 1] ];
      gen = [EvolvableIPDagent createBegin: [self getZone]];
      [gen copyAgent: parent];
      if ( ! (gen = [gen createEnd] ) ) {
	[InvalidCombination raiseEvent: "Error creating chromosome"];
      }
      [popList addFirst: gen];
    }
    // Mutation
    for ( i = 0; i < popSize*mut; i ++ ) {
      EvolvableIPDagent *gen, *parent;
      parent = [evalPopList atOffset:
			  [uniformUnsRand getUnsignedWithMin: 0L 
					  withMax:[evalPopList count] - 1] ];
      gen = [EvolvableIPDagent createBegin: [self getZone]];
      [gen copyAgent: parent];
      if ( ! (gen = [gen createEnd] ) ) {
	[InvalidCombination raiseEvent: "Error creating chromosome"];
      }
      [gen bitMutate];
      [popList addFirst: gen];
    }

    // Duplication
    for ( i = 0; i < popSize*dup; i ++ ) {
      EvolvableIPDagent *gen, *parent;
      parent = 
	    [evalPopList atOffset:
		       [uniformUnsRand getUnsignedWithMin: 0L 
				       withMax:[evalPopList count] - 1]];
	
      gen = [EvolvableIPDagent createBegin: [self getZone]];
      [gen copyAgent: parent];
      if ( ! (gen = [gen createEnd] ) ) {
	[InvalidCombination raiseEvent:"Error creating chromosome\n"];
      }
      [gen doubleMutate];
      [popList addFirst: gen];
    } 

    // Split
    for ( i = 0; i < popSize*split; i ++ ) {
      EvolvableIPDagent *gen, *parent;
      parent = 
	    [evalPopList atOffset:
			   [uniformUnsRand getUnsignedWithMin: 0L 
					   withMax:[evalPopList count] - 1]];
      gen = [EvolvableIPDagent createBegin: [self getZone]];
      [gen copyAgent: parent];
      if ( ! (gen = [gen createEnd] ) ) {
	[InvalidCombination raiseEvent:"Error creating chromosome\n"];
      }
      [gen splitMutate];
      [popList addFirst: gen];
    } 
     
    return self;
}

-getBest{			// Returns the best genome
    return [evalPopList first];
}

-(double) getBestFitness{		// Fitness of the best genome
  float tmp;
  id index = [evalPopList begin: [self getZone]];// First member
  char key[STRSIZE];
  [index next];
  strcpy( key, (char * ) [index getKey] );
  [index drop];
  sscanf( key, "%f", &tmp );
  return tmp;
}

-(id) getList {
  return evalPopList;
}

@end

