// -*- mode: objc; -*-
// Copyright  1995-1999 The Santa Fe Institute.
// No warranty implied, see LICENSE for terms.


#import "BinChrom.h"

#include <sys/types.h>
#include <malloc.h>
#include <random.h>

unsigned char Wgts[9] = { 0,128,64,32,16,8,4,2,1 }; 
				   // 1st number does not mind
@implementation BinChromosome


// Creating-----------------------------------------------------------

-setLength: (unsigned) len {
    length = len;
    genotype = (char*) calloc ( length, sizeof(char));
    return self;
}

-setRandomInit {
    unsigned i;
    for ( i = 0; i < length; i++ ) {
	genotype[i] = 
	  [uniformUnsRand getUnsignedWithMin: 0L withMax: MAX8BITS - 1 ];
    }
    return self;
}

- setID: (const char*)_ID
{
  strcpy (ID, _ID);
  return self;
}

// Override the createEnd function from SwarmObject
-createEnd {
// And check everything is correct
  if ( !genotype || !length) {
    [InvalidCombination raiseEvent: "Genotype not initialized properly\n"];
    return nil;
  } else
    return self;
}
  

// Setting ---------------------------------------------------

-setFitness: (double) _fitness {
  fitness = _fitness;
  return self;
}
 

// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Useful but not user-visible functions
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

void swap( unsigned* _s1, unsigned* _s2 ) { 
    unsigned tmp = *_s1; 
    *_s1 = *_s2; 
    *_s2 = tmp;
}

-changeBit: (unsigned long) chBit {
//  -- changes a single bit in the string
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

    unsigned long chByte = chBit >> 3;
    // bit in chBit position is altered (by XORing it)
    genotype[chByte] ^= Wgts[(chBit & LASTBYTEN )+1];
    return self;
}

// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
-swapChunkByte:	(unsigned char) _scbStart
	    To:	(unsigned char) _scbEnd
	offset: (unsigned) _idxByte
	  With: (char) _scbByte 
// --	swaps a chunk of bits between the internal byte of index idxByte
//	and the input byte
//      _scbStart and _scbEnd must be less than 7
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
{
    unsigned char mask=0;
    unsigned char oneLeft = MAX8BITS/2;
    unsigned char oneRight = 1;
    unsigned char i;

    if ( ( _scbStart >= BITSINBYTE ) || ( _scbEnd >= BITSINBYTE ) ) {
	  [WarningMessage raiseEvent: "WarningMessage: swapping chunk bigger than byte!\n"];
    }
    
    // first, create mask from left
    for ( i = 0; i < _scbStart; i ++ ) {
	mask |= oneLeft;
	oneLeft >>=1;
    }
    
    // now, from right
    for ( i = LASTBYTEN; i>  _scbEnd; i-- ) {
	mask |= oneRight;
	oneRight <<=1;
    }
    
    // mask created :-) Now interchange.
    genotype[_idxByte] &= mask;
    _scbByte &= ~mask;
    genotype[_idxByte]|=_scbByte;

    return self;
}


// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
-swapChunk: (unsigned long) _swBitStart 
	To: (unsigned long) _swBitEnd
      With: ( char *) inStr {
//  --	swaps all the bytes between the start bit and the end bit;
//	ie. from 0 to _swBit_start from the inside string; from _swBitStart
//	to _swBitEnd from the second string, and from _swBitEnd to the final
//	bit from the inside string
//	Good for 2-point crossover
//	Boundary check is on the user
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

    unsigned byteStart = (unsigned)( _swBitStart / BITSINBYTE );
    unsigned byteEnd   = (unsigned)( _swBitEnd / BITSINBYTE );
    unsigned i;

    if ( byteStart == byteEnd )
	[self swapChunkByte: (unsigned char) _swBitStart%BITSINBYTE
	      To: (unsigned char) _swBitEnd %BITSINBYTE
	      offset: byteStart With: inStr[byteStart] ];
    else  { // different
	// start byte
	[ self swapChunkByte: (unsigned char) ( _swBitStart%BITSINBYTE )
	       To:  LASTBYTEN
	       offset: byteStart With: inStr[byteStart] ];
	
	// middle bytes
	for (  i = byteStart + 1; i < byteEnd; i ++ )
	    genotype[i] = inStr[i];
	    
	// end byte
	[ self swapChunkByte: 0
	       To: (unsigned char) (_swBitEnd %BITSINBYTE) 
	       offset: byteEnd With: inStr[byteEnd] ];
    }
    return self;
}


// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
-mutate: (float) _mutationRate {
// --	mutates genome string with given probability
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

    unsigned i;
    for ( i = 0; i < length*BITSINBYTE; i ++ )
      if ( (float)[uniformDblRand getDoubleWithMin: 0.0L withMax: 1.0L ] < _mutationRate )
	[self changeBit: i ];
    return self;
}

// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
- Clone : (id<Chromosome>) _chrom withMutation: (float) _mutProb {

    unsigned auxL = [_chrom getLength];     // computes length
	
    genotype = (char*) calloc( auxL, sizeof(char)); // allocs memory
    if (!genotype) {
	[OutOfMemory raiseEvent: "No more Memory in binChromosome\n"];
	exit(1);
    }
    length= auxL;
    memcpy( genotype, (char*) [_chrom getGenotype], length); // copies into genome
    [self mutate: _mutProb ];
    return self;
}

// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
- Duplicate : (id<Chromosome>) _chrom 
	From: (unsigned) _l1 
	  To: (unsigned) _l2 
withMutation: (float) _mutProb {
// -- Duplicate a chunk of bytes with mutation. Takes a copy of the 
//    bytes from l1 to _l2, and mutates them
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

    unsigned sizeChunk = _l2 - _l1 + 1, charIdx, i;
    char* tmp;

    length = [_chrom getLength];

    if ( (_l1 > length) || (_l2 >length) )
      return nil;

    // everything is fine; now go on

    tmp = (char*) [_chrom getGenotype];

    length += sizeChunk;
    genotype = (char*)calloc(length, sizeof( char ));
    if (genotype == 0) {
	[SourceMessage raiseEvent: "Calloc failed in dupAllele - genotype\n"];
	return nil;
    }

    charIdx = _l1;
    memcpy( genotype, tmp, charIdx ); // copies old string, up to _l1
    memcpy( genotype + charIdx, tmp + charIdx, sizeChunk ); // dups allele
    memcpy( genotype + charIdx + sizeChunk, tmp + charIdx, length - charIdx );

    for ( i = (_l2+1)*BITSINBYTE; i < (_l2+ sizeChunk+1)*BITSINBYTE; i ++ )
      if ( (float)[uniformDblRand getDoubleWithMin: 0.0L withMax: 1.0L ] < _mutProb )
	[self changeBit: i ];

    return self;
}


// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
- Kill : (id<Chromosome>) _chrom
   From: (unsigned) _l1 
     To: (unsigned) _l2 {
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

    unsigned sizeChunk = _l2 - _l1 + 1, charIdx;
    char *tmp;
    length = [_chrom getLength];

    if ( ( _l1 > length ) || ( _l2 > length ) ) // _pos is in byte units
      return nil;
    
   if ( length <= sizeChunk ) {	   // return if there is nothing left
       [WarningMessage raiseEvent: "Small enough, thanks\n"];
       return nil;
   }

   // everything is fine; now go on
   tmp = (char *) [_chrom getGenotype];
   length -= sizeChunk;
   genotype = (char*) calloc( length, sizeof( char ) );
   if (genotype == 0) {
       [SourceMessage raiseEvent: "Malloc failed in killChunk"];
       return nil;
   }

   charIdx = _l1;
   memcpy( genotype, tmp, charIdx );	   // copies old string, up to idx
   memcpy( genotype+charIdx, tmp+charIdx+sizeChunk, length-charIdx-sizeChunk);
				   // and from idx

   return self;
}


// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
- Swap : (id<Chromosome>) _chrom 
       : (unsigned) _l1 
   with: (unsigned) _l2 
   size: (unsigned) _bytes {
// Remove a chunk of _bytes and reinsert it somewhere else
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

       char *chunk;
       length = [_chrom getLength];

       if ( ( _l1 + _bytes > length ) 
	    || ( _l2 + _bytes > length ) ) // _pos is in byte units
	   return nil;
    
       genotype = (char*) calloc( length, sizeof( char ) );
       if (genotype == 0) {
	   [SourceMessage raiseEvent: "Malloc failed in swap"];
	   return nil;
       }
       memcpy( genotype, [_chrom getGenotype], length * sizeof(char));

       chunk = (char*) calloc( _bytes, sizeof(char));

       memcpy( chunk, genotype+_l2, _bytes);
       memcpy( genotype+_l2, genotype + _l1, _bytes);
       memcpy( genotype+_l1, chunk, _bytes );

       free ( chunk );
       return self;
}

// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
- RemoveReinsert:  (id<Chromosome>) _chrom 
		: (unsigned) _l1 
	      To: (unsigned) _l2
	    size: (unsigned) _bytes {	
// Swap whole byte/loci, l1 is the smaller
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

		char *chunk;
		length = [_chrom getLength];

		if ( ( _l1 + _bytes > length ) 
		     || ( _l2 + _bytes > length ) ) // _pos is in byte units
		    return nil;
    
		if ( _l1 > _l2 )
		    swap( &_l1, &_l2 );

		genotype = (char*) calloc( length, sizeof( char ) );
		if (genotype == 0) {
		    [SourceMessage raiseEvent: "Malloc failed in swap"];
		    return nil;
		}
		memcpy( genotype, [_chrom getGenotype], length * sizeof(char));

		chunk = (char*) calloc( _bytes, sizeof(char));

		memcpy( chunk, genotype+_l1, _bytes);
		memcpy( genotype+_l1, genotype + _l1 + _bytes, _l2 - _l1 );
		memcpy( genotype+_l2, chunk, _bytes );

		free ( chunk );
		return self;
}


// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
-Mate: (id<Chromosome>) _chrom1 
  And: (id<Chromosome>) _chrom2
withMutation: (float) _mutProb 
// --	Builds a genome from 2 others using 2 point crossover with points
//	generated randomly. 
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
{
    // Starting with gene1 == parent A
    unsigned auxL = length = [ _chrom1 getLength]; // computes length
    unsigned auxL2 = [_chrom2 getLength];
    unsigned startPoint, endPoint;

    if (!(genotype = (char *) calloc( auxL, sizeof(char)))) { // allocs memory
	[SourceMessage raiseEvent: "Malloc failed in  genotype\n" ];
    }

    memcpy( genotype, [_chrom1 getGenotype], length);	// copies into genome

    if ( auxL2 < auxL )			     // take smaller one
      auxL = auxL2;
    
    startPoint = 
      [uniformUnsRand getUnsignedWithMin: 0L withMax: length*BITSINBYTE - 1];
    endPoint = 
      [uniformUnsRand getUnsignedWithMin: 0L withMax: length*BITSINBYTE - 1];

    // swapping as needed
    if ( startPoint > endPoint ) 
      swap( &startPoint, &endPoint );
    
    // check bounds; auxL is the smaller of the two
    if (endPoint > auxL*BITSINBYTE )
      endPoint = auxL*BITSINBYTE;
	    
    // now perform interchange
    [ self swapChunk: startPoint To: endPoint With: (char *) [_chrom2 getGenotype] ];
    
    // and then mutate
    [ self mutate: _mutProb ];
//    printf("\n");
//    [_chrom1 hexaPrint];  [_chrom2 hexaPrint];  [self hexaPrint];
    return self;
}

// ------------------------------------------------
// Using Functions --------------------------------
// ------------------------------------------------

- (unsigned) getLength {
    return length;
}

- getGenotype {
    return (id) genotype;
}

- (unsigned char) getGene: (unsigned) offset {
    return genotype[offset];
}

- hexaPrint {
    unsigned i;
    for ( i = 0; i < length; i++ )
	printf( "%x ", genotype[i] );
    return self;
}

-(char* ) getID {
  return ID;
}

-(double) getFitness {
  return fitness;
}

// Override the drop function from SwarmObject
-(void) drop {
  free( genotype );
  [super drop];
}
@end
