// Swarm library. Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

// Class "MoXGrid2d" by Sven Thommesen 

// This class adds to the MoGrid2d base class the ability to return
// lists of cells in a specified neighborhood of a given cell. 

// You may ask for either a list of vacant (unoccupied) sites only, a list
// of occupied sites only, or for the whole neighborhood. These lists do
// *not* include the center cell. If the returned list would be empty,
// we return the value "nil" rather than an actual empty List.

// If you ask for a list of neighbors (occupied cells), the list returned
// contains pointers to the lists found at the respective cells: 
// you get a list of lists!

// If you ask for a list of empty cells, a list full of NIL pointers would
// be rather useless. Pointers to the cells themselves is also useless,
// since the cells do not know what (x,y) position they occupy. My solution
// is to encode the position into a long int (id) thus: 
// 	xy = (X << 16) + Y OR xy = 65536*X + Y. 
// These values may be unpacked this way:
// 	x = xy / 65536     and  y = xy % 65536, OR 
// 	x = (xy & 0xffff)  and  y = (xy >> 16).

// This trick limits the maximum size of a grid to 64k x 64k cells.

// You may specify default values for neighborhood type, neighborhood
// size, and whether you want the order of the items in a list randomized.
// You can then call for lists using these default values, or you may
// call for lists using one-time values for these parameters.

// On randomization: this procedure uses a Shuffler object, which in
// turn uses the default random generators. If you do not randomize,
// the cells for a given neighborhood come back in the same order every
// time. This may introduce a bias in the movement of agents etc.
// (As in the current Heatbugs and derivatives.)

// XGrid2d currently recognizes these neighborhood types:
//   -AxesOnly: cells along the N-S, E-W axes only (i.e. "von Neumann")
//   -DiagsOnly: cells along the NW-SE, NE-SW diagonals only
//   -AxesNDiags: combination of the first two (i.e. "Moore")
//   -aStar: cells along the axes within the given size,
//		plus cells along the diagonals within (size/2)
//   -aBox: cells within a square box [-size .. +size] of center
//   -aCircle: cells within an approximated circle of radius 'size'
//

// (c) 1997 Sven Thommesen <sthomme@humsci.auburn.edu>

#undef DOALLTESTS
// Comment out the line below to speed things up
// at the expense of less extensive testing of parameters:
#define DOALLTESTS 1

// Largest neighborhood we can deal with:
// (Makes space for arrays -- expand as needed.)
#define MAXSIZE 5

// Largest number of cells in a neighborhood of size MAXSIZE
// (currently the aBox neighborhood):
#define MAXPOINTS ((2*MAXSIZE+1)*(2*MAXSIZE+1)-1)

// Enumerate valid neighborhood types:

typedef enum {			// (enum is always the size of an int)
	AxesOnly=1,		// main axes only (vonNeumann)
	DiagsOnly=2,		// main diagonals only
	AxesNDiags=3,		// axes+diagonals (Moore)
	aStar=4,		// axes=size, diagonals=size/2
	aBox=5,			// square box
	aCircle=6,		// approximated circle with radius s
	// (Insert new types here)
        InvalidType		// = largest valid type + 1
} NeighborhoodType;


#import "MoGrid2d.h"

@interface MoXGrid2d: MoGrid2d {

   id shuffleObject;			// Shuffler to randomize lists

// Default parameters:

   int              defaultSize;	// default neighborhood size
   NeighborhoodType defaultType;	// default neighborhood type
   BOOL             defaultShuffle;	// default value for Shuffled?

// Data for each neighborhood cell selected:

   id            pArray[MAXPOINTS];	// holds pointers to the cells
   id          objArray[MAXPOINTS];	// holds pointers to objects at cells
   unsigned int  xArray[MAXPOINTS];	// holds x-values
   unsigned int  yArray[MAXPOINTS];	// holds y-values
   unsigned int xyArray[MAXPOINTS];	// holds packed x+y values

   int thisType, thisSize;		// current values
   int thisX, thisY;			// current values
   int index;			// working variable (pointer into objArray etc.)

}

// Private methods:

-addAxesAtX:   (int) x Y: (int) y Size: (int) s;
-addDiagsAtX:  (int) x Y: (int) y Size: (int) s;
-addBoxAtX:    (int) x Y: (int) y Size: (int) s;
-addCircleAtX: (int) x Y: (int) y Size: (int) s;
-addCells;

// Public methods:

+createWithDefaults: aZone // type=AxesOnly, size=1, Shuffled=NO (vonNeumann)
	setSizeX: (int) x Y: (int) y;

+create: aZone 	setSizeX: (int) x Y: (int) y
		withType: (NeighborhoodType) type 
		Size: (int) size 
		Shuffled: (BOOL) s;

// Set default parameters:
-setNeighborhoodType: (NeighborhoodType) type Size: (int) size;
-setShuffled: (BOOL) s;

// Return current values of default parameters:
-(int)  getNeighborhoodType;
-(int)  getNeighborhoodSize;
-(BOOL) getShuffled;

// Return a list of cells in the specified neighborhood,
// using the previously set default parameters,
// or (nil) if the list would be empty:

-getNeighborhoodAtX:  (int) x Y: (int) y;	// all cells
-getNeighborListsAtX: (int) x Y: (int) y;	// occupied cells
-getVacanciesAtX:     (int) x Y: (int) y;	// unoccupied cells

// Return a list of cells in the specified neighborhood,
// using one-time parameter values,
// or (nil) if the list would be empty:

-getNeighborhoodAtX: (int) x Y: (int) y 
	    Type: (int) type Size: (int) size Shuffled: (BOOL) s;

-getNeighborListsAtX: (int) x Y: (int) y 
	    Type: (int) type Size: (int) size Shuffled: (BOOL) s;

-getVacanciesAtX: (int) x Y: (int) y 
	    Type: (int) type Size: (int) size Shuffled: (BOOL) s;

@end
