// Copyright (C) 1999 University of Washington, Columbia Basin Research.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.


// This class implements a container which allows quick solution of the
//  "find the nearest" problem in a two-dimensional space.
//  Min and max x & y values for objects in that space must be known at 
//  the time a SearchableSpace object is created. Once objects have been
//  inserted into the SearchableSpace, their x,y positions must not change.

// The objects to be stored in this container need not all be of the same
//  type, but they must all support the same interface:
//  (int) getIntX;
//  (int) getIntY;   // Only if intCoords == TRUE
//  (double) getDoubleX;
//  (double) getDoubleY;  // only if intCoords == FALSE
//  setNext: anObject;
//  getNext;         // Only if allowDup == TRUE or useTree == FALSE
//  (double) getVal;    // Gets a characteristic value from the object,
//                      //  used for searches which exclude some objects.

// If allowDups is TRUE or useTree is FALSE, the container is "intrusive",
//  in other words it relies on the contained objects maintaining a pointer
//  which is accessed by setNext and getNext. This would mean that objects
//  can only reside in one SearchableSpace at a time.

#import <objectbase/SwarmObject.h>

// The comparison constants (for filtering objects by their getVal values).
#define ALL 0
#define EQ 1
#define NEQ 2
#define GT 3
#define GE 4
#define LT 5
#define LE 6

@interface SearchableSpace: SwarmObject {
  id  gridSpace;     // SearchableGridSpace object (if that option selected).
  id  treeBase;      // Base SearchableTreeNode object (if that opt selected).
  double minX, minY, maxX, maxY;  // Bounding box of underlying space.
  double gridSpacing;  // used with grid, only.
  BOOL useTree;      // TRUE = use BSP tree; FALSE = use grid.
  BOOL intCoords;    // TRUE = object coordinates are integers; FALSE = doubles
  BOOL allowDups;    // used only if useTree == TRUE: if TRUE, allow two
                     //  objects to have equal coordinates; if FALSE, forbid.
  id  freeList;      // List of free SearchableTreeNode objects single-linked
                     //  by their child[0] pointers.

  // Saved variables for a "find the nearest" search.
  double searchX, searchY;
  double minDistSq;
  id minDistObj;
  double compareTo;  // Limiting value, if any, from "getVal" on objects.
  int valueSwitch;   // How that limit is applied.

  // Saved variables for an iterated enumeration of all the objects within
  //  a bounding box.
  double bbMinX, bbMaxX, bbMinY, bbMaxY;
  id currTreeNode;   // Tree node currently being iterated.
  id currObject;     // Current object in a list of co-located objects being
                     //  iterated.
  int iterX, iterY;  //  current loop counters in a grid iteration.
  int cellMinX, cellMaxX, cellMinY, cellMaxY; // loop limits.
}

// Messages to populate or depopulate the space (may be sent many times).
- putObjects: (id <List>) aList; // add a bunch of objects to space.
- clear;                         // clear all pointers in space.
- clearAndPutObjects: (id <List>) aList; // both of above.
- clearAndFree;  // clear all pointers and free dynamic objects (tree nodes)
- clearFreeAndPutObjects: (id <List>) aList; // all of above.

// Internal messages to populate or depopulate the space.
- addObject: anObject toTree: aTreeNode;
- createLeafNodeFor: object atMinX: (double) minx maxX: (double) maxx
                              minY: (double) miny maxY: (double) maxy;
- dropTreeNode: aTreeNode; // really drops object
- freeTreeNode: aTreeNode; // goes to internal free list for reuse.

// Messages for initialization (sent once only).
+ createBegin: aZone;
- setUseTree: (BOOL) bUse;
- setAllowDups: (BOOL) bAllow;
- setIntCoords: (BOOL) bInt;
- setMinX: (double) minx maxX: (double) maxx
     minY: (double) miny maxY: (double) maxy;
- setGridSpacing: (double) spacing;
- createEnd;

// Destroy self and any allocated objects.
- (void) drop;

// Messages to find the nearest object to a given point.
- getNearestObjectToX: (double) x Y: (double) y;
- getNearestObjectToX: (double) x Y: (double) y WithValue: (int) comparison
                                                To: (double) compareValue;

// Messages used internally during search for nearest object to a given point.
- getClosest;
- getClosestIn: aTreeNode;

// Messages used to iterate through all objects in a bounding box.
// "iterate..." sets up the process. "next" gets each object in turn until
//  they run out, after which it returns nil.
// Only one iteration can be supported by a given SearchableSpace object
//  at a given time.
- iterateMinX: (double) mnX maxX: (double) mxX 
         minY: (double) mnY maxY: (double) mxY;
- next;

@end


