// (C) 1999 University of Washington, Columbia Basin Research.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.
 
#import <space.h>
#import <space/Discrete2d.h>
#import "SearchableObject.h"
 
// SearchableGridSpace is an extension of a Discrete2D space. The grid has
//  a fixed spacing in x,y which is chosen with reference to the underlying
//  problem: hopefully most of the grid cells will be empty or have only
//  a few objects in them, without taking up too much space for grid
//  pointers or having most grid cells empty.

// The grid can contain more then one object per cell. The objects must be
//  able to keep a pointer to the next object in the grid cell and be able
//  to respond to the following messages:
//  (int) getIntX;
//  (int) getIntY;      // x,y position in underlying simulated space.
//  (double) getDoubleX;
//  (double) getDoubleY;// Only the "Int" or "Double" messages will be used,
//                      //  depending on an initialization option.
//  setNext: anObject;  // sets "next object" pointer, returns self.
//  getNext;            // returns next object in same grid cell.

// The grid may require that no two objects have the same x,y location
//  (the same results from "get*X" and "get*Y"). To do this, send the 
//  setAllowDups message with FALSE (TRUE to allow duplicates).
//  If duplicates are not allowed and an attempt is made to add an object
//  at the same location as another object, the attempt fails with an 
//  error message.
// The grid may assume that the underlying object coordinates are integers
//  or real numbers (doubles). Send the setIntCoords message with TRUE
//  to use the integer coordinates, FALSE to use the doubles. The "get*"
//  methods for the coordinate type not being used will never be called,
//  and so do not need to be defined.

@interface SearchableGridSpace : Discrete2d {
  double minX, minY, maxX, maxY;   // Limits of underlying space (in meters).
  double gridSpacing;              // Size of a (square) grid cell, in meters.
  int sizeX, sizeY;                // number of grid cells across and down.
  BOOL allowDups;       // allow same get*X,get*Y results for multiple objects
  BOOL intCoords;       // TRUE = object coords are integers; FALSE = doubles
}

// Messages to create a SearchableGridSpace (sent only once).
+ createBegin: aZone;
- setAllowDups: (BOOL) bAllow;
- setIntCoords: (BOOL) bInt;
- setMinX: (double) minx maxX: (double) maxx minY: (double) miny
     maxY: (double) maxy gridSpacing: (double) spacing;
- createEnd;

// Messages to populate or depopulate the space (may be sent many times).
- putObjects: (id <List>) aList; // add a bunch of objects to grid.
- clear;                         // clear all pointers in grid.
- clearAndPutObjects: (id <List>) aList; // both of above.

// Messages to help access objects stored in the space.
//  An object with position x,y will be in grid cell:
//     (int)((x - minX) / gridSpacing), (int)((y - minY) / gridSpacing)
- (double) getSpacing;
- (double) getMinX;
- (double) getMinY;

@end

