// MNGVideo.m - (c) 2001 R. Stephan <ralf@ark.in-berlin.de>
/* Most mng* funcs (C) 1997 Rasca Gmelch, Berlin, EMail: thron@gmx.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
// $Id: MNGVideo.m,v 1.23 2001/04/14 12:28:05 ralf Exp $

#include <stdlib.h>
#include <zlib.h>
#define PNG_INTERNAL
#include <png.h>
#import <space/Discrete2d.h>
#import <tkobjc/Colormap.h>
#import "MNGVideo/MNGVideo.h"

static int fill_png_palette (Colormap* cm, png_color* pp, int n);
static void mng_write_sig (png_structp png_ptr);
static void mng_write_MHDR (png_structp png_ptr, int width, int height, 
			    int tick, int simp);
static void mng_write_MEND (png_structp png_ptr);
static void mng_write_DEFI (png_structp png_ptr, unsigned short idnr, int conc);
static void mng_write_DHDR (png_structp png_ptr, unsigned short idnr, char img_type,
			    char delta_type, int w, int h, int x, int y);
static void mng_write_FRAM (png_structp png_ptr, char framing_mode, char *name,
			    char chg_interframe_delay, char chg_sync_timeout,
			    char chg_frame_clipping_boundaries, char chg_sync_id_list,
			    png_uint_32 interframe_delay);
static void mng_write_MAGN (png_structp png_ptr, unsigned short idnr, unsigned short mag);
static void mng_write_empty_PLTE (png_structp png_ptr);
static void mng_write_SAVE (png_structp png_ptr);
static void mng_set_IHDR(png_structp png_ptr, png_uint_32 width, 
			 png_uint_32 height,   int bit_depth, int color_type, 
			 int compression_type, int filter_type,
			 int interlace_type);

png_byte mng_sig[8] = {138, 77, 78, 71, 13, 10, 26, 10};

@implementation MNGVideo

//----------------------------------------------------

+create: aZone ColorMap: (id <Colormap>) cm MagFactor: (unsigned) m;
{
  MNGVideo *obj = [MNGVideo createBegin: aZone];
  [obj setColormap: cm];
  [obj setMagnification: m];
  return [obj createEnd];
}

+create: aZone ColorMap: (id <Colormap>) cm MagFactor: (unsigned) m 
   FrameRate: (unsigned) rate 
   NFrames: (int) nf 
   Basename: (unsigned char*) name;
{
  MNGVideo *obj = [MNGVideo createBegin: aZone];
  [obj setColormap: cm];
  [obj setMagnification: m];
  [obj setFrameRate: rate];
  [obj setNFrames: nf];
  [obj setBasename: name];
  return [obj createEnd];
}

+createBegin: aZone
{
  MNGVideo *obj = [super createBegin: aZone];
  obj->width = obj->height = obj->mag = obj->fps = obj->n_frames = 0;
  obj->pngbuf = 0;
  obj->cbuf1 = obj->cbuf2 = 0;
  obj->img_nr = -1;
  obj->ogrid = obj->vgrid = obj->ocoll = nil;
  obj->omsg = (SEL)nil;
  obj->mfac = obj->mc = obj->ncolors = 0;
  obj->cmap = nil;
  obj->delta_f = NO;
  obj->basename = obj->filename = nil;
  obj->file_ptr = NULL;
  return obj;
}

-createEnd
{ 
  [self setMagnification: mag];
  [self setFrameRate: fps];
  [self setNFrames: n_frames];
  if (basename==nil) [self setBasename: NULL]; // opens file too

  return [super createEnd];
}

//---------
-setBasename: (char*) filebase
{
  if (basename) [basename drop];
  if (filename) [filename drop];

  if (filebase == NULL)
    filebase = "swarm";
  basename = [String create: [self getZone] setC: filebase];
  filename = [String create: [self getZone] setC: filebase];
  [filename catC: ".mng"];

  if (file_ptr != NULL) fclose (file_ptr);

  // check if we can write file
  file_ptr = fopen ([filename getC], "w");
  if (!file_ptr)
    fprintf (stderr, "Couldn't fopen file %s\n", [filename getC]);

  return self;
}

-setColormap: (id <Colormap>) cm
{
  int i;

  if (cm==nil)
    fprintf (stderr, "Warning: attempt to set MNGVideo::cmap=nil\n");
  
  cmap = cm;
  for (i=0; i<MAXCOLORS-1; ++i)
    if (((Colormap*)cmap)->isSet[i])
      ++ncolors;

  ++ncolors;
  return self;
}

-setMagnification: (unsigned) m
{
  mag = m;
  if (mag==0) mag=1;
  return self;
}

-setFrameRate: (unsigned) rate
{
  fps = rate;
  if (fps==0) fps=10;
  return self;
}

-setNFrames: (unsigned) nf
{
  n_frames = nf;
  if (n_frames==0) n_frames=100;
  return self;
}

-setWidth: (unsigned) w
{
  width = w;
  return self;
}

-setHeight: (unsigned) h
{
  height = h;
  return self;
}

//-----------------------------------------------------
-(void)setValueLayerGrid: (id <Discrete2d>) d MappingM: (int)m C: (int)c
{
  vgrid = d;
  mfac = m;
  if (mfac==0) mfac=1;
  mc = c;
  if (width==0 || width > [d getSizeX])
    width = [d getSizeX];
  if (height==0 || height > [d getSizeY])
    height = [d getSizeY];
}

-(void)setObjectLayerGrid: (id <Discrete2d>) d ObjectCollection: c Message: (SEL)sel
{
  if (c!=nil && sel==(SEL)nil)
    { fprintf (stderr, "MNGVideo: collection set but no message!\n"); return; }

  ogrid = d;
  //  printf("ocoll set to %ld\n", (long)ocoll);
  ocoll = c;
  omsg = sel;
  if (width==0 || width > [d getSizeX])
    width = [d getSizeX];
  if (height==0 || height > [d getSizeY])
    height = [d getSizeY];
}

//-----------------------------------------------------
-(int) initVideo
{
// returns -1 on failure

  png_color palette[256];
  int y;

  if (vgrid==nil && ogrid==nil)
    {
      fprintf (stderr, "MNGVideo: no layers set!\n");
      return -1;
    }

  png_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING, 
				    (png_voidp)NULL, 
				    (png_error_ptr)NULL, 
				    (png_error_ptr)NULL);
  if (!png_ptr) return -1;

  info_ptr = png_create_info_struct (png_ptr);
  if (!info_ptr)
    return -1;
  
  if (setjmp (png_ptr->jmpbuf))
    return -1;

  png_init_io (png_ptr, file_ptr);
  mng_write_sig (png_ptr);
  mng_write_MHDR (png_ptr, width*mag, height*mag, fps, delta_f? 615:583);
  mng_write_FRAM (png_ptr, 1, NULL, 2, 0, 0, 0, 1);

  palette[0].red = palette[0].green = palette[0].blue = 0;
  if (vgrid==nil && omsg==(SEL)nil)
    {
      palette[1].red = palette[1].green = palette[1].blue = 255;
      ncolors = 2;
    }
  else
    {
      // shift colors 0-254 to 1-255, 0 becoming black
      if (fill_png_palette (cmap, palette+1, 250) < 0)
	return -1;
    }
  png_write_PLTE (png_ptr, palette, ncolors);  

  if (ncolors>16) nbits=8;
  else if (ncolors>4) nbits=4;
  else if (ncolors>2) nbits=2;
  else nbits=1;

  // lazy allocation of buffers -- we have one/two Color buffers
  // plus one for final png data

  pngbuf = [[self getZone] alloc: sizeof(png_byte*)*height];
  if (!pngbuf) 
    {
      fprintf (stderr, "MNGVideo.m: Couldn't Zone::alloc pngbuf!\n");
      return -1;
    }
  for (y=0; y<height; ++y)
    {
      pngbuf[y] = [[self getZone] alloc: sizeof(png_byte)*((nbits*width)/8)+1];
      if (pngbuf[y] == NULL)
	{
	  fprintf (stderr, "MNGVideo.m: Couldn't Zone::alloc buf!\n");
	  return -1;
	}
    }

  cbuf1 = [[self getZone] alloc: sizeof(Color*)*height];
  if (!cbuf1) 
    {
      fprintf (stderr, "MNGVideo.m: Couldn't Zone::alloc buf!\n");
      return -1;
    }
  for (y=0; y<height; ++y)
    {
      cbuf1[y] = [[self getZone] alloc: sizeof(Color)*width];
      if (cbuf1[y] == NULL)
	{
	  fprintf (stderr, "MNGVideo.m: Couldn't Zone::alloc buf!\n");
	  return -1;
	}
    }
  cbuf = cbuf1;

  if (delta_f)
    {
      cbuf2 = [[self getZone] alloc: sizeof(Color*)*height];
      if (!cbuf2) 
	{
	  fprintf (stderr, "MNGVideo.m: Couldn't Zone::alloc buf!\n");
	  return -1;
	}
      for (y=0; y<height; ++y)
	{
	  cbuf2[y] = [[self getZone] alloc: sizeof(Color)*width];
	  if (cbuf2[y] == NULL)
	    {
	      fprintf (stderr, "MNGVideo.m: Couldn't Zone::alloc buf!\n");
	      return -1;
	    }
	}
    }

  return 0;
}

-(void) takeAFrame
{
  if (file_ptr == 0) { return; }

  if (++img_nr >= n_frames) {
    [self finish];
    --img_nr;
    return;
  }

  if (img_nr == 0)  // first call
    if ([self initVideo] < 0)
      fprintf (stderr, "MNGVideo.m: Couldn't init libpng!\n");

  if (png_ptr)
    png_destroy_write_struct (&png_ptr, (png_infopp)NULL);
  png_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING,
				(void *)NULL, NULL, NULL);
  if (!png_ptr) 
    fprintf (stderr, "MNGVideo.m: Couldn't make png_ptr\n");

  png_init_io (png_ptr, file_ptr);

  png_set_IHDR (png_ptr, info_ptr, width, height, nbits, 
		PNG_COLOR_TYPE_PALETTE,
		PNG_INTERLACE_NONE,
		PNG_COMPRESSION_TYPE_DEFAULT,
		PNG_FILTER_TYPE_DEFAULT);

  png_set_compression_level (png_ptr, Z_BEST_COMPRESSION);
  png_set_sig_bytes (png_ptr, 8);
  png_set_filter (png_ptr, 0,
		  PNG_FILTER_NONE | PNG_FILTER_SUB |
		  PNG_FILTER_PAETH);
  png_set_packing (png_ptr);

  if (delta_f==NO || img_nr==0)
    {
      mng_write_DEFI (png_ptr, img_nr+1, delta_f? 1:0);
      if (delta_f)
	mng_write_SAVE (png_ptr);
      mng_write_MAGN (png_ptr, img_nr+1, mag);
    }
  else
    mng_write_DHDR (png_ptr, 1, 1, 1, width, height, 0, 0);

  if (delta_f == NO || img_nr==0)
    {
      png_write_info_before_PLTE (png_ptr, info_ptr);
      mng_write_empty_PLTE (png_ptr);   // need no palette, have global one
    }
  else
   mng_set_IHDR(png_ptr, info_ptr->width, info_ptr->height,
      info_ptr->bit_depth, info_ptr->color_type, info_ptr->compression_type,
      info_ptr->filter_type, 0);


  // Now first fill Color buffers.

  if (delta_f)                         // switch buffer if delta
    {
      if (cbuf == cbuf1) cbuf = cbuf2;
      else cbuf = cbuf1;
    }

  if (vgrid != nil)
    {                                  // fill cbuf from value grid first
      int x,y;
      long *lat = (long*)[vgrid getLattice];
      long *off = [vgrid getOffsets];
      for (y=0; y<height; ++y)
	for (x=0; x<width; ++x)
	  cbuf[y][x] = ((*discrete2dSiteAt(lat,off,x,y))/mfac)+mc;
    }

  if (ogrid != nil)                    // draw objects
    {
      id *lat = [ogrid getLattice];
      long *off = [ogrid getOffsets];

      if (vgrid == nil)                // clear buffer
	{
	  int y;
	  for (y=0; y<height; ++y)
	    memset (cbuf[y], 0, sizeof(Color)*width);
	}

      if (omsg != (SEL)nil)
	if (ocoll != nil)
	  [ocoll forEach: omsg: self];
        else
	  {
	    int x,y;
	    for (y=0; y<height; ++y)
	      for (x=0; x<width; ++x) 
		{
		  id obj = *discrete2dSiteAt(lat,off,x,y);
		  if (obj != nil)
		    [obj perform: omsg with: self];
		}
	  }
      else
	if (vgrid == nil)
	  {                            // the case vgrid=ocoll=omsg=nil
	    int x,y;
	    for (y=0; y<height; ++y)
	      for (x=0; x<width; ++x) 
		cbuf[y][x] = ((*discrete2dSiteAt(lat,off,x,y))!=nil);
	  }
    }

  // fill png buffer
  {
    int x,y, chunks_per_byte = 8/nbits;

    if (!delta_f)
      {
	for (y=0; y<height; ++y)
	  for (x=0; x<width; )
	    {
	      int chunk, byte=0;
	      for (chunk=0; chunk<chunks_per_byte; ++chunk, ++x)
		byte = (byte<<nbits) + (cbuf[y][x] % (1<<nbits));
	      pngbuf[y][(x-1)/chunks_per_byte] = byte;
	    }
      }
    else
      {
	Color **obuf = cbuf1;
	if (cbuf==cbuf1) obuf=cbuf2;
	for (y=0; y<height; ++y)
	  for (x=0; x<width; ++x)
	    pngbuf[y][(x-1)/chunks_per_byte] = obuf[y][x]^cbuf[y][x];
      }
  }

  // write the buffer
  png_write_image (png_ptr, pngbuf);
  png_write_end (png_ptr, info_ptr);

  if (delta_f) 
    {
      if (cbuf==cbuf1) cbuf = cbuf2;
      else cbuf = cbuf1;
    }
}

-(void) drawPointX: (int)x Y: (int)y Color: (Color)c
// this message is called by the message [agent <omsg>]
{
  cbuf[y][x] = c+1;
}

-(void) finish
{
  if (file_ptr) 
    {
      printf ("MNGVideo: file '%s' written, closing now...\n",
	      [filename getC]);
      mng_write_MEND (png_ptr);
      fclose (file_ptr);
      file_ptr = 0;
    }
}

-(void) drop
{
  [self finish];

  if (pngbuf) { [[self getZone] free: pngbuf]; pngbuf=0; }
  if (cbuf1) { [[self getZone] free: cbuf1]; cbuf1=0; }
  if (cbuf2) { [[self getZone] free: cbuf2]; cbuf2=0; }
  if (basename) { [basename drop]; basename=nil; }
  if (filename) { [filename drop]; filename=nil; }
}

//---------------------png/mng functions-----------------------------
static int fill_png_palette (Colormap* cm, png_color* pp, int n)
// does this work for WIN32?
{
  XColor xc;
  Display *display = Tk_Display (cm->tkwin);
  int i;

  for (i=0; i<n; ++i)
    {
      int ret;

      if (cm->isSet[i] == NO) continue;
      xc.pixel = cm->map[i];
      ret = XQueryColor (display, cm->cmap, &xc);
      if (ret<0) return -1;
      pp[i].red = xc.red/256;
      pp[i].green = xc.green/256;
      pp[i].blue = xc.blue/256;
      //     printf("color %02d: red %03d green %03d blue %03d\n",
      //     i, xc.red, xc.green, xc.blue);
    }
  return 0;
}

/*
 * write MNG signature
 */
static void
mng_write_sig (png_structp png_ptr)
{
  /* write the rest of the 8 byte signature */
  png_write_data (png_ptr, &mng_sig[png_ptr->sig_bytes],
		  (png_size_t)8 - png_ptr->sig_bytes);
}


/*
 * write the MHDR header to the file
 */
static void
mng_write_MHDR (png_structp png_ptr, int width, int height, int tick, int simp)
{
  png_byte buff[32];
  png_save_uint_32 (buff,   width);
  png_save_uint_32 (buff+4, height);
  png_save_uint_32 (buff+8, tick);
  png_save_uint_32 (buff+12, 0);
  png_save_uint_32 (buff+16, 0);
  png_save_uint_32 (buff+20, 0);
  png_save_uint_32 (buff+24, simp);
  png_write_chunk (png_ptr, "MHDR", buff, (png_size_t)28);
}

/*
 * write the MEND chunk to the file
 */
static void
mng_write_MEND (png_structp png_ptr)
{
  png_write_chunk (png_ptr, "MEND", NULL, (png_size_t)0);
}

/*
 * short version .. not all parameters for the DEFI chunk
 * are supported
 */
static void
mng_write_DEFI (png_structp png_ptr, unsigned short idnr, int conc)
{
  png_byte buff[4];
  png_save_uint_16 (buff, idnr);
  buff[2] = 0;
  buff[3] = conc;
  png_write_chunk (png_ptr, "DEFI", buff, (png_size_t)4);
}

static void mng_write_DHDR (png_structp png_ptr, unsigned short idnr, 
			    char img_type, char delta_type, int w, int h,
			    int x, int y)
{
  png_byte buff[20];
  png_save_uint_16 (buff, idnr);
  buff[2] = img_type;
  buff[3] = delta_type;
  png_save_uint_32 (buff+4, w);
  png_save_uint_32 (buff+8, h);
  png_save_uint_32 (buff+12, x);
  png_save_uint_32 (buff+16, y);
  png_write_chunk (png_ptr, "DHDR", buff, (png_size_t)20);
  
}
#define FRAMBUFLEN 256
/*
 * short version .. not all parameters for the FRAM chunk
 * are supported
 */
static void
mng_write_FRAM (png_structp png_ptr, char framing_mode, char *name,
	char chg_interframe_delay, char chg_sync_timeout,
	char chg_frame_clipping_boundaries, char chg_sync_id_list,
	png_uint_32 interframe_delay)
{
  png_byte buff[FRAMBUFLEN];
  int len = 1;

  if (name) {
    len = strlen (name) + 1;
    if (len > FRAMBUFLEN-10) {
      fprintf (stderr, "MNGVideo error: filename longer %d chars, bailing out...\n", 
	       FRAMBUFLEN-10);
      return;
    }
  }

  buff[0] = framing_mode;
  if (name)
    strcpy (buff+1, name);
  else
    buff[1] = 0;
  buff[len+1] = chg_interframe_delay;
  buff[len+2] = chg_sync_timeout;
  buff[len+3] = chg_frame_clipping_boundaries;
  buff[len+4] = chg_sync_id_list;
  png_save_uint_32 (buff+len+5, interframe_delay);
  png_write_chunk (png_ptr, "FRAM", buff, (png_size_t)(len+9));
}

static void
mng_write_MAGN (
		png_structp png_ptr,
		unsigned short idnr,
		unsigned short mag)
{
  png_byte buff[32];

  png_save_uint_16 (buff, idnr);
  png_save_uint_16 (buff+2, idnr);
  png_save_uint_16 (buff+4, 1); // pixel replication
  png_save_uint_16 (buff+6, mag);
  //  png_save_uint_16 (buff+7, 0);
  //  png_save_uint_16 (buff+9, 0);
  //  png_save_uint_16 (buff+11, 0);
  //  png_save_uint_16 (buff+13, 0);
  //  png_save_uint_16 (buff+15, 0);
  //  buff[17] = 0;

  png_write_chunk (png_ptr, "MAGN", buff, 8);
}  

static void 
mng_write_empty_PLTE (png_structp png_ptr)
{
  png_write_chunk (png_ptr, "PLTE", 0, 0);
}

static void 
mng_write_SAVE (png_structp png_ptr)
{
  png_write_chunk (png_ptr, "SAVE", 0, 0);
}

/* Write the IHDR chunk, and update the png_struct with the necessary
 * information.  Note that the rest of this code depends upon this
 * information being correct.
 */

static void
mng_set_IHDR(png_structp png_ptr, png_uint_32 width, png_uint_32 height,
   int bit_depth, int color_type, int compression_type, int filter_type,
   int interlace_type)
{
#ifdef PNG_USE_LOCAL_ARRAYS
   PNG_IHDR;
#endif
   png_byte buf[13]; /* buffer to store the IHDR info */

   png_debug(1, "in png_write_IHDR\n");
   /* Check that we have valid input data from the application info */
   switch (color_type)
   {
      case PNG_COLOR_TYPE_GRAY:
         switch (bit_depth)
         {
            case 1:
            case 2:
            case 4:
            case 8:
            case 16: png_ptr->channels = 1; break;
            default: png_error(png_ptr,"Invalid bit depth for grayscale image");
         }
         break;
      case PNG_COLOR_TYPE_RGB:
         if (bit_depth != 8 && bit_depth != 16)
            png_error(png_ptr, "Invalid bit depth for RGB image");
         png_ptr->channels = 3;
         break;
      case PNG_COLOR_TYPE_PALETTE:
         switch (bit_depth)
         {
            case 1:
            case 2:
            case 4:
            case 8: png_ptr->channels = 1; break;
            default: png_error(png_ptr, "Invalid bit depth for paletted image");
         }
         break;
      case PNG_COLOR_TYPE_GRAY_ALPHA:
         if (bit_depth != 8 && bit_depth != 16)
            png_error(png_ptr, "Invalid bit depth for grayscale+alpha image");
         png_ptr->channels = 2;
         break;
      case PNG_COLOR_TYPE_RGB_ALPHA:
         if (bit_depth != 8 && bit_depth != 16)
            png_error(png_ptr, "Invalid bit depth for RGBA image");
         png_ptr->channels = 4;
         break;
      default:
         png_error(png_ptr, "Invalid image color type specified");
   }

   if (compression_type != PNG_COMPRESSION_TYPE_BASE)
   {
      png_warning(png_ptr, "Invalid compression type specified");
      compression_type = PNG_COMPRESSION_TYPE_BASE;
   }

   /* Write filter_method 64 (intrapixel differencing) only if
    * 1. Libpng was compiled with PNG_MNG_FEATURES_SUPPORTED and
    * 2. Libpng did not write a PNG signature (this filter_method is only
    *    used in PNG datastreams that are embedded in MNG datastreams) and
    * 3. The application called png_permit_mng_features with a mask that
    *    included PNG_FLAG_MNG_FILTER_64 and
    * 4. The filter_method is 64 and
    * 5. The color_type is RGB or RGBA
    */
   if (
#if defined(PNG_MNG_FEATURES_SUPPORTED)
      !((png_ptr->mng_features_permitted & PNG_FLAG_MNG_FILTER_64) &&
      ((png_ptr->mode&PNG_HAVE_PNG_SIGNATURE) == 0) &&
      (color_type == PNG_COLOR_TYPE_RGB || 
       color_type == PNG_COLOR_TYPE_RGB_ALPHA) &&
      (filter_type == PNG_INTRAPIXEL_DIFFERENCING)) &&
#endif
      filter_type != PNG_FILTER_TYPE_BASE)
   {
      png_warning(png_ptr, "Invalid filter type specified");
      filter_type = PNG_FILTER_TYPE_BASE;
   }

#ifdef PNG_WRITE_INTERLACING_SUPPORTED
   if (interlace_type != PNG_INTERLACE_NONE &&
      interlace_type != PNG_INTERLACE_ADAM7)
   {
      png_warning(png_ptr, "Invalid interlace type specified");
      interlace_type = PNG_INTERLACE_ADAM7;
   }
#else
   interlace_type=PNG_INTERLACE_NONE;
#endif

   /* save off the relevent information */
   png_ptr->bit_depth = (png_byte)bit_depth;
   png_ptr->color_type = (png_byte)color_type;
   png_ptr->interlaced = (png_byte)interlace_type;
   //   png_ptr->filter_type = (png_byte)filter_type;
   png_ptr->width = width;
   png_ptr->height = height;

   png_ptr->pixel_depth = (png_byte)(bit_depth * png_ptr->channels);
   png_ptr->rowbytes = ((width * (png_size_t)png_ptr->pixel_depth + 7) >> 3);
   /* set the usr info, so any transformations can modify it */
   png_ptr->usr_width = png_ptr->width;
   png_ptr->usr_bit_depth = png_ptr->bit_depth;
   png_ptr->usr_channels = png_ptr->channels;

   /* pack the header information into the buffer */
   png_save_uint_32(buf, width);
   png_save_uint_32(buf + 4, height);
   buf[8] = (png_byte)bit_depth;
   buf[9] = (png_byte)color_type;
   buf[10] = (png_byte)compression_type;
   buf[11] = (png_byte)filter_type;
   buf[12] = (png_byte)interlace_type;

   /* write the chunk */
   //   png_write_chunk(png_ptr, (png_bytep)png_IHDR, buf, (png_size_t)13);

   /* initialize zlib with PNG info */
   png_ptr->zstream.zalloc = png_zalloc;
   png_ptr->zstream.zfree = png_zfree;
   png_ptr->zstream.opaque = (voidpf)png_ptr;
   if (!(png_ptr->do_filter))
   {
      if (png_ptr->color_type == PNG_COLOR_TYPE_PALETTE ||
         png_ptr->bit_depth < 8)
         png_ptr->do_filter = PNG_FILTER_NONE;
      else
         png_ptr->do_filter = PNG_ALL_FILTERS;
   }
   if (!(png_ptr->flags & PNG_FLAG_ZLIB_CUSTOM_STRATEGY))
   {
      if (png_ptr->do_filter != PNG_FILTER_NONE)
         png_ptr->zlib_strategy = Z_FILTERED;
      else
         png_ptr->zlib_strategy = Z_DEFAULT_STRATEGY;
   }
   if (!(png_ptr->flags & PNG_FLAG_ZLIB_CUSTOM_LEVEL))
      png_ptr->zlib_level = Z_DEFAULT_COMPRESSION;
   if (!(png_ptr->flags & PNG_FLAG_ZLIB_CUSTOM_MEM_LEVEL))
      png_ptr->zlib_mem_level = 8;
   if (!(png_ptr->flags & PNG_FLAG_ZLIB_CUSTOM_WINDOW_BITS))
      png_ptr->zlib_window_bits = 15;
   if (!(png_ptr->flags & PNG_FLAG_ZLIB_CUSTOM_METHOD))
      png_ptr->zlib_method = 8;
   deflateInit2(&png_ptr->zstream, png_ptr->zlib_level,
      png_ptr->zlib_method, png_ptr->zlib_window_bits,
      png_ptr->zlib_mem_level, png_ptr->zlib_strategy);
   png_ptr->zstream.next_out = png_ptr->zbuf;
   png_ptr->zstream.avail_out = (uInt)png_ptr->zbuf_size;

   png_ptr->mode = PNG_HAVE_IHDR;
}

@end


