//EPDExperimentSwarm.m EPD
//Copyright James Marshall 1998-2004. Freely distributable under the GNU General Public Licence

#import "EPDExperimentSwarm.h"

@implementation ParameterManager

  // The ParameterManager handles the parameter manipulation for
  //     the series of models run during the experiment


- initializeParameters
{
  //create the probe map for the parameter manager
  pmProbeMap = [EmptyProbeMap createBegin: [self getZone]];
  [pmProbeMap setProbedClass: [self class]];
  pmProbeMap = [pmProbeMap createEnd];

  //add probes for the parameter manager's variables
  [pmProbeMap addProbe: [probeLibrary getProbeForVariable: "movementProbability" inClass: [self class]]];
  [pmProbeMap addProbe: [probeLibrary getProbeForVariable: "moveProbInc" inClass: [self class]]];
  [pmProbeMap addProbe: [probeLibrary getProbeForVariable: "moveProbMax" inClass: [self class]]];
  [pmProbeMap addProbe: [probeLibrary getProbeForVariable: "runNumber" inClass: [self class]]];
  [pmProbeMap addProbe: [probeLibrary getProbeForVariable: "runNumberMax" inClass: [self class]]];

  //set the probe map for the parameter manager
  [probeLibrary setProbeMap: pmProbeMap For: [self class]];

  //load parameters from file 'experiment.setup'
  [ObjectLoader load: self fromFileNamed: "experiment.setup"];

  //create display for parameter manager probe
  [probeDisplayManager createProbeDisplayFor: self];

  return self;
}

- initializeModel: (id)theModel
{
  [theModel setMoveProbability: movementProbability]; //set the movement probability for agents in the model

  return self;
}


- stepParameters
{
  runNumber++; //increment the run number for the current parameter settings
  //sweep to the next parameter values and reset the run number if the all the runs have completed for these values
  if (runNumber>runNumberMax)
  {
    runNumber=1;
    movementProbability += moveProbInc;
 
    if (movementProbability > moveProbMax)
    {
      return nil;
    }
  }

  return self;
}

- printParameters: (id <OutFile>)anOutFile
{
  //store the parameter values in a file
  [anOutFile putNewLine];
  [ObjectSaver save: self to: anOutFile withTemplate: pmProbeMap];
  [anOutFile putNewLine];

  return self;
}

@end

@implementation EPDExperimentSwarm

+createBegin: (id) aZone
{
  EPDExperimentSwarm *obj;
  id <ProbeMap> probeMap;

  obj=[super createBegin: aZone];

  //create the probe map for the experiment Swarm
  probeMap=[EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [self class]];
  probeMap=[probeMap createEnd];

  //set the probe map for the experiment Swarm
  [probeLibrary setProbeMap: probeMap For: [self class]];

  return obj;
}

-createEnd
{
  return [super createEnd];
}

-buildObjects
{
  id modelZone;

  [super buildObjects];

  //create the parameter manager
  parameterManager=[ParameterManager create: self];
  [parameterManager initializeParameters];

  //create the Swarm
  modelZone=[Zone create: [self getZone]];
  epdSwarm=[EPDSwarm create: modelZone];

  //create the display for the experiment Swarm probe
  [probeDisplayManager createProbeDisplayFor: self];

  [controlPanel setStateRunning];

  //create the log file
  logFile=[OutFile create: self setName: "log.file"];

  return self;
}

-buildActions
{
  [super buildActions];

  displayActions=[ActionGroup create: [self getZone]];

  //create the action group to build and run the models and log results
  [displayActions createActionTo: self message: M(buildModel)];
  [displayActions createActionTo: self message: M(runModel)];
  [displayActions createActionTo: self message: M(doStats)];
  [displayActions createActionTo: self message: M(showStats)];
  [displayActions createActionTo: self message: M(logResults)];
  [displayActions createActionTo: self message: M(dropModel)];
  [displayActions createActionTo: self message: M(checkToStop)];

  [displayActions createActionTo: probeDisplayManager message: M(update)];

  [displayActions createActionTo: actionCache message: M(doTkEvents)];

  //create the schedule to execute the action group
  displaySchedule=[Schedule createBegin: [self getZone]];
  [displaySchedule setRepeatInterval: 1];
  displaySchedule=[displaySchedule createEnd];

  //schedule the action group
  [displaySchedule at: 0 createAction: displayActions];

  return self;
}

-activateIn: (id)swarmContext
{
  [super activateIn: swarmContext];

  [displaySchedule activateIn: self];

  return [self getSwarmActivity];
}

- buildModel
{
  //create the model Swarm
  epdSwarm = [EPDSwarm create: self];

  // If this is the first model, create a custom probeMap for modelSwarm 
  // and construct a graph displaying model results

  if (numModelRuns == 0)
    {
      modelProbeMap = [EmptyProbeMap createBegin: self];
      [modelProbeMap setProbedClass: [EPDSwarm class]];
      modelProbeMap = [modelProbeMap createEnd];

      [modelProbeMap addProbe: [probeLibrary getProbeForVariable: "worldXsize" inClass: [EPDSwarm class]]];
      [modelProbeMap addProbe: [probeLibrary getProbeForVariable: "worldYsize" inClass: [EPDSwarm class]]];
      [modelProbeMap addProbe: [probeLibrary getProbeForVariable: "maxAgents" inClass: [EPDSwarm class]]];
      [modelProbeMap addProbe: [probeLibrary getProbeForVariable: "maxLocalAgents" inClass: [EPDSwarm class]]];
      [modelProbeMap addProbe: [probeLibrary getProbeForVariable: "startAgents" inClass: [EPDSwarm class]]];
      [modelProbeMap addProbe: [probeLibrary getProbeForVariable: "deathProbability" inClass: [EPDSwarm class]]];
      [modelProbeMap addProbe: [probeLibrary getProbeForVariable: "moveProbability" inClass: [EPDSwarm class]]];
      [modelProbeMap addProbe: [probeLibrary getProbeForVariable: "mutationRate" inClass: [EPDSwarm class]]];
      [modelProbeMap addProbe: [probeLibrary getProbeForVariable: "crossProbability" inClass: [EPDSwarm class]]];
      [modelProbeMap addProbe: [probeLibrary getProbeForVariable: "startEnergy" inClass: [EPDSwarm class]]];
      [modelProbeMap addProbe: [probeLibrary getProbeForVariable: "livingCost" inClass: [EPDSwarm class]]];

      [probeLibrary setProbeMap: modelProbeMap For: [EPDSwarm class]];
      
  }
  
  //initialise the model Swarm according to the values in the parameter manager
  [parameterManager initializeModel: epdSwarm];

  //build and activate the model swarm
  [epdSwarm buildObjects];
  [epdSwarm buildActions];
  [epdSwarm activateIn: nil];

  return self;
}  


- runModel
{
  printf("\nStarting model %d \n", numModelRuns+1);

  [[epdSwarm getActivity] run];

  printf("Model %d is done \n", numModelRuns+1);

  numModelRuns++;

  return self;
}


- doStats
{
  //display the model run results on standard output
  int l;
  
  modelTime =  [[epdSwarm getWorld] getWorldTime];
  averageCooperation=[[epdSwarm getWorld] getAverageCooperationLevel];
  averageRelatedness=[[epdSwarm getWorld] getAverageRelatedness];
  averageInteractionLength=[[epdSwarm getWorld] getAverageInteractionLength];
  averageCellPopulation=[[epdSwarm getWorld] getAverageLocalPopSize];
  averageNumberOfPartners=[[epdSwarm getWorld] getAverageNumberOfPartners];

  printf("Length of this run = %d \n", modelTime);
  printf("Movement probability = %f\n",[[epdSwarm getWorld] getMoveProbability]);
  printf("Average cooperation = %d\n", averageCooperation);
  printf("Average relatedness = %d\n", averageRelatedness);
  printf("Average interaction length = %f\n", averageInteractionLength);
  printf("Average cell population = %f\n", averageCellPopulation);
  printf("Average number of partners per agent = %f\n", averageNumberOfPartners);
  for (l=0; l<[[epdSwarm getWorld] getStrategyLength]; l++)
  {
    printf("Average allele %d frequency = %d\n", l, [[epdSwarm getWorld] getAverageAlleleFrequency: l]);
  }

  return self;
}


- (int)getModelTime
{
  // This method feeds the EZGraph "runTime" sequence  
  // this always points to the current modelSwarm 

   return [[epdSwarm getWorld] getWorldTime];
}


- logResults
{
  //store the model run results in the log file
  
  int l;
  
  [logFile putInt: averageCooperation];
  [logFile putString: ","];
  [logFile putInt: averageRelatedness];
  [logFile putString: ","];
  [logFile putDouble: averageInteractionLength];
  [logFile putString: ","];
  [logFile putInt: modelTime];
  [logFile putString: ","];
  [logFile putDouble: averageCellPopulation];
  [logFile putString: ","];
  [logFile putDouble: averageNumberOfPartners];
  for (l=0; l<[[epdSwarm getWorld] getStrategyLength]; l++)
  {
    [logFile putString: ","];  
    [logFile putInt: [[epdSwarm getWorld] getAverageAlleleFrequency: l]];
  }

  [logFile putNewLine];

  return self;
}


- showStats
{
//  [resultGraph step];
  return self;
}


- dropModel
{
  [[epdSwarm getActivity] drop];
  [epdSwarm drop];

  return self;
}


- checkToStop
{
  //check if the current model run has reached the termination point (defined as an execution length)
  if ([parameterManager stepParameters]==nil)
    {
      [probeDisplayManager update];
      [actionCache doTkEvents];
      
      printf("\n All the models have run!\n");
      
      [logFile drop];
      [controlPanel setStateQuit];
    }
  
  return self;
}

@end
