//Copyright Simon Tokumine 2003. Freely distributable under the GNU General Public Licence

import java.text.NumberFormat;

/**
 * @author 	S. Tokumine
 * @date	22-Jun-2003
 * @time	20:01:57
 * @file	Timeslice.java
 * 
 * @desc 	Container Class for all the strategies 
 * 			and their associated frequencies and fitness
 *  		of a timeslice
 *
 */

public class Timeslice {

	private static HeightMapFactory mapFact = new HeightMapFactory();
	private static ColorMapFactory colorFact;
	private static ScoreMapFactory scoreFact;
	private int[] freqList; //frequency list
	private double[] freqListConv; //double freq list for normalised freq scores
	private double[] fitList; //fitness list
	public static Payoffs payOffFact = new Payoffs();
	

	/**
	 * constructor sets up timeslice, including 2d height map
	 * @param Array Inputlist of strategy frequencies
	 */
	public Timeslice(int[] freqList_in, ColorMapFactory colorFact_in, ScoreMapFactory scoreMapFactory_in) {
		
		//Dynamically assign size of arrays
		freqList = new int[freqList_in.length];
		fitList = new double[freqList.length];
		freqListConv = new double[freqList.length];
		
		//Copy XML frequency output array to local array
		System.arraycopy(freqList_in, 0, freqList, 0, freqList_in.length);

		colorFact = colorFact_in;		
		scoreFact = scoreMapFactory_in;
		
		//Call the fitness array populator
		populateList();
	}

	/**	
	 *  Create FITNESS scores
	 *	possible speed issue with using another array list:
	 *  replaced with standard array declared in constructor.
	 */
	public void populateList() {
		double totalFreq = 0;
		
		//sum the total size of the population
		for (int y = 0; y < freqList.length; y++) {
			totalFreq = totalFreq + freqList[y];
		}
		//use total size to calculate a normalised frequency array
		for (int i = 0; i < freqList.length; i++) {
			freqListConv[i] = (double) freqList[i] / totalFreq;
		}
		//use normalised frequency list to calculate fitness 
		for (int i = 0; i < freqListConv.length; i++) {
			fitList[i] = payOffFact.calculateFitness(i,freqListConv);
		}
	}

	/**
	 * Access methods, length, fitness and frequency. -1 for error.
	 * @return
	 */

	public int getTypeNo() {
		return freqList.length;
	}

	public int getFreq(int gType) {
		try {
			return freqList[gType];
		} catch (ArrayIndexOutOfBoundsException ai) {
			return -1;
		}
	}

	public double getFit(int gType) {
		try {
			return fitList[gType];
		} catch (ArrayIndexOutOfBoundsException ai) {
			return -1;
		}
	}

	/**
	 * calls heightMapFactory, returns 1d array of ARGB values.
	 * @return
	 */
	public int[] getHeightMap() {
		return mapFact.fit2HeightMap(scoreFact.get2DArray(fitList));
	}

	/**
	* calls colorMapFactory, returns 1d array of ARGB values.
	* @return
	*/
	public int[] getColorMap() {
		return colorFact.freq2ColorMap(scoreFact.get2DArray(freqListConv));
	}

	/**
	* calls scoreMapFactory, returns the size of the 2D array x axis.
	* @return
	*/
	public int getArraySizeX() {
		return scoreFact.getArraySizeX();
	}

	/**
	* calls scoreMapFactory, returns the size of the 2D array y axis.
	* @return
	*/
	public int getArraySizeY() {
		return scoreFact.getArraySizeY();
	}


	/**
	 * 
	 *Print outs for testing
	 */
	public void printAll(TextPanel textPanel) {
    double totalFreq = 0;
    
    //sum the total size of the population
    for (int y = 0; y < freqList.length; y++) {
      totalFreq = totalFreq + freqList[y];
    }
		//set up a number formater to ensure output is to 3dp..
		NumberFormat nf	= NumberFormat.getInstance();
		nf.setMaximumFractionDigits(3);
		nf.setMinimumFractionDigits(3);
		
		for (int i = 0; i < fitList.length; i++) {
			textPanel.print("Genotype " + i + " Frequency:    " + (double) freqList[i] / totalFreq);
			textPanel.print("Genotype " + i + " Fitness:      " + nf.format(fitList[i]));
		}
	}
}
