//Copyright Simon Tokumine 2003. Freely distributable under the GNU General Public Licence

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.Dimension;
import java.awt.HeadlessException;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import javax.swing.JFrame;
import javax.swing.UIManager;

/**
 * @author 	S. Tokumine
 * @date	24-Jun-2003
 * @time	00:14:35
 * @file	E3DFrame.java
 * 
 * @desc Root level frame for GUI and E3D system,
 *
 */
public class E3DFrame extends JFrame {

	private MainPanel mainPanel; //2D images
	private SlidePanel slidePanel; //slider panel
	private TextPanel textPanel; //output panel
	private JFileChooser fc; //filechooser for opening files
	private SliceList slist; //list of timesteps
	private XMLReader XMLRead; //SAX based XML parser
	private JTabbedPane tabbedPane; //high level container for inner panels
	private StatusBar statusBar; //status bar at bottom of program
	private ColorChooser colorChooser; //colour chooser panel
	private static ColorMapFactory colorMapFactory = new ColorMapFactory();
	private static ScoreMapFactory scoreMapFactory = new ScoreMapFactory();
	private ImagePanel panel3D; // area for 3D scene
	private SSMPanel sSMPanel; //search space matrix descriptor panel
	private JFrame progress; // test progress bar window
	private JProgressBar progressBar; //test progressbar
	private Timer timer; //used for testing progress bar

	//	private TeVi tevi;

	/**
	 * Constructor sets up all the panels and objects used in the E3D. Also sets up window
	 * behaviours
	 * @throws java.awt.HeadlessException
	 */
	public E3DFrame() throws HeadlessException {

		/*Setup name, decorations,layout manager and behavious of E3D window*/
		super("E3D - A 3D visualisation tool for the EPD - v0.8");
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				System.exit(0);
			}
		});

		this.setResizable(false); //cant maximise window	

		Image icon = Toolkit.getDefaultToolkit().getImage("icon3.gif");
		this.setIconImage(icon); //icon for top leftof window

		getContentPane().setLayout(new BorderLayout());

		/*create a status bar*/
		statusBar = new StatusBar();

		/* Now create a main panel and set its size */
		mainPanel = new MainPanel(scoreMapFactory);
		mainPanel.setPreferredSize(new Dimension(545, 340)); //336

		/* Now create a text output panel*/
		textPanel = new TextPanel();

		/* Set up timeslice list */
		slist = new SliceList(10, textPanel, colorMapFactory, scoreMapFactory);

		/* Create a colorChooser class */
		colorChooser = new ColorChooser(mainPanel, colorMapFactory, statusBar);
		colorChooser.setPreferredSize(new Dimension(545, 420));

		//init 3d panel for the slider panel pass
		panel3D = new ImagePanel();

		/* Now create a control panel and set its size */
		slidePanel =
			new SlidePanel(
				slist,
				textPanel,
				mainPanel,
				colorChooser,
				statusBar,
				panel3D /*, test3D*/
		);
		slidePanel.setPreferredSize(new Dimension(545, 100));

		/* Now create a combo panel of main and controls*/
		JPanel mainPlusControl = new JPanel();
		mainPlusControl.setPreferredSize(new Dimension(545, 450));
		mainPlusControl.add(mainPanel, BorderLayout.CENTER);
		mainPlusControl.add(slidePanel, BorderLayout.SOUTH);

		/*Create a SSMatrix explanation class*/
		sSMPanel = new SSMPanel(scoreMapFactory, slidePanel);

		/*set up help panel*/
		HTMLPanel instructionPanel = new HTMLPanel();

		/* create a tabbed pane set of windows for the main pane and the other panes*/
		tabbedPane = new JTabbedPane();
		tabbedPane.addTab("Control Panel", mainPlusControl);
		tabbedPane.setSelectedIndex(0);

		tabbedPane.addTab("Colour Schemes", colorChooser);

		tabbedPane.addTab("Search Space Matrix", sSMPanel);

		tabbedPane.addTab("Numerical Output", textPanel);

		tabbedPane.addTab("Instructions & Help", instructionPanel);

		//link in the scoremapfactory with the slidepanel so that it can refresh on movement only
		scoreMapFactory.setSlider(slidePanel);

		//create frame for the 3d vis
		JPanel rightSide = new JPanel();
		rightSide.setPreferredSize(new Dimension(455, 450));
		panel3D.setPreferredSize(new Dimension(445, 455));
		rightSide.setBorder(BorderFactory.createEmptyBorder(16, 5, 5, 5));
		panel3D.setBorder(BorderFactory.createLineBorder(Color.GRAY, 1));

		//set splash image in the 3d vis window
		URL splash3DURL = MainPanel.class.getResource("resources/3dsplash.jpg");
		Toolkit toolkit = Toolkit.getDefaultToolkit();
		panel3D.SetImage(toolkit.getImage(splash3DURL));

		//add panel3d to the rightside 
		rightSide.add(panel3D);

		/* add the panels to the main frame's content pane. */
		getContentPane().add(tabbedPane, BorderLayout.WEST);
		getContentPane().add(rightSide, BorderLayout.EAST);
		getContentPane().add(statusBar, BorderLayout.SOUTH);

		/* add menu bar to the frame */
		addMenu();

		/* Initialise JFileChooser */
		initFileChooser();

		/* Make sure stuff gets displayed. */
		pack();
		textPanel.requestFocusInWindow();
		setVisible(true);
		show();
	}

	/**
	 * Initialises the filechooser with the XML filter class
	 *
	 */
	private void initFileChooser() {

		fc = new JFileChooser();
		fc.addChoosableFileFilter(new XMLFilter());
	}

	/**
	 * Adds the menu to the E3D frame, and sets up the open file and exit dialogs
	 *
	 */
	private void addMenu() {
		JMenuBar menuBar;
		JMenu menu;
		JMenuItem menuItem;

		//Create the menu bar.
		menuBar = new JMenuBar();
		setJMenuBar(menuBar);

		//Build the first menu.
		menu = new JMenu("File");
		menu.setMnemonic(KeyEvent.VK_F);
		menu.getAccessibleContext().setAccessibleDescription(
			"Main File Access Menu");
		menuBar.add(menu);

		//Open XML file menu button
		menuItem = new JMenuItem("Open XML file", KeyEvent.VK_O);
		menuItem.setAccelerator(
			KeyStroke.getKeyStroke(KeyEvent.VK_1, ActionEvent.ALT_MASK));
		menuItem.getAccessibleContext().setAccessibleDescription(
			"Opens XML file from EPD");
		menuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				try {
					openButtonPress(evt);
				} catch (IOException ioe) {
				}
			}
		});

		menu.add(menuItem);

		//Exit E3D menu button
		menuItem = new JMenuItem("Exit E3D", KeyEvent.VK_E);
		menuItem.setAccelerator(
			KeyStroke.getKeyStroke(KeyEvent.VK_2, ActionEvent.ALT_MASK));
		menuItem.getAccessibleContext().setAccessibleDescription("Exits EPD");
		menuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				try {
					exit(evt);
				} catch (IOException ioe) {
				}
			}
		});
		menu.add(menuItem);
	}

	/**
	 * action to perform on exit menu item selection
	 * @param evt
	 * @throws IOException
	 */
	private void exit(ActionEvent evt) throws IOException {
		System.exit(1);
	}

	/**
	 * performed when file open button pressed
	 * open the xml file, parse it, populate timeslice datastructures
	 * also: initialise the controls and main image panels
	 * @param evt
	 * @throws IOException
	 */
	private void openButtonPress(ActionEvent evt) throws IOException {

		int returnVal = fc.showOpenDialog(E3DFrame.this);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fc.getSelectedFile();
			openXML(file);
			slidePanel.initControls(0, slist.size() - 1);
			mainPanel.initMovement();
		}
	}

	/**
	 * takes the file from the JFilechooser and parses it. Also, set up flags for 
	 * file loaded status in the colorchoosers, and initialise mose movement in the ssmpanel.
	 * @param XMLFile
	 */
	private void openXML(File XMLFile) {
		slist.clear();
		textPanel.clearPanel();
		XMLRead = new XMLReader(XMLFile, slist, statusBar);
		colorChooser.fileLoaded();
		sSMPanel.initMouse();
	}

	/**
	 * set up E3D with the systems Look and feel (good on XP systems with 1.4.2 RE)
	 * @param args
	 */
	public static void main(String[] args) {
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());

		} catch (Exception e) {
		}
		E3DFrame mainFrame = new E3DFrame();
	}
}
