//Copyright Simon Tokumine 2003. Freely distributable under the GNU General Public Licence

/**
 * @author 	S. Tokumine
 * @date	30-Jun-2003
 * @time	11:13:01
 * @file	ColorMapFactory.java
 * 
 * @desc In charge of converting scoremaps of population density to 
 * 		 1D arrays of colours (represented by integers in aRGB format)
 * 		 Also is used to apply grids
 *
 */

public class ColorMapFactory {

	int noColors = 256; //number of colours in the pallette
	int[] colorMap = new int[noColors]; //create pallete
	int isGrid = 3; //sets up the grid flag (0 = no grid)
	int white; //used as a reference white colour
	int seeThru; //test to see if J3D can use alpha blended transparent colours
	//- got to be too simple!

	/**
	 * Constructor initialises colorMap.
	 */
	public ColorMapFactory() {
		initMap();
	}

	/**
	 * sets color pallette - usually called form the colourchooser set
	 * @param colorMap_in The pallete to change to
	 */
	public void setColorMap(int[] colorMap_in) {
		colorMap = colorMap_in;
	}

	/**
	 * sets grid flags - does a bit of error checking.
	 * @param gridType integer 0 -> 3.
	 * 
	 * 0 = no grid
	 * 1 = white lines
	 * 2 = inverted lines
	 * 3 = inverted dots		  
	 */
	public void setGrid(int gridType) {
		if (gridType >= 0 && gridType <= 3) {
			isGrid = gridType;
		}
	}

	/**
	 * calibrates color colour pallete to default fire color scheme
	 */
	private void initMap() {
		int Alpha = 255; //integers defining colour values. 
		int Red = 0; //RGB set to 0 (black)
		int Green = 0;
		int Blue = 0;

		//set up white color
		white = (255 << 24) | (255 << 16) | (255 << 8) | 255;

		//set up see thru color
		seeThru = (0 << 24) | (0 << 16) | (0 << 8) | 0;

		//set up step amount (when we swap incrementing colours)
		int colorband = (int) noColors / 3;

		//increment red first (silly multiplications by 0 and 1 left for clarity)
		for (int i = 0 * colorband; i < 1 * colorband; i++) {
			colorMap[i] = (Alpha << 24) | (Red << 16) | (Green << 8) | Blue;
			Red += 3;
		}

		//then green
		for (int i = 1 * colorband; i < 2 * colorband; i++) {
			colorMap[i] = (Alpha << 24) | (Red << 16) | (Green << 8) | Blue;
			Green += 3;
		}

		//finally blue
		for (int i = 2 * colorband; i < 3 * colorband; i++) {
			colorMap[i] = (Alpha << 24) | (Red << 16) | (Green << 8) | Blue;
			Blue += 3;
		}
	}

	/**
	 * converts a 2D score array into a 1D integer color array ready
	 * for conversion to an image
	 * @param freqMap (popualtion density scores) 2D array input
	 * @return 1D integer color array ready for image conversion
	 */
	public int[] freq2ColorMap(double[][] freqMap) {

		//2D integer array of equal size to the freqmap array
		//use it to store double scores as integer colours pre convertion to 1D
		int[][] temp = new int[freqMap.length][freqMap[0].length];

		//1D integer array used to hold 2D integer colour array (temp) values when 
		//converted to 1D array ready for image conversion
		int[] array1D = new int[freqMap.length * freqMap[0].length];

		//records position within 1D array
		int index = 0;

		//convert freqMap of double scores to a 2D array of integer colors
		//also overlays grid dependant on flags 
		for (int i = 0; i < freqMap.length; i++) {
			for (int j = 0; j < freqMap[0].length; j++) {
				//added to trim edges - stops texture bleed in 3D
				if (j > 64 && j < 190) {

					//gets colour for the score and places it in the 2D integer array
					temp[i][j] = getColor(freqMap[i][j]);

					//white lines grid creation
					if (isGrid == 1) {
						if (i != 0) {
							if (i % 32 == 0) {
								temp[i][j] = white;
							}
						}
						if (j != 0) {
							if (j % 32 == 0) {
								temp[i][j] = white;
							}
						}
					}

					//inverted lines grid creation
					if (isGrid == 2) {
						if (i != 0) {
							if (i % 32 == 0) {
								temp[i][j] = invertColor(temp[i][j]);
							}
						}
						if (j != 0) {
							if (j % 32 == 0) {
								temp[i][j] = invertColor(temp[i][j]);
							}
						}
					}

					//inverted dots grid creation
					if (isGrid == 3) {
						if (i != 0) {
							if (i % 32 == 0) {
								if (j % 2 == 0) {
									temp[i][j] = invertColor(temp[i][j]);
								}
							}
						}
						if (j != 0) {
							if (j % 32 == 0) {
								if (i % 2 == 0) {
									temp[i][j] = invertColor(temp[i][j]);
								}
							}
						}
						if (i != 0 && j != 0) {
							if (i % 32 == 0 && j % 32 == 0) {
								temp[i][j] = invertColor(temp[i][j]);
							}
						}
					}
					
				} else {
					//outside of trim area, set colour to transparent.
					temp[i][j] = seeThru;
				}
			}
		}

		//convert 2D color array to 1D array and return, trimming a pixel around
		for (int y = 1; y < temp.length - 1; y++) {
			for (int x = 1; x < temp[0].length - 1; x++) {
				array1D[index++] = temp[y][x];
			}
		}
		return array1D;
	}

	/**
	 * unpack color bits from the aRGB integer values and invert them
	 * @param color_in the colour to be inverted
	 * @return inverted colour
	 */
	private int invertColor(int color_in) {

		//unpack colour bits from colour_in
		int red = ((color_in >> 16) & (255));
		int green = ((color_in >> 8) & (255));
		int blue = (color_in & 255);

		//return them inverted(255-x)
		return (
			(255 << 24)
				| ((255 - red) << 16)
				| ((255 - green) << 8)
				| (255 - blue));
	}

	/**
	 * return the colour for the score, as per the pallette
	 * truncates any decimal points, but ok, as wont notice visually
	 * @param height Height value between 1 and 0.
	 * @return	a color value between 0 and 255 (ARGB)
	 */
	private int getColor(double height) {
		return colorMap[(int) (height * 255)];
	}
}
