//EPDAgent.m EPD
//Copyright James Marshall 1998-2002. Freely distributable under the GNU General Public Licence

#import "EPDAgent.h"

@implementation EPDAgent

-setupWorld: world x: (int) xStart y: (int) yStart
{
  //initialise agent with information on the world they inhabit, their position within it, and their state
  id <List> tempList;

  alive=YES; //the agent is alive
  played=NO; //the agent has not played the PD this round
  mated=NO; //the agent has not mated this round
  myWorld=world; //the world the agent inhabits
  //create the agent's strategy chromosome
  strategy=[Array create: [self getZone] setCount: [world getStrategyLength]];
  //create the agent's history (memory of previous PD interactions with other agents
  tempList=[List create: [self getZone]];
  history=[tempList begin: [tempList getZone]];
  x=xStart; //the agent's x-coordinate in the world
  y=yStart; //the agent's y-coordinate in the world
  fitness=0;
  totalInteractions=0; //the total number of PD interactions the agent has had
  totalFitness=0; //the total payoffs resulting from PD interactions the agent has had
  energy=[myWorld getStartEnergy]; //the agent's energy level
  return self;
}

-createEnd
{
  return [super createEnd];
}

-move
{
  id <ListIndex> myList;
  int l, xChange, yChange, stratNum;

  if ([uniformDblRand getDoubleWithMin: 0 withMax: 1]>[myWorld getDeathProbability] && energy>0)
  {
    //the agent has sufficient energy and has not died through 'natural causes'... the agent lives
    played=NO;
    mated=NO;
    energy=energy-[myWorld getLivingCost]; //decrease agent's energy level
    if ([uniformDblRand getDoubleWithMin: 0 withMax: 1]<[myWorld getMoveProbability]) //agent moves with certain probability
    {
      //agent moves
      //remove agent from current local cell list in world
      myList=[myWorld getObjectAtX: x Y: y];
      [myList setLoc: Start];
      if ([myList findNext: self])
      {
        [myList remove];
      }
      else
      {
        printf("Unable to remove agent from list\n");
      }

      //randomly determine neighbouring cell to move into
      xChange=0;
      yChange=0;
      while (xChange==0 && yChange==0)
      {
        xChange=[uniformIntRand getIntegerWithMin: -1 withMax: 1];
        yChange=[uniformIntRand getIntegerWithMin: -1 withMax: 1];
      }
      x=x+xChange;
      y=y+yChange;
      x=(x+[myWorld getSizeX])%[myWorld getSizeX];
      y=(y+[myWorld getSizeY])%[myWorld getSizeY];

      //add agent to new local cell list in world
      myList=[myWorld getObjectAtX: x Y: y];
      [[myList getCollection] addFirst: self];
    }
  }
  else
  {
    //agent dies

    alive=NO;

    //decrease frequency of agent's alleles in gene pool
    for (l=0; l<[myWorld getStrategyLength]; l++)
    {
      if ((int)[[self getStrategy] atOffset: l]==1)
      {
        [myWorld decreaseAlleleFrequency: l];
      }
    }

    //decrease frequency of agent's strategy in population
    stratNum=(int)[[self getStrategy] atOffset: 2];
    stratNum+=(int)[[self getStrategy] atOffset: 3]*2;
    stratNum+=(int)[[self getStrategy] atOffset: 0]*4;
    [myWorld decreaseStrategyFrequency: stratNum];

    //remove agent from local cell list in world
    myList=[myWorld getObjectAtX: x Y: y];
    [myList setLoc: Start];
    if ([myList findNext: self])
    {
      [myList remove];
    }
    else
    {
      printf("Couldn't find agent on local list\n");
    }
  }

  return self;
}

-play
{
  int agentsHere;
  int agentToPlayWith;
  int myPreviousChoice, theirPreviousChoice;
  int myCurrentChoice, theirCurrentChoice;
  int myPayoff, theirPayoff;
  int interactionLength;
  id historyItem1, historyItem2, potentialPartner, partner, localList;
#ifdef relatedness_measure3
  int l1;
  id alleles;
  double diversity;
#endif
#ifdef interaction_method2
  int l2;
#endif

  //here to prevent compiler warnings
  myCurrentChoice=0;
  theirCurrentChoice=0;
  myPreviousChoice=0;
  theirPreviousChoice=0;
  historyItem1=nil;
  historyItem2=nil;
  partner=nil;
  interactionLength=0;

  //find how many potential free partners are here from the agent's local cell list
  agentsHere=0;
  localList=[myWorld getObjectAtX: x Y: y];
  [localList setLoc: Start];
  if (played==NO)
  {
    while ((potentialPartner=[localList next]))
    {
      if (potentialPartner!=self && [potentialPartner hasPlayed]==NO) //check if this partner is not the agent, and if they haven't played the PD this round
      {
        agentsHere++; //this is a valid partner
      }
    }
  }
  //if there are other free partners here, pick one at random to play against
  if (agentsHere>0)
  {
    agentToPlayWith=[uniformIntRand getIntegerWithMin: 0 withMax: agentsHere-1]; //randomly select a partner
    [localList setLoc: Start];
    //find that partner in the local cell list
    while ((potentialPartner=[localList next]))
    {
      if (potentialPartner!=self && [potentialPartner hasPlayed]==NO)
      {
        if (agentToPlayWith==0)
        {
          partner=potentialPartner;
        }
        else	
        {
          agentToPlayWith--;
        }
      }
    }

    #ifdef relatedness_measure3
      //calculate the interacting agent's allelic diversity
      alleles=[Array create: [self getZone] setCount: [myWorld getStrategyLength]];
      diversity=0;
      #ifdef relatedness_measure4
        for (l1=2; l1<4; l1++)
        {
          [alleles atOffset: l1 put: [DoubleData create: [self getZone]]];
          [[alleles atOffset: l1] setDoubleData: (double)((int)[strategy atOffset: l1]+(int)[[partner getStrategy] atOffset: l1])/2];
          diversity=diversity+1-(4*((0.5-[[alleles atOffset: l1] getDoubleData])*(0.5-[[alleles atOffset: l1] getDoubleData])));
        }
        diversity=diversity/2;
      #else
        for (l1=0; l1<[myWorld getStrategyLength]; l1++)
        {
          [alleles atOffset: l1 put: [DoubleData create: [self getZone]]];
          [[alleles atOffset: l1] setDoubleData: (double)((int)[strategy atOffset: l1]+(int)[[partner getStrategy] atOffset: l1])/2];
          diversity=diversity+1-(4*((0.5-[[alleles atOffset: l1] getDoubleData])*(0.5-[[alleles atOffset: l1] getDoubleData])));
        }
        diversity=diversity/[myWorld getStrategyLength];
      #endif
    #endif

    #ifdef interaction_method2
      //repeat PD interaction for 3 iterations
      for (l2=0; l2<3; l2++)
      {
    #endif
    #ifdef pure_kin_selection
      //randomly assign roles between agents (used by each agent to choose which strategy locus to use in this interaction)
      if ([uniformDblRand getDoubleWithMin: 0 withMax: 1]<0.5)
      {
        myCurrentChoice=(int)[strategy atOffset: 0];
        theirCurrentChoice=(int)[[partner getStrategy] atOffset: 1];
      }
      else
      {
        myCurrentChoice=(int)[strategy atOffset: 1];
        theirCurrentChoice=(int)[[partner getStrategy] atOffset: 0];
      }
    #else
      //see if the agent has played this partner before by checking the agent's interaction history
      myPreviousChoice=-1;
      theirPreviousChoice=-1;
      interactionLength=0;
      [history setLoc: Start];
      [[partner getHistory] setLoc: Start];
      while ((historyItem1=[history next]) && [historyItem1 atOffset: 0]!=partner)
      {
      }
      if (historyItem1!=nil && [historyItem1 atOffset: 0]==partner)
      {
        //this partner has been played before, recover the actions from the last interaction
        myPreviousChoice=(int)[historyItem1 atOffset: 1];
        theirPreviousChoice=(int)[historyItem1 atOffset: 2];
        interactionLength=(int)[historyItem1 atOffset: 3];
        //find partner's memory of last interaction
        while ((historyItem2=[[partner getHistory] next]) && [historyItem2 atOffset: 0]!=self)
        {
        }
      }
      if (myPreviousChoice!=-1 && theirPreviousChoice!=-1)
      {
        //these agents have played before
        if (myPreviousChoice==0 && theirPreviousChoice==0)
        {
          //last time the agent played B and their partner played B
          myCurrentChoice=(int)[strategy atOffset: 4];
          theirCurrentChoice=(int)[[partner getStrategy] atOffset: 4];
        }
        if (myPreviousChoice==0 && theirPreviousChoice==1)
        {
          //last time the agent played B and their partner played A
          myCurrentChoice=(int)[strategy atOffset: 2];
          theirCurrentChoice=(int)[[partner getStrategy] atOffset: 3];
        }
        if (myPreviousChoice==1 && theirPreviousChoice==1)
        {
          //last time the agent played A and their partner played A
          myCurrentChoice=(int)[strategy atOffset: 1];
          theirCurrentChoice=(int)[[partner getStrategy] atOffset: 1];
        }
        if (myPreviousChoice==1 && theirPreviousChoice==0)
        {
          //last time the agent played A and their partner played B
          myCurrentChoice=(int)[strategy atOffset: 3];
          theirCurrentChoice=(int)[[partner getStrategy] atOffset: 2];
        }
      }
      else
      {
        //these agents have never met before
        myCurrentChoice=(int)[strategy atOffset: 0];
        theirCurrentChoice=(int)[[partner getStrategy] atOffset: 0];
        historyItem1=[Array create: [self getZone] setCount: 4];
        [[history getCollection] addFirst: historyItem1];
        historyItem2=[Array create: [self getZone] setCount: 4];
        [[[partner getHistory] getCollection] addFirst: historyItem2];
      }
      //add this interaction to my history
      interactionLength++;
      [historyItem1 atOffset: 0 put: partner];
      [historyItem1 atOffset: 1 put: (id)myCurrentChoice];
      [historyItem1 atOffset: 2 put: (id)theirCurrentChoice];
      [historyItem1 atOffset: 3 put: (id)interactionLength];
      //add this interaction to their history
      [historyItem2 atOffset: 0 put: self];
      [historyItem2 atOffset: 1 put: (id)theirCurrentChoice];
      [historyItem2 atOffset: 2 put: (id)myCurrentChoice];
      [historyItem2 atOffset: 3 put: (id)interactionLength];
    #endif
    //update fitnesses
    myPayoff=[myWorld getPayoffAction1: myCurrentChoice action2: theirCurrentChoice];
    theirPayoff=[myWorld getPayoffAction1: theirCurrentChoice action2: myCurrentChoice];
    [self updateFitness: myPayoff];
    [partner updateFitness: theirPayoff];
    //update energy levels
    [self increaseEnergy: myPayoff];
    [partner increaseEnergy: theirPayoff];
    //update cooperation level
    #ifdef consider_interaction_subset
      if ((myPreviousChoice==0 && theirPreviousChoice==1) || (myPreviousChoice==1 && theirPreviousChoice==0))
      {
    #endif
        #ifdef role_swap_cooperation
          if ((myCurrentChoice==0 && theirCurrentChoice==1) || (myCurrentChoice==1 && theirCurrentChoice==0))
          {
            [myWorld updateCoopLevelMyAction: 1 TheirAction: 1];
            [myWorld incrementTotalCooperativeActionsAtX: x Y: y NumCoopActions: 2];
          }
          else
          {
            [myWorld updateCoopLevelMyAction: 0 TheirAction: 0];
          }
          [myWorld incrementTotalActionsAtX: x Y: y NumActions: 2];
        #else
          [myWorld updateCoopLevelMyAction: myCurrentChoice TheirAction: theirCurrentChoice];
          [myWorld incrementTotalCooperativeActionsAtX: x Y: y NumCoopActions: myCurrentChoice+theirCurrentChoice];
          [myWorld incrementTotalActionsAtX: x Y: y NumActions: 2];
        #endif
    #ifdef consider_interaction_subset
      }
    #endif
    //set agents' current action
    currentAction=myCurrentChoice;
    [partner setCurrentAction: theirCurrentChoice];

    #ifdef relatedness_measure3
      #ifdef consider_interaction_subset
        if ((myPreviousChoice==0 && theirPreviousChoice==1) || (myPreviousChoice==1 && theirPreviousChoice==0))
        {
      #endif
          [myWorld increaseGlobalRelatedness: 1-diversity];
          [myWorld incrementTotalDiversityAtX: x Y: y Diversity: diversity];
      #ifdef consider_interaction_subset
        }
      #endif
    #endif
    #ifdef interaction_method2
      } //terminate for loop for iterated PD interaction
    #endif

    #ifdef relatedness_measure3
      //drop the allele frequency array
      for (l1=0; l1<[myWorld getStrategyLength]; l1++)
      {
        [[alleles atOffset: l1] drop];
      }
      [alleles drop];
    #endif
  }

  return self;
}

-mate
{
  int agentsHere, agentToMateWith, stratNum;
  int l1;
#ifdef global_mating_pool
  int newX, newY;
#endif
  id potentialPartner, partner, matingList, newAgent, mainParent, otherParent;

  //here to prevent compiler warning
  partner=nil;

  //find how many potential free partners are here from the agent's local cell list
  agentsHere=0;
#ifdef global_mating_pool
  matingList=[myWorld getAllAgents];
#else
  matingList=[myWorld getObjectAtX: x Y: y];
#endif
  [matingList setLoc: Start];
#ifdef global_population_regulation
  if (mated==NO && [uniformDblRand getDoubleWithMin: 0 withMax: 1]<1-((double)[myWorld getWorldPopulation]/(double)[myWorld getMaxAgents])) //agent can mate if it hasn't already done so, and with a probability inversely proportional to the global population size
#else
  if (mated==NO && [uniformDblRand getDoubleWithMin: 0 withMax: 1]<1-((double)[[matingList getCollection] getCount]/(double)[myWorld getMaxLocalAgents])) //agent can mate if it hasn't already done so, and with a probability inversely proportional to the local population size
#endif
  {
    while ((potentialPartner=[matingList next]))
    {
      if (potentialPartner!=self && [potentialPartner hasMated]==NO) //check if this partner is not the agent, and if they have not already mated
      {
        agentsHere++;
      }
    }
  }
  //if there are other free partners here, pick one at random to mate with
  if (agentsHere>0)
  {
    agentToMateWith=[uniformIntRand getIntegerWithMin: 0 withMax: agentsHere-1]; //randomly select a partner
    [matingList setLoc: Start];
    while ((potentialPartner=[matingList next]))
    {
      if (potentialPartner!=self && [potentialPartner hasMated]==NO)
      {
        if (agentToMateWith==0)
        {
          partner=potentialPartner;
        }
        else
        {
          agentToMateWith--;
        }
      }
    }

    //set agent and partner as mated
    [self setMated];
    [partner setMated];

    //create an offspring agent
    newAgent=[EPDAgent createBegin: [self getZone]];
#ifdef global_mating_pool
    newX=[uniformIntRand getIntegerWithMin: 0 withMax: [myWorld getSizeX]-1];
    newY=[uniformIntRand getIntegerWithMin: 0 withMax: [myWorld getSizeY]-1];
    [newAgent setupWorld: myWorld x: newX y: newY];
#else
    [newAgent setupWorld: myWorld x: x y: y];
#endif
    newAgent=[newAgent createEnd];
    //set offspring as mated so no other agents mate with it during this time step
    [newAgent setMated];
    //initialise offspring agent's chromosome with parameterised crossover and mutation
    //choose main parent with 50% probability (avoids bias in transmission of parental genetic material)
    if ([uniformDblRand getDoubleWithMin: 0 withMax: 1]<0.5)
    {
      mainParent=self;
      otherParent=partner;
    }
    else
    {
      mainParent=partner;
      otherParent=self;
    }
    //combine parental genetic material with parameterised crossover operator
    for (l1=0; l1<[myWorld getStrategyLength]; l1++)
    {
      if ([uniformDblRand getDoubleWithMin: 0 withMax: 1]<[myWorld getCrossProbability])
      {
        //allele taken from other parent
        [newAgent setStrategyAtOffset: l1 value: (int) [[otherParent getStrategy] atOffset: l1]];
      }
      else
      {
        //allele taken from main parent
        [newAgent setStrategyAtOffset: l1 value: (int) [[mainParent getStrategy] atOffset: l1]];
      }
      //apply mutation operator
      if ([uniformDblRand getDoubleWithMin: 0 withMax: 1]<[myWorld getMutationRate])
      {
        //allele mutates
        [newAgent setStrategyAtOffset: l1 value: (int) 1-(int)[[newAgent getStrategy] atOffset: l1]];
      }
      //update frequency of allele in gene pool
      if ((int)[[newAgent getStrategy] atOffset: l1]==1)
      {
        [myWorld increaseAlleleFrequency: l1];
      }
    }

    //update frequency of strategy in population
    stratNum=(int)[[newAgent getStrategy] atOffset: 2];
    stratNum+=(int)[[newAgent getStrategy] atOffset: 3]*2;
    stratNum+=(int)[[newAgent getStrategy] atOffset: 0]*4;
    [myWorld increaseStrategyFrequency: stratNum];

  #ifdef global_mating_pool
    [[[myWorld getObjectAtX: newX Y: newY] getCollection] addLast: newAgent]; //add offspring agent to random cell list
  #else
    [[matingList getCollection] addLast: newAgent]; //add offspring agent to local cell list
  #endif
    [myWorld addAgent: newAgent]; //add offspring agent to global agent list
  }

  return self;
}

-(float)getFitness
{
  //return agent's fitness level
  return fitness;
}

-(id)getHistory
{
  //return agent's interaction history
  return history;
}

-(id)getStrategy
{
  //return agent's strategy chromosome
  return strategy;
}

-(void) setStrategyAtOffset: (int) offset value: (int) value
{
  //set allele at locus on agent's strategy chromosome
  [strategy atOffset: offset put: (id)value];
}

-(BOOL)isAlive
{
  //return the agent's status (YES=alive, NO=dead)
  return alive;
}

-(BOOL)hasPlayed
{
  //return the agent's play status (YES=has played this round, NO=has not played)
  return played;
}

-(BOOL)hasMated
{
  //return the agent's mating status (YES=has mated this round, NO=has not mated)
  return mated;
}

-(void)setMated
{
  //set the agent's status to indicate it has mated this round
  mated=YES;
}

-(int)getCurrentAction
{
  //return the agent's action (C or D) for this round
  return currentAction;
}

-(void)setCurrentAction: (int) action
{
  //set the agent's current action (C or D) for this round
  currentAction=action;
}

-(void)updateFitness: (int) payoff
{
  //update the agent's fitness level according to payoff received from PD interaction
  totalInteractions++;
  totalFitness=totalFitness+payoff;
  fitness=totalFitness/totalInteractions;
  played=YES; //set the agent's status to indicate it has played this round
}

-(void)increaseEnergy: (int) payoff
{
  //increase the agent's energy level
  energy=energy+payoff;
}

-(void)drop
{
  //destroy the agent object
  [strategy drop]; //drop the agent's strategy chromosome
  //delete the contents of the agent's interaction history
  [[history getCollection] deleteAll];
  [[history getCollection] drop];
  [history drop]; //drop the agent's interaction history

  [super drop];
}

@end
