package space;

import java.awt.Dimension;

import collection.DoubleMatrix;
import collection.BaseMatrix;

public class Diffuse2D implements Discrete2DSpace {

  public static final int MAX = 0x7FFF;
  private final int RATE_MULTIPLIER = 1000;

  private int diffusionConstantMultiplier;
  private int evapRateMultiplier;
  protected int matrix[][];
  private int xSize, ySize;

  private int x, prevX, nextX;
  private int y, prevY, nextY;

  public Diffuse2D(int xSize, int ySize) {
    this (1.0, 1.0, xSize, ySize);
  }

  public Diffuse2D (double diffusionConstant, double evaporationRate,
                    int xSize, int ySize)
  {
    diffusionConstantMultiplier =
      (int) (diffusionConstant * RATE_MULTIPLIER);
    evapRateMultiplier = (int) (evaporationRate * RATE_MULTIPLIER);
    this.xSize = xSize;
    this.ySize = ySize;
    matrix = new int[ySize][xSize];
  }
  
  private void computeVal () {
    int val = matrix[y][x];
    int sum;
    
    sum = matrix[prevY][prevX];
    sum += 4 * matrix[prevY][x];
    sum += matrix[prevY][nextX];
    sum += 4 * matrix[y][prevX];
    sum += 4 * matrix[y][nextX];
    sum += matrix[nextY][prevX];
    sum += 4 * matrix[nextY][x];
    sum += matrix[nextY][nextX];
    sum -= 20 * val;
    
    int d = val + (((sum / 20)
                    * diffusionConstantMultiplier) / RATE_MULTIPLIER);
    d *= evapRateMultiplier;
    d /= RATE_MULTIPLIER;

    if (d < 0)
      val = 0;
    else
      val = (d > MAX) ? MAX : d;
    matrix[y][x] = val;
  }

  private void computeRow () {
    int endX = xSize - 1;
    prevX = endX;
    x = 0;

    while (x < endX) {
      nextX = x + 1;

      computeVal ();
      
      prevX = x; 
      x = nextX;
    }
    nextX = 0;
    computeVal ();
  }
  
  public void diffuse() {
    int endY = ySize - 1;

    prevY = endY;
    y = 0;
    while (y < endY) {
      nextY = y + 1;

      computeRow ();
      prevY = y;
      y = nextY;
    }
    nextY = 0;
    computeRow ();
  }

  public int getSizeX () {
    return xSize;
  }

  public int getSizeY () {
    return ySize;
  }

  public Dimension getSize() {
    return new Dimension(xSize, ySize);
  }

  public Object getObjectAt(int x, int y) {
    return new Long ((long) matrix[y][x]);
  }

  public double getValueAt(int x, int y) {
    return matrix[y][x];
  }

  public void putObjectAt(int x, int y, Object object) {
    if (!(object instanceof Number)) {
      throw new IllegalArgumentException ("object must be a Number");
    }
    Number number = (Number)object;
    matrix[y][x] = (int) number.doubleValue ();
  }

  public void putValueAt(int x, int y, double value) {
    matrix[y][x] = (int) value;
  }

  protected int xnorm(int x) {
    while (x < 0) x += xSize;
    return x % xSize;
  }

  protected int ynorm(int y) {
    while (y < 0) y += ySize;
    return y % ySize;
  }
  
  public BaseMatrix getMatrix () {
    double newMatrix[][] = new double[xSize][ySize];
    int x, y;

    for (x = 0; x < xSize; x++)
      for (y = 0; y < ySize; y++)
        newMatrix[x][y] = matrix[y][x];
    return new DoubleMatrix (newMatrix, xSize, ySize);
  }
}

