/*$$
 * packages uchicago.src.*
 * Copyright (c) 1999, Trustees of the University of Chicago
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with 
 * or without modification, are permitted provided that the following 
 * conditions are met:
 *
 *	 Redistributions of source code must retain the above copyright notice,
 *	 this list of conditions and the following disclaimer.
 *
 *	 Redistributions in binary form must reproduce the above copyright notice,
 *	 this list of conditions and the following disclaimer in the documentation
 *	 and/or other materials provided with the distribution.
 *
 *	 Neither the name of the University of Chicago nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE TRUSTEES OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Nick Collier
 * nick@src.uchicago.edu
 *
 * packages cern.jet.random.*
 * Copyright (c) 1999 CERN - European Laboratory for Particle
 * Physics. Permission to use, copy, modify, distribute and sell this
 * software and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice appear in
 * supporting documentation. CERN makes no representations about the 
 * suitability of this software for any purpose. It is provided "as is" 
 * without expressed or implied warranty. 
 *
 * Wolfgang Hoschek
 * wolfgang.hoschek@cern.ch
 *$$*/
package gui;

import java.util.ArrayList;
import java.util.ListIterator;

import space.Discrete2DSpace;
import collection.BaseMatrix;

/**
 * Displays Discrete2DSpaces and the objects contained within them. Implements
 * probeable so that the objects within the Discrete2DSpace can be probed. All
 * the objects within the space are expected to have implemented Drawable.
 * Objects are displayed on a grid, a Discrete2DSpace.
 *
 * @author Nick Collier
 * @version $Revision: 1.1.1.1 $ $Date: 2000/03/10 03:18:18 $
 */

public class Object2DDisplay extends Display2D implements Probeable {

  protected ArrayList objList = null;
  private boolean view = true;

  /**
   * Creates an Object2DDisplay for displaying the specified Discrete2DSpace.
   *
   * @grid the space to display
   */
  public Object2DDisplay(Discrete2DSpace grid) {
    super(grid);
  }

  /**
   * Sets the list of objects to display. If a space is sparsely populated then
   * rather than iterating over the entire space looking for objects to draw,
   * this Object2DDisplay can iterate only over the specified list and draw
   * those objects. This list is expected to contain objects implementing the
   * drawable interface.
   *
   * @param objectList the list of objects to draw
   */
  public void setObjectList(ArrayList objectList) {
    objList = objectList;
  }

  // Probeable interface
  /**
   * Gets the object at the specified screen coordinates for probing.
   *
   * @param x the x coordinate
   * @param y the y coordinate
   * @return the list of objects at x,y
   */
  public ArrayList getObjectsAt(int x, int y) {

    if (y != 0)
      y /= DisplayConstants.CELL_HEIGHT;
    if (x != 0)
      x /= DisplayConstants.CELL_WIDTH;

    ArrayList list = new ArrayList();
    if (x > grid.getSizeX() || x < 0 || y > grid.getSizeY() || y < 0) {
      return list;
    } else {
      list.add(grid.getObjectAt(x, y));
    }

    return list;
  }

  // Displayable interface
  /**
   * Draws the contained space, either by iterating over the entire space
   * and calling draw(SimGraphics g) on the Drawables contained therein, or
   * by iterating through a list of Drawables and calling draw(SimGraphics g)
   * on them. This method should never by called directly by a user.
   *
   * @param g the graphics context on which to draw
   */
  public void drawDisplay(SimGraphics g) {
    if (!view) {
      return;
    }
    int xTrans = DisplayConstants.CELL_WIDTH;
    int yTrans = DisplayConstants.CELL_HEIGHT;
    // without synchronization get lots of concurrent modification errors.
    Drawable d = null;
    if (objList == null) {
      BaseMatrix matrix = grid.getMatrix();
      synchronized (matrix) {
        for (int i = 0; i < grid.getSizeX(); i++) {
          for (int j = 0; j < grid.getSizeY(); j++) {
            d = (Drawable)matrix.get(i, j);
            if (d != null) {
              // make the translation and set the coordinates
              g.setDrawingCoordinates(i * xTrans, j * yTrans, 0);
              d.draw(g);
            }
          }
        }
      }
    } else {
      ArrayList t;
      synchronized(objList) {
        t = (ArrayList)objList.clone();
      }
      ListIterator li = t.listIterator();
      while (li.hasNext()) {
        d = (Drawable)li.next();
        g.setDrawingCoordinates(d.getX() * xTrans, d.getY() * yTrans, 0);
        d.draw(g);
      }
    }
  }

  /**
   * Gets a list of the DisplayInfo object associated with this Object2DDisplay.
   */
  public ArrayList getDisplayableInfo() {
    ArrayList list = new ArrayList();
    list.add(new DisplayInfo("", TOGGLE_VIEW, this));
    return list;
  }

  
  /**
   * Invoked when a viewEvent for this display is fired by the
   * DisplaySurface.
   */
  public void viewEventPerformed(ViewEvent evt) {
    view = evt.showView();
  }
}
