/****************************************************************
*
*                  Container World Project
*
*           (c) 2002-2004 Imperial College London
*
* $RCSfile: SequenceAverager.java,v $
* $Revision: 1.1 $
* $Date: 2003/03/20 16:00:03 $
* $Author: marsha02 $
* Log: See end of file
*
****************************************************************/

package cosmic.util;

import java.lang.Integer;
import java.lang.IllegalArgumentException;
import java.util.ListIterator;
import java.util.LinkedList;

/**
 * A class for efficiently averaging over a set number of the most recent values in a continuous sequence.
 * Author: James.A.Marshall@imperial.ac.uk
 *
 */

public class SequenceAverager
{
	/** The total of the full sequence */
	private double mFullSequenceTotal;
	/** The length of the full sequence */
	private int mFullSequenceLength;
	/** The sequence to be averaged over */
	private LinkedList mAverageSequence;
	/** The length of the sequence over which averages should be calculated */
	private int mAverageSequenceLength;
	/** The history of the sequence prior to the sequence currently being averaged over */
	private LinkedList mSequenceHistory;
	/** The number of values added since the last time getAverage() was called */
	private int mNumValuesAddedSinceLastGetAverage;
	/** The total calculated before averaging when getAverage() was last called */
	private double mLastCalculatedTotal;

	/**
	 * Adds a value to the sequence
	 *
	 * @param integer value to be added to the sequence
	 *
	 * @return void
	 *
	 */
	public void addToSequence(int value)
	{
		Integer integerValue = new Integer(value);

		mFullSequenceTotal += value;
		mFullSequenceLength++;
		mAverageSequence.addLast(integerValue);
		mNumValuesAddedSinceLastGetAverage++;
	}

	/**
	 * Adds a value to the sequence
	 *
	 * @param double value to be added to the sequence
	 *
	 * @return void
	 *
	 */
	public void addToSequence(double value)
	{
		Double doubleValue = new Double(value);

		mFullSequenceTotal += value;
		mFullSequenceLength++;
		mAverageSequence.addLast(doubleValue);
		mNumValuesAddedSinceLastGetAverage++;
	}

	/**
	 * Efficiently gets the average over the specified last number of values (If the sequence is not long enough the average over the entire available sequence is returned)
	 *
	 * @return Average over set sequence
	 *
	 */
	public double getAverage()
	{
		Integer value;
		ListIterator i;
		int total;

		// the efficiency of this algorithm is contingent on efficient creation of a listIterator referencing the tail of the list
		i = mAverageSequence.listIterator(mAverageSequence.size());
		while (mNumValuesAddedSinceLastGetAverage > 0)
		{
			value = (Integer) i.previous();
			mLastCalculatedTotal += value.intValue();
			mNumValuesAddedSinceLastGetAverage--;
		}
		while (mAverageSequence.size() > mAverageSequenceLength)
		{
			value = (Integer) mAverageSequence.removeFirst();
			mLastCalculatedTotal -= value.intValue();
			mSequenceHistory.addLast(value);
		}

		if (mAverageSequence.size() < mAverageSequenceLength)
		{
			return (double) mLastCalculatedTotal / mAverageSequence.size();
		}
		else
		{
			return (double) mLastCalculatedTotal / mAverageSequenceLength;
		}
	}

	/**
     * Efficiently gets the average over the entire sequence
     *
     * @return The average over the entire sequence
     *
     */
	public double getEntireSequenceAverage()
	{
		return (double) mFullSequenceTotal / mFullSequenceLength;
	}

	/**
     * Gets the average over the specified sequence (this method involves iteration over the entire list and is consequently inefficient)
     *
     * @param sequenceStart (0 <= sequenceStart <= sequenceEnd)
     * @param sequenceEnd (sequenceStart <= sequenceEnd < getSequenceLength())
     *
     * @return Average over specified sequence
     *
     */
	public double getAverageOverSequence(int sequenceStart, int sequenceEnd)
	{
		if (sequenceStart < 0)
		{
			throw new IllegalArgumentException("Attempt to construct SequenceAverager with sequenceStart out of range (sequenceStart = " + sequenceStart + ")");
		}
		if (sequenceStart > sequenceEnd)
		{
			throw new IllegalArgumentException("Attempt to construct SequenceAverager with sequenceStart out of range (sequenceStart = " + sequenceStart + ")");
		}
		if (sequenceEnd >= mAverageSequence.size() + mSequenceHistory.size())
		{
			throw new IllegalArgumentException("Attempt to construct SequenceAverager with sequenceEnd out of range (sequenceEnd = " + sequenceEnd + ")");
		}

		Integer value;
		ListIterator i;
		int l, total = 0, valuesAveraged = 0;

		if (sequenceStart < mSequenceHistory.size())
		{
			i = mSequenceHistory.listIterator(sequenceStart);
			if (sequenceEnd < mSequenceHistory.size())
			{
				for (l = sequenceStart; l <= sequenceEnd; l++)
				{
					value = (Integer) i.next();
					total += value.intValue();
					valuesAveraged++;
				}
			}
			else
			{
				while (i.hasNext())
				{
					value = (Integer) i.next();
					total += value.intValue();
					valuesAveraged++;
				}
				i = mAverageSequence.listIterator();
				while (valuesAveraged < (sequenceEnd - sequenceStart) + 1)
				{
					value = (Integer) i.next();
					total += value.intValue();
					valuesAveraged++;
				}
			}
		}
		else
		{
			i = mAverageSequence.listIterator(sequenceStart - mSequenceHistory.size());
			for (l = sequenceStart; l <= sequenceEnd; l++)
			{
				value = (Integer) i.next();
				total += value.intValue();
				valuesAveraged++;
			}
		}

		return (double) total / valuesAveraged;
	}

	/**
     * Gets the length of the entire sequence
     *
     * @return Length of the entire sequence
     *
     */
	public int getEntireSequenceLength()
	{
		return mFullSequenceLength;
	}

	/**
     * Gets the total of the entire sequence
     *
     * @return Total of the entire sequence
     *
     */
	public double getEntireSequenceTotal()
	{
		return mFullSequenceTotal;
	}

	/**
     * Gets the length of the sequence
     *
     * @return Sequence length
     *
     */
	public int getSequenceLength()
	{
		return mAverageSequence.size();
	}

	/**
	 * SequenceAverager constructor
	 *
	 * @param Length of sequence over which averages should be calculated (> 0)
	 *
	 */
	public SequenceAverager(int sequenceLength)
	{
		if (sequenceLength <= 0)
		{
			throw new IllegalArgumentException("Attempt to construct SequenceAverager object with sequenceLength out of range (sequenceLength = " + sequenceLength + ")");
		}

		mFullSequenceTotal = 0;
		mFullSequenceLength = 0;
		mAverageSequenceLength = sequenceLength;
		mAverageSequence = new LinkedList();
		mSequenceHistory = new LinkedList();
		mNumValuesAddedSinceLastGetAverage = 0;
		mLastCalculatedTotal = 0;
	}
}

/****************************************************************
*
*                              File log
*
* $Log: SequenceAverager.java,v $
* Revision 1.1  2003/03/20 16:00:03  marsha02
*
* Revision 1.4  2002/11/27 15:10:21  marsha02
* James Marshall: undid changes made in previous commit
*
* Revision 1.2  2002/11/11 11:50:06  marsha02
* James Marshall: improved efficiency of getting average over entire sequence, and added methods to get total and length of entire sequence
*
* Revision 1.1  2002/09/03 12:55:08  marsha02
* James Marshall: new class for averaging over sequences
*
*
*
****************************************************************/