/****************************************************************
*
*                  Container World Project
*
*           (c) 2002-2004 Imperial College London
*
* $RCSfile: Adjudicator.java,v $
* $Revision: 1.1 $
* $Date: 2003/03/20 16:00:03 $
* $Author: marsha02 $
* Log: See end of file
*
****************************************************************/

package cosmic.util;

import cosmic.util.ImmutableIterator;
import swarm.objectbase.SwarmObject;
import swarm.objectbase.ProbeMap;
import swarm.objectbase.MessageProbe;
import swarm.random.UniformIntegerDist;
import swarm.random.UniformIntegerDistImpl;
import swarm.Globals;
import java.util.LinkedList;
import java.util.Iterator;

/**
 * A class for choosing the 'best' among several candidates according to some user-supplied criterion, with ties resolved randomly.
 * Author: James.A.Marshall@imperial.ac.uk
 *
 */

public class Adjudicator
{
	/** The class of candidates to be chosen between */
	private Class mClass;
	/** The message probe used to determine the winner */
	private MessageProbe mMessageProbe;
	/** The name of the message probe used to determine the winner */
	private String mMessageProbeString;
	/** The flag that indicates if the winner is the lowest or highest value (true = lowest, false = highest) */
	private boolean mLowestWins;
	/** The best int value found so far */
	private int mBestInt;
	/** The best double value found so far */
	private double mBestDouble;
	/** The list of candidates to be chosen between */
	private LinkedList mCandidates;
	/** The list of candidates that are tied */
	private LinkedList mTiedCandidates;
	/** The winner of the adjudication */
	private Object mWinner;
	/** The flag that indicates if the instantiation is useable */
	private boolean mUseable;
	/** Instances of the Adjudicator class */
	private static LinkedList msInstances;
	/** Unused instances of the Adjudicator class */
	private static LinkedList msUnusedInstances;
	/** Random number generator for all instances of adjudicator */
	private static UniformIntegerDist msDistribution;
	/** Constant to denote dropping no candidates when releasing Adjudicator instance */
	public static final int dropNone = 0;
	/** Constant to denote dropping losing candidates when releasing Adjudicator instance */
	public static final int dropLosers = 1;
	/** Constant to denote dropping all candidates when releasing Adjudicator instance */
	public static final int dropAll = 2;

	static
	{
		msInstances = new LinkedList();
		msUnusedInstances = new LinkedList();
		msDistribution = new UniformIntegerDistImpl(Globals.env.globalZone);
	}

	/**
     * Adds a candidate for selection
     *
     * @param candidate (!= null)
     *
     * @return void
     *
     * @throws RuntimeException if instance is not useable
     * @throws RuntimeException if called after getWinner
     * @throws RuntimeException if message probe specified in constructor has unhandled return type (only int and double are valid)
     * @throws RuntimeException if candidate does not respond to message specified in constructor
     * @throws RuntimeException if called without first calling newAdjudication
     *
     */
	public void addCandidate(Object candidate)
	{
		if (mUseable == false)
		{
			throw new RuntimeException("instance of Adjudicator is not useable");
		}
		if (candidate == null)
		{
			throw new IllegalArgumentException("addCandidate called with null candidate");
		}
		if (mClass != null && candidate.getClass() != mClass)
		{
			throw new IllegalArgumentException("addCandidate called with candidate of different class to already added candidates");
		}
		if (mMessageProbeString == null)
		{
			throw new RuntimeException("addCandidate called without first calling newAdjudication");
		}
		if (mWinner != null)
		{
			throw new RuntimeException("addCandidate cannot be called after getWinner");
		}

		ProbeMap probeMap;
		String string;
		int intValue;
		double doubleValue;

		if (mClass == null)
		{
			mClass = candidate.getClass();
			mMessageProbe = Globals.env.getProbeLibrary().getProbeForMessage$inClass(mMessageProbeString, mClass);
			if (mMessageProbe == null)
			{
				throw new RuntimeException("candidate does not respond to message");
			}
		}
		if (mLowestWins)
		{
			// the lowest value wins
			if (mMessageProbe.getProbedType().equals("i8@0:4"))
			{
				// return type is int
				if (mCandidates.size() == 0)
				{
					// this is the first candidate, so the best value is set to this candidate's value
					mBestInt = mMessageProbe.longDynamicCallOn(candidate);
				}
				else
				{
					intValue = mMessageProbe.longDynamicCallOn(candidate);
					if (intValue < mBestInt)
					{
						// this candidate's value is better than the best added candidate so far
						mBestInt = intValue;
					}
				}
			}
			else if (mMessageProbe.getProbedType().equals("d8@0:4"))
			{
				// return type is double
				if (mCandidates.size() == 0)
				{
					// this is the first candidate, so the best value is set to this candidate's value
					mBestDouble = mMessageProbe.doubleDynamicCallOn(candidate);
				}
				else
				{
					doubleValue = mMessageProbe.doubleDynamicCallOn(candidate);
					if (doubleValue < mBestDouble)
					{
						// this candidate's value is better than the best added candidate so far
						mBestDouble = doubleValue;
					}
				}
			}
			else
			{
				throw new RuntimeException("Message probe has unhandled return type (only int and double are valid)");
			}
		}
		else
		{
			// the highest value wins
			if (mMessageProbe.getProbedType().equals("i8@0:4"))
			{
				// return type is int
				if (mCandidates.size() == 0)
				{
					// this is the first candidate, so the best value is set to this candidate's value
					mBestInt = mMessageProbe.longDynamicCallOn(candidate);
				}
				else
				{
					intValue = mMessageProbe.longDynamicCallOn(candidate);
					if (intValue > mBestInt)
					{
						// this candidate's value is better than the best added candidate so far
						mBestInt = intValue;
					}
				}
			}
			else if (mMessageProbe.getProbedType().equals("d8@0:4"))
			{
				// return type is double
				if (mCandidates.size() == 0)
				{
					// this is the first candidate, so the best value is set to this candidate's value
					mBestDouble = mMessageProbe.doubleDynamicCallOn(candidate);
				}
				else
				{
					doubleValue = mMessageProbe.doubleDynamicCallOn(candidate);
					if (doubleValue > mBestDouble)
					{
						// this candidate's value is better than the best added candidate so far
						mBestDouble = doubleValue;
					}
				}
			}
			else
			{
				throw new RuntimeException("Message probe has unhandled return type (only int and double are valid)");
			}
		}
		mCandidates.addLast(candidate);
	}

	/**
     * Gets an iterator over the candidates in the Adjudicator
     *
     * @return Iterator over candidates in Adjudicator
     *
     * @throws RuntimeException if instance is not useable
     * @throws RuntimeException if called before getWinner
     *
     */
	public ImmutableIterator getIteratorOnCandidates()
	{
		if (mUseable == false)
		{
			throw new RuntimeException("instance of Adjudicator is not useable");
		}
		if (mCandidates.size() > 0 && mWinner == null)
		{
			throw new RuntimeException("getIteratorOnCandidates called before getWinner");
		}

		return new ImmutableIterator(mCandidates.iterator());
	}

	/**
     * Gets the winner of the adjudication (multiple calls will all return the same winner)
     *
     * @return The winner of the adjudication
     *
     * @throws RuntimeException if instance is not useable
     * @throws RuntimeException if called without first calling newAdjudication
     *
     */
	public Object getWinner()
	{
		if (mUseable == false)
		{
			throw new RuntimeException("instance of Adjudicator is not useable");
		}
		if (mMessageProbeString == null)
		{
			throw new RuntimeException("getWinner called without first calling newAdjudication");
		}
		UniformIntegerDist distribution;
		Object candidate;
		Iterator i;
		int winner;

		if (mCandidates.size() == 0)
		{
			return null;
		}
		if (mWinner == null)
		{
			i = mCandidates.iterator();
			while (i.hasNext())
			{
				candidate = i.next();
				if (mMessageProbe.getProbedType().equals("i8@0:4"))
				{
					// return type is int
					if (mMessageProbe.longDynamicCallOn(candidate) == mBestInt)
					{
						mTiedCandidates.addLast(candidate);
					}
				}
				else
				{
					// return type is double
					if (mMessageProbe.doubleDynamicCallOn(candidate) == mBestDouble)
					{
						mTiedCandidates.addLast(candidate);
					}
				}
			}

			// randomly choose winner from among tied candidates
			winner = msDistribution.getIntegerWithMin$withMax(0, mTiedCandidates.size() - 1);
			mWinner = mTiedCandidates.get(winner);
		}

		return mWinner;
	}

	/**
	 * Releases the Adjudicator instantiation for re-use
	 *
	 * @param dropPolicy (dropNone || dropLosers || dropAll)
	 *
	 * @return void
	 *
	 * @throws RuntimeException if instance is not useable
	 *
	 */
	public void releaseInstance(int dropPolicy)
	{
		if (mUseable == false)
		{
			throw new RuntimeException("instance of Adjudicator is not useable");
		}
		switch (dropPolicy)
		{
			case dropNone:
			case dropLosers:
			case dropAll:
				break;
			default:
				throw new IllegalArgumentException("releaseInstance called with invalid drop policy (dropPolicy = " + dropPolicy + ")");
		}
		SwarmObject swarmObject;
		Iterator i;

		if (dropPolicy != dropNone)
		{
			i = mCandidates.iterator();
			while (i.hasNext())
			{
				swarmObject = (SwarmObject) i.next();
				if (dropPolicy == dropAll || swarmObject != mWinner)
				{
					swarmObject.drop();
				}
			}
		}
		msUnusedInstances.addLast(this);
		mCandidates.clear();
		mTiedCandidates.clear();
		mMessageProbe = null;
		mClass = null;
		mBestInt = 0;
		mBestDouble = 0;
		mWinner = null;
		mUseable = false;
	}

	/**
     * Gets an unused instance of the class Adjudicator for adjudication according to the supplied criteria
     *
     * @param messageProbeString - the name of the function to be used in evaluating the winner
     * @param lowestWins - true indicates the lowest value wins, false indicates the highest value wins
     *
     * @return Unused instance of Adjudicator
     *
     */
	public static Adjudicator getFreeInstance(String messageProbeString, boolean lowestWins)
	{
		Adjudicator instance;
		if (msUnusedInstances.size() == 0)
		{
			instance = new Adjudicator();
			msInstances.addLast(instance);
		}
		else
		{
			instance = (Adjudicator) msUnusedInstances.removeFirst();
		}
		instance.mMessageProbeString = messageProbeString;
		instance.mLowestWins = lowestWins;
		instance.mUseable = true;

		return instance;
	}

	/**
     * Private constructor for the class Adjudicator
     *
     */
	private Adjudicator()
	{
		mCandidates = new LinkedList();
		mTiedCandidates = new LinkedList();
		mMessageProbe = null;
		mClass = null;
		mBestInt = 0;
		mBestDouble = 0;
		mWinner = null;
		mUseable = false;
	}
}

/****************************************************************
*
*                              File log
*
* $Log: Adjudicator.java,v $
* Revision 1.1  2003/03/20 16:00:03  marsha02
* James Marshall: new re-organised ITM version approaching readiness for release to partners, with various small updates
*
* Revision 1.11  2003/03/18 14:25:20  marsha02
* James Marshall: updated doc comments
*
* Revision 1.10  2003/01/23 13:26:51  marsha02
* James Marshall: gave Adjudicator class a single static random number generator to fix memory leak
*
* Revision 1.9  2002/11/27 14:54:51  marsha02
* James Marshall: undid changes made in previous commit
*
* Revision 1.7  2002/11/15 20:28:22  pksr
* proshun: Wrote redistributeEmptyContainers in Shipline.java + many smaller additions to implement redistribution of empty containers
*
* Revision 1.3  2002/09/23 13:06:06  marsha02
* James Marshall: temporarily hard-coded to treat candidates as ShippingContract objects and choose the lowest value returned by getTotalShippingCharge, due to Swarm Probe memory leak problems. Changed class to use Object Pool design pattern.
*
* Revision 1.2  2002/09/12 16:46:00  marsha02
* James Marshall: removed need for candidates to inherit from SwarmObject, changed getWinner to return null if no candidates have been added, added exception if added candidate does not respond to message specified in constructor
*
* Revision 1.1  2002/09/12 14:09:18  marsha02
* James Marshall: renamed from TieBreaker and extended to choose the "best" from several candidates according to some user-supplied criterion, with random breaking of ties.
*
* Revision 1.1  2002/09/11 11:24:05  marsha02
* James Marshall: new class for randomly breaking ties
*
*
*
*
****************************************************************/