// Copyright Imperial College London 2002. Freely distributable under the GNU General Public Licence

package buffon;

import java.lang.Integer;
import java.lang.IllegalArgumentException;
import java.util.ListIterator;
import java.util.LinkedList;

/**
 * A class for efficiently averaging over a set number of the most recent values in a continuous sequence of integers
 *
*/

class SequenceAverager
{
	/** The total of the full sequence */
	private int mFullSequenceTotal;
	/** The length of the full sequence */
	private int mFullSequenceLength;
	/** The sequence to be averaged over */
	private LinkedList mAverageSequence;
	/** The length of the sequence over which averages should be calculated */
	private int mAverageSequenceLength;
	/** The history of the sequence prior to the sequence currently being averaged over */
	private LinkedList mSequenceHistory;
	/** The number of values added since the last time getAverage() was called */
	private int mNumValuesAddedSinceLastGetAverage;
	/** The total calculated before averaging when getAverage() was last called */
	private int mLastCalculatedTotal;

	/**
	 * Adds a value to the sequence
	 *
	 * @param value to be added to the sequence
	 *
	 * @return void
	 *
	 */
	public void addToSequence(int value)
	{
		Integer integerValue = new Integer(value);

		mFullSequenceTotal += value;
		mFullSequenceLength++;
		mAverageSequence.addLast(integerValue);
		mNumValuesAddedSinceLastGetAverage++;
	}

	/**
	 * Efficiently gets the average over the specified last number of values (If the sequence is not long enough the average over the entire available sequence is returned)
	 *
	 * @return Average over set sequence
	 *
	 */
	public double getAverage()
	{
		Integer value;
		ListIterator i;
		int total;

		// the efficiency of this algorithm is contingent on efficient creation of a listIterator referencing the tail of the list
		i = mAverageSequence.listIterator(mAverageSequence.size());
		while (mNumValuesAddedSinceLastGetAverage > 0)
		{
			value = (Integer) i.previous();
			mLastCalculatedTotal += value.intValue();
			mNumValuesAddedSinceLastGetAverage--;
		}
		while (mAverageSequence.size() > mAverageSequenceLength)
		{
			value = (Integer) mAverageSequence.removeFirst();
			mLastCalculatedTotal -= value.intValue();
			mSequenceHistory.addLast(value);
		}

		if (mAverageSequence.size() < mAverageSequenceLength)
		{
			return (double) mLastCalculatedTotal / mAverageSequence.size();
		}
		else
		{
			return (double) mLastCalculatedTotal / mAverageSequenceLength;
		}
	}

	/**
     * Efficiently gets the average over the entire sequence
     *
     * @return The average over the entire sequence
     *
     */
	public double getEntireSequenceAverage()
	{
		return (double) mFullSequenceTotal / mFullSequenceLength;
	}

	/**
     * Gets the average over the specified sequence (this method involves iteration over the entire list and is consequently inefficient)
     *
     * @param sequenceStart (0 <= sequenceStart <= sequenceEnd)
     * @param sequenceEnd (sequenceStart <= sequenceEnd < getSequenceLength())
     *
     * @return Average over specified sequence
     *
     */
	public double getAverageOverSequence(int sequenceStart, int sequenceEnd)
	{
		if (sequenceStart < 0)
		{
			throw new IllegalArgumentException("Attempt to construct SequenceAverager with sequenceStart out of range (sequenceStart = " + sequenceStart + ")");
		}
		if (sequenceStart > sequenceEnd)
		{
			throw new IllegalArgumentException("Attempt to construct SequenceAverager with sequenceStart out of range (sequenceStart = " + sequenceStart + ")");
		}
		if (sequenceEnd >= mAverageSequence.size() + mSequenceHistory.size())
		{
			throw new IllegalArgumentException("Attempt to construct SequenceAverager with sequenceEnd out of range (sequenceEnd = " + sequenceEnd + ")");
		}

		Integer value;
		ListIterator i;
		int l, total = 0, valuesAveraged = 0;

		if (sequenceStart < mSequenceHistory.size())
		{
			i = mSequenceHistory.listIterator(sequenceStart);
			if (sequenceEnd < mSequenceHistory.size())
			{
				for (l = sequenceStart; l <= sequenceEnd; l++)
				{
					value = (Integer) i.next();
					total += value.intValue();
					valuesAveraged++;
				}
			}
			else
			{
				while (i.hasNext())
				{
					value = (Integer) i.next();
					total += value.intValue();
					valuesAveraged++;
				}
				i = mAverageSequence.listIterator();
				while (valuesAveraged < (sequenceEnd - sequenceStart) + 1)
				{
					value = (Integer) i.next();
					total += value.intValue();
					valuesAveraged++;
				}
			}
		}
		else
		{
			i = mAverageSequence.listIterator(sequenceStart - mSequenceHistory.size());
			for (l = sequenceStart; l <= sequenceEnd; l++)
			{
				value = (Integer) i.next();
				total += value.intValue();
				valuesAveraged++;
			}
		}

		return (double) total / valuesAveraged;
	}

	/**
     * Gets the length of the entire sequence
     *
     * @return Length of the entire sequence
     *
     */
	public int getEntireSequenceLength()
	{
		return mFullSequenceLength;
	}

	/**
     * Gets the total of the entire sequence
     *
     * @return Total of the entire sequence
     *
     */
	public int getEntireSequenceTotal()
	{
		return mFullSequenceTotal;
	}

	/**
     * Gets the length of the sequence
     *
     * @return Sequence length
     *
     */
	public int getSequenceLength()
	{
		return mAverageSequence.size();
	}

	/**
	 * SequenceAverager constructor
	 *
	 * @param Length of sequence over which averages should be calculated (> 0)
	 *
	 */
	public SequenceAverager(int sequenceLength)
	{
		if (sequenceLength <= 0)
		{
			throw new IllegalArgumentException("Attempt to construct SequenceAverager object with sequenceLength out of range (sequenceLength = " + sequenceLength + ")");
		}

		mFullSequenceTotal = 0;
		mFullSequenceLength = 0;
		mAverageSequenceLength = sequenceLength;
		mAverageSequence = new LinkedList();
		mSequenceHistory = new LinkedList();
		mNumValuesAddedSinceLastGetAverage = 0;
		mLastCalculatedTotal = 0;
	}
}