/* 
 * $Id: local-vc.c,v 1.9 2004/03/18 14:03:59 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <vorbis/vorbisfile.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include "gettext.h"
#include "vcedit.h"
#include "local.h"

/*** constant definitions ****************************************************/

static const struct
{
  const char	*field_name;
  int		field_id;
  unsigned int	offset;
} mappings[] = {
  { "TITLE",			LOCAL_FIELD_TITLE,	G_STRUCT_OFFSET(LocalStream, title) },
  { "ARTIST",			LOCAL_FIELD_ARTIST,	G_STRUCT_OFFSET(LocalStream, artist) },
  { "ALBUM",			LOCAL_FIELD_ALBUM,	G_STRUCT_OFFSET(LocalStream, album) },
  { "DATE",			LOCAL_FIELD_YEAR,	G_STRUCT_OFFSET(LocalStream, year) },
  { "GENRE",			LOCAL_FIELD_GENRE,	G_STRUCT_OFFSET(LocalStream, genre) },
  { "DESCRIPTION",		LOCAL_FIELD_COMMENT,	G_STRUCT_OFFSET(LocalStream, comment) }
};

/*** function declarations ***************************************************/

static void local_vc_read_comments (LocalStream *stream,
				    const vorbis_comment *comments);
static void local_vc_write_comments (LocalStream *stream,
				     vorbis_comment *comments,
				     GSList *fields,
				     GSList *values);

static char *local_vc_build_comment (const char *name, const char *value);
static gboolean local_vc_parse_comment (const char *comment,
					char **name,
					char **value);

static gboolean local_vc_write (LocalStream *stream,
				vcedit_state *state,
				GError **err);

/*** implementation **********************************************************/

static void
local_vc_read_comments (LocalStream *stream, const vorbis_comment *comments)
{
  int i;

  g_return_if_fail(stream != NULL);
  g_return_if_fail(comments != NULL);

  for (i = 0; i < comments->comments; i++)
    {
      char *name;
      char *value;

      if (local_vc_parse_comment(comments->user_comments[i], &name, &value))
	{
	  char **str = NULL;
	  int j;

	  for (j = 0; j < G_N_ELEMENTS(mappings); j++)
	    if (! strcasecmp(name, mappings[j].field_name))
	      {
		str = (void *) stream + mappings[j].offset;
		break;
	      }
	  
	  if (str)
	    {
	      char *new = *str
		? g_strdup_printf("%s, %s", *str, value)
		: g_strdup(value);

	      g_free(*str);
	      *str = new;
	    }

	  g_free(name);
	  g_free(value);
	}
    }
}

static void
local_vc_write_comments (LocalStream *stream,
			 vorbis_comment *comments,
			 GSList *fields,
			 GSList *values)
{
  GSList *f = fields;
  GSList *v = values;
  GSList *original_comments = NULL;
  GSList *custom_fields = NULL;
  int i;
  GSList *l;

  g_return_if_fail(stream != NULL);
  g_return_if_fail(comments != NULL);

  /* keep the original comments for later on */
  for (i = 0; i < comments->comments; i++)
    original_comments = g_slist_append(original_comments, g_strdup(comments->user_comments[i]));

  vorbis_comment_clear(comments);
  vorbis_comment_init(comments);

  while (f && v)
    {
      STHandlerField *field = f->data;
      const GValue *value = v->data;
      const char *field_name;
      char **str = NULL;

      for (i = 0; i < G_N_ELEMENTS(mappings); i++)
	if (field->id == mappings[i].field_id)
	  {
	    field_name = mappings[i].field_name;
	    str = (void *) stream + mappings[i].offset;
	    break;
	  }

      if (str)
	{
	  char *comment;

	  custom_fields = g_slist_append(custom_fields, (gpointer) field_name);

	  comment = local_vc_build_comment(field_name, g_value_get_string(value));
	  vorbis_comment_add(comments, comment);
	  g_free(comment);

	  g_free(*str);
	  *str = g_value_dup_string(value);
	}

      f = f->next;
      v = v->next;
    }

  /* restore the original comments */
  for (l = original_comments; l; l = l->next)
    {
      char *name;
      char *value;

      if (local_vc_parse_comment(l->data, &name, &value))
	{
	  GSList *m;
	  gboolean found = FALSE;

	  for (m = custom_fields; m; m = m->next)
	    if (! strcasecmp(name, m->data))
	      {
		found = TRUE;
		break;
	      }

	  if (! found)
	    vorbis_comment_add(comments, l->data);

	  g_free(name);
	  g_free(value);
	}

      g_free(l->data);
    }

  g_slist_free(original_comments);
  g_slist_free(custom_fields);
}

static char *
local_vc_build_comment (const char *name, const char *value)
{
  g_return_val_if_fail(name != NULL, NULL);
  g_return_val_if_fail(value != NULL, NULL);

  return g_strconcat(name, "=", value, NULL);
}

static gboolean
local_vc_parse_comment (const char *comment, char **name, char **value)
{
  char *equal_sign;

  g_return_val_if_fail(comment != NULL, FALSE);
  g_return_val_if_fail(name != NULL, FALSE);
  g_return_val_if_fail(value != NULL, FALSE);

  equal_sign = strchr(comment, '=');
  if (equal_sign)
    {
      *name = g_strndup(comment, equal_sign - comment);
      *value = g_strdup(equal_sign + 1);

      return TRUE;
    }
  else
    return FALSE;
}

void
local_vc_read (LocalStream *stream)
{
  FILE *file;
  OggVorbis_File vf;
  int status;
  const vorbis_comment *comments;
  double total_time;

  g_return_if_fail(stream != NULL);

  /* open the file */

  file = fopen(stream->pathname, "r");
  if (! file)
    {
      st_notice(_("Local: unable to open %s: %s"), stream->pathname, g_strerror(errno));
      return;
    }

  /* ov_open() the file */

  status = ov_open(file, &vf, NULL, 0);
  if (status < 0)
    {
      const char *err;

      switch (status)
	{
	case OV_EREAD:		err = _("a read from media returned an error"); break;
	case OV_ENOTVORBIS:	err = _("bitstream is not Vorbis data"); break;
	case OV_EVERSION:	err = _("Vorbis version mismatch"); break;
	case OV_EBADHEADER:	err = _("invalid Vorbis bitstream header"); break;
	case OV_EFAULT:		err = _("internal logic fault"); break;
	default:		err = _("unknown error"); break;
	}
      
      st_notice(_("Local: unable to ov_open() %s: %s"), stream->pathname, err);
      fclose(file);

      return;
    }

  /* read the comments */

  comments = ov_comment(&vf, -1);
  local_vc_read_comments(stream, comments);

  /* read the total time */

  total_time = ov_time_total(&vf, -1);
  if (total_time != OV_EINVAL)
    stream->duration = LOCAL_DURATION(total_time);
  else
    st_notice(_("Local: unable to read duration of %s"), stream->pathname);
  
  /* close the file */
  
  ov_clear(&vf);
}

static gboolean
local_vc_write (LocalStream *stream, vcedit_state *state, GError **err)
{
  char *tmp_pathname;
  int fd;
  FILE *file;
  gboolean status = TRUE;

  g_return_val_if_fail(stream != NULL, FALSE);
  g_return_val_if_fail(state != NULL, FALSE);

  tmp_pathname = g_strconcat(stream->pathname, ".XXXXXX", NULL);
  fd = g_mkstemp(tmp_pathname);
  if (fd < 0)
    {
      g_set_error(err, 0, 0, _("unable to create a temporary file: %s"), g_strerror(errno));
      g_free(tmp_pathname);
      return FALSE;
    }

  file = fdopen(fd, "w");
  if (! file)
    {
      g_set_error(err, 0, 0, _("unable to create a temporary file: %s"), g_strerror(errno));
      status = FALSE;
      goto end;
    }

  if (vcedit_write(state, file) < 0)
    {
      g_set_error(err, 0, 0, _("unable to write temporary file: %s"), vcedit_error(state));
      status = FALSE;
      goto end;
    }

  if (g_file_test(stream->pathname, G_FILE_TEST_EXISTS))
    {
      if (unlink(stream->pathname) < 0)
	{
	  g_set_error(err, 0, 0, _("unable to unlink file: %s"), g_strerror(errno));
	  status = FALSE;
	  goto end;
	}
    }
  
  if (rename(tmp_pathname, stream->pathname) >= 0)
    {				/* ok */
      g_free(tmp_pathname);
      tmp_pathname = NULL;
    }
  else
    {
      g_set_error(err, 0, 0, _("unable to rename temporary file: %s"), g_strerror(errno));
      status = FALSE;
      goto end;
    }

 end:
  if (file)
    {
      if (fclose(file) != 0 && status)
	{
	  g_set_error(err, 0, 0, _("unable to close file: %s"), g_strerror(errno));
	  status = FALSE;
	}
    }
  else
    close(fd);			/* don't check for errors (already an exception) */

  if (tmp_pathname)
    unlink(tmp_pathname);	/* don't check for errors (already an exception) */
  
  return status;
}

gboolean
local_vc_modify (LocalStream *stream,
		 GSList *fields,
		 GSList *values,
		 GError **err)
{
  FILE *file;
  vcedit_state *state;
  gboolean status = TRUE;
  vorbis_comment *comments;

  g_return_val_if_fail(stream != NULL, FALSE);

  file = fopen(stream->pathname, "r");
  if (! file)
    {
      g_set_error(err, 0, 0, _("unable to open file: %s"), g_strerror(errno));
      return FALSE;
    }

  state = vcedit_new_state();
  if (vcedit_open(state, file) < 0)
    {
      g_set_error(err, 0, 0, _("unable to open file: %s"), vcedit_error(state));
      status = FALSE;
      goto end;
    }

  comments = vcedit_comments(state);
  local_vc_write_comments(stream, comments, fields, values);

  status = local_vc_write(stream, state, err);

 end:
  vcedit_clear(state);
  if (fclose(file) != 0 && status)
    {
      g_set_error(err, 0, 0, _("unable to close file: %s"), g_strerror(errno));
      status = FALSE;
    }

  return status;
}
